#define ORBITER_MODULE

#include "V2iWreck.h"
#include <stdio.h>
#include <stdlib.h>
#include <math.h>


// constructor
V2iWreck::V2iWreck (OBJHANDLE hObj, int fmodel): VESSEL2 (hObj, fmodel)
{
	// This gives us a chance to delay the finalization of the physics 
	// until the firts pre-step, when we will know are we landed or inflight.
	bJustDeployed = true;
	DebugFlag = 0;
}



// called when new vessel is initialized, before reading scenario file
void V2iWreck::SetClassCaps (FILEHANDLE cfg)
{
	// set common physics, same for landed and inflight
	SetSize (7.);
	SetSurfaceFrictionCoeff(100., 100.);
	SetTouchdownPoints (_V(15, 0, 0), _V(-15, -15, 0), _V(-15, 15, 0));
	SetCameraOffset(_V(0, 0, 1));

	// default state for runtime creation: start with 'to blast'!
	BlastState = TOBLAST;

	// For 'landed', default only needed for older scn compatibility.
	// It will be always set by either original V2 or scenario.
	bLanded = true;

	// For inflight phase, original V2 always assumes 1.
	// Only scenario can override it to 2.
	bInflight1 = true;
}


// tis is called tbefore the first timestep, 
// to finalize the physical and logical configutration.
void V2iWreck::SetAdditionalConfig()
{
	if (bLanded)
		SetPhysicsLanded();

	else if (bInflight1)
		SetPhysicsInflight1();

	else
		SetPhysicsInflight2();

	bJustDeployed = false;
}



// this gets called not only during initialisation,
// but also after the actual touchdown of the falling wreckage.
void V2iWreck::SetPhysicsLanded()
{
	SetEmptyMass (100000);
	SetCW (2500, 2500, 2500, 2500);

	double Radius;
	GetEquPos(Longitude, Latitude, Radius);

	VESSELSTATUS vs;
	GetStatus(vs);

	vs.status = 1;
	vs.vdata[0].x = Longitude;
	vs.vdata[0].y = Latitude;
	vs.vdata[0].z = 0;

	DefSetState(&vs);

	bLanded = true;
	bInflight1 = true;
}


// 1-st stage of breakup: hard aerodynamic break
void V2iWreck::SetPhysicsInflight1()
{
	SetEmptyMass (1000);

	SetCW (1.0, 1.0, 1.0, 1.0);
	SetPMI (_V(1., 1., 1.));
	SetRotDrag (_V(1., 1., 1.));
	SetCrossSections (_V(20.0, 20.0, 20.0));

	SetLiftCoeffFunc (0);
	SetPitchMomentScale (1e-4);
	SetBankMomentScale (1e-4);

	bLanded = false;
	bInflight1 = true;
	TimeInflight1 = oapiGetSimTime();
}


// 2-nd stage of breakup: faster freefall
void V2iWreck::SetPhysicsInflight2()
{
	SetEmptyMass (1000);

	SetCW (1.0, 1.0, 1.0, 1.0);
	SetPMI (_V(1., 1., 1.));
	SetRotDrag (_V(1., 1., 1.));
	SetCrossSections (_V(2.0, 2.0, 2.0));

	SetLiftCoeffFunc (0);
	SetPitchMomentScale (1e-4);
	SetBankMomentScale (1e-4);

	bLanded = false;
	bInflight1 = false;
}


void V2iWreck::clbkSaveState(FILEHANDLE scn)
{
	char BufferStr[256];

	// default vessel parameters
	SaveDefaultState (scn);

	// state
	oapiWriteScenario_int (scn, "LANDED", bLanded);
	if (!bLanded)
		oapiWriteScenario_int (scn, "FALLPHASE", bInflight1);

	// store analysis vars
	sprintf(BufferStr, "%.13f", Latitude*180/PI);
	oapiWriteScenario_string(scn, "LATITUDE_LAST", BufferStr);
	sprintf(BufferStr, "%.13f", Longitude*180/PI);
	oapiWriteScenario_string(scn, "LONGITUDE_LAST", BufferStr);
	oapiWriteScenario_float (scn, "MAX_ALTITUDE", MaxAltitude);
	oapiWriteScenario_float (scn, "FLIGHT_TIME", FlightTime);
	oapiWriteScenario_float (scn, "IMPACT_VELOCITY", ImpactVelocity);
	sprintf(BufferStr, "%.13f", LatitudeT);
	oapiWriteScenario_string(scn, "LATITUDE_TARGET", BufferStr);
	sprintf(BufferStr, "%.13f", LongitudeT);
	oapiWriteScenario_string(scn, "LONGITUDE_TARGET", BufferStr);
	oapiWriteScenario_float (scn, "RANGE_ACTUAL", RealRange);
	oapiWriteScenario_float (scn, "AZIMUTH_ACTUAL", RealAzimuth);
	oapiWriteScenario_float (scn, "RANGE_MISS", MissRange);
	oapiWriteScenario_float (scn, "AZIMUTH_MISS", MissAzimuth);
	oapiWriteScenario_float (scn, "LATERAL_MISS", MissLateral);
	oapiWriteScenario_float (scn, "LONGITUDAL_MISS", MissLongitudal);
	sprintf(BufferStr, "%.13f", LatitudeL*180/PI);
	oapiWriteScenario_string(scn, "LATITUDE_LAUNCH", BufferStr);
	sprintf(BufferStr, "%.13f", LongitudeL*180/PI);
	oapiWriteScenario_string(scn, "LONGITUDE_LAUNCH", BufferStr);

	// store target name
	if (strlen(TargetName) > 0)
	  oapiWriteScenario_string(scn, "TARGET", TargetName);

	// store failure string
	if (strlen(TargetName) > 0)
	  oapiWriteScenario_string(scn, "FAILURE", FailureString);

	// store GPS track points
	for (int i = 0; i < gpsCount; i++)
	{
		sprintf(TrackPoint, "%d %.13f %.13f %.13f %d", i, Longitudes[i], Latitudes[i], Elevations[i], TrackStages[i]);
		oapiWriteScenario_string(scn, "TRACK", TrackPoint);
	}
}



// called when reading the scenario file - this does not happen on runtime creation
void V2iWreck::clbkLoadStateEx (FILEHANDLE scn, void *status)
{
	// when restoring, do not blast, just start smoke
	BlastState = TOSMOKE;


	//Read the scenario file line by line
    char *line;
	while (oapiReadScenario_nextline (scn, line)) 
	{
        if (!strnicmp (line, "LANDED", 6)) 
		{
            sscanf (line+6, "%d", &bLanded);
		}
        else if (!strnicmp (line, "FALLPHASE", 9)) 
		{
            sscanf (line+9, "%d", &bInflight1);
		}
        else if (!strnicmp (line, "LATITUDE_LAST", 13)) 
		{
            sscanf (line+13, "%lf", &Latitude);
			Latitude = Latitude*PI/180;
		}
		else if (!strnicmp (line, "LONGITUDE_LAST", 14)) 
		{
            sscanf (line+14, "%lf", &Longitude);
			Longitude = Longitude*PI/180;
		}
		else if (!strnicmp (line, "MAX_ALTITUDE", 12)) 
		{
            sscanf (line+12, "%lf", &MaxAltitude);
		}
		else if (!strnicmp (line, "FLIGHT_TIME", 11)) 
		{
            sscanf (line+11, "%lf", &FlightTime);
		}
		else if (!strnicmp (line, "IMPACT_VELOCITY", 15)) 
		{
            sscanf (line+15, "%lf", &ImpactVelocity);
		}
		else if (!strnicmp (line, "LATITUDE_TARGET", 15)) 
		{
            sscanf (line+15, "%lf", &LatitudeT);
		}
		else if (!strnicmp (line, "LONGITUDE_TARGET", 16)) 
		{
            sscanf (line+16, "%lf", &LongitudeT);
		}
		else if (!strnicmp (line, "RANGE_ACTUAL", 12)) 
		{
            sscanf (line+12, "%lf", &RealRange);
		}
		else if (!strnicmp (line, "AZIMUTH_ACTUAL", 14)) 
		{
            sscanf (line+14, "%lf", &RealAzimuth);
		}
		else if (!strnicmp (line, "RANGE_MISS", 10)) 
		{
            sscanf (line+10, "%lf", &MissRange);
		}
		else if (!strnicmp (line, "AZIMUTH_MISS", 12)) 
		{
            sscanf (line+12, "%lf", &MissAzimuth);
		}
		else if (!strnicmp (line, "LATERAL_MISS", 12)) 
		{
            sscanf (line+12, "%lf", &MissLateral);
		}
		else if (!strnicmp (line, "LONGITUDAL_MISS", 15)) 
		{
            sscanf (line+15, "%lf", &MissLongitudal);
		}
		else if (!strnicmp (line, "TARGET", 6)) 
		{
            sscanf (line+6, "%s", TargetName);
		}
		else if (!strnicmp (line, "FAILURE", 7)) 
		{
            sscanf (line+7, "%s", FailureString);
		}
		else if (!strnicmp (line, "LATITUDE_LAUNCH", 15)) 
		{
            sscanf (line+15, "%lf", &LatitudeL);
			LatitudeL = LatitudeL*PI/180;
		}
		else if (!strnicmp (line, "LONGITUDE_LAUNCH", 16)) 
		{
            sscanf (line+16, "%lf", &LongitudeL);
			LongitudeL = LongitudeL*PI/180;
		}
		else if (!strnicmp (line, "TRACK", 5)) 
		{
			double lon, lat, elev;
			int idx, col;
            sscanf (line+5, "%ld %lf %lf %lf %ld", &idx, &lon, &lat, &elev, &col);

			gpsCount = idx+1;
			Longitudes[idx] = lon, 
			Latitudes[idx] = lat, 
			Elevations[idx] = elev;
			TrackStages[idx] = col;
		}
		else
		{
			// generic scenario parser
            ParseScenarioLineEx (line, status);
        }
	}
}


// Now that we know additional details, we can finalize the configuration and physics
// to differentiate between landed or flying states.
// This only needs to run once, before the very first time step.
void V2iWreck::clbkPreStep(double simt, double simdt, double mjd)
{
	if (bJustDeployed)
		SetAdditionalConfig();
}


// simple, fast, short
void V2iWreck::clbkPostStep(double simt, double simdt, double mjd)
{

	// blast control
	if (BlastState != SMOKING)
	{
		if (BlastState == TOBLAST)
			StartBlast();

		else if ((BlastState == BLASTING) && (StopBlastTime < simt))
			StopBlast();

		else if (BlastState == TOSMOKE)
			StartSmoke();
	}

	// when still falling, we will need to go through some physical and state transformations...
	if (!bLanded) 
	{
		if ((bInflight1) && ((simt - TimeInflight1) > 13 ))
			SetPhysicsInflight2();

		if (GroundContact())
			SetPhysicsLanded();
	}
}



void V2iWreck::StartBlast()
{
	// create a very quickly expanding white "fireball"
	PARTICLESTREAMSPEC V2BlastFire = {
			0, 1, 0.5, 0, 0, 2, 400, 0, PARTICLESTREAMSPEC::EMISSIVE,
			PARTICLESTREAMSPEC::LVL_FLAT, 1, 1,
			PARTICLESTREAMSPEC::ATM_FLAT, 1, 1
	};

	// start the "fireball" by running a small virtual "engine"
	BlastTank = CreatePropellantResource(1);
	BlastEngine = CreateThruster(_V(0,0, 0.5), _V(0,0,-1), 0, BlastTank, 1, 49000000);
	AddExhaustStream (BlastEngine, _V(0,0,0), &V2BlastFire);
	SetThrusterLevel(BlastEngine, 0.1);

	// change flags to stop the blast after one second
	StopBlastTime = oapiGetSimTime() + 1;
	BlastState = BLASTING;
}



void V2iWreck::StopBlast()
{
	// shut down the "virtual" engine and change flags to start smoke
	SetThrusterLevel(BlastEngine, 0);
	BlastState = TOSMOKE;
}



void V2iWreck::StartSmoke()
{
	// create a smoke cloud from the wreck
	PARTICLESTREAMSPEC V2BlastSmoke = {
			0, 1, 10, 0, 0, 2, 20, 5, PARTICLESTREAMSPEC::DIFFUSE,
			PARTICLESTREAMSPEC::LVL_PSQRT, 0, 0.5,
			PARTICLESTREAMSPEC::ATM_FLAT, 1.22, 1.25
	};

	// start the "smoking gun" by running a small virtual "engine"
	SmokeTank = CreatePropellantResource(1);
	SmokeEngine = CreateThruster(_V(0,0,0), _V(0,0,-1), 0, SmokeTank, 1, 49000000);
	AddExhaustStream (SmokeEngine, _V(0, 0, 0), &V2BlastSmoke);
	SetThrusterLevel(SmokeEngine, 0.1);

	// close the last flag, no timestep activity anymore
	BlastState = SMOKING;
}


// Keyboard interface handler, only for debug
int V2iWreck::clbkConsumeBufferedKey(DWORD key, bool down, char *kstate)
{
	if (!down) return 0; // only process keydown events

	if (KEYMOD_SHIFT (kstate)) 
	{
		return 0; // shift combinations are reserved
	}

	if ( !KEYMOD_CONTROL (kstate) && (key == OAPI_KEY_D) )
	{
		DebugSomething();
		return 0;
	}

	return 0;
}


void V2iWreck::DebugSomething()
{
//	sprintf(oapiDebugString(), "debug %d, debLat %f, Lat: %f, Lon: %f", DebugFlag, debLat, Latitude, Longitude);
}



// Vessel dll callbacks

DLLCLBK VESSEL *ovcInit (OBJHANDLE hVessel, int flightmodel)
{
	return new V2iWreck (hVessel,flightmodel);
}

DLLCLBK void ovcExit (VESSEL *vessel)
{
	if (vessel) 
		delete (V2iWreck*)vessel;
}









