#define ORBITER_MODULE

#include "V2i.h"
#include "V2iWreck.h"
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <math.h>


// produce a random double number from given value of a given span (length)
double DRand(double from, double  span)
{
	return (double)rand() / (double)RAND_MAX * span + from;
}


V2i::V2i (OBJHANDLE hObj, int fmodel): VESSEL2 (hObj, fmodel)
{
	;
}


// This V2 class initialization is called for both runtime and scenario created objects.
// Realistically, I DO NOT expect V2 object to be created at runtime.
// For scenario object,is is called BEFORE the .scn file is read!
void V2i::clbkSetClassCaps(FILEHANDLE cfg)
{
	// create dust cloud at liftoff
	PARTICLESTREAMSPEC V2Dust = 
	{
		0,			// flags, reserved
		1,			// srcsize, particle size at creation, m
		25,			// srcrate, average particle generation rate, Hz  
		5.0,		// V0, average particle emission velocity, m/s
		0,			// srcspread, emission velocity distribution randomisation
		5,			// lifetime, average particle lifetime [s]
		20,			// growthrate, particle growth rate [m/s]
		5,			// atmslowdown, deceleration rate b in atmosphere, defined as v = v0 e- bdt 
		PARTICLESTREAMSPEC::DIFFUSE,
		PARTICLESTREAMSPEC::LVL_PSQRT, 
		0, 
		0.5,
		PARTICLESTREAMSPEC::ATM_PLOG, 
		1.22, 
		1.25
	};

	// Set only generic physics here. 
	// More physics (realistic or ideal) will be added after scn file is loaded.
	SetSize (7.0);
	SetEmptyMass (4008);
	SetTouchdownPoints (_V(0,-1,-7.013), _V(-.7,.7,-7.013), _V(.7,.7,-7.013));
	SetPitchMomentScale (1e-4);
	SetBankMomentScale (1e-4);
	SetLiftCoeffFunc (0);
	SetCameraOffset(_V(0, 1.512, 0));
	SetSurfaceFrictionCoeff(.5,.5);

	// set main engine
	PROPELLANT_HANDLE MainTank = CreatePropellantResource(8797);
	MainEngine = CreateThruster(_V(0,0,-6.613),  _V(0,0,1), 315000, MainTank, 2520, 2000);
	MainTH[0] = MainEngine;
	CreateThrusterGroup(MainTH,1,THGROUP_MAIN);
	AddExhaust(MainTH[0], 6, 0.8, _V(0,0,-8), _V(0,0,-1));
	AddExhaustStream (MainTH[0], _V(0,0,-6.613), &V2Dust); 
	
	// Initial defaults and intermediates. 

	met = 0;
	metGps = 0;

	// Everything below MAY/WILL be overridden in the scenario file

	Range = 500000.;				// until fuel lasts
	ExternalLaunchDelay = 0;		// no countdown delay, go right away

	// Default option flags.
	// These may be overridden in both manual and (auto)saved scenario file.
	bAutopilot = true;
	bRealistic = true;
	Reliability = 75;  // %

	// Default state flags, initialised to "before launch" state.
	// These may be overridden in the (auto)saved scenario file.
	bRocketLaunched = false;

	bCountdown = false;
	bTimeToLaunch = false;
	bPoweredFlight = false;
	bShowingContrail = false;

	bYawFailed = false;
	bRollFailed = false;
	bPitchFailed = false;

	//
	*TargetName = NULL;
	LatitudeT = 0;
	LongitudeT = 0;

	*FailureString = NULL;
	NumFailParam = 0;
}


// updating the configuration from scn file
void V2i::clbkLoadStateEx (FILEHANDLE scn, void *status)
{
    char *line;
	int StateFlags = -1;
	bool bOverrideAutopilot = false; 
	bool bOverrideRealistic = false; 
	bool bOverrideReliability = false; 

	// read the scenario file
	while (oapiReadScenario_nextline (scn, line)) 
	{
		// --- auto-generated scenario variables
		if (!strnicmp (line, "RANGE", 5)) 
		{
            sscanf (line+5, "%lf", &Range);
		}
		else if (!strnicmp (line, "LAUNCHDELAY", 11)) 
		{
            sscanf (line+11, "%lf", &ExternalLaunchDelay);
		}
		else if (!strnicmp (line, "REALISTIC", 9)) 
		{
            sscanf (line+9, "%ld", &bRealistic);
			bOverrideRealistic = true;
		}
		else if (!strnicmp (line, "LATITUDE_TARGET", 15)) 
		{
            sscanf (line+15, "%lf", &LatitudeT);
		}
		else if (!strnicmp (line, "LONGITUDE_TARGET", 16)) 
		{
            sscanf (line+16, "%lf", &LongitudeT);
		}
		else if (!strnicmp (line, "TARGET", 6)) 
		{
            sscanf (line+6, "%s", TargetName);
		}

		// --- manual scenario overrides
		else if (!strnicmp (line, "AUTOPILOT", 9)) 
		{
            sscanf (line+9, "%ld", &bAutopilot);
			bOverrideAutopilot = true;
		}
		else if (!strnicmp (line, "RELIABILITY", 11)) 
		{
            sscanf (line+11, "%lf", &Reliability);

			if (Reliability < 0)
				Reliability = 0;

			else if (Reliability > 100)
				Reliability = 100;

			bOverrideReliability = true;
		}
		
		// --- restore autosaved entries

		// launch-initialized vars
		else if (!strnicmp (line, "STATEFLAGS", 10)) 
		{
            sscanf (line+10, "%ld", &StateFlags);
		}
		else if (!strnicmp (line, "POS0", 4)) 
		{
            sscanf (line+4, "%lf %lf %lf", &Longitude0, &Latitude0, &Radius0);
		}
		else if (!strnicmp (line, "HEADING_IDEAL", 13)) 
		{
            sscanf (line+13, "%lf", &LaunchHeadingIdeal);
		}
		else if (!strnicmp (line, "HEADING_REALISTIC", 17)) 
		{
            sscanf (line+17, "%lf", &LaunchHeadingRealistic);
		}
		else if (!strnicmp (line, "RVEL0", 5)) 
		{
            sscanf (line+5, "%lf %lf %lf", &rvel0.x, &rvel0.y, &rvel0.z);
		}
		else if (!strnicmp (line, "COSS", 4)) 
		{
            sscanf (line+4, "%lf %lf %lf", &dCosZs, &dCosYs, &dCosXs);
		}
		else if (!strnicmp (line, "SINS", 4)) 
		{
            sscanf (line+4, "%lf %lf %lf", &dSinZs, &dSinYs, &dSinXs);
		}
		else if (!strnicmp (line, "MET_MECO", 8)) 
		{
            sscanf (line+8, "%lf", &metMECO);
		}


		// running vars
		else if (!strnicmp (line, "MET", 3)) 
		{
            sscanf (line+3, "%lf", &met);
		}
		else if (!strnicmp (line, "MAX_ALTITUDE", 12)) 
		{
            sscanf (line+12, "%lf", &MaxAltitude);
		}

		// GPS
		else if (!strnicmp (line, "MET_GPS", 7)) 
		{
            sscanf (line+7, "%lf", &metGps);
		}
		else if (!strnicmp (line, "TRACK", 5)) 
		{
			double lon, lat, elev;
			int idx, col;
            sscanf (line+5, "%ld %lf %lf %lf %ld", &idx, &lon, &lat, &elev, &col);

			gpsIndex = idx+1;
			Longitudes[idx] = lon, 
			Latitudes[idx] = lat, 
			Elevations[idx] = elev;
			TrackStages[idx] = col;
		}

		// Failures
		else if (!strnicmp (line, "FAILDOUBLES", 11)) 
		{
            sscanf (line+11, "%lf %lf %lf", &FailTime, &FailDynPressure, &NumFailParam);
		}
		else if (!strnicmp (line, "CONTROLSFIXED", 13)) 
		{
            sscanf (line+13, "%lf %lf %lf", &YawFixed, &RollFixed, &PitchFixed);
		}
		else if (!strnicmp (line, "FAILSTRING", 10)) 
		{
            sscanf (line+610, "%s", FailureString);
		}
		else if (!strnicmp (line, "FAILIDX1", 8)) 
		{
            sscanf (line+8, "%ld", &FailIdx);
		}
		else if (!strnicmp (line, "FAILIDX2", 8)) 
		{
            sscanf (line+8, "%ld", &SubFailIdx);
		}
		else if (!strnicmp (line, "FAILTYPE", 8)) 
		{
            sscanf (line+8, "%ld", &FailType);
		}

		// --- skip visual separator
		else if (!strnicmp (line, "------", 6)) 
		{
            ; 
		}

		// --- unrecognised option - pass to Orbiter's generic parser
		else 
		{
            ParseScenarioLineEx (line, status);
        }
	}

	// now that we know final configuration, finish configurng the new vessel

	// update flags from scenario file
	if (StateFlags >= 0)
	{
		bCountdown = 1 & (StateFlags >> 2);
		bRocketLaunched = 1 & (StateFlags >> 3);
		bTimeToLaunch = 1 & (StateFlags >> 5);
		bPoweredFlight = 1 & (StateFlags >> 6);
//		bShowingContrail = 1 & (StateFlags >> 7);	- skip this: contrail will auto-create itself if needed

		// account for possibility of manual override on these two flags
		if (!bOverrideAutopilot)
			bAutopilot = 1 & (StateFlags >> 9);
		if (!bOverrideRealistic)
			bRealistic = 1 & (StateFlags >> 4);

		// restore failure flags
		bFail = 1 & (StateFlags >> 10);
		bYawFailed = 1 & (StateFlags >> 11);
		bRollFailed = 1 & (StateFlags >> 12);
		bPitchFailed = 1 & (StateFlags >> 13);
	}

	// create vessel with either realistic or "ideal" configuration
	if (bRealistic)
	{
		// set virtual RCS (to simulate vanes/rudders)
		MakeRealisticRCS();

		CalcMECOTime();

		SetPhysicsToRealistic();
	}
	else
	{
		MakeIdealRCS();

		SetPhysicsToIdeal();

		if (!bOverrideReliability)
			Reliability = 100;  // % 

	}

	// if we don't have target name - create it!
	if (strlen(TargetName) == 0)
		sprintf(TargetName, "%s-Target", GetName());

	// for now, only one mesh, but with future configurations varieties this it a place to update
	// both configuration and physics
	int MeshID = AddMesh (oapiLoadMeshGlobal("NGV2R"), &OFS);
	SetMeshVisibilityMode (MeshID, MESHVIS_ALWAYS);
}


// saving configuration to scn file
void V2i::clbkSaveState (FILEHANDLE scn)
{
	char BufferStr[256];

	// default vessel parameters
	SaveDefaultState (scn);

	// store visual separator
	oapiWriteScenario_string (scn, "------", "");

	// store auto-generated entries, except for bRealistic (will be part of the StateFlags)
	oapiWriteScenario_float (scn, "RANGE", Range);
	oapiWriteScenario_float (scn, "LAUNCHDELAY", ExternalLaunchDelay);
	sprintf(BufferStr, "%.13f", LatitudeT);
	oapiWriteScenario_string(scn, "LATITUDE_TARGET", BufferStr);
	sprintf(BufferStr, "%.13f", LongitudeT);
	oapiWriteScenario_string(scn, "LONGITUDE_TARGET", BufferStr);
	oapiWriteScenario_string(scn, "TARGET", TargetName);

	// store visual separator
	oapiWriteScenario_string (scn, "------", "");

	// combine state flags into one variable
	int StateFlags = 0;
	StateFlags += (bCountdown << 2);
	StateFlags += (bRocketLaunched << 3);

	StateFlags += (bTimeToLaunch << 5);
	StateFlags += (bPoweredFlight << 6);
	StateFlags += (bShowingContrail << 7);
	StateFlags += (bAutopilot << 9);
	StateFlags += (bRealistic << 4);
	// add fail flags
	StateFlags += (bFail << 10);
	StateFlags += (bYawFailed << 11);
	StateFlags += (bRollFailed << 12);
	StateFlags += (bPitchFailed << 13);

	// store state flags
	oapiWriteScenario_int (scn, "STATEFLAGS", StateFlags);

	// store reliability (can be manually overridden)
	oapiWriteScenario_float (scn, "RELIABILITY", Reliability);

	// store in-flight variables only if we are launched
	if (bRocketLaunched)
	{
		// initial launch settings
		sprintf(BufferStr, "%.13f %.13f %.13f", Longitude0, Latitude0, Radius0);
		oapiWriteScenario_string(scn, "POS0", BufferStr);
		oapiWriteScenario_float (scn, "HEADING_IDEAL", LaunchHeadingIdeal);
		oapiWriteScenario_float (scn, "HEADING_REALISTIC", LaunchHeadingRealistic);
		oapiWriteScenario_vec   (scn, "RVEL0", rvel0);
		VECTOR3 cosS = {dCosZs, dCosYs, dCosXs};
		VECTOR3 sinS = {dSinZs, dSinYs, dSinXs};
		oapiWriteScenario_vec   (scn, "COSS", cosS);
		oapiWriteScenario_vec   (scn, "SINS", sinS);
		oapiWriteScenario_float (scn, "MET_MECO", metMECO);

		// running vars
		oapiWriteScenario_float (scn, "MET", met);
		oapiWriteScenario_float (scn, "MAX_ALTITUDE", MaxAltitude);

		// store visual separator
		oapiWriteScenario_string (scn, "------", "");

		// failures
		VECTOR3 FailDoubles = {FailTime, FailDynPressure, NumFailParam};
		oapiWriteScenario_vec   (scn, "FAILDOUBLES", FailDoubles);
		VECTOR3 ControlsFixed = {YawFixed, RollFixed, PitchFixed};
		oapiWriteScenario_vec   (scn, "CONTROLSFIXED", ControlsFixed);
		oapiWriteScenario_string(scn, "FAILSTRING", FailureString);
		oapiWriteScenario_int   (scn, "FAILIDX1", FailIdx);
		oapiWriteScenario_int   (scn, "FAILIDX2", SubFailIdx);
		oapiWriteScenario_int   (scn, "FAILTYPE", FailType);

		// store visual separator
		oapiWriteScenario_string (scn, "------", "");

		// store GPS track points
		oapiWriteScenario_float (scn, "MET_GPS", metGps);
		char TrackPoint[250];
		for (int i = 0; i < gpsIndex; i++)
		{
			sprintf(TrackPoint, "%d %.13f %.13f %.13f %d", i, Longitudes[i], Latitudes[i], Elevations[i], TrackStages[i]);
			oapiWriteScenario_string(scn, "TRACK", TrackPoint);
		}
	}
}


// Keyboard interface handler
int V2i::clbkConsumeBufferedKey(DWORD key, bool down, char *kstate)
{
	if (!down) return 0; // only process keydown events

	if (KEYMOD_SHIFT (kstate)) 
	{
		return 0; // shift combinations are reserved
	}

	if ( !KEYMOD_CONTROL (kstate) && (key == OAPI_KEY_O))
	{
		// launch all other V2s around!
  		OBJHANDLE targetHandle;
		int VCount = oapiGetVesselCount();
		for (int i=0; i<VCount; i++)
		{
			targetHandle = oapiGetVesselByIndex(i);				
			try
			{
				((V2i*)oapiGetVesselInterface(targetHandle))->ExternalLaunch();
			}
			catch (...)
			{	// will catch errors for non-V2 vessels
			}
		}
		return 1;
	}

	if ( !KEYMOD_CONTROL (kstate) && (key == OAPI_KEY_A))
	{
		// toggle autopilot
		if (bAutopilot)
			ShutdownControls();
		bAutopilot = !bAutopilot;

		return 1;
	}

	if ( !KEYMOD_CONTROL (kstate) && (key == OAPI_KEY_D) )
	{
		DebugSomething();
		return 0;
	}

	return 0;
}



// simpliest RCS, symmetrical forces
void V2i::MakeIdealRCS()
{
	// RSC tuning, important for pitch program
	const RCSThrust = 30;

	// set virtual RCS (to simulate vanes/rudders)
	PROPELLANT_HANDLE RCSTank  = CreatePropellantResource (1);
	const RCSISP = 49000000;
	THRUSTER_HANDLE thRot[2];


	thRot[0] = CreateThruster (_V(-6, 0, 0), _V(0,  1,0), RCSThrust, RCSTank, RCSISP);
	thRot[1] = CreateThruster (_V( 6, 0, 0), _V(0, -1,0), RCSThrust, RCSTank, RCSISP);
	RightRoll = CreateThrusterGroup (thRot, 2, THGROUP_ATT_BANKRIGHT);

	thRot[0] = CreateThruster (_V( 6, 0, 0), _V(0,  1,0), RCSThrust, RCSTank, RCSISP);
	thRot[1] = CreateThruster (_V(-6, 0, 0), _V(0, -1,0), RCSThrust, RCSTank, RCSISP);
	LeftRoll = CreateThrusterGroup (thRot, 2, THGROUP_ATT_BANKLEFT);

	thRot[0] = CreateThruster (_V( 6,0,0), _V(0, 0,-1), RCSThrust, RCSTank, RCSISP);
	thRot[1] = CreateThruster (_V(-6,0,0), _V(0, 0, 1), RCSThrust, RCSTank, RCSISP);
	RightYaw = CreateThrusterGroup (thRot, 2, THGROUP_ATT_YAWRIGHT);

	thRot[0] = CreateThruster (_V(-6,0,0), _V(0, 0,-1), RCSThrust, RCSTank, RCSISP);
	thRot[1] = CreateThruster (_V( 6,0,0), _V(0, 0, 1), RCSThrust, RCSTank, RCSISP);
	LeftYaw = CreateThrusterGroup (thRot, 2, THGROUP_ATT_YAWLEFT);

	thRot[0] = CreateThruster (_V(0,0, 6), _V(0,  1, 0), RCSThrust, RCSTank, RCSISP);
	thRot[1] = CreateThruster (_V(0,0,-6), _V(0, -1, 0), RCSThrust, RCSTank, RCSISP);
	PitchUp = CreateThrusterGroup (thRot, 2, THGROUP_ATT_PITCHUP);

	thRot[0] = CreateThruster (_V(0,0, 6), _V(0, -1, 0), RCSThrust, RCSTank, RCSISP);
	thRot[1] = CreateThruster (_V(0,0,-6), _V(0,  1, 0), RCSThrust, RCSTank, RCSISP);
	PitchDown = CreateThrusterGroup (thRot, 2, THGROUP_ATT_PITCHDOWN);

	// to debug thrust location, visualize it like this:
	//AddExhaust (thRot[0], 0.5, 0.1);
}



// more realistic RCS locations at the base of the rocket
void V2i::MakeRealisticRCS()
{
	// RSC tuning
	const RCSThrust = 200;

	// set virtual RCS (to simulate vanes/rudders)
	PROPELLANT_HANDLE RCSTank  = CreatePropellantResource (1);
	const RCSISP = 49000000;
	THRUSTER_HANDLE thRot[2];

	// roll vanes

	thRot[0] = CreateThruster (_V(-1.6, 0, -7), _V(0,  1,0), RCSThrust*2, RCSTank, RCSISP);
	thRot[1] = CreateThruster (_V( 1.6, 0, -7), _V(0, -1,0), RCSThrust*2, RCSTank, RCSISP);
	RightRoll = CreateThrusterGroup (thRot, 2, THGROUP_ATT_BANKRIGHT);

	thRot[0] = CreateThruster (_V( 1.6, 0, -7), _V(0,  1,0), RCSThrust*2, RCSTank, RCSISP);
	thRot[1] = CreateThruster (_V(-1.6, 0, -7), _V(0, -1,0), RCSThrust*2, RCSTank, RCSISP);
	LeftRoll = CreateThrusterGroup (thRot, 2, THGROUP_ATT_BANKLEFT);

	// yaw rudders

	thRot[0] = CreateThruster (_V( 0,  0.5, -6.5), _V(-1, 0,0), RCSThrust, RCSTank, RCSISP);
	RightYawUpper = CreateThrusterGroup (thRot, 1, THGROUP_USER);

	thRot[0] = CreateThruster (_V( 0, -0.5, -6.5), _V(-1, 0,0), RCSThrust, RCSTank, RCSISP);
	RightYawLower = CreateThrusterGroup (thRot, 1, THGROUP_ATT_YAWRIGHT);

	thRot[0] = CreateThruster (_V( 0, -0.5, -6.5), _V(1, 0, 0), RCSThrust, RCSTank, RCSISP);
	LeftYawLower = CreateThrusterGroup (thRot, 1, THGROUP_ATT_YAWLEFT);
	
	thRot[0] = CreateThruster (_V( 0,  0.5, -6.5), _V(1, 0, 0), RCSThrust, RCSTank, RCSISP);
	LeftYawUpper = CreateThrusterGroup (thRot, 1, THGROUP_USER);

	// yaw vanes

	thRot[0] = CreateThruster (_V( 0,  1.6, -7), _V(-1, 0,0), RCSThrust*2, RCSTank, RCSISP);
	RightYawUpperVane = CreateThrusterGroup (thRot, 1, THGROUP_USER);

	thRot[0] = CreateThruster (_V( 0, -1.6, -7), _V(-1, 0,0), RCSThrust*2, RCSTank, RCSISP);
	RightYawLowerVane = CreateThrusterGroup (thRot, 1, THGROUP_USER);

	thRot[0] = CreateThruster (_V( 0, -1.6, -7), _V(1, 0, 0), RCSThrust*2, RCSTank, RCSISP);
	LeftYawLowerVane = CreateThrusterGroup (thRot, 1, THGROUP_USER);
	
	thRot[0] = CreateThruster (_V( 0,  1.6, -7), _V(1, 0, 0), RCSThrust*2, RCSTank, RCSISP);
	LeftYawUpperVane = CreateThrusterGroup (thRot, 1, THGROUP_USER);

	// pitch rudders

	thRot[0] = CreateThruster (_V( 0.5, 0, -6.5), _V(0, -1, 0), RCSThrust, RCSTank, RCSISP);
	thRot[1] = CreateThruster (_V(-0.5, 0, -6.5), _V(0, -1, 0), RCSThrust, RCSTank, RCSISP);
	PitchUp = CreateThrusterGroup (thRot, 2, THGROUP_ATT_PITCHUP);

	thRot[0] = CreateThruster (_V( 0.5, 0, -6.5), _V(0, 1, 0), RCSThrust, RCSTank, RCSISP);
	thRot[1] = CreateThruster (_V(-0.5, 0, -6.5), _V(0, 1, 0), RCSThrust, RCSTank, RCSISP);
	PitchDown = CreateThrusterGroup (thRot, 2, THGROUP_ATT_PITCHDOWN);
}


// make the vessel aerodynamically symmetrical, like a warhead instead of dart, 
// for better (more "ideal") shooting accuracy 
void V2i::SetPhysicsToIdeal()
{
	SetPMI (_V(.35, .35, .35));
	SetCrossSections (_V(1.07, 1.07, 1.07));
	SetCW (0.55, 0.55, 0.55, 0.55);
	SetRotDrag (_V(0.1, 0.1, 0.1));
}


void V2i::SetPhysicsToRealistic()
{
	SetPMI (_V(1.41, 1.41, .35));
	SetCrossSections (_V(12.0, 12.0, 1.07));
	SetCW (0.55, 0.7, 1.5, 1.5);
	SetRotDrag (_V(0.7, 0.7, 0.1));
}


void V2i::clbkPostStep(double simt, double simdt, double mjd)
{

	// still waiting on the ground
	if (!bRocketLaunched) 
	{
		// is countdown still on?
		if (bCountdown)
			UpdateLaunchCountdown();

		// time to launch?
		else if (bTimeToLaunch)
			Launch();

		// try to detect manual launch (user pressed '+' - this bypasses launch automation!)
		else if (GetEngineLevel(ENGINE_MAIN) > 0.1)
			Launch();

		// DO NOT process any additional conditions!
		return;
	}


	// if we got here - we are in flight for sure!

	// did we hit the ground? if yes, no need to go any further...
	// Just wait 3 seconds, or it sometimes blows up at the launch
	if ((GroundContact() == 1) && (met > 3))
	{
		Explode(true);
		return;
	}

	// update met, do it only here!
	met += simdt;

	// update/store altitudes (and other vars?) - will be used downstream
	Altitude = GetAltitude();
	if (Altitude > MaxAltitude)
		MaxAltitude = Altitude;

	// record GPS flight data for GE, every 1 sec, no longer than 400 sec.
	if ((met >= metGps) && (gpsIndex < 400))
		WriteGpsTrackPoint();
	
	// check some operations and conditions only in powered flight (contrail, autopilot, etc.)
	if (bPoweredFlight)
	{
		// try to detect manual shutdown
		if (GetEngineLevel(ENGINE_MAIN) == 0)
		{
			EngineShutdown();
			return;
		}

		if (bRealistic)
		{
	 		dSimt = simdt;

			if (bAutopilot)
				AutopilotRealistic();
		}
		else if (bAutopilot)
			AutopilotIdeal();

		// also called directly from Shutdown
		UpdateContrail();
	}

	// try to detect manual in-flight engine restart 
	else if (GetEngineLevel(ENGINE_MAIN) > 0.1)
		Launch();

	// Check for failure! (always, except for when waiting for launch):
	if ((bFail) && FailCondition())
			FailAction();
}



// this is called externally, from this or some other V2 rocket
void V2i::ExternalLaunch()
{
	if (!bRocketLaunched) 
		bCountdown = true;
}



void V2i::UpdateLaunchCountdown()
{
	// decrement countdown delay
	ExternalLaunchDelay -= oapiGetSimStep();

	if (ExternalLaunchDelay < 0)
	{
		bCountdown = false;
		bTimeToLaunch = true;
	}
}


void V2i::Launch()
{
	// not in-flight manual restart?
	if (!bRocketLaunched)
	{
		bTimeToLaunch = false;
		bRocketLaunched = true;

		InitStartLocalRot();

		VESSELSTATUS VS0;
		GetStatus(VS0);
		rvel0 = VS0.rvel;

		// store the launch state
		GetEquPos(Longitude0, Latitude0, Radius0);

		// launch heading
		oapiGetHeading(GetHandle(), &LaunchHeadingIdeal);
		LaunchHeadingIdeal = LaunchHeadingIdeal/PI*180;
		LaunchHeadingRealistic = LaunchHeadingIdeal;

		// if not passed from scenario, calculate target location, dg
		if ((LatitudeT == 0) || (LongitudeT ==0))
		{
			CoordsFromDistanceAzimuthKmDg(Latitude0, Longitude0, Range, LaunchHeadingIdeal, &LatitudeT, &LongitudeT);
			LatitudeT *=DEG;
			LongitudeT *= DEG;
		}

		if (bRealistic)
			LaunchHeadingRealistic += DRand(-0.25, 0.5);
	}

	bPoweredFlight = true;

	// full throttle!
	if (bAutopilot)
		SetThrusterLevel(MainEngine, 1);

	InitializeFailures();
}



// Replace myself (V2 object) with the V2iWreck object, pass all needed data
void V2i::Explode(bool bLanded)
{
	// only if landed: just in case, make myself huge and heavy to kill any bouncing
	if (bLanded)
	{
		SetCW(2500, 2500, 2500, 2500);
		SetEmptyMass(100000);
	}

	// get actual location, rad
	double Latitude, Longitude, Radius;
	GetEquPos(Longitude, Latitude, Radius);

	// adjust elevation for the last track point
	if (bLanded)
		Radius = 0;
	else
		Radius -= Radius0;

	// store it as the last track point
	Longitudes[gpsIndex] = Longitude;
	Latitudes[gpsIndex] = Latitude;
	Elevations[gpsIndex] = Radius;
	TrackStages[gpsIndex] = 3;
	gpsIndex++;

	// calculate actual range and azimuth, km+dg
	double RealRange, RealAzimuth; 
	DistanceAzimuthFromCoordPairKmDg(Latitude0, Longitude0, Latitude, Longitude, &RealRange, &RealAzimuth);
	
	// calculate miss and azimuth from target to actual, km+dg
	double MissRange, MissAzimuth; 
	DistanceAzimuthFromCoordPairKmDg(LatitudeT*PI/180, LongitudeT*PI/180, Latitude, Longitude, &MissRange, &MissAzimuth);

	// calculate lateral and longitudal misses, km
	double dAzimuthRad = (MissAzimuth - LaunchHeadingIdeal)*PI/180;
	double MissLateral = MissRange * sin(dAzimuthRad);
	double MissLongitudal = MissRange * cos(dAzimuthRad);

	// prepare a name for a wreck object 
	char Name[100];
	strcpy(Name, GetName());
	strcat(Name, "-Wreck");

	// clone wreck status from the current one
	VESSELSTATUS vs;
	GetStatus(vs);

	// only if landed: force the status to pin the new object to the ground
	if (bLanded)
	{
		vs.status = 1;
		vs.vdata[0].x = Longitude;
		vs.vdata[0].y = Latitude;
		vs.vdata[0].z = 0;
	}

	// now, create a wreck object 
	void *hWreck;
	hWreck = oapiCreateVessel(Name, "V2iWreck", vs);

	// transfer loggable values to the new wreck object
	VESSEL *Wreck = oapiGetVesselInterface(hWreck);
	((V2iWreck*)Wreck)->Latitude = Latitude;
	((V2iWreck*)Wreck)->Longitude = Longitude;
	((V2iWreck*)Wreck)->FlightTime = met;
	((V2iWreck*)Wreck)->MaxAltitude = MaxAltitude;
	((V2iWreck*)Wreck)->ImpactVelocity = GetAirspeed();
	((V2iWreck*)Wreck)->RealRange = RealRange;
	((V2iWreck*)Wreck)->RealAzimuth = RealAzimuth;
	((V2iWreck*)Wreck)->LatitudeT = LatitudeT;
	((V2iWreck*)Wreck)->LongitudeT = LongitudeT;
	((V2iWreck*)Wreck)->MissRange = MissRange;
	((V2iWreck*)Wreck)->MissAzimuth = MissAzimuth;
	((V2iWreck*)Wreck)->MissLateral = MissLateral;
	((V2iWreck*)Wreck)->MissLongitudal = MissLongitudal;
	((V2iWreck*)Wreck)->LatitudeL = Latitude0;
	((V2iWreck*)Wreck)->LongitudeL = Longitude0;
	((V2iWreck*)Wreck)->gpsCount = gpsIndex;

	// transfer GPS points
	for (int i = 0; i < gpsIndex; i++)
	{
		((V2iWreck*)Wreck)->Longitudes[i] = Longitudes[i];
		((V2iWreck*)Wreck)->Latitudes[i] = Latitudes[i];
		((V2iWreck*)Wreck)->Elevations[i] = Elevations[i];
		((V2iWreck*)Wreck)->TrackStages[i] = TrackStages[i];
	}

	if (TargetName != NULL)
	  strcpy(((V2iWreck*)Wreck)->TargetName, TargetName);

	if (FailureString != NULL)
	  strcpy(((V2iWreck*)Wreck)->FailureString, FailureString);

	((V2iWreck*)Wreck)->bLanded = bLanded;

	if (oapiGetFocusObject() == GetHandle())
		oapiSetFocusObject(hWreck);

	// Finally, delete myself
	oapiDeleteVessel (GetHandle(), 0);
}


void V2i::EngineShutdown()
{
	SetThrusterLevel(MainEngine, 0);
	ShutdownControls();
	bPoweredFlight = false;
	UpdateContrail();
}


void V2i::ShutdownControls()
{
	SetThrusterGroupLevel(RightRoll, 0);
	SetThrusterGroupLevel(LeftRoll, 0);

	if (bRealistic)
	{
		SetThrusterGroupLevel(RightYawUpper, 0);
		SetThrusterGroupLevel(LeftYawUpper, 0);
		SetThrusterGroupLevel(RightYawLower, 0);
		SetThrusterGroupLevel(LeftYawLower, 0);

		SetThrusterGroupLevel(RightYawUpperVane, 0);
		SetThrusterGroupLevel(LeftYawUpperVane, 0);
		SetThrusterGroupLevel(RightYawLowerVane, 0);
		SetThrusterGroupLevel(LeftYawLowerVane, 0);
	}
	else
	{
		SetThrusterGroupLevel(RightYaw, 0);
		SetThrusterGroupLevel(LeftYaw, 0);
	}

	SetThrusterGroupLevel(PitchUp, 0);
	SetThrusterGroupLevel(PitchDown, 0);
}


void V2i::UpdateContrail()
{
	// create the contrail
	PARTICLESTREAMSPEC V2Con = {
		0,		// flags, reserved
		5.00,	// srcsize, particle size at creation, m
		100,		// srcrate, average particle generation rate, Hz  
		500.0,		// V0, average particle emission velocity, m/s
		0,		// srcspread, emission velocity distribution randomisation
		30.,	// lifetime, average particle lifetime [s]
		2.,		// growthrate, particle growth rate [m/s]
		1000.,		// atmslowdown, deceleration rate b in atmosphere, defined as v = v0 e- bdt 
		PARTICLESTREAMSPEC::EMISSIVE,
		PARTICLESTREAMSPEC::LVL_PSQRT, 
		0, 
		0.5,
		PARTICLESTREAMSPEC::ATM_PLOG, 
		1e-6, 
		0.1
	};

	// it mostly depends on altitude
	bool bContrailAltitude = (Altitude > 4000) && (Altitude < 20000);

	if (bShowingContrail)		// already showing
	{
		if ( (!bContrailAltitude) || (!bPoweredFlight) )	// need to stop?
		{
 			DelExhaustStream(ContrailExhStream);
			bShowingContrail = false;
		}
	}
	else						// not showing yet
	{
		if ( (bContrailAltitude) && (bPoweredFlight) )		// time to show?
		{
			bShowingContrail = true;
 			ContrailExhStream = AddExhaustStream (MainTH[0], _V(0,0,-25), &V2Con);
		}
	}
}


void V2i::AutopilotIdeal()
{
	// vertical flight - do nothing!
	if (met < TIME_VERT)
		return;

	// attitude control
	YawCorrectionIdeal();
	RollCorrectionIdeal();
	PitchProgramIdeal();

	// range
	UpdateCurrentRangeIdeal();
}


void V2i::RollCorrectionIdeal()
{
	double DesiredBank = 0;

	// failed?
	if (bRollFailed)
		DesiredBank = RollFixed;

	if (GetBank() < DesiredBank)
	{
		SetThrusterGroupLevel(RightRoll, 0);
		SetThrusterGroupLevel(LeftRoll, 1);
	}
	else
	{
		SetThrusterGroupLevel(RightRoll, 1);
		SetThrusterGroupLevel(LeftRoll, 0);
	}
}


void V2i::YawCorrectionIdeal()
{
	// get real flight parameters
	double Heading;
	oapiGetHeading(GetHandle(), &Heading);
	Heading = Heading/PI*180;
	
	// failed?
	if (bYawFailed)
		Heading = YawFixed;

	// calculate direction of correction
	bool bYawTurnLeft;
	if (LaunchHeadingIdeal < 180)
		bYawTurnLeft = (Heading > LaunchHeadingIdeal) && (Heading < (180+LaunchHeadingIdeal));
	else
		bYawTurnLeft = (Heading > LaunchHeadingIdeal) || (Heading < (LaunchHeadingIdeal-180));

	// correct the flight
	if (bYawTurnLeft)
	{
		SetThrusterGroupLevel(RightYaw, 0);
		SetThrusterGroupLevel(LeftYaw, 1);
	}
	else
	{
		SetThrusterGroupLevel(RightYaw, 1);
		SetThrusterGroupLevel(LeftYaw, 0);
	}
}


// more ideal pitch program: fast pitchover for about 20 sec (may depend on RCS thrust), then follow aoa
void V2i::PitchProgramIdeal()
{
	// tuning
	const TIME_PITCHOVER = 20;		// sec
	const ANGLE_PITCHOVER = -10;	// deg, does not really reach so deep

	// get real angle of attack
	double aoa = GetAOA()*180./PI;

	// get required angle of attack
	double PlannedAoa;
	if (met < TIME_PITCHOVER)
		PlannedAoa = ANGLE_PITCHOVER;
	else
		PlannedAoa = 0;

	if (bPitchFailed)
		PlannedAoa = PitchFixed - 45;

	// do pitch control
	if (PlannedAoa > aoa) 
	{
		SetThrusterGroupLevel(PitchUp, 1);
		SetThrusterGroupLevel(PitchDown, 0);
	}		
	else  
	{
		SetThrusterGroupLevel(PitchUp, 0);
		SetThrusterGroupLevel(PitchDown, 1);
	}		
}



void V2i::AutopilotRealistic()
{
	// vertical flight - do nothing!
	if (met < TIME_VERT)
		return;

	// turn attitude controls from previous step
	TurnAttitudeServos(); 

	// attitude control
	YawCorrectionRealistic();
	RollCorrectionRealistic();
	PitchProgramRealistic();

	// range
	if (met > metMECO)
		EngineShutdown();
}



double SERVOS_SWITCHOVER_TIME = 0.125; // sec
const ANGLE_SENSOR_MAX = 3; // dg
const MAX_DYN_PRESSURE = 65000;  


void V2i::TurnAttitudeServos()
{

	// how far we can move a servo during the current step
	double dServoPos = dSimt / SERVOS_SWITCHOVER_TIME;


	// ROLL

	if (ActualRollTrimsPos < DesiredRollTrimsPos)			// right negative, left positive
	{
		ActualRollTrimsPos += dServoPos;
		if (ActualRollTrimsPos > 1)
			ActualRollTrimsPos = 1;
	}
	else
	{
		ActualRollTrimsPos -= dServoPos;
		if (ActualRollTrimsPos < -1)
			ActualRollTrimsPos = -1;
	}

	double VaneEffectivenessPercent = GetDynPressure() / MAX_DYN_PRESSURE;
	double VaneEffectiveness = ActualRollTrimsPos * VaneEffectivenessPercent;

	SetThrusterGroupLevel(LeftRoll, VaneEffectiveness);  
	SetThrusterGroupLevel(RightRoll, -VaneEffectiveness);


	// YAW

	if (ActualYawUpperServoPos < DesiredYawUpperServoPos)
	{
		ActualYawUpperServoPos += dServoPos;
		if (ActualYawUpperServoPos > 1)
			ActualYawUpperServoPos = 1;
	}
	else
	{
		ActualYawUpperServoPos -= dServoPos;
		if (ActualYawUpperServoPos < -1)
			ActualYawUpperServoPos = -1;
	}

	if (ActualYawLowerServoPos < DesiredYawLowerServoPos)
	{
		ActualYawLowerServoPos += dServoPos;
		if (ActualYawLowerServoPos > 1)
			ActualYawLowerServoPos = 1;
	}
	else
	{
		ActualYawLowerServoPos -= dServoPos;
		if (ActualYawLowerServoPos < -1)
			ActualYawLowerServoPos = -1;
	}
	
	// calc vanes effectiveness
	double UpperVaneEffectiveness = ActualYawUpperServoPos * VaneEffectivenessPercent;
	double LowerVaneEffectiveness = ActualYawLowerServoPos * VaneEffectivenessPercent;

	// apply controls

	// upper rudder
	SetThrusterGroupLevel(RightYawUpper, ActualYawUpperServoPos);
	SetThrusterGroupLevel(LeftYawUpper, -ActualYawUpperServoPos);

	// lower rudder
	SetThrusterGroupLevel(RightYawLower, ActualYawLowerServoPos);
	SetThrusterGroupLevel(LeftYawLower, -ActualYawLowerServoPos);

	// upper vane
	SetThrusterGroupLevel(RightYawUpperVane, UpperVaneEffectiveness);
	SetThrusterGroupLevel(LeftYawUpperVane, -UpperVaneEffectiveness);

	// lower vane
	SetThrusterGroupLevel(RightYawLowerVane, LowerVaneEffectiveness);
	SetThrusterGroupLevel(LeftYawLowerVane, -LowerVaneEffectiveness);


	// PITCH

	if (ActualPitchServosPos < DesiredPitchServosPos)
	{
		ActualPitchServosPos += dServoPos;
		if (ActualPitchServosPos > 1)
			ActualPitchServosPos = 1;
	}
	else
	{
		ActualPitchServosPos -= dServoPos;
		if (ActualPitchServosPos < -1)
			ActualPitchServosPos = -1;
	}

	SetThrusterGroupLevel(PitchDown, ActualPitchServosPos);  
	SetThrusterGroupLevel(PitchUp, - ActualPitchServosPos);
}


// note: part of roll correction is also done by yaw controls
void V2i::RollCorrectionRealistic()
{
	// get actual roll angle
	double RollSensor = GetBank()/PI*180;
	
	// failed?
	if (bRollFailed)
		RollSensor += RollFixed;

	// calculate potentiometer, set its limitations
	if (RollSensor > ANGLE_SENSOR_MAX)
		RollSensor = ANGLE_SENSOR_MAX;
	else if (RollSensor < -ANGLE_SENSOR_MAX)
		RollSensor = -ANGLE_SENSOR_MAX;

	DesiredRollTrimsPos = -RollSensor/ANGLE_SENSOR_MAX;	// right negative, left positive
}


void V2i::YawCorrectionRealistic()
{

	// get actual yaw angle
	double ActualYaw;
	oapiGetHeading(GetHandle(), &ActualYaw);
	ActualYaw = ActualYaw/PI*180;	

	// calculate potentiometer, set its limitations
	double YawSensor = ActualYaw - LaunchHeadingRealistic;				// right positive, left negative

	// failed?
	if (bYawFailed)
		YawSensor += YawFixed;

	if (YawSensor > 180)
		YawSensor -= 360;
	else if (YawSensor < -180)
		YawSensor += 360;

	if (YawSensor > ANGLE_SENSOR_MAX)
		YawSensor = ANGLE_SENSOR_MAX;
	else if (YawSensor < -ANGLE_SENSOR_MAX)
		YawSensor = -ANGLE_SENSOR_MAX;


	DesiredYawUpperServoPos = -YawSensor/ANGLE_SENSOR_MAX;	// right negative, left positive
	DesiredYawLowerServoPos = DesiredYawUpperServoPos;   	// right negative, left positive

	// disable one rudder/vane for roll correction
	if (GetBank() > 0)							// banked to right
	{
		if (DesiredYawUpperServoPos < 0)	// yawing to right
			DesiredYawLowerServoPos = 0;
		else								// yawing to left
			DesiredYawUpperServoPos = 0;
	}
	else										// banked to right
	{
		if (DesiredYawUpperServoPos < 0)	// yawing to right
			DesiredYawUpperServoPos = 0;
		else								// yawing to left
			DesiredYawLowerServoPos = 0;
	}
}


// realistic pitch program: 4 seconds vertical, 43 seconds to 47 degrees, then hold steady on 47
void V2i::PitchProgramRealistic()
{
	// tuning
	const TIME_PITCHOVER = 47;		// sec

	// get real pitch angle
	double ActualPitch = GetPitch()*180./PI;

	// get required pitch angle
	double PlannedPitch;
	if (met < TIME_PITCHOVER)
		PlannedPitch = 90 - (met-TIME_VERT);
	else
		PlannedPitch = 47;

	// calc values for next step

	double PitchSensor = PlannedPitch - ActualPitch;		//dg	// nose down positive, nose up negative

	// failed?
	if (bPitchFailed)
		PitchSensor += (PitchFixed-45);
	
	if (PitchSensor > ANGLE_SENSOR_MAX)
		PitchSensor = ANGLE_SENSOR_MAX;
	if (PitchSensor < -ANGLE_SENSOR_MAX)
		PitchSensor = -ANGLE_SENSOR_MAX;

	DesiredPitchServosPos = -PitchSensor/ANGLE_SENSOR_MAX;	// nose down negative, nose up positive
}



void V2i::UpdateCurrentRangeIdeal()
{
	VESSELSTATUS VS;
	GetStatus(VS);
	VECTOR3 Vg = VS.rvel-rvel0; //global vel vector relative to stored launch point.
	VECTOR3 Vl;  // same rotated to local vessel coords
	LocalRot(Vg, Vl);
	double V = sqrt(Vl.x*Vl.x + Vl.y*Vl.y + Vl.z*Vl.z);
	VECTOR3 Vs;  // same rotated to start vessel coords
	StartLocalRot(Vg, Vs);
	double VHorHor = sqrt(Vs.x*Vs.x + Vs.y*Vs.y);

	// current location
	double Longitude;
	double Latitude;
	double Radius;
	GetEquPos(Longitude, Latitude, Radius);

	double h = Radius*VHorHor;

	// major semiaxis
	double a = 1 / ((1/Radius - V*V/(2*3.986E14)) * 2);

	// eccentricity
	double e = sqrt(1 - ( h*h /3.985651951E+14/a));


	// teta from perigee to the launch point
	double TetaGround0 = acos(((a * (1-e*e) / Radius0) - 1) / e); //rad

	// orbital ground track length 
	double Range0 = (PI-TetaGround0) *2 *Radius0 - RANGE_LOSS_IDEAL;

	// are we done? 
	if (Range0 >= Range)
		EngineShutdown();
}


void V2i::DistanceAzimuthFromCoordPairRad(double Lat1, double Lon1, double Lat2, double Lon2, double* pDistance, double* pAzimuth)
{
	double t = sin(Lat1)*sin(Lat2) + cos(Lat1)*cos(Lat2)*cos(Lon2-Lon1);
	if (t > 1)
		t = 1;
	else if (t < -1)
		t = -1;

	*pDistance = acos(t);
	
	t = (sin(Lat2) - sin(Lat1)*cos(*pDistance))/(sin(*pDistance)*cos(Lat1));
	if (t > 1)
		t = 1;
	else if (t < -1)
		t = -1;

	*pAzimuth = acos(t);
	
	if (Lon2 < Lon1) 
		*pAzimuth = 2*PI - *pAzimuth ;
}


void V2i::DistanceAzimuthFromCoordPairKmDg(double Lat1, double Lon1, double Lat2, double Lon2, double* pDistance, double* pAzimuth)
{
	DistanceAzimuthFromCoordPairRad(Lat1, Lon1, Lat2, Lon2, pDistance, pAzimuth);
	*pDistance *= Radius0;
	*pAzimuth = *pAzimuth*180/PI;
}


void V2i::CoordsFromDistanceAzimuthRad(double Lat1, double Lon1, double Distance, double Azimuth, double* pLat2, double* pLon2)
{
	double t = (cos(Azimuth)*sin(Distance)*cos(Lat1)) + sin(Lat1)*cos(Distance);
	if (t > 1)
		t = 1;
	else if (t < -1)
		t = -1;

	*pLat2 = asin(t);

	t = (cos(Distance) - sin(Lat1)*sin(*pLat2)) / (cos(Lat1)*cos(*pLat2));
	if (t > 1)
		t = 1;
	else if (t < -1)
		t = -1;

	double DeltaLon = acos(t);

	if (Azimuth < (PI))
	  *pLon2 = Lon1 + DeltaLon;
	else
	  *pLon2 = Lon1 - DeltaLon;
}


void V2i::CoordsFromDistanceAzimuthKmDg(double Lat1, double Lon1, double Distance, double Azimuth, double* pLat2, double* pLon2)
{
	// note: this depends on Radius0 being initialized!
	CoordsFromDistanceAzimuthRad(Lat1, Lon1, Distance/Radius0, Azimuth*PI/180, pLat2, pLon2);
}


void V2i::CalcMECOTime()
{
	// 45, 50, 55, 60, 65, 70 sec
	double Ranges[6] = { 41184.08, 63753.38, 99145.27, 150090.56, 224036.94, 324799.44 }; 
	int i;

	if (Range < Ranges[0])
		metMECO = 45;
	else if (Range > Ranges[5])
		metMECO = 75;
	else
	{
		for (i = 0; i < 4; i++)
			if (Range > Ranges[i])
				continue;

		double dRangeInterval = Ranges[i+1] - Ranges[i];
		double dRangeActual = Range - Ranges[i];
		double dTime = dRangeActual/dRangeInterval * 5;

		metMECO = 45 + i*5 + dTime;
	}
}


void V2i::LocalRot(const VECTOR3 vecGlob, VECTOR3 &vecLoc)
{
	VESSELSTATUS2 vs;
	double dCosZ, dSinZ, dCosY, dSinY, dCosX, dSinX;
	vs.version = 2;
	vs.flag = 0;
	GetStatusEx((void*) &vs);

	dCosZ = cos(vs.arot.z);
	dSinZ = sin(vs.arot.z);
	dCosY = cos(vs.arot.y);
	dSinY = sin(vs.arot.y);
	dCosX = cos(vs.arot.x);
	dSinX = sin(vs.arot.x);

	vecLoc.x = vecGlob.x*dCosZ*dCosY + vecGlob.y*(-dSinZ*dCosX+dCosZ*dSinY*dSinX) + vecGlob.z*(dSinZ*dSinX+dCosZ*dSinY*dCosX);
	vecLoc.y = vecGlob.x*dSinZ*dCosY + vecGlob.y*(dCosZ*dCosX+dSinZ*dSinY*dSinX) + vecGlob.z*(-dCosZ*dSinX+dSinZ*dSinY*dCosX);
	vecLoc.z = vecGlob.x*-dSinY + vecGlob.y*dCosY*dSinX + vecGlob.z*dCosY*dCosX;
}


void V2i::InitStartLocalRot()
{
	VESSELSTATUS2 vs;
	vs.version = 2;
	vs.flag = 0;
	GetStatusEx((void*) &vs);

	dCosZs = cos(vs.arot.z);
	dSinZs = sin(vs.arot.z);
	dCosYs = cos(vs.arot.y);
	dSinYs = sin(vs.arot.y);
	dCosXs = cos(vs.arot.x);
	dSinXs = sin(vs.arot.x);
}


void V2i::StartLocalRot(const VECTOR3 vecGlob, VECTOR3 &vecLoc)
{
	vecLoc.x = vecGlob.x*dCosZs*dCosYs + vecGlob.y*(-dSinZs*dCosXs+dCosZs*dSinYs*dSinXs) + vecGlob.z*(dSinZs*dSinXs+dCosZs*dSinYs*dCosXs);
	vecLoc.y = vecGlob.x*dSinZs*dCosYs + vecGlob.y*(dCosZs*dCosXs+dSinZs*dSinYs*dSinXs) + vecGlob.z*(-dCosZs*dSinXs+dSinZs*dSinYs*dCosXs);
	vecLoc.z = vecGlob.x*-dSinYs + vecGlob.y*dCosYs*dSinXs + vecGlob.z*dCosYs*dCosXs;
}


// This is called in the late stages of initialization.
// Failure is pre-determined before start (quite realistic!).
// Are we going to fail? What type of failure? At what point of flight? 
void V2i::InitializeFailures()
{
	// start clean
	bFail = false;
	FailType = NONE;

	// NN% chances to fail...
	if (DRand(0, 100) > Reliability)
		bFail = true;

	// fail!
	if (bFail)
	{
		// how?
		double rnd = DRand(0, 90);
		if (rnd < 30)
		{
			FailType = BREAKUP;
			FailIdx = 0;

			// Dyn pressure. In ver 2, this should be a part of a more general structure tolerances.
			// Ideal: burns at 40000-90000 at launch, can be up to 800000 at re-entry.
			// Realistic: 40000-65000 at launch, can be up to 350000 at reentry.

			// 5 % probability: break up at launch, remaining 95% - at reentry, 
			// but try to cover whole range 
			if (DRand(0, 100) < 5)
			{
				if (bRealistic)
					FailDynPressure = DRand(50000, 20000);
				else
					FailDynPressure = DRand(50000, 50000);

				SubFailIdx = 0;
				NumFailParam = FailDynPressure;
			}
			else
			{
				if (bRealistic)
					FailDynPressure = DRand(70000, 330000);
				else
					FailDynPressure = DRand(100000, 700000);

				SubFailIdx = 1;
				NumFailParam = FailDynPressure;
			}
		}

		else if (rnd > 60)
		{
			FailType = ENGINE;
			FailIdx = 1;

			// engine can fail anytime during its max 70 sec burn
			FailTime = DRand(0, 70);
		}

		else
		{
			FailType = CONTROL;
			FailIdx = 2;

			// control can fail anytime during max 70 sec burn
			FailTime = DRand(0, 70);
		}
	}
}


bool V2i::FailCondition()
{
	if (FailType == BREAKUP)
		return ConditionBreakup();

	else if (FailType == ENGINE)
		return ConditionEngine();

	else 
		return ConditionControls();
}


bool V2i::ConditionBreakup()
{
	return GetDynPressure() > FailDynPressure;
}


bool V2i::ConditionEngine()
{
	return bPoweredFlight && (met > FailTime);
}


bool V2i::ConditionControls()
{
	return bPoweredFlight && (met > FailTime);
}


// write strng comment on failure
void V2i::WriteFailComment()
{
	sprintf(FailureString, "%s %s (%.0f) at %.0f sec.", sFailures[FailIdx], sSubFailuresBreakup[SubFailIdx], NumFailParam, met);
}


void V2i::FailAction()
{
	if (FailType == BREAKUP)
		ActionBreakup();

	else if (FailType == ENGINE)
		ActionEngine();

	else 
		ActionControls();
}


void V2i::ActionBreakup()
{
	WriteFailComment();
	bFail = false;
	Explode(false);
}


void V2i::ActionEngine()
{
	bFail = false;

	// engine explosion
	if (DRand(0, 30) < 10)
	{
		SubFailIdx = 2;
		WriteFailComment();
		Explode(false);
	}

	// premature engine shutdown
	else if (DRand(0, 30) > 20)
	{
		SubFailIdx = 3;
		EngineShutdown();
		WriteFailComment();
	}

	// engine low thrust
	else
	{
		SubFailIdx = 4;
		NumFailParam = DRand(0, 1);
		SetThrusterLevel(MainEngine, NumFailParam);
		NumFailParam *=100;
		WriteFailComment();
	}
}


void V2i::ActionControls()
{
	bFail = false;

	// yaw channel fail
	if (DRand(0, 30) < 10)
	{
		YawFixed = DRand(-180, 360);
		bYawFailed = true;
		SubFailIdx = 5;
		NumFailParam = YawFixed;
	}

	// roll channel fail
	else if (DRand(0, 30) > 20)
	{
		RollFixed = DRand(-180, 360);
		bRollFailed = true;
		SubFailIdx = 6;
		NumFailParam = RollFixed;
	}

	// pitch channel fail
	else
	{
		PitchFixed = DRand(0, 90);
		bPitchFailed = true;
		SubFailIdx = 7;
		NumFailParam = PitchFixed;
	}

	WriteFailComment();
}



// called from timestep, but only once a second
void V2i::WriteGpsTrackPoint()
{
	metGps = met + 1;

	GetEquPos(Longitudes[gpsIndex], Latitudes[gpsIndex], Elevations[gpsIndex]);
	Elevations[gpsIndex] -= Radius0;

	if (bShowingContrail)
		TrackStages[gpsIndex] = 2;

	else if (bPoweredFlight)
		TrackStages[gpsIndex] = 1;

	else
		TrackStages[gpsIndex] = 3;

	gpsIndex++;
}


void V2i::DebugSomething()
{
//	Explode(false);
//	sprintf(oapiDebugString(), "bFail: %d, FailType: %d, FailTime: %f", bFail, FailType, FailTime);
	sprintf(oapiDebugString(), "Dtn: %f", GetDynPressure());
}


DLLCLBK VESSEL *ovcInit (OBJHANDLE hVessel, int flightmodel)
{
	srand(time(NULL));
	return new V2i (hVessel,flightmodel);
}


DLLCLBK void ovcExit (VESSEL *vessel)
{
	if (vessel) delete (V2i*)vessel;
}

