#define ORBITER_MODULE


#include "wreck.h"
#include "OrbiterSoundSDK35.h"
#include "stdio.h"
#include "math.h"


// constructor
R7_wreck::R7_wreck (OBJHANDLE hObj, int fmodel): VESSEL2 (hObj, fmodel)
{
	// marker for cross-vessel signature
	VesselSignature = 'W';		

	// focus defaults: allow by default, unless Vessel overrides it
	bFocusOffByDefault = false;
	bFocus = true;
	bPermanentlyHidden = false;

	// mesh
	MeshIdx = 0;				// most generic wreck mesh

	// important "scenario vs. runtime" flag
	bJustDeployed = true;		// run-rime creation is more likely than the scenario loading

	// important ot properly process landing
	bFlying = true;				// in-flight burn-up is more likely than ground crash
	
	// video effects off by default
	bExplode = false;
	BlastLevel = 0;
	SmokeLevel = 0;
	DustLevel = 0;
	Timer = -1;
	Pressure = 0;		//+--

	// sound
	SoundLibID = -1;

	// outcome
	bShowOutcome = false;
	OutcomeTimer = -1;
	sprintf(Outcome.OutcomeMessage, "");
}



// called when reading the scenario file - this does not happen on runtime creation
void R7_wreck::clbkLoadStateEx (FILEHANDLE scn, void *status)
{
    char *line;

	while (oapiReadScenario_nextline (scn, line)) 
		ParseScenarioLineEx(line, status);

	// reset runtime flag
	bJustDeployed = false;

	// additional post-load configuration here
	FinishConfiguration();
}


void R7_wreck::ParseScenarioLineEx(char *line, void *status)
{
	double x, y, z;
	double Size, Mass;

	// mesh
	if (!_strnicmp (line, "MESH_IDX", 8))
		sscanf (line+8, "%d", &MeshIdx);

	// default focus?
	else if (!_strnicmp (line, "CANNOTFOCUS", 11))
	{
		bFocusOffByDefault = true;
		bFocus = false;
	}

	// non-default focus?
	else if (!_strnicmp (line, "FOCUS", 5))
		bFocus = true;

		// physics, only when flying in atmoshpere

	else if (!_strnicmp (line, "SIZE", 4))
	{
		sscanf (line+4, "%lf", &Size);
		SetSize(Size);
	}

	else if (!_strnicmp (line, "MASS", 4))
	{
		sscanf (line+4, "%lf", &Mass);
		SetEmptyMass(Mass);
	}

	else if (!_strnicmp (line, "CROSSSEC", 8))
	{
		sscanf (line+8, "%lf %lf %lf", &x, &y, &z);
		SetCrossSections(_V(x, y, z));
	}

	// default Orbiter parser, but intercept landing status
	else 
	{
		// are we landed? 
		if (!_strnicmp (line, "STATUS", 6))
		{
			if (!_strnicmp (line+7, "Landed", 6))
			bFlying = false;
		}

		// run default parser
		VESSEL2::ParseScenarioLineEx (line, status);
	}
}




// Save status to scenario file
void R7_wreck::clbkSaveState (FILEHANDLE scn)
{
	// save default vessel parameters
	SaveDefaultState (scn);

	// visual separator
	oapiWriteScenario_string (scn, "=========== WRECK vars", "");

	// save custom parameters
	if (MeshIdx != 0)
		oapiWriteScenario_int (scn, "MESH_IDX", MeshIdx);

	if (bFocusOffByDefault)
		oapiWriteScenario_string (scn, "CANNOTFOCUS", "");

	// only store focus in scenario if different from default
	if (bFocusOffByDefault == bFocus)
			oapiWriteScenario_int (scn, "FOCUS", bFocus);

	// if still flying, save physics
	if (bFlying)
	{
		oapiWriteScenario_float (scn, "SIZE", GetSize());
		oapiWriteScenario_float (scn, "MASS", GetEmptyMass());
		VECTOR3 cs;
		GetCrossSections(cs);
		oapiWriteScenario_vec (scn, "CROSSSEC", cs);
	}
}


// this function runs both for scenario and runtime creation
void R7_wreck::FinishConfiguration()
{	
	// set focus (or not)
	SetEnableFocus(bFocus);

	// set touchdown points
	SetTouchdownPoints (_V( 1, 0, 0), _V( -1, 0, 1), _V(-1, 0, -1));

	// load mesh
	char MeshName[256];
	sprintf(MeshName, "%s%.2d", "r7_S\\r7_wreck", MeshIdx);
	AddMesh (oapiLoadMeshGlobal(MeshName));

	// getconfiguration if runtime-created
	if (bJustDeployed)
	{
		VESSELSTATUS vs;
		memset(&vs, 0, sizeof(vs)); // Orbiter bug workaround
		GetStatus(vs);

		// still flying? loaded gets it from scenario
		bFlying = (vs.status != 1);
		
		// have atmosphere?
		oapiGetAtmPressureDensity(GetHandle(), &Pressure, &Density);
	}

	// for loaded, set timer to 0.5 sec to re-read the atmosphere and restart smoke, if needed
	else
		Timer = 0.5;

	// start explosion only if explicitly commanded during runtime creation
	if (bExplode)
		StartBlast();

	// start dust cloud if created at runtime on the ground
	if (bJustDeployed && !bFlying)
		StartDust();

	// start smoke if threre is atmosphere
	if (Pressure > PRESSURE_SMOKE)
		StartSmoke();

	// finally, reset the flag, so that configuration runs only once
	bJustDeployed = false;
}


// duplicating some interface functionality from VESSEL2M
int R7_wreck::clbkConsumeBufferedKey(DWORD key, bool down, char *kstate)
{
	// only process keydown events
	if (!down) 
		return 0; 

	// Control combinations
	if (KEYMOD_CONTROL (kstate))
	{
		// toggle focus on secondary items
		if (key == OAPI_KEY_F) 
			ToggleEnableFocus();

		return 0; 
	}

	// shift combinations 
	if (KEYMOD_SHIFT (kstate)) 
	{
		// Shift+TAB - switch focus to PREVIOUS vessel in the list
		if (key == OAPI_KEY_TAB) 
		{
			SwitchFocus(FOCUS_PREVIOUS);
			return 0;
		}

		return 0; 
	}

	// TAB - switch focus to NEXT vessel in the list
	if (key == OAPI_KEY_TAB) 
	{
		SwitchFocus(FOCUS_NEXT);
		return 0;
	}

	return 0;
}

// simple, fast, short
void R7_wreck::clbkPostStep(double simt, double simdt, double mjd)
{
	// Now that we know additional details, we can finalize the configuration and physics
	// to differentiate between landed or flying states.
	// This only needs to run once, before the very first time step.
	if (bJustDeployed)
		FinishConfiguration();

	// timer may be activated for video effects: blast, smoke, dust, etc.
	if (Timer > 0)
	{
		Timer -= simdt;

		if (Timer <= 0)
			RunTimerEvent();
	}

	// landing? Land!
	if (bFlying)
		if (GroundContact())
			Land();

	if (bExplode)
		if (BlastLevel > 0)
			if (SoundLibID != -1)
			{
				bExplode = false;
//				StopVesselWave3(SoundLibID, SND_BLAST_LONG);
				PlayVesselWave3(SoundLibID, SND_BLAST_LONG);
			}

	if (bShowOutcome)
		if ((OutcomeTimer-=oapiGetSimStep()) <= 0)	
		{
			bShowOutcome = false;
			sprintf(oapiDebugString(), Outcome.OutcomeMessage);
		}

// debug output
//	if (GetHandle() == oapiGetFocusObject())
//		sprintf(oapiDebugString(), "%f", Pressure);
}


void R7_wreck::RunTimerEvent()
{
	if (BlastLevel > 0)
		StopBlast();

	else if (DustLevel > 0)
		StopDust();

	// check for atmoshpere
	else
		UpdateSmokeState();
}

// run this every second to know if we have atmoshpere around
void R7_wreck::UpdateSmokeState()
{
	// update atmosphere vars
	oapiGetAtmPressureDensity(GetHandle(), &Pressure, &Density);
	
	// smoking when pressure is not sufficient? stop!
	if ((Pressure < PRESSURE_SMOKE) && (SmokeLevel > 0))
		StopSmoke();

	// not smoking when pressure is sufficient? start!
	else if ((Pressure > PRESSURE_SMOKE) && (SmokeLevel == 0))
		StartSmoke();

	// reset timer for the next check, but only if not landed
	if (bFlying)
		Timer = 5;
}


void R7_wreck::Land()
{
	// force landing by updating vessel status
	double Longitude, Latitude, Radius;
	VESSELSTATUS vs;
	memset(&vs, 0, sizeof(vs)); // Orbiter bug workaround
	GetStatus(vs);

	GetEquPos(Longitude, Latitude, Radius);

	vs.status = 1;
	vs.vdata[0].x = Longitude;
	vs.vdata[0].y = Latitude; 
	oapiGetHeading(GetHandle(), &vs.vdata[0].z);

	DefSetState(&vs);

	// reset fly/land flag
	bFlying = false;

	// reconfigure smoke
	StopSmoke();
	StartSmoke();
}


void R7_wreck::StartBlast()
{
	// create a very quickly expanding white "fireball"
	PARTICLESTREAMSPEC BlastFire = {
		0, 
		0.1, 
		4, 
		0, 
		0, 
		2, 
		100, 
		0, 
		PARTICLESTREAMSPEC::EMISSIVE,
		PARTICLESTREAMSPEC::LVL_FLAT, 
		1, 
		1,
		PARTICLESTREAMSPEC::ATM_FLAT, 
		1, 
		1
	};

	AddParticleStream (&BlastFire, _V(0,0,0), _V(0,1,0), &BlastLevel);
	BlastLevel = 1;

	// set timer to 1-second blast
	Timer = 1;
}



void R7_wreck::StopBlast()
{
	// kill blast particle stream
	BlastLevel = 0;

	// reset timer for smoke
	Timer = 5;
}


void R7_wreck::StartSmoke()
{
	// smoke should be running only in atmosphere or on Earth surface, not on the Moon or in space!
	oapiGetAtmPressureDensity(GetHandle(), &Pressure, &Density);
	if (Pressure < PRESSURE_SMOKE)
		return;

	// create smoke stream
	PARTICLESTREAMSPEC BlastSmoke = {
		0,			// flags, reserved
		0.5,		// srcsize, particle size at creation, m
		10,			// srcrate, average particle generation rate, Hz   
		0,			// V0, average particle emission velocity, m/s
		0,			// srcspread, emission velocity distribution randomisation
		2,			// lifetime, average particle lifetime [s]
		10,			// growthrate, particle growth rate [m/s]
		5,			// atmslowdown, deceleration rate b in atmosphere, defined as v = v0 e- bdt 
		PARTICLESTREAMSPEC::DIFFUSE,
		PARTICLESTREAMSPEC::LVL_PSQRT, 
		0, 
		0.5,
		PARTICLESTREAMSPEC::ATM_FLAT, 
		1.22, 
		1.25,
	};

	AddParticleStream (&BlastSmoke, _V(0,0,0), _V(0,1,0), &SmokeLevel);

	// add small fire
	SURFHANDLE FlameTex = oapiRegisterParticleTexture("r7_S\\TrenchFlame");
	PARTICLESTREAMSPEC FlameStream = { 0,			// flags, reserved
		1,			// srcsize, particle size at creation, m
		10,			// srcrate, average particle generation rate, Hz   
		10,			// V0, average particle emission velocity, m/s
		0.01,			// srcspread, emission velocity distribution randomisation
		0.1,			// lifetime, average particle lifetime [s]
		5,				// growthrate, particle growth rate [m/s]
		0.1,			// atmslowdown, deceleration rate b in atmosphere, defined as v = v0 e- bdt 
		PARTICLESTREAMSPEC::EMISSIVE, PARTICLESTREAMSPEC::LVL_PSQRT, -1, 0.5, PARTICLESTREAMSPEC::ATM_PLIN, -1, 0.5, 
		FlameTex };	
	AddParticleStream (&FlameStream, _V(0,0,0), _V(0,1,0), &SmokeLevel);


	// full smoke works for inflight situations
	SmokeLevel = 1;

	// when landed, throttle smoke down
	if (!bFlying)
		SmokeLevel = 0.001;
}


void R7_wreck::StopSmoke()
{
	SmokeLevel = 0;
}


void R7_wreck::StartDust()
{
	PARTICLESTREAMSPEC Dust = {
		0,			// flags, reserved
		0.1,			// srcsize, particle size at creation, m
		20,			// srcrate, average particle generation rate, Hz   
		1,			// V0, average particle emission velocity, m/s
		0,			// srcspread, emission velocity distribution randomisation
		4,			// lifetime, average particle lifetime [s]
		3,			// growthrate, particle growth rate [m/s]
		5,			// atmslowdown, deceleration rate b in atmosphere, defined as v = v0 e- bdt 
		PARTICLESTREAMSPEC::DIFFUSE,
		PARTICLESTREAMSPEC::LVL_PSQRT, 
		0, 
		0.5,
		PARTICLESTREAMSPEC::ATM_FLAT, 
		1.22, 
		1.25
	};

	AddParticleStream (&Dust, _V(0,0,0), _V(0,1,0), &DustLevel);
	DustLevel = 1;

	Timer = 0.2;
}



void R7_wreck::StopDust()
{
	// stop dust particle stream
	DustLevel = 0;

	// reset timer for smoke
	Timer = 5;
}


void R7_wreck::SwitchFocus(focus_Dirs FocusDir)
{
	// get total count of vessels in the scenario, should be more than one to continue.
	int VesCnt = oapiGetVesselCount();
	if (VesCnt == 1) 
		return;

	// find our own index in the list of vessles
	int MyIdx = -1;
	for (int i = 0; i < VesCnt; i++)
	{
		// get next vessel in the list
		OBJHANDLE hVes = oapiGetVesselByIndex(i);

		// found ourselves!
		if (hVes == GetHandle())
		{
			MyIdx = i;
			break;
		}
	}

	// find next
	bool bSearching = true;
	OBJHANDLE hVes = NULL;
	while (bSearching)
	{
		// move index ahead one at a time, wrapping around if needed

		// previous 
		if (FocusDir == FOCUS_PREVIOUS)
		{
			MyIdx--;
			if (MyIdx < 0)
				MyIdx = VesCnt-1;
		}

		// next
		else
		{
			MyIdx++;
			if (MyIdx >= VesCnt)
				MyIdx = 0;
		}

		// get vessesl at the new index, skip if it looks wrong
		hVes = oapiGetVesselByIndex(MyIdx);
		if (hVes == NULL)
			continue;

		// get vessel's interface
		VESSEL *Intf;
		Intf = oapiGetVesselInterface(hVes);

		// skip vessels with disabled focus
		if (!Intf->GetEnableFocus())
			continue;

		// skip vessels that are neither VESSEL2M nor Wreck
		if ((!IsVessel2M(Intf)) && (!IsWreck(Intf)))
			continue;

		// if we got here, we have a good find!
		bSearching = false;
	}

	// switch focus to the new vessel
	oapiSetFocusObject(hVes);
}


void R7_wreck::ToggleEnableFocus()
{
	// that is, we are enabling rather than disabling...
	bool bEnableFocuses = true;

	// get total count of vessels in the scenario
	int VesCnt = oapiGetVesselCount();

	// go through the list of the vessles
	for (int i = 0; i < VesCnt; i++)
	{
		// get next vessel in the list
		OBJHANDLE hVes = oapiGetVesselByIndex(i);

		// get vessel's interface
		VESSEL *Intf;
		Intf = oapiGetVesselInterface(hVes);

		// skip vessels that are neither VESSEL2M nor Wreck
		if ((!IsVessel2M(Intf)) && (!IsWreck(Intf)))
			continue;

		// if default focus is off, we simply TOGGLE the currently set focus
		if ( ((R7_wreck*)Intf)->bFocusOffByDefault )
		{
			((R7_wreck*)Intf)->bFocus = !((R7_wreck*)Intf)->bFocus;
			Intf->SetEnableFocus(((R7_wreck*)Intf)->bFocus);

			// either keep the default 'enabling' flag or overriding it with the 'disabling';
			bEnableFocuses = ((R7_wreck*)Intf)->bFocus;
		}
	}

	// show screen message
	if (bEnableFocuses)
		sprintf(oapiDebugString(), "Focus is ENABLED on all secondary items in the scenario.");
	else
		sprintf(oapiDebugString(), "Focus is DISABLED on all secondary items in the scenario.");
}


void R7_wreck::ShowMissionOutcome()
{
	;
}



void R7_wreck::clbkPostCreation (void)
{
	SoundLibID = ConnectToOrbiterSoundDLL3(GetHandle());

	SoundOptionOnOff3(SoundLibID, PLAYMAINTHRUST, true);
	SoundOptionOnOff3(SoundLibID, PLAYATTITUDETHRUST, true);
	SoundOptionOnOff3(SoundLibID, PLAYLANDINGANDGROUNDSOUND, true);
	SoundOptionOnOff3(SoundLibID, PLAYWINDAIRSPEED, true);
	SoundOptionOnOff3(SoundLibID, PLAYWINDAMBIANCEWHENLANDED, true);

	SoundOptionOnOff3(SoundLibID, PLAYCOUNTDOWNWHENTAKEOFF, false);
	SoundOptionOnOff3(SoundLibID, PLAYWHENATTITUDEMODECHANGE, false);
	SoundOptionOnOff3(SoundLibID, PLAYGPWS, false);
	SoundOptionOnOff3(SoundLibID, PLAYHOVERTHRUST, false);
	SoundOptionOnOff3(SoundLibID, PLAYDOCKINGSOUND, false);
	SoundOptionOnOff3(SoundLibID, PLAYRADARBIP, false);
	SoundOptionOnOff3(SoundLibID, PLAYDOCKLANDCLEARANCE, false);
	SoundOptionOnOff3(SoundLibID, PLAYCABINAIRCONDITIONING, false);
	SoundOptionOnOff3(SoundLibID, PLAYCABINRANDOMAMBIANCE, false);
	SoundOptionOnOff3(SoundLibID, PLAYRADIOATC, false);
	SoundOptionOnOff3(SoundLibID, DISPLAYTIMER, false);

	RequestLoadVesselWave3(SoundLibID, SND_BLAST_LONG, "Sound\\r7_S\\BlastLong.wav", BOTHVIEW_FADED_FAR);
	RequestLoadVesselWave3(SoundLibID, SND_BLAST_MEDIUM, "Sound\\r7_S\\BlastMedium.wav", BOTHVIEW_FADED_FAR);
}


// Vessel dll callbacks

DLLCLBK VESSEL *ovcInit (OBJHANDLE hVessel, int flightmodel)
{
	return new R7_wreck (hVessel,flightmodel);
}

DLLCLBK void ovcExit (VESSEL *vessel)
{
	if (vessel) 
		delete (R7_wreck*)vessel;
}









