#include <stdio.h>
#include <time.h>
#include "vessel2m.h"




VESSEL2M::VESSEL2M (OBJHANDLE hObj, int fmodel): VESSEL2 (hObj, fmodel)
{
	// marker for cross-vessel signature
	VesselSignature = 'M';

	// focus defaults: allow by default, unless Vessel overrides it
	bFocusOffByDefault = false;
	bFocus = true;
	bPermanentlyHidden = false;

	// set default structural limits, in case derived object forgets to do so...
	VvCrash = 1; 
	VhCrash = 1;
	VMaxBlast = 1000; 
	QMax = 20000;

	// set default wrecks, can be overridden by derived objects
	WreckSingleIdx = 5;				// default to genereric shape
	WreckCount = 1;					// default to one wreck
	for (int i=0; i<10; i++)
		WreckIdxs[i] = 5;			// default to genereric shape

	// by default, we are not attached 
	bAttached = false;
	
	// this will be overriden by the vessel if needed
	EmptyMassWithoutAttachments = 0;

	// seed random generator
	srand(int(time(NULL)));

	// generate a small cloud of dust at touchdown
	PARTICLESTREAMSPEC Dust = {
		0,			// flags, reserved
		0.1,			// srcsize, particle size at creation, m
		20,			// srcrate, average particle generation rate, Hz   
		1,			// V0, average particle emission velocity, m/s
		0,			// srcspread, emission velocity distribution randomisation
		4,			// lifetime, average particle lifetime [s]
		3,			// growthrate, particle growth rate [m/s]
		5,			// atmslowdown, deceleration rate b in atmosphere, defined as v = v0 e- bdt 
		PARTICLESTREAMSPEC::DIFFUSE,
		PARTICLESTREAMSPEC::LVL_PSQRT, 
		0, 
		0.5,
		PARTICLESTREAMSPEC::ATM_FLAT, 
		1.22, 
		1.25
	};
	bGeneratingDust = false;
	bLandedPermanently = false;
	DustTimer = -1;
	DustLevel = 0;
	AddParticleStream (&Dust, _V(0,0,0), _V(0,1,0), &DustLevel);

	// annotation window
	MessagePort = oapiCreateAnnotation ( true, 0.5, _V(1, 1, 1) );
	oapiAnnotationSetPos (MessagePort, 0, 0.1, 1, 1);
	msgStartTime = 0;
	bMessageDisplayed = false;

	// kml
	kmlHandle = NULL;
	kmlPlotColor = KML_CLR_GREEN;
	kmlNoPlotting = true;

	// cameras
	CameraID = 0;
	CamerasCount = 1;
	SetCameraData();

	// no self-destructor by default
	bHaveSelfDestructor = false;
	bDelayedExplode = false;
	DelayedExplodeTimer = -1;

	// sound
	SoundLibID = -1;

	// mission outcome
	bOutcomeCriteriaSet = false;
	OutcomeTimer = -1;
	OutcomeCriteria = OC_FAIL;

	// failures
	bFail = false;
	Reliability = 100;
	failTimer = -1;
	Fire.FireLevel = 0;

	// debug
	CockpitCameraTilt = 0;
	bDebugInternalCameraMove = false;
}

// destructor
VESSEL2M::~VESSEL2M ()
{
	// clean up output
	oapiAnnotationSetText (MessagePort, "");
	sprintf(oapiDebugString(), "");
}


void VESSEL2M::clbkSaveState (FILEHANDLE scn)
{
	WriteLastKmlEntry();

	// attached: save only some defaults from VESSEL2
	if (bAttached)
	{
		WriteStatusToScenario(scn);
		WritePropellantsToScenario(scn);
	}
	// not attached: save all defaults from VESSEL2
	else
		SaveDefaultState (scn);

	// visual separator
	oapiWriteScenario_string (scn, "=========== VESSEL2M vars", "");

	// focus enabled?
	bFocus = GetEnableFocus();

	// only store focus in scenario if different from default
	if (bFocusOffByDefault == bFocus)
			oapiWriteScenario_int (scn, "FOCUS", bFocus);

	// non-default camera?
	if (CameraID != 0) 
		oapiWriteScenario_int (scn, "CAMERA", CameraID);

	// failure flag(s)
	if (bFail) 
		oapiWriteScenario_int (scn, "FLAGS_MI", bFail);

	// failure timer
	if (failTimer > 0) 
		oapiWriteScenario_float (scn, "FLAGS_MD_1", failTimer);

	// fire params
	if (Fire.FireLevel > 0) 
	{
		oapiWriteScenario_float (scn, "FLAGS_MD_2", Fire.FireLevel);
		oapiWriteScenario_vec (scn, "FLAGS_MD_3", _V(Fire.tFireUp, Fire.tFireSteady, Fire.tThrustDown));
	}
}


void VESSEL2M::clbkLoadStateEx (FILEHANDLE scn, void *status)
{
    char *line;

	while (oapiReadScenario_nextline (scn, line)) 
		ParseScenarioLineEx(line, status);

	// call this when we loaded everything from the scenario 
	FinishConfiguration();
}


void VESSEL2M::ParseScenarioLineEx(char *line, void *status)
{
	// focus
	if (!_strnicmp (line, "FOCUS", 5))
		bFocus = true;

	else if (!strnicmp (line, "CAMERA", 6))
		sscanf (line+6, "%d", &CameraID);

	else if (!strnicmp (line, "RELIABILITY", 11))
		sscanf (line+11, "%lf", &Reliability);

	else if (!strnicmp (line, "FLAGS_MI", 8))
		sscanf (line+8, "%d", &bFail);

	else if (!strnicmp (line, "FLAGS_MD_1", 10))
		sscanf (line+10, "%lf", &failTimer);

	else if (!strnicmp (line, "FLAGS_MD_2", 10))
		sscanf (line+10, "%lf", &Fire.FireLevel);

	else if (!strnicmp (line, "FLAGS_MD_3", 10))
		sscanf (line+10, "%lf %lf %lf", &Fire.tFireUp, &Fire.tFireSteady, &Fire.tThrustDown);

	// default Orbiter processing, but intercept STATUS and ATTACHED 
	else 
	{
		// are we landed? we want to know to avoid dusting
		if (!_strnicmp (line, "STATUS", 6))
		{
			if (!_strnicmp (line+7, "Landed", 6))
				bLandedPermanently = true;
		}

		// are we attached to parent?
		else if (!_strnicmp (line, "ATTACHED", 8))
			bAttached = true;

		//pass to Orbiter's generic parser
		VESSEL2::ParseScenarioLineEx (line, status);
	}
}



void VESSEL2M::FinishConfiguration()
{
	SetEnableFocus(bFocus);

	if (bAttached)
		bLandedPermanently = false;
	
	// enable kml plotting for freeflying vessels
	if ((!bLandedPermanently) && (!bAttached))
		kmlNoPlotting = false;

	// initialize failures, if any
	failInitialize();

	// restore fire, if any
	if (Fire.FireLevel > 0)
		failStartFireVisual();
}



void VESSEL2M::clbkPostCreation (void)
{
	// set up sound
	SoundLibID = ConnectToOrbiterSoundDLL3(GetHandle());

	SoundOptionOnOff3(SoundLibID, PLAYMAINTHRUST, true);
	SoundOptionOnOff3(SoundLibID, PLAYATTITUDETHRUST, true);
	SoundOptionOnOff3(SoundLibID, PLAYLANDINGANDGROUNDSOUND, true);
	SoundOptionOnOff3(SoundLibID, PLAYWINDAIRSPEED, true);
	SoundOptionOnOff3(SoundLibID, PLAYWINDAMBIANCEWHENLANDED, true);

	SoundOptionOnOff3(SoundLibID, PLAYCOUNTDOWNWHENTAKEOFF, false);
	SoundOptionOnOff3(SoundLibID, PLAYWHENATTITUDEMODECHANGE, false);
	SoundOptionOnOff3(SoundLibID, PLAYGPWS, false);
	SoundOptionOnOff3(SoundLibID, PLAYHOVERTHRUST, false);
	SoundOptionOnOff3(SoundLibID, PLAYDOCKINGSOUND, false);
	SoundOptionOnOff3(SoundLibID, PLAYRADARBIP, false);
	SoundOptionOnOff3(SoundLibID, PLAYDOCKLANDCLEARANCE, false);
	SoundOptionOnOff3(SoundLibID, PLAYCABINAIRCONDITIONING, false);
	SoundOptionOnOff3(SoundLibID, PLAYCABINRANDOMAMBIANCE, false);
	SoundOptionOnOff3(SoundLibID, PLAYRADIOATC, false);
	SoundOptionOnOff3(SoundLibID, DISPLAYTIMER, false);

	RequestLoadVesselWave3(SoundLibID, SND_PYRO, "Sound\\r7_S\\Pyro.wav", BOTHVIEW_FADED_FAR);
	RequestLoadVesselWave3(SoundLibID, SND_GAS, "Sound\\r7_S\\Gas.wav", BOTHVIEW_FADED_FAR);

	RequestLoadVesselWave3(SoundLibID, SND_RING_MOVE, "Sound\\r7_S\\RingMove.wav", BOTHVIEW_FADED_FAR);
	RequestLoadVesselWave3(SoundLibID, SND_BOOMS_MOVE, "Sound\\r7_S\\BoomsMove.wav", BOTHVIEW_FADED_FAR);
	RequestLoadVesselWave3(SoundLibID, SND_SUPPORTS_MOVE, "Sound\\r7_S\\SupportsMove.wav", BOTHVIEW_FADED_FAR);
	RequestLoadVesselWave3(SoundLibID, SND_PLATFORM_MOVE, "Sound\\r7_S\\BoomsMove.wav", BOTHVIEW_FADED_FAR);
	RequestLoadVesselWave3(SoundLibID, SND_CABLE_MAST_MOVE, "Sound\\r7_S\\CableMastMove.wav", BOTHVIEW_FADED_FAR);
	RequestLoadVesselWave3(SoundLibID, SND_CLANG_LIGHT, "Sound\\r7_S\\ClangLight.wav", BOTHVIEW_FADED_FAR);
	RequestLoadVesselWave3(SoundLibID, SND_CLANG_HEAVY, "Sound\\r7_S\\ClangHeavy.wav", BOTHVIEW_FADED_FAR);

	RequestLoadVesselWave3(SoundLibID, SND_BIP, "Sound\\r7_S\\Bip.wav", BOTHVIEW_FADED_FAR);
	RequestLoadVesselWave3(SoundLibID, SND_RADIO, "Sound\\r7_S\\Radio.wav", BOTHVIEW_FADED_FAR);
	RequestLoadVesselWave3(SoundLibID, SND_DOG, "Sound\\r7_S\\Dog.wav", BOTHVIEW_FADED_FAR);

	RequestLoadVesselWave3(SoundLibID, SND_CLANG, "Sound\\r7_VL\\Clang.wav", BOTHVIEW_FADED_FAR);

	RequestLoadVesselWave3(SoundLibID, SND_BLAST1, "Sound\\r7_S\\Blast1.wav", DEFAULT);
	RequestLoadVesselWave3(SoundLibID, SND_BLAST2, "Sound\\r7_S\\Blast2.wav", DEFAULT);

	RequestLoadVesselWave3(SoundLibID, SND_ENGINE, "Sound\\Vessel\\mainextloud.wav", BOTHVIEW_FADED_FAR);
	RequestLoadVesselWave3(SoundLibID, SND_ALARM, "Sound\\r7_S\\Alarm.wav", BOTHVIEW_FADED_FAR);

	RequestLoadVesselWave3(SoundLibID, SND_BLAST_LONG, "Sound\\r7_S\\BlastLong.wav", BOTHVIEW_FADED_FAR);
	RequestLoadVesselWave3(SoundLibID, SND_BLAST_MEDIUM, "Sound\\r7_S\\BlastMedium.wav", BOTHVIEW_FADED_FAR);

	RequestLoadVesselWave3(SoundLibID, SND_FUELING_MAST_MOVE, "Sound\\r7_S\\CableMastMove.wav", BOTHVIEW_FADED_FAR);

	// register ourselves with kml writer, if any
	kmlHandle = (OBJHANDLE)BroadcastVesselMessage(VM_KML_REGISTER, (int)GetHandle());
}


void VESSEL2M::SendVesselWaveToFocused(int SND_ID, int SND_MODE, int Volume)
{
	if (oapiGetFocusObject() == NULL)
		return;

	// get focused vessel interface
	VESSEL* FocusedVesselIntf = oapiGetFocusInterface();

	// pass on it if not VESSEL2M or Wreck
	if (!IsVessel2M(FocusedVesselIntf) && !IsWreck(FocusedVesselIntf))
		return;

	// get SoundLibID of the focused vessel
	VESSEL2M * FocusedVesselIntf2M = (VESSEL2M*)FocusedVesselIntf;
	int SoundLibID = FocusedVesselIntf2M->SoundLibID;
	if (SoundLibID == -1)
		return;


	// check if the focused vessel is too far from the sound source 
	VECTOR3 Pos;
	GetRelativePos(FocusedVesselIntf2M->GetHandle(), Pos);
	double Dist = length(Pos);
	if (Dist > 3300)
		return;

	// limit fade the sound by the distance till 1 km
	double dVol = (1-Dist/3000);
	Volume = (int)(Volume * dVol);

	// play or stop sound on the focused vessel
	if (SND_MODE == SND_STOP)
		StopVesselWave3(SoundLibID, SND_ID);
	else
	{
		if (SND_ID == SND_BLAST_LONG)
		{
			if (IsPlaying3(SoundLibID, SND_ID))
				StopVesselWave3(SoundLibID, SND_ID);
		}
	
		PlayVesselWave3(SoundLibID, SND_ID, SND_MODE, Volume);
	}
}


int VESSEL2M::clbkConsumeBufferedKey(DWORD key, bool down, char *kstate)
{
	// only process keydown events
	if (!down) 
		return 0; 

	// control combinations are reserved
	if (KEYMOD_CONTROL (kstate))
	{
		// toggle focus on secondary items
		if (key == OAPI_KEY_F) 
		{
			ToggleEnableFocus();
			return 1; 
		}

		// self-destruct?
		if (key == OAPI_KEY_D) 
		{
			if (bHaveSelfDestructor)
				Break();

			return 1; 
		}

		return 0; 
	}

	// shift combinations!
	if (KEYMOD_SHIFT (kstate)) 
	{
		// Shift+C - debug cockpit camera placement
		if (key == OAPI_KEY_C) 
		{
			// toggle debug flag
			bDebugInternalCameraMove = !bDebugInternalCameraMove;

			// save original cockpit settings
			if (bDebugInternalCameraMove)
			{
				GetCameraOffset(CockpitCameraPos);
				GetCameraDefaultDirection(CockpitCameraDir);
				DebugCockpitCameraTilt = CockpitCameraTilt;
			}

			// restore original cockpit settings
			else
			{
				SetCameraOffset(CockpitCameraPos);
				SetCameraDefaultDirection(CockpitCameraDir, CockpitCameraTilt);
				sprintf(oapiDebugString(), "");
			}

			return 1;
		}

		// Shift+TAB - switch focus to PREVIOUS vessel in the list
		if (key == OAPI_KEY_TAB) 
		{
			SwitchFocus(FOCUS_PREVIOUS);
			return 1;
		}

		return 0;
	}

	// TAB - switch focus to NEXT vessel in the list
	if (key == OAPI_KEY_TAB) 
	{
		SwitchFocus(FOCUS_NEXT);
		return 1;
	}

	// cycle to next camera
	if (key == OAPI_KEY_C)
	{
		CycleOnboardCamera();
		return 1;
	}

	return 0;
}





// this NEVER gets ignored internally!
// To ignore VESSEL2 processing, just don't call it from the derived object.
void VESSEL2M::TimestepForVESSEL2M()
{
	// %%% this should be commented out for the release!
	//if (bDebugInternalCameraMove)
	//	DebugInternalCameraMove();

	// message display timer - always give it a chance to finish
	if (bMessageDisplayed)
	{
		if ( (oapiGetSysTime() - msgStartTime) > MSG_DISPLAY_TIME )
		{
			oapiAnnotationSetText (MessagePort, "");
			bMessageDisplayed = false;
		}
	}

	// failures
	if (bFail)
		failUpdate();

	// slightly delayed explosion because of something else exploding nearby
	if (bDelayedExplode)
		if ( (DelayedExplodeTimer -= oapiGetSimStep()) < 0 )
			Break();

	// don't do any further processing when still attached to parent
	if (bAttached)
		return;

	// don't do any further processing when still/already landed
	if (bLandedPermanently)
		return;

	// did we just touched down?
	if (GroundContact())
		TouchingTheGround();

	// aerodynamic breakup?
	if (GetDynPressure() > QMax)
		Burn();

	// generating dust? Thus goes for a short time after ground contact,
	// when sitting on the ground or flying after a bounce.
	// If by its end of dusting we are still grounded - landing will become permanent)
	if (bGeneratingDust)
		if ( (DustTimer -= oapiGetSimStep()) < 0 )
			StopDust();

	// show mission success or failure outcome
	if (bOutcomeCriteriaSet)
		if ((OutcomeTimer-=oapiGetSimStep()) <= 0)	
			ShowMissionOutcome(oapiGetFocusObject());
}


void VESSEL2M::ShowMissionOutcome(OBJHANDLE hVessel)
{
	VESSEL *VIntf;
	VIntf = oapiGetVesselInterface(hVessel);
	
	if (IsVessel2M(VIntf))
		SendVesselMessage(hVessel, VM_SHOW_OUTCOME, (int)(&Outcomes[OutcomeCriteria]));
	else if (IsWreck(VIntf))
	{
		R7_wreck *WIntf;
		WIntf = (R7_wreck*)VIntf;
		OUTCOME_OBJECT *pDest;
		pDest = &WIntf->Outcome;
		OUTCOME_OBJECT *pSrc;
		pSrc = &Outcomes[OutcomeCriteria];
		memcpy(pDest, pSrc, sizeof(OUTCOME_OBJECT));
		((R7_wreck*)VIntf)->bShowOutcome = true;
		((R7_wreck*)VIntf)->OutcomeTimer = 4;
	}

	bOutcomeCriteriaSet = false;
}


// this may be both the first contact and the subsequent sitting on the ground
void VESSEL2M::TouchingTheGround()
{
	// start short dust cloud 
	if (DustTimer <= 0) 
		StartDust();

	// get landing velocities
	VECTOR3 HAV; 
	GetHorizonAirspeedVector(HAV);
	double V = length(HAV);
	double Vv = fabs(HAV.y);
	double Vh = sqrt(HAV.x*HAV.x + HAV.z*HAV.z);

	// explosive high velocity impact?
	if (V > VMaxBlast) 
		Explode();

	// crash velocity impact?
	else if ( (Vv > VvCrash) || (Vh > VhCrash) )
		Break();

	// safe or at least non-destructive landiing
	else 
		Land();
}


// returns true if fuel mass is more than 10% of the remaining vessel weight
bool VESSEL2M::bFuelExplode()
{
	double FuelMass = 0;
	for (DWORD i=0; i<GetPropellantCount(); i++)
		FuelMass += GetPropellantMass(GetPropellantHandleByIndex(i));

	return ((FuelMass / GetMass()) > 0.1);
}


// explosive breakup, blast and smoke
void VESSEL2M::Explode()
{
	ReplaceWithWreck(EXPLODE);
}


// non-explosive breakup, smoking fragments
void VESSEL2M::Crash()
{
	ReplaceWithWreck(NO_EXPLODE);
}


// aerodynamic breakup
void VESSEL2M::Burn()
{
	Break();
}


// Depending on fuel amount, either breaks to pieces, or explodes to nothing.
// May be called externally (for mutual destruct) or internally (for self-destruct).
// Also, is used (called) by other automated destruction modes (burn, crash-land)
void VESSEL2M::Break()
{
	// if still attached, tell parent to detach ourselves in a graceful way!
	if (bAttached)
		SendVesselMessage(GetAttachmentStatus(GetAttachmentHandle(TOPARENT, 0)), VM_REQUEST_DETACH, (int)GetHandle());

	// if we have any non-VESSEL2M children, drop them. 
	// to gracefully drop VESSEL2M children, preempt this portion of code in the VESSEL2M parent 
	for (DWORD i = 0; i < AttachmentCount(TOCHILD); i++)
	{
		ATTACHMENTHANDLE hAttToChild = GetAttachmentHandle(TOCHILD, i);
		OBJHANDLE hChild = GetAttachmentStatus(hAttToChild);
		if (hChild != NULL)
			DetachChild(hAttToChild, 0);
	}

	// record kml event and stop plotting here.
	kmlAddEvent(KML_WRECK, GetName());
	SendVesselMessage(kmlHandle, VM_KML_STOP_PLOTTING, (int)GetHandle());

	// enough fuel to explode or have explosive charge?
	if (bFuelExplode() || bHaveSelfDestructor)
		Explode();

	// just crash?
	else
		Crash();
}


// do nothing here yet...
void VESSEL2M::Land()
{
	SendVesselMessage(kmlHandle, VM_KML_STOP_PLOTTING, (int)GetHandle());
}


void VESSEL2M::ReplaceWithWreck(bool bExplode)
{
	// 1. PREPARATION

	// do we have atmoshpere around? 
	double Pressure;
	double Density;
	oapiGetAtmPressureDensity(GetHandle(), &Pressure, &Density);

	// 2. BLAST EVERYTHING AROUND
	if (bExplode)
	{
		// play blast sound on focused vessel, in case we are not focused
		// (if we are, this won't work, as we are about to disappear, but then 
		// the other  mechanism should play the sound in the wreck).
		if (!bHaveFocus)
			SendVesselWaveToFocused(SND_BLAST_LONG);

		// delay-explode all exploadable vessels in the 200 m surrounding
		for (DWORD i=0; i<oapiGetVesselCount(); i++)
		{
			OBJHANDLE hV = oapiGetVesselByIndex(i);
			if (!oapiIsVessel(hV))
				continue;

			if (hV == GetHandle())
				continue;

			VECTOR3 Pos;
			GetRelativePos(hV, Pos);
			double Dist = length(Pos);

			if (Dist > 200)
				continue;

			SendVesselMessage(hV, VM_DELAY_EXPLODE);
		}
	}


	// 3. WRECK OBJECT CREATIONS

	// GroundContact: create only one wreck object
	if (GroundContact() && (WreckCount > 1))
	{
		WreckCount = 1;
		WreckIdxs[0] = WreckSingleIdx;
	}

	// if more than one wreck, and we have focus, select the new focused object
	int FocusedIdx = 0;
	OBJHANDLE hWreckToFocus = NULL;
	if (bHaveFocus)
		if (WreckCount > 1)
			FocusedIdx = (int)DRand(0, WreckCount);

	// create N wreck objects
	for (int i=0; i<WreckCount; i++)
	{
		// prepare a name for a wreck object 
		char Name[100];
		strcpy(Name, GetName());
		strcat(Name, "-Wreck");

		// if more than one wreck, add index to the name
		if (WreckCount > 1)
		{
			char sIdx[3];
			sprintf(sIdx, "-%.2d", i);
			strcat(Name, sIdx);
		}

		// clone wreck status from the current one
		VESSELSTATUS vs;
		memset(&vs, 0, sizeof(vs)); // Orbiter bug workaround
		GetStatus(vs);

		// for touchdown, force landing in the new status by directly updating vessel status
		if (GroundContact())
		{
			double Longitude, Latitude, Radius;
			GetEquPos(Longitude, Latitude, Radius);

			vs.status = 1;
			vs.vdata[0].x = Longitude;
			vs.vdata[0].y = Latitude;
			oapiGetHeading(GetHandle(), &vs.vdata[0].z);
		}

		// for flying debree, randomize flight parameters
		else
		{
			const double WRK_VMIN = -100; 
			const double WRK_VMAX = -WRK_VMIN; 
			const double WRK_VSPAN = WRK_VMAX - WRK_VMIN; 

			const double WRK_OMIN = -100*RAD; 
			const double WRK_OMAX = -WRK_OMIN; 
			const double WRK_OSPAN = WRK_OMAX - WRK_OMIN; 

			vs.vrot = _V(DRand(WRK_OMIN, WRK_OSPAN), DRand(WRK_OMIN, WRK_OSPAN), DRand(WRK_OMIN, WRK_OSPAN));
			vs.rvel += _V(DRand(WRK_VMIN, WRK_VSPAN), DRand(WRK_VMIN, WRK_VSPAN), DRand(WRK_VMIN, WRK_VSPAN));
		}

		// create a wreck object - this will run its constructor 
		void *hWreck;
		hWreck = oapiCreateVessel(Name, "r7_S\\Wreck", vs); 

		// get wreck object interface
		VESSEL *VesselIntf;
		R7_wreck* Intf;
		VesselIntf = oapiGetVesselInterface(hWreck);
		Intf = (R7_wreck*)VesselIntf;

		// WRECK OBJECT INITIALIZATION 
		// this is a runtime alternative to the loading object from scenario

			// blast? 
		if (bExplode)
			Intf->bExplode = bExplode;			

		// update and slightly modify physics based on the originating object (self)
		Intf->SetSize(GetSize());
		Intf->SetEmptyMass(GetEmptyMass() * 0.95); // decrease mass by 5 %
		VECTOR3 cs;
		GetCrossSections(cs);
		Intf->SetCrossSections(cs * 1.1); // increase cross-section by 10 %

		// set mesh 
		Intf->MeshIdx = WreckIdxs[i];

		// can focus wreck?
		Intf->bFocus = bFocus;
		Intf->bFocusOffByDefault = bFocusOffByDefault;

		// is this the wreck we'll be switching to?
		if (FocusedIdx == i)
			hWreckToFocus = hWreck;

		// finalize wreck object initialization, like after loading from scenario
		Intf->bJustDeployed = true;

		SendVesselMessage(kmlHandle, VM_KML_REGISTER, (int)hWreck);
	} // end of cycle by wrecks

	// SWITCH TO WRECK OBJECT 
	if (bHaveFocus)
	{
		// cleanup output
		sprintf(oapiDebugString(), "");

		// switch focus
		oapiSetFocusObject(hWreckToFocus);

		// show mission outcome in the wreck
		if (bOutcomeCriteriaSet)
			ShowMissionOutcome(hWreckToFocus);
	}

	// not focused but have to show mission outcome? show in whatever is focused!
	else if (bOutcomeCriteriaSet)
		ShowMissionOutcome(oapiGetFocusObject());


	//CLEANUP OLD OBJECT 

	// Finally, delete myself!
	oapiDeleteVessel (GetHandle(), 0);
}


void VESSEL2M::UpdateOrbitalMissionCriteria()
{
	OBJHANDLE hEarth = oapiGetGbodyByName("Earth");
	ELEMENTS OrbEls;
	ORBITPARAM OrbParam;
	GetElements(hEarth, OrbEls, &OrbParam);

	if ((OrbParam.PeD - RADIUS_E) > 180000)
		OutcomeCriteria = OC_SUCCESS;

	// 
	bOutcomeCriteriaSet = true;
	OutcomeTimer = 7;
}

void VESSEL2M::ShowAnnotationMessage(char *Message)
{
	oapiAnnotationSetText (MessagePort, Message);
	msgStartTime = oapiGetSysTime();
	bMessageDisplayed = true;
}


void VESSEL2M::SwitchFocus(focus_Dirs FocusDir)
{
	// get total count of vessels in the scenario, should be more than one to continue.
	int VesCnt = oapiGetVesselCount();
	if (VesCnt == 1) 
		return;

	// find our own index in the list of vessles
	int MyIdx = -1;
	for (int i = 0; i < VesCnt; i++)
	{
		// get next vessel in the list
		OBJHANDLE hVes = oapiGetVesselByIndex(i);

		// found ourselves!
		if (hVes == GetHandle())
		{
			MyIdx = i;
			break;
		}
	}

	// find next
	bool bSearching = true;
	OBJHANDLE hVes = NULL;
	while (bSearching)
	{
		// move index ahead one at a time, wrapping around if needed

		// previous 
		if (FocusDir == FOCUS_PREVIOUS)
		{
			MyIdx--;
			if (MyIdx < 0)
				MyIdx = VesCnt-1;
		}

		// next
		else
		{
			MyIdx++;
			if (MyIdx >= VesCnt)
				MyIdx = 0;
		}

		// get vessesl at the new index, skip if it looks wrong
		hVes = oapiGetVesselByIndex(MyIdx);
		if (hVes == NULL)
			continue;

		// get vessel's interface
		VESSEL *Intf;
		Intf = oapiGetVesselInterface(hVes);

		// skip vessels with disabled focus
		if (!Intf->GetEnableFocus())
			continue;

		// skip vessels that are neither VESSEL2M nor Wreck
		if ((!IsVessel2M(Intf)) && (!IsWreck(Intf)))
			continue;

		// if we got here, we have a good find!
		bSearching = false;
	}

	// switch focus to the new vessel
	oapiSetFocusObject(hVes);
}



void VESSEL2M::ToggleEnableFocus()
{
	// that is, we are enabling rather than disabling...
	bool bEnableFocuses = true;

	// get total count of vessels in the scenario
	int VesCnt = oapiGetVesselCount();

	// go through the list of the vessles
	for (int i = 0; i < VesCnt; i++)
	{
		// get next vessel in the list
		OBJHANDLE hVes = oapiGetVesselByIndex(i);

		// get vessel's interface
		VESSEL *Intf;
		Intf = oapiGetVesselInterface(hVes);

		// skip vessels that are neither VESSEL2M nor Wreck
		if ((!IsVessel2M(Intf)) && (!IsWreck(Intf)))
			continue;

		// skip permanently hidden vessels
		if ( ((R7_wreck*)Intf)->bPermanentlyHidden )
			continue;

		// if default focus is off, we simply TOGGLE the currently set focus
		if ( ((R7_wreck*)Intf)->bFocusOffByDefault )
		{
			((R7_wreck*)Intf)->bFocus = !((R7_wreck*)Intf)->bFocus;
			Intf->SetEnableFocus(((R7_wreck*)Intf)->bFocus);

			// either keep the default 'enabling' flag or overriding it with the 'disabling';
			bEnableFocuses = ((R7_wreck*)Intf)->bFocus;
		}
	}

	// show screen message
	if (bEnableFocuses)
		ShowAnnotationMessage("Focus is ENABLED on all secondary items in the scenario.");
	else
		ShowAnnotationMessage("Focus is DISABLED on all secondary items in the scenario.");
}



void VESSEL2M::AttachObject(ATTACHED_OBJECT* AttObject)
{
	// create attachment point on us - always, regardless of the actual presense of the object (or it screws many things)
	AttObject->hAttachment = CreateAttachment(TOCHILD,	AttObject->AttParamsOnUs.AttPoint, 
														AttObject->AttParamsOnUs.AttDir, 
														AttObject->AttParamsOnUs.AttRot, 
														"", NOTLOOSE);

	// do we have a object at all?
	if (AttObject->ObjectName[0] == 0)
		return;

	// does the object exist? get it!
	AttObject->hObject = oapiGetObjectByName(AttObject->ObjectName);
	if (!AttObject->hObject)
		return;

	// get payload vessel interface
	AttObject->ObjectIntf2M = (VESSEL2M*)oapiGetVesselInterface(AttObject->hObject);

	// create attachment point on the object, pointed to us
	ATTACHMENTHANDLE hToMe = AttObject->ObjectIntf2M->CreateAttachment( TOPARENT,	AttObject->AttParamsOnObject.AttPoint, 
																					AttObject->AttParamsOnObject.AttDir, 
																					AttObject->AttParamsOnObject.AttRot, 
																					"", NOTLOOSE);

	// Attach!
	AttachChild(AttObject->hObject, AttObject->hAttachment, hToMe);

	// explicitly set bAttached flag for VESSEL2M objects
	AttObject->ObjectIntf2M->bAttached = true;
	AttObject->ObjectIntf2M->bLandedPermanently = false;
	AttObject->ObjectIntf2M->kmlNoPlotting = true;

	// finally, update empty mass by recounting all children
	UpdateEmptyMass();
}


void VESSEL2M::UpdateEmptyMass()
{
	double AttachedMass = 0;

	for (DWORD i=0; i<AttachmentCount(TOCHILD); i++)
	{
		ATTACHMENTHANDLE hAttachment = GetAttachmentHandle(TOCHILD, i);
		OBJHANDLE hObject = GetAttachmentStatus(hAttachment);

		if (hObject != NULL)
		{
			VESSEL* ObjectIntf;
			ObjectIntf = oapiGetVesselInterface(hObject);
			double ObjectMass = ObjectIntf->GetMass();

			AttachedMass += ObjectMass;
		}
	}

	SetEmptyMass(EmptyMassWithoutAttachments + AttachedMass);
}


void VESSEL2M::WriteAttachedObjectToScenario(FILEHANDLE scn, ATTACHED_OBJECT* AttObject)
{
	// is there anything to write at all?
	if (AttObject->ObjectName[0] == 0)
		return;

	// write object name
	oapiWriteScenario_string(scn, AttObject->ScnVarNames.ObjectName, AttObject->ObjectName);

	// write attachment params, but only if different from defaults
	if (!VectorEqual(AttObject->AttParamsOnObject.AttPoint, AttObject->AttParamsOnObjectDefault.AttPoint))
		oapiWriteScenario_vec(scn, AttObject->ScnVarNames.PointName, AttObject->AttParamsOnObject.AttPoint);
	if (!VectorEqual(AttObject->AttParamsOnObject.AttDir, AttObject->AttParamsOnObjectDefault.AttDir))
		oapiWriteScenario_vec(scn, AttObject->ScnVarNames.DirName, AttObject->AttParamsOnObject.AttDir);
	if (!VectorEqual(AttObject->AttParamsOnObject.AttRot, AttObject->AttParamsOnObjectDefault.AttRot))
		oapiWriteScenario_vec(scn, AttObject->ScnVarNames.RotName, AttObject->AttParamsOnObject.AttRot);
}


bool VESSEL2M::ReadAttachedObjectLineFromScenario(char* line, ATTACHED_OBJECT* AttObject)
{
	int VarLen = strlen(AttObject->ScnVarNames.ObjectName);
	if (!strnicmp (line, AttObject->ScnVarNames.ObjectName, VarLen))
	{
		sscanf (line + VarLen, "%s", &AttObject->ObjectName);
		return true;
	}

	VarLen = strlen(AttObject->ScnVarNames.PointName);
	if (!strnicmp (line, AttObject->ScnVarNames.PointName, VarLen))
	{
		sscanf (line + VarLen, "%lf %lf %lf",	&AttObject->AttParamsOnObject.AttPoint.x, 
												&AttObject->AttParamsOnObject.AttPoint.y, 
												&AttObject->AttParamsOnObject.AttPoint.z);
		return true;
	}

	VarLen = strlen(AttObject->ScnVarNames.DirName);
	if (!strnicmp (line, AttObject->ScnVarNames.DirName, VarLen))
	{
		sscanf (line + VarLen, "%lf %lf %lf",	&AttObject->AttParamsOnObject.AttDir.x, 
												&AttObject->AttParamsOnObject.AttDir.y, 
												&AttObject->AttParamsOnObject.AttDir.z);
		return true;
	}

	VarLen = strlen(AttObject->ScnVarNames.RotName);
	if (!strnicmp (line, AttObject->ScnVarNames.RotName, VarLen))
	{
		sscanf (line + VarLen, "%lf %lf %lf",	&AttObject->AttParamsOnObject.AttRot.x, 
												&AttObject->AttParamsOnObject.AttRot.y, 
												&AttObject->AttParamsOnObject.AttRot.z);
		return true;
	}

	return false;
}


void VESSEL2M::WritePropellantsToScenario(FILEHANDLE scn)
{
	char Values[255];

	strcpy(Values, "");
	bool bHaveFuel = false;

	for (int i=0; i<(int)GetPropellantCount(); i++)
	{
		PROPELLANT_HANDLE ph = GetPropellantHandleByIndex(i);
		double Mass = GetPropellantMass(ph);
		double MaxMass = GetPropellantMaxMass(ph);
		double Value = Mass/MaxMass;

		if (Value > 0)
			bHaveFuel = true;

		if (Values[0] != 0)
			strcat(Values, ",");

		sprintf(Values, "%s%d:%f", Values, i, Value);
	}

	if (bHaveFuel)
		oapiWriteScenario_string(scn, "PRPLEVEL", Values);
}


void VESSEL2M::WriteStatusToScenario(FILEHANDLE scn)
{
	VESSELSTATUS vs;
	GetStatus(vs);
	
	if (vs.status == 1)
		oapiWriteScenario_string(scn, "STATUS", "Landed Earth");
	else
		oapiWriteScenario_string(scn, "STATUS", "Orbiting Earth");
}



int VESSEL2M::SendVesselMessage(OBJHANDLE hTarget, IntVesselMessage VM, int Value)
{
	VESSEL *Intf;

//	if ((hTarget != NULL) && (VM != 0xcccccccc))
	if (oapiIsVessel(hTarget))
	{
		Intf = oapiGetVesselInterface(hTarget);
		if (Intf != NULL)
			if (IsVessel2M(Intf))
				return ((VESSEL2M*)Intf)->GetVesselMessage(VM, Value);
	}

	return 0;
}


int VESSEL2M::GetVesselMessage(IntVesselMessage VM, int Value)
{
	if (VM == VM_DETACHED)
		bAttached = false;

	else if (VM == VM_DELAY_EXPLODE)
	{
		if ( (bFuelExplode() || bHaveSelfDestructor) && (!bDelayedExplode) )
		{
			bDelayedExplode = true;
			DelayedExplodeTimer = DRand(1.5, 3);
		}
	}

	else if (VM == VM_SHOW_OUTCOME)
	{
		memcpy(&Outcomes[OC_FOREIGN], (OUTCOME_OBJECT*)Value, sizeof(OUTCOME_OBJECT));
		ShowAnnotationMessage(Outcomes[OC_FOREIGN].OutcomeMessage);
	}

	// unsupported message
	return 0;
}


int VESSEL2M::BroadcastVesselMessage(IntVesselMessage VM, int Value)
{
	int result = 0;
	int tempResult = 0;

	for (DWORD i=0; i<oapiGetVesselCount(); i++)
	{
		if ((tempResult = SendVesselMessage(oapiGetVesselByIndex(i), VM, Value)) != 0)
			result = tempResult;
	}

	return result;
}


double VESSEL2M::SendVesselMessage(OBJHANDLE hTarget, DblVesselMessage VM, double Value)
{
	VESSEL *Intf;

	if (hTarget != NULL)
	{
		Intf = oapiGetVesselInterface(hTarget);
		if (Intf != NULL)
			if (IsVessel2M(Intf))
				return ((VESSEL2M*)Intf)->GetVesselMessage(VM, Value);
	}

	return 0;
}



double VESSEL2M::GetVesselMessage(DblVesselMessage VM, double Value)
{
	return 0;
}


void VESSEL2M::StartDust()
{
	bGeneratingDust = true;
	DustTimer = 0.2;
	DustLevel = 1;
}


void VESSEL2M::StopDust()
{
	bGeneratingDust = false;
	DustTimer = -1;
	DustLevel = 0;

	// if by the time of the "dust-off" we are still sitting on the ground, consider us landed.
	bLandedPermanently = GroundContact();
}


void VESSEL2M::SetCameraData(int Idx, VECTOR3 Off, VECTOR3 Dir, double Tilt, bool bEnabled)
{
	Cameras[Idx].Off = Off;
	Cameras[Idx].Dir = Dir;
	Cameras[Idx].Tilt = Tilt;
	Cameras[Idx].bEnabled = bEnabled;
}


void VESSEL2M::CycleOnboardCamera()
{
	// get new camera ID 
	do
	{
		CameraID++;
		if (CameraID == CamerasCount)
			CameraID = 0;
	}
	while (!Cameras[CameraID].bEnabled);

	// set new camera
	ActivateCurrentCamera();
}


void VESSEL2M::ActivateCurrentCamera()
{
	SetCameraOffset(Cameras[CameraID].Off);
	SetCameraDefaultDirection(Cameras[CameraID].Dir, Cameras[CameraID].Tilt);
}


// SetCameraDefaultDirection has no effect when called from clbkSetClassCaps - bug in Orbiter? 
// Enforce camera dir switch here:
bool VESSEL2M::clbkLoadGenericCockpit ()
{
	oapiSetDefRCSDisplay(0);
	ActivateCurrentCamera();
	return true;
}




void VESSEL2M::DebugInternalCameraMove()
{
	VECTOR3 Pos, Dir;
	GetCameraOffset(Pos);
	GetCameraDefaultDirection(Dir);
	double NewLenXY;

	// camera travels in timestep, rad
	double SimDt = oapiGetSimStep();

	if (GetAttitudeMode() == ATTMODE_LIN)
	{
		double dShift = 0.1 * SimDt;

		double ShiftUp = GetManualControlLevel(THGROUP_ATT_UP, MANCTRL_LINMODE, MANCTRL_KEYBOARD);
		double ShiftDown = GetManualControlLevel(THGROUP_ATT_DOWN, MANCTRL_LINMODE, MANCTRL_KEYBOARD);
		double ShiftLeft = GetManualControlLevel(THGROUP_ATT_LEFT, MANCTRL_LINMODE, MANCTRL_KEYBOARD);
		double ShiftRight = GetManualControlLevel(THGROUP_ATT_RIGHT, MANCTRL_LINMODE, MANCTRL_KEYBOARD);
		double ShiftForward = GetManualControlLevel(THGROUP_ATT_FORWARD, MANCTRL_LINMODE, MANCTRL_KEYBOARD);
		double ShiftBack = GetManualControlLevel(THGROUP_ATT_BACK, MANCTRL_LINMODE, MANCTRL_KEYBOARD);

		Pos.x += (ShiftForward-ShiftBack)*dShift;
		Pos.y += (ShiftLeft-ShiftRight)*dShift;
		Pos.z += (-ShiftUp+ShiftDown)*dShift;

		SetCameraOffset(Pos);
	}

	else if (GetAttitudeMode() == ATTMODE_ROT)
	{
		double dAngle = 2.5*RAD * SimDt;

		double PitchUp = GetManualControlLevel(THGROUP_ATT_PITCHUP, MANCTRL_ATTMODE, MANCTRL_KEYBOARD);
		double PitchDown = GetManualControlLevel(THGROUP_ATT_PITCHDOWN, MANCTRL_ATTMODE, MANCTRL_KEYBOARD);
		double YawLeft = GetManualControlLevel(THGROUP_ATT_YAWLEFT, MANCTRL_ATTMODE, MANCTRL_KEYBOARD);
		double YawRight = GetManualControlLevel(THGROUP_ATT_YAWRIGHT, MANCTRL_ATTMODE, MANCTRL_KEYBOARD);
		double BankLeft = GetManualControlLevel(THGROUP_ATT_BANKLEFT, MANCTRL_ATTMODE, MANCTRL_KEYBOARD);
		double BankRight = GetManualControlLevel(THGROUP_ATT_BANKRIGHT, MANCTRL_ATTMODE, MANCTRL_KEYBOARD);

		// pitch

		double Pitch = acos(Dir.z);		
		double PitchSign = PitchUp-PitchDown;
		Pitch += PitchSign*dAngle;
		
		if (Pitch > PI)
			Pitch = -Pitch+PI;
		else if (Pitch < -PI)
			Pitch = -PI - Pitch;
		
		Dir.z = cos(Pitch); 
		
		if (Dir.x+Dir.y == 0)
			Dir.y = 1-fabs(Dir.z);
		else
		{
			NewLenXY = sqrt(1-Dir.z*Dir.z);
			double OldLenXY = sqrt(Dir.x*Dir.x + Dir.y*Dir.y);
			double Ratio = NewLenXY / OldLenXY;

			Dir.x *= Ratio;
			Dir.y *= Ratio;
		}
		
		// yaw
		double Yaw = asin(Dir.y);		
		double YawSign = YawRight-YawLeft;
		Yaw += YawSign*dAngle;

		if (Yaw > PI)
			Yaw = -Yaw+PI;
		else if (Yaw < -PI)
			Yaw = -PI - Yaw;

		Dir.y = sin(Yaw);

		double xs = NewLenXY*NewLenXY - Dir.y*Dir.y;
		if (xs > 0)
			Dir.x = -sqrt(xs);
		else
			Dir.x = 0;

		// bank
		DebugCockpitCameraTilt += (BankLeft-BankRight)*dAngle;

		SetCameraDefaultDirection(Dir, DebugCockpitCameraTilt);
	}

	if (bHaveFocus)
		sprintf(oapiDebugString(), "Pos: %f, %f, %f    Dir: %f, %f, %f,    Tilt: %f", Pos.x, Pos.y, Pos.z, Dir.x, Dir.y, Dir.z, DebugCockpitCameraTilt*DEG);
}


void VESSEL2M::LocalRot(const VECTOR3 vecGlob, VECTOR3 &vecLoc)
{
	VESSELSTATUS2 vs;
	double dCosZ, dSinZ, dCosY, dSinY, dCosX, dSinX;
	vs.version = 2;
	vs.flag = 0;
	GetStatusEx((void*) &vs);

	dCosZ = cos(vs.arot.z);
	dSinZ = sin(vs.arot.z);
	dCosY = cos(vs.arot.y);
	dSinY = sin(vs.arot.y);
	dCosX = cos(vs.arot.x);
	dSinX = sin(vs.arot.x);

	vecLoc.x = vecGlob.x*dCosZ*dCosY + vecGlob.y*(-dSinZ*dCosX+dCosZ*dSinY*dSinX) + vecGlob.z*(dSinZ*dSinX+dCosZ*dSinY*dCosX);
	vecLoc.y = vecGlob.x*dSinZ*dCosY + vecGlob.y*(dCosZ*dCosX+dSinZ*dSinY*dSinX) + vecGlob.z*(-dCosZ*dSinX+dSinZ*dSinY*dCosX);
	vecLoc.z = vecGlob.x*-dSinY + vecGlob.y*dCosY*dSinX + vecGlob.z*dCosY*dCosX;
}




void VESSEL2M::kmlAddEvent(kmlEvents kmlEvent, char* Label, double LatDeg, double LonDeg, OBJHANDLE hOrigin)
{
	// safety
	if (!oapiIsVessel(kmlHandle))
		return;

	// we will be populating this empty event with new values...
	KML_EVENT Event;

	// initialize Originator, if skipped - with ourselves
	if (hOrigin == NULL)
	{
		// select originator, service or ourselves
		OBJHANDLE hOriginator;
		if (kmlEvent < KML_FLIGHT)
			hOriginator = (OBJHANDLE)666;
		else
			hOriginator = GetHandle();

		// use what we selected
		Event.hOrigin = hOriginator;
	}

	// if originator passed explisitly, use it
	else
		Event.hOrigin = hOrigin;

	// initialize skipped coordinates to our current values
	if ((LatDeg == 0) || (LonDeg == 0))
	{
		double Radius;
		GetEquPos(LonDeg, LatDeg, Radius);
		LatDeg *= DEG;
		LonDeg *= DEG;
	}

	// populate kml event object 
	Event.Lat = LatDeg;
	Event.Lon = LonDeg;
	Event.Alt = GetAltitude();
	Event.Time = oapiGetSimTime();
	Event.Event = kmlEvent;
	strcpy(Event.Label, Label);

	// send kml event to kmlWriter
	SendVesselMessage(kmlHandle, VM_KML_ADD_EVENT, (int)(&Event));
}



void VESSEL2M::kmlSetColor(kmlPlotColors Color)
{
	// don't do anything if no change
	if (kmlPlotColor == Color)
		return;

	KML_COLOR_STRUCT ColorStruct;

	ColorStruct.kmlPlotColor = Color;
	ColorStruct.hOriginator = GetHandle();

	// send kml event to kmlWriter
	SendVesselMessage(kmlHandle, VM_KML_SET_COLOR, (int)(&ColorStruct));

	// update internal variable
	kmlPlotColor = Color;
}


// this gets called durig 'save scenario' operation, can be overridden by the vessel
void VESSEL2M::WriteLastKmlEntry()
{
	kmlAddEvent(KML_FLIGHT, GetName());
}


// FAILURES


// placeolder
bool VESSEL2M::failInitialize()
{
	// NN% chances to fail...
	if (DRand(0., 100.) < Reliability)
		return true;

	// fail!
	bFail = true;

	// need to continue processing in the derived vessel
	return false;
}


bool VESSEL2M::failUpdate()
{
	// no thrust or filre - no developing in falure!
	if ((GetThrusterGroupLevel(THGROUP_MAIN) == 0) && (Fire.FireLevel == 0))
		return true;

	// time to fail? 
	if (failTimer > 0)
	{
		if ((failTimer -= oapiGetSimStep()) > 0)
			return true;

		// activate whatever failure is set
		failActivate();
	}

	// need to continue processing in the derived vessel
	return false;
}


// placeholder
void VESSEL2M::failActivate()
{
	// slow down to not to miss the show
	if ( oapiGetTimeAcceleration() > 1)
		oapiSetTimeAcceleration(1);

	// sound alarm
	SendVesselWaveToFocused(SND_ALARM);
}


void VESSEL2M::failStartFire()
{
	// lengths of fire stages
	Fire.tFireUp = DRand(10, 10);
	Fire.tFireSteady = DRand(5, 5);
	Fire.tThrustDown = DRand(10, 10);

	// addiing smoke and flame
	Fire.FireLevel = 0.1;
	failStartFireVisual();

	kmlAddEvent(KML_FLIGHT, "Engine fire detected!");
	ShowAnnotationMessage("Engine fire detected!");
}


void VESSEL2M::failStartFireVisual()
{
	// smoke and flame for the engine fire
	SURFHANDLE SmokeTex = oapiRegisterParticleTexture("r7_S\\Contrail3");
	SURFHANDLE FlameTex = oapiRegisterParticleTexture("r7_S\\TrenchFlame");

	PARTICLESTREAMSPEC SmokeStream = {
		0,			// flags, reserved
		0.1,		// srcsize, particle size at creation, m
		10,			// srcrate, average particle generation rate, Hz   
		0,			// V0, average particle emission velocity, m/s
		0,			// srcspread, emission velocity distribution randomisation
		2,			// lifetime, average particle lifetime [s]
		10,			// growthrate, particle growth rate [m/s]
		5,			// atmslowdown, deceleration rate b in atmosphere, defined as v = v0 e- bdt 
		PARTICLESTREAMSPEC::DIFFUSE, PARTICLESTREAMSPEC::LVL_PSQRT, 0, 0.5, PARTICLESTREAMSPEC::ATM_FLAT, 1.22, 1.25,
		SmokeTex	
	};

	PARTICLESTREAMSPEC FlameStream = { 0,			// flags, reserved
		1,			// srcsize, particle size at creation, m
		10,			// srcrate, average particle generation rate, Hz   
		10,			// V0, average particle emission velocity, m/s
		0.01,			// srcspread, emission velocity distribution randomisation
		0.1,			// lifetime, average particle lifetime [s]
		5,				// growthrate, particle growth rate [m/s]
		0.1,			// atmslowdown, deceleration rate b in atmosphere, defined as v = v0 e- bdt 
		PARTICLESTREAMSPEC::EMISSIVE, PARTICLESTREAMSPEC::LVL_PSQRT, -1, 0.5, PARTICLESTREAMSPEC::ATM_PLIN, -1, 0.5, 
		FlameTex };	

	AddParticleStream (&SmokeStream, Fire.Pos, Fire.Dir,  &Fire.FireLevel);
	AddParticleStream (&FlameStream, Fire.Pos, Fire.Dir,  &Fire.FireLevel);
}


void VESSEL2M::failUpdateFire()
{
	// reduce thrust due to fuel leaks
	if (GetThrusterGroupLevel(THGROUP_MAIN) > 0.1)
		SetThrusterGroupLevel(THGROUP_MAIN, GetThrusterGroupLevel(THGROUP_MAIN) - oapiGetSimStep()*0.9/Fire.tThrustDown);

	// increase fire level
	if (Fire.FireLevel < 1)
		Fire.FireLevel += (oapiGetSimStep()*0.9/Fire.tFireUp);

	// if fire level is saturated, decrement flat fire timer until total failure
	else if ((Fire.tFireSteady -= oapiGetSimStep()) < 0)
		Break();
}



