#include	"math.h"
#include	"stdio.h"
#include	"Launchpad.h"



R7_launchpad::R7_launchpad(OBJHANDLE hObj, int fmodel): VESSEL2M(hObj, fmodel)
{
	// default launcher attachment params
	Launcher.ObjectName[0] = 0;
	Launcher.hObject = NULL;
	Launcher.AttParamsOnUs = _AP(LAUNCHER_OFS, LAUNCHER_DIR, xplus);
	Launcher.AttParamsOnObjectDefault = _AP(xyz0, zminus, _V(-COS45D,COS45D,0));
	Launcher.AttParamsOnObject = Launcher.AttParamsOnObjectDefault;
	Launcher.ScnVarNames = _ASVN("LAUNCHER", "UNDEFINED", "UNDEFINED", "UNDEFINED");

	// timer events and autopilot
	Timer = TIMER_NOT_SET;
	bAutopilot = true;
	bLaunchSequence = false;
	LastTE = TE_NONE;
	LaunchMJD = 0;

	// targeting
	PayloadType = PAYLOAD_UNKNOWN;
	PanBalTgtIdx = 7;
	TargetLat = 0;						
	TargetLon = 0;						
	TargetName[0] = 0;				
	Apogee = 0;							
	Perigee = 0;						
	Inc = 0;		
	TrajectoryType = TRAJECTORY_UNKNOWN;
	bTrajectoryIsSet = false;
	hEarth = oapiGetGbodyByName("Earth");
	RangeAdjustment = 0;
	AzimuthAdjustment = 0;
	sScenarioWarning[0] = 0;
	sTimeWarning[0] = 0;

	// all cameras
	CamerasCount = 3;
	double X, Y, Z;

	// first camera, table-level 
	X = 0;
	Z = 0.95;
	Y = sqrt(1-X*X-Z*Z);
	SetCameraData(0, _V(-8, -11.6, -26.5), _V(X, Y, Z));

	// second camera, platform-level 
	X = 0;
	Y = 0.3;
	Z = sqrt(1-X*X-Y*Y);
	SetCameraData(1, _V(-0.8788, -20.657, -10.9614), _V(X, Y, Z));

	// third camera, under-the-table 
	X = 0.76;
	Y = -0.3;
	Z = sqrt(1-X*X-Y*Y);
	SetCameraData(2, _V(-13.78, -14.88, -11.837), _V(X, Y, Z));

	// animations 
	RingRotPercent = 0;
	RingTurning = 0;
	bRingToAzimuth = false;

	BoomRotPercent = 1;
	BoomTurning = 0;
	bBoomsTurnDown = false;

	SupportRotPercent = 1;
	SupportTurning = 0;
	bSupportFalling = false;
	SupportDropOmega = 0;

	PlatformTranslPercent = 1;
	PlatformTranslating = 0;
	bPlatformRetract = false;

	CableMastRotPercent = 1;
	CableMastTurning = 0;
	bCableMastFalling = false;
	CableMastDropOmega = 0;

	FuelingMastRotPercent = 0;						
	FuelingMastTurning = 0;
	bFuelingMastFalling = false;
	FuelingMastDropOmega = 0;

	// mesh configurations
	BoomsMeshName[0] = 0;
	CableMastMeshName[0] = 0;
	FuelingMastMesh = 0;
	TableTopMeshName[0] = 0;

	// additional engine smokes
	SURFHANDLE SmokeTex = oapiRegisterParticleTexture("r7_S\\Contrail3");
	SURFHANDLE FlameTex = oapiRegisterParticleTexture("r7_S\\TrenchFlame");

	PARTICLESTREAMSPEC MainSmokeStream = { 0,			// flags, reserved
		7.5,			// srcsize, particle size at creation, m
		10,				// srcrate, average particle generation rate, Hz   
		80,				// V0, average particle emission velocity, m/s
		0.01,			// srcspread, emission velocity distribution randomisation
		4,				// lifetime, average particle lifetime [s]
		10,				// growthrate, particle growth rate [m/s]
		0.01,			// atmslowdown, deceleration rate b in atmosphere, defined as v = v0 e- bdt 
		PARTICLESTREAMSPEC::DIFFUSE, PARTICLESTREAMSPEC::LVL_PSQRT, 0, 1.0, PARTICLESTREAMSPEC::ATM_PLIN, 0, 0, 
		SmokeTex };	

	PARTICLESTREAMSPEC FlameStream = { 0,			// flags, reserved
		5,			// srcsize, particle size at creation, m
		5,				// srcrate, average particle generation rate, Hz   
		30,				// V0, average particle emission velocity, m/s
		0.01,			// srcspread, emission velocity distribution randomisation
		1,				// lifetime, average particle lifetime [s]
		1,				// growthrate, particle growth rate [m/s]
		0.1,			// atmslowdown, deceleration rate b in atmosphere, defined as v = v0 e- bdt 
		PARTICLESTREAMSPEC::EMISSIVE, PARTICLESTREAMSPEC::LVL_PSQRT, 0, 1.0, PARTICLESTREAMSPEC::ATM_PLIN, 0, 0, 
		FlameTex };	

	PARTICLESTREAMSPEC SideSmokeStream = { 0,			// flags, reserved
		5,			// srcsize, particle size at creation, m
		10,				// srcrate, average particle generation rate, Hz   
		60,				// V0, average particle emission velocity, m/s
		0.01,			// srcspread, emission velocity distribution randomisation
		4,				// lifetime, average particle lifetime [s]
		4,				// growthrate, particle growth rate [m/s]
		0.1,			// atmslowdown, deceleration rate b in atmosphere, defined as v = v0 e- bdt 
		PARTICLESTREAMSPEC::DIFFUSE, PARTICLESTREAMSPEC::LVL_PSQRT, 0, 1.0, PARTICLESTREAMSPEC::ATM_PLIN, 0, 0, 
		SmokeTex };	

	// source of additional flames and smokes
	const VECTOR3 ENGINE_POS = { 0, -H_TD_POINTS, 0.5};

	// flames, not too intense, one goes UP 
	FlameLevel = 0;
	AddParticleStream (&FlameStream, ENGINE_POS, yplus,  &FlameLevel);
	AddParticleStream (&FlameStream, ENGINE_POS, xplus,  &FlameLevel);
	AddParticleStream (&FlameStream, ENGINE_POS, xminus, &FlameLevel);
	AddParticleStream (&FlameStream, ENGINE_POS, zplus,  &FlameLevel);

	// smokes, main is more intense than sides 
	SmokeLevelMain = 0;
	SmokeLevelSides = 0;
	AddParticleStream (&SideSmokeStream, ENGINE_POS, xplus,  &SmokeLevelSides);
	AddParticleStream (&SideSmokeStream, ENGINE_POS, xminus, &SmokeLevelSides);
	AddParticleStream (&MainSmokeStream, ENGINE_POS, zplus, &SmokeLevelMain);


		// LIGHTS

	// virtual fuel tank and light "engines" 
	MainTank = CreatePropellantResource(PropellantMass, PropellantMass);
	SetDefaultPropellantResource(MainTank);
	LightsTimer = 598; // 2 sec before check
	// the rest of the lights will be created later in CreateSpotlights called from FinishConfiguration

	// beacons
	BeaconPos[0] = _V(-18.79,42.253,19.383);
	BeaconPos[1] = _V(106.496,98.666,47.1717);
	BeaconPos[2] = _V(-64.145,99.666,-40.182);
	BeaconPos[3] = _V(-104.217,40.926,83.027);
	BeaconPos[4] = _V(106.973,40.926,69.672);
	BeaconPos[5] = _V(69.4,45.84,-129.52);
	BeaconPos[6] = _V(-88.9,45.84,-129.07);

	bcol = _V(1,0,0);

	for (int i = 0; i < BEACON_COUNT; i++)
	{
		Beacons[i].shape = BEACONSHAPE_DIFFUSE;
		Beacons[i].falloff = 0;
		Beacons[i].pos = &BeaconPos[i];
		Beacons[i].col = &bcol;
		Beacons[i].size = 0.5;
		Beacons[i].active = false;

		AddBeacon (&Beacons[i]);
	}
}


void R7_launchpad::CreateSpotlights()
{
	// light texture
	SURFHANDLE LightTex = oapiRegisterParticleTexture("r7_S\\Lights");

	// virtual thrusters to initiate lights
	THRUSTER_HANDLE	th_lights[8];	

	// nominal R-7 lights geometry
	double ltDElevR = LT_D_ELEV_R;
	double ltDAzmR =  LT_D_AZM_R;
	double ltWid  = LT_WID;

	// lights geometry for future higher 3-stage rockets
	if (PayloadType == PAYLOAD_3ST)
	{
		ltDElevR = LT_D_ELEV_3ST_R;
		ltDAzmR =  LT_D_AZM_3ST_R;
		ltWid  = LT_WID_3ST;
	}

	// 4 light towers
	for (int t=0; t<4; t++)		
	{
		// tower pos
		VECTOR3 VPos = LT_POS[t];

		// calc tower dir components
		double dirY = sin(LT_ELEV_R);

		double dirXZ = cos(LT_ELEV_R);
		double azmR = LT_AZM[t] * RAD;
		double dirX = sin(azmR);
		double dirZ = cos(azmR);
		dirZ = dirXZ * dirZ;
		dirX = dirXZ * dirX;

		// tower dir
		VECTOR3 VDir = _V(dirX, dirY, dirZ);
	
		// two virtual thrustesr in opposite directions
		th_lights[t*2]   = CreateThruster(VPos,  VDir, 1, MainTank, 10000);
		th_lights[t*2+1] = CreateThruster(VPos, -VDir, 1, MainTank, 10000);
	
		// pre-cycle initialize
		VECTOR3 VPosTex = VPos;
		double azmStepR = azmR + PI/2;
		double dx = LT_DHOR * sin(azmStepR);
		double dz = LT_DHOR * cos(azmStepR);
		VECTOR3 VDirTex = VDir;

		// 6 columns
		for (int col=0; col<6; col++)
		{
			// reset col
			VPosTex.y = VPos.y;
			VDirTex = VDir;

			// 5 rows
			for (int row=0; row<5; row++)
			{
				AddExhaust (th_lights[t*2], LT_LEN, ltWid, VPosTex, VDirTex, LightTex);
				
				VPosTex.y += LT_DVER;

				VDirTex.y = sin(LT_ELEV_R + ltDElevR*row);
				dirXZ = cos(LT_ELEV_R + ltDElevR*row);
				double azmTexR = azmR + ltDAzmR * (col+0.5-2);
				dirX = sin(azmTexR);
				dirZ = cos(azmTexR);
				VDirTex.z = dirXZ * dirZ;
				VDirTex.x = dirXZ * dirX;
			}

			// update col
			VPosTex.x += dx;
			VPosTex.z += dz;
		}
	}

	thg_lights = CreateThrusterGroup(th_lights, 8, THGROUP_USER); 
}


void R7_launchpad::clbkSetClassCaps(FILEHANDLE cfg)
{
	// generic physics and geopmetry
	SetSize(100);
	SetEmptyMass(2e6);
	SetSurfaceFrictionCoeff(1e5, 1e5);
	SetTouchdownPoints(_V(0, H_TD_POINTS, -1), _V(1, H_TD_POINTS, 0), _V(-1, H_TD_POINTS, 0));
}


void R7_launchpad::clbkSaveState (FILEHANDLE scn)
{
	// do a small hack with bAttached to limit the amount of output by the parent function
	bAttached = true;
	VESSEL2M::clbkSaveState(scn);
	bAttached = false;

	// write separator
	oapiWriteScenario_string (scn, "=========== LAUNCHPAD vars", "");

	// animations
	if (RingRotPercent != 0)
		oapiWriteScenario_float (scn, "RING", RingRotPercent);
	if (SupportRotPercent != 1)
		oapiWriteScenario_float (scn, "SUPPORTS", SupportRotPercent);
	if (BoomRotPercent != 1)
		oapiWriteScenario_float (scn, "BOOMS", BoomRotPercent);
	if (CableMastRotPercent != 1)
		oapiWriteScenario_float (scn, "CABLE_MAST", CableMastRotPercent);
	if (FuelingMastRotPercent != 0)
		oapiWriteScenario_float (scn, "FUELING_MAST", FuelingMastRotPercent);
	if (PlatformTranslPercent != 1)
		oapiWriteScenario_float (scn, "PLATFORM", PlatformTranslPercent);

	// custom meshes
	if (BoomsMeshName[0] != 0)
		oapiWriteScenario_string (scn, "BOOMS_MESHNAME", BoomsMeshName);
	if (CableMastMeshName[0] != 0)
		oapiWriteScenario_string (scn, "CABLEMAST_MESHNAME", CableMastMeshName);
	if (TableTopMeshName[0] != 0)
		oapiWriteScenario_string (scn, "TABLETOP_MESHNAME", TableTopMeshName);

	// attachments
	WriteAttachedObjectToScenario(scn, &Launcher);

	// do not save the rest unless we have a rocket attached %%% or far enough to be "done" with the pad
	if (Launcher.ObjectName[0] == 0)
		return;

	// ====== 

	// timer
	if (Timer != TIMER_NOT_SET)
	{
		oapiWriteScenario_float (scn, "TIMER", Timer);

		if (LaunchMJD > 0) 
		{
			char sTemp[256];
			sprintf(sTemp, "%lf", LaunchMJD);
			oapiWriteScenario_string (scn, "LAUNCH_MJD", sTemp);
		}

		// non-default autopilot setting
		if (!bAutopilot) 
			oapiWriteScenario_string (scn, "MANUAL_CONTROL", "");

		if (bLaunchSequence) 
			oapiWriteScenario_string (scn, "LAUNCH_SEQUENCE", "");
	}

	// target name
	if (TargetName[0] != 0)
		oapiWriteScenario_string (scn, "TARGET_NAME", TargetName);
	
	// target coords
	if ( (PanBalTgtIdx == 7) && ((TargetLat + TargetLon) != 0) )
	{
		char TargetPos[255];
		sprintf(TargetPos, "%.13f %.13f", TargetLat, TargetLon);
		oapiWriteScenario_string(scn, "TARGET_POS", TargetPos);
	}

	// orbit params
	else if ((Apogee + Perigee + Inc) != 0)
		oapiWriteScenario_vec(scn, "ORBIT", _V(Perigee, Apogee, Inc));
}


void R7_launchpad::ParseScenarioLineEx(char *line, void *status)
{
	// animations
	if (!strnicmp (line, "RING", 4))
		sscanf (line+4, "%lf", &RingRotPercent);
	else if (!strnicmp (line, "SUPPORTS", 8))
		sscanf (line+8, "%lf", &SupportRotPercent);
	else if (!strnicmp (line, "BOOMS", 5))
		sscanf (line+5, "%lf", &BoomRotPercent);
	else if (!strnicmp (line, "CABLE_MAST", 10))
		sscanf (line+10, "%lf", &CableMastRotPercent);
	if (!strnicmp (line, "FUELING_MAST", 12))
		sscanf (line+12, "%lf", &FuelingMastRotPercent);
	else if (!strnicmp (line, "PLATFORM", 8))
		sscanf (line+8, "%lf", &PlatformTranslPercent);

	// custom meshes
	else if (!strnicmp (line, "BOOMS_MESHNAME", 14))
		sscanf (line+14, "%s", &BoomsMeshName);
	else if (!strnicmp (line, "CABLEMAST_MESHNAME", 18))
		sscanf (line+18, "%s", &CableMastMeshName);
	else if (!strnicmp (line, "TABLETOP_MESHNAME", 17))
		sscanf (line+17, "%s", &TableTopMeshName);

	// attachments
	else if (ReadAttachedObjectLineFromScenario(line, &Launcher))
		;

	// timer
	else if (!strnicmp (line, "TIMER", 5))
		sscanf (line+5, "%lf", &Timer);
	else if (!strnicmp (line, "LAUNCH_MJD", 10))
		sscanf (line+10, "%lf", &LaunchMJD);
	else if (!strnicmp (line, "MANUAL_CONTROL", 14))
		bAutopilot = false;
	else if (!strnicmp (line, "LAUNCH_SEQUENCE", 15))
		bLaunchSequence = true;

	// targeting
	else if (!strnicmp (line, "TARGET_NAME", 11))
		sscanf (line+11, "%s", &TargetName);
	else if (!strnicmp (line, "TARGET_POS", 10))
		sscanf (line+10, "%lf %lf", &TargetLat, &TargetLon);
	else if (!strnicmp (line, "ORBIT", 5))
		sscanf (line+5, "%lf %lf %lf", &Perigee, &Apogee, &Inc);

	// VESSEL2M stuff
	else 
		VESSEL2M::ParseScenarioLineEx (line, status);
}


void R7_launchpad::FinishConfiguration()
{
	// default launch azimuth, deg
	Azimuth = LP_AZM;

	// extract year from scenario date
	ScnYear = GetScenarioYear();

	// finish attaching objects, update empty mass
	AttachObject(&Launcher);

	// load config-specific meshes 
	LoadMeshes();

	// set animations
	DefineAnimations();

	// verify target or orbital destination as read from the scenario
	VerifyTargeting();

	// autopilot can stay enabled only when we have launch time set!
	if (Timer == TIMER_NOT_SET)
		bAutopilot = false;

	// set vents
	if (Launcher.ObjectName[0] != 0)
	{
		if (CableMastRotPercent == 1)
			SendVesselMessage(Launcher.hObject, VM_SET_PRE_LAUNCH_VENT, 1);
		if (FuelingMastRotPercent == 1)
			SendVesselMessage(Launcher.hObject, VM_SET_PRE_LAUNCH_VENT_3ST, 1);
	}

	// set lights
	CreateSpotlights();

	//VESSEL2M stuff
	VESSEL2M::FinishConfiguration();
}



void R7_launchpad::clbkPostCreation()
{
	//VESSEL2M stuff
	VESSEL2M::clbkPostCreation();

	// add button sound
	RequestLoadVesselWave3(SoundLibID, SND_BUTTON, "Sound\\r7_S\\Button.wav", BOTHVIEW_FADED_FAR);

	// move pad to the proper historic location via direct status update
	VESSELSTATUS vs;
	memset(&vs, 0, sizeof(vs)); // Orbiter bug workaround
	GetStatus(vs);

	vs.status = 1;
	vs.vdata[0].x = LP_LON * RAD;
	vs.vdata[0].y = LP_LAT * RAD;
	vs.vdata[0].z = LP_AZM * RAD;

	DefSetState(&vs);

	// always restore propellant for the virtual engines
	SetPropellantMass(MainTank, PropellantMass);
}


int R7_launchpad::clbkConsumeBufferedKey(DWORD key, bool down, char *kstate)
{
	// VESSEL2M stuff
	int result = VESSEL2M::clbkConsumeBufferedKey(key, down, kstate);
	if (result != 0)
		return result;

	// only process keydown events
	if (!down) 
		return 0; 

	// shift combinations are reserved
	if (KEYMOD_SHIFT (kstate)) 
		return 0; 

	// control combinations 
	if (KEYMOD_CONTROL (kstate)) 
	{
		// launch escape
		if (key == OAPI_KEY_A) 
			SendVesselMessage(Launcher.hObject, VM_LAUNCH_ESCAPE);

		return 0; 
	}

	// release supports manually (if rocket not installed)
	if (key == OAPI_KEY_G) 
	{
		if (Launcher.ObjectName[0] == 0)
		{
			bSupportFalling = true;
			SupportDropOmega = SUPPORT_DROP_OMEGA_START;
		}
		else
			ShowAnnotationMessage("Rocket supports cannot be released when rocket is on the pad!");
	}

	// drop cable mast manually (in manual mode)
	if (key == OAPI_KEY_K) 
	{
		if (!bAutopilot)
		{
			bCableMastFalling = true;
			CableMastDropOmega = CABLE_MAST_DROP_OMEGA_START;
			SendVesselMessage(Launcher.hObject, VM_SET_PRE_LAUNCH_VENT, 0);
		}
		else
			ShowAnnotationMessage("Manual controls are disabled. To enable, use M key.");
	}


	// drop fueling mast manually (in manual mode)
	if (key == OAPI_KEY_N) 
	{
		// fuel mast, only if present
		if (FuelingMastMesh > 0)
		{
			if (!bAutopilot)
			{
				bFuelingMastFalling = true;
				FuelingMastDropOmega = FUELING_MAST_DROP_OMEGA_START;
					SendVesselMessage(Launcher.hObject, VM_SET_PRE_LAUNCH_VENT_3ST, 0);
			}
			else
				ShowAnnotationMessage("Manual controls are disabled. To enable, use M key.");
		}
	}

	// (re)start/pause autopilot (or rather, automatic launch sequence)
	if (key == OAPI_KEY_M) 
		ToggleManualAndAutoplilot();

	// skip full lengthy launch sequence, launch immediately!
	if (key == OAPI_KEY_L) 
	{
		if (Launcher.ObjectName[0] != 0)
		{
			if (bAutopilot)
			{
				AnimateRingToAzimuth();
				AnimateRing(true);
			}
			else
				bLaunchSequence = true;

			BoomRotPercent = 0;
			CableMastRotPercent = 0;
			FuelingMastRotPercent = 0;
			PlatformTranslPercent = 0;
			SetAnimation (BoomRotAnim, BoomRotPercent);
			SetAnimation (CableMastRotAnim, CableMastRotPercent);
			SetAnimation (PlatformTranslAnim, PlatformTranslPercent);

			SendVesselMessage(Launcher.hObject, VM_SET_PRE_LAUNCH_VENT, 0);

			if (FuelingMastMesh > 0)
			{
				SetAnimation (FuelingMastRotAnim, FuelingMastRotPercent);
				SendVesselMessage(Launcher.hObject, VM_SET_PRE_LAUNCH_VENT_3ST, 0);
			}

			Timer = -4;
		}
	}

	if (key == OAPI_KEY_D) 
		DebugSomething();

	return 0;
}


int R7_launchpad::clbkConsumeDirectKey (char *keystate)
{
	// override "engine" keys
	RESETKEY(keystate, OAPI_KEY_ADD);
	RESETKEY(keystate, OAPI_KEY_SUBTRACT);
	RESETKEY(keystate, OAPI_KEY_MULTIPLY);

	// don't allow regular "steady" animations if someting is being dropped, or in autopilot!
	if ( bAutopilot || bSupportFalling || bCableMastFalling)
	{
		if	(	KEYDOWN (keystate, OAPI_KEY_LEFT ) + KEYDOWN (keystate, OAPI_KEY_RIGHT ) + 
				KEYDOWN (keystate, OAPI_KEY_UP ) + KEYDOWN (keystate, OAPI_KEY_DOWN ) + 
				KEYDOWN (keystate, OAPI_KEY_NUMPAD8 ) + KEYDOWN (keystate, OAPI_KEY_NUMPAD2 ) + 
				KEYDOWN (keystate, OAPI_KEY_NUMPAD4 ) + KEYDOWN (keystate, OAPI_KEY_NUMPAD6 ) + 
				KEYDOWN (keystate, OAPI_KEY_LBRACKET ) + KEYDOWN (keystate, OAPI_KEY_RBRACKET ) 
			)		
			ShowAnnotationMessage("Manual controls disabled. To enable, use M key.");
		
		// fuel mast, only if present
		if (FuelingMastMesh > 0)
			if	(KEYDOWN (keystate, OAPI_KEY_COMMA ) + KEYDOWN (keystate, OAPI_KEY_PERIOD ))
				ShowAnnotationMessage("Manual controls disabled. To enable, use M key.");
		
		return 0;
	}

	if	( KEYDOWN (keystate, OAPI_KEY_LEFT ) )
		RingTurning = 1;

	if	( KEYDOWN (keystate, OAPI_KEY_RIGHT ) )
		RingTurning = -1;

	if	( KEYDOWN (keystate, OAPI_KEY_UP ) )
		BoomTurning = 1;

	if	( KEYDOWN (keystate, OAPI_KEY_DOWN ) )
		BoomTurning = -1;

	if	( KEYDOWN (keystate, OAPI_KEY_NUMPAD8 ) )
	{
		if (Launcher.ObjectName[0] == 0)
			SupportTurning = 1;
		else
			ShowAnnotationMessage("Rocket supports cannot be released when rocket is on the pad!");
	}

	if	( KEYDOWN (keystate, OAPI_KEY_NUMPAD2 ) )
	{
		if (Launcher.ObjectName[0] == 0)
			SupportTurning = -1;
		else
			ShowAnnotationMessage("Rocket supports cannot be released when rocket is on the pad!");
	}

	if	( KEYDOWN (keystate, OAPI_KEY_NUMPAD4 ) )
		PlatformTranslating = -1;

	if	( KEYDOWN (keystate, OAPI_KEY_NUMPAD6 ) )
		PlatformTranslating = 1;

	if	( KEYDOWN (keystate, OAPI_KEY_LBRACKET ) )
		CableMastTurning = -1;

	if	( KEYDOWN (keystate, OAPI_KEY_RBRACKET ) )
		CableMastTurning = 1;

	if	( KEYDOWN (keystate, OAPI_KEY_COMMA ) )
	{
		if (FuelingMastMesh > 0)
			FuelingMastTurning = 1;
	}

	if	( KEYDOWN (keystate, OAPI_KEY_PERIOD ) )
	{
		if (FuelingMastMesh > 0)
			FuelingMastTurning = -1;
	}

	return 0;
}


void R7_launchpad::ToggleManualAndAutoplilot()
{
	// implement switching to the appropriate mode
	if (bAutopilot)
		TrySetManualControl();
	else
		TrySetAutopilotControl();
}

void R7_launchpad::TrySetManualControl()
{
	// don't allow manual control after ignition %%% later, after "launch key"
	if ((Timer >= TIMER_IGNITION) && (Timer <= 2))
	{
		ShowAnnotationMessage("Cannot switch to manual control mode after ignition!");
		return;
	}

	ShowAnnotationMessage("Manual controls enabled. To return to autopilot, use M key again.");
	sprintf(oapiDebugString(), "");

	bAutopilot = false;

	// if we have rocket, tell it the new autopilot mode
	SendVesselMessage(Launcher.hObject, VM_SET_AUTOPILOT, 0);
}


void R7_launchpad::TrySetAutopilotControl()
{
	if (Timer == TIMER_NOT_SET)
	{
		ShowAnnotationMessage("Cannot switch to auto sequence mode: launch time is not set!");
		return;
	}

	if (!bTrajectoryIsSet)
	{
		ShowAnnotationMessage("Cannot switch to auto sequence mode: trajectory is not set!");
		return;
	}

	if (Launcher.ObjectName[0] == 0)
	{
		ShowAnnotationMessage("Cannot switch to auto sequence mode: rocket is not installed!");
		return;
	}

	ShowAnnotationMessage("Manual controls disabled. To enable, use M key again.");

	bAutopilot = true;

	// if we have rocket, tell it the new autopilot mode
	SendVesselMessage(Launcher.hObject, VM_SET_AUTOPILOT, 1);
}


void R7_launchpad::InitializeBallisticTargets()
{
	// initialzie pre-defined target coords (%%% good candidate for config file)

	strcpy(BalTargets[0].NameCamel, "Kura");
	strcpy(BalTargets[0].NameCaps, "KURA");
	BalTargets[0].Lat = 57.33333;
	BalTargets[0].Lon = 161.83333;
	BalTargets[0].RangeAdjustment = 2.4;
	BalTargets[0].AzimuthAdjustment = 1.05;

	strcpy(BalTargets[1].NameCamel, "NewYork");
	strcpy(BalTargets[1].NameCaps, "NEW YORK");
	BalTargets[1].Lat =  40.7120;
	BalTargets[1].Lon = -74.0129;
	BalTargets[1].RangeAdjustment = 2.33;
	BalTargets[1].AzimuthAdjustment = 0.9;

	strcpy(BalTargets[2].NameCamel, "Washington");
	strcpy(BalTargets[2].NameCaps, "WASHINGTON");
	BalTargets[2].Lat =  38.8977;
	BalTargets[2].Lon = -77.0365;
	BalTargets[2].RangeAdjustment = 2.45;
	BalTargets[2].AzimuthAdjustment = 0.85;

	strcpy(BalTargets[3].NameCamel, "Chicago");
	strcpy(BalTargets[3].NameCaps, "CHICAGO");
	BalTargets[3].Lat =  41.8889;
	BalTargets[3].Lon = -87.6245;
	BalTargets[3].RangeAdjustment = 2.31;
	BalTargets[3].AzimuthAdjustment = 0.85;

	strcpy(BalTargets[4].NameCamel, "LosAngeles");
	strcpy(BalTargets[4].NameCaps, "LOS ANGELES");
	BalTargets[4].Lat =   34.0521;
	BalTargets[4].Lon = -118.2541;
	BalTargets[4].RangeAdjustment = 2.52;
	BalTargets[4].AzimuthAdjustment = 0.45;

	strcpy(BalTargets[5].NameCamel, "Canaveral");
	strcpy(BalTargets[5].NameCaps, "CANAVERAL");
	BalTargets[5].Lat = 28.58606;
	BalTargets[5].Lon = -80.65115;
	BalTargets[5].RangeAdjustment = 2.2;
	BalTargets[5].AzimuthAdjustment = 0.6;

	// check which one of those do we have in scenario, defaulting to "other"
	PanBalTgtIdx = 7;
	for (int i=0; i<6; i++)
		if (!strnicmp(TargetName, BalTargets[i].NameCamel, strlen(BalTargets[i].NameCamel)))
		{
			TargetLat = BalTargets[i].Lat;
			TargetLon = BalTargets[i].Lon;
			RangeAdjustment = BalTargets[i].RangeAdjustment;
			AzimuthAdjustment = BalTargets[i].AzimuthAdjustment;
			PanBalTgtIdx = i+1;
		}
}


void R7_launchpad::VerifyTargeting()
{
	// disable autopilot until we decide it is OK
	bool bAutopilotCached = bAutopilot; 
	bAutopilot = false;

	// ignore this function if there is no rocket
	if (Launcher.ObjectName[0] == 0)
		return;

	// get payload type
	PayloadType = (VM_PayloadTypes)SendVesselMessage(Launcher.hObject, VM_PAYLOAD_TYPE);
	
	// find trajectory type
	if ( (PayloadType == PAYLOAD_UNKNOWN) || (PayloadType >= PAYLOAD_SAT1) )
		TrajectoryType = TRAJECTORY_ORBITAL;
	
	else
	{
		TrajectoryType = TRAJECTORY_BALLISTIC;
		InitializeBallisticTargets();
	}
	
		// sanity check of loaded scenario

	// ballistic trajectory checks
	if (TrajectoryType == TRAJECTORY_BALLISTIC)
		VerifyTargetingBallistic(TargetName, TargetLat, TargetLon);

	// orbital trajectory checks
	else 
		VerifyTargetingOrbital(Perigee, Apogee, Inc);

	// get timer candidate setting
	double tempTimer = TimerFromMJD();

	// launch time checks
	VerifyLaunchTime(tempTimer, LaunchMJD);

	// if timer is OK, set it and cancel LaunchMJD as not needed anymore
	if (sTimeWarning[0] == 0)
	{
		Timer = tempTimer;
		LaunchMJD = 0;
	}

	// if timer is not OK, report it, unless target is not OK as well
	if (sScenarioWarning [0] == 0)
		strcpy(sScenarioWarning, sTimeWarning);

	// were there any targeting errors in scenario
	if (sScenarioWarning[0] != 0)
	{
		char strTemp[256];
		sprintf(strTemp, "Scenario warning: %s Flight program not set.", sScenarioWarning);
		strcpy(sScenarioWarning, strTemp);

		bScenarioWarning = true;

		return;
	}

	// verify ballistic numbers 
	if (TrajectoryType == TRAJECTORY_BALLISTIC)
	{
		// calculate all trajectory
		CalcBallisticAzimuth();

		// pass trajectory params to the rocket
		SendVesselMessage(Launcher.hObject, VM_SET_RANGE, Range);
		SendVesselMessage(Launcher.hObject, VM_SET_AZIMUTH, Azimuth*RAD);
		SendVesselMessage(Launcher.hObject, VM_SET_LAT, LP_LAT * RAD);
		SendVesselMessage(Launcher.hObject, VM_SET_LON, LP_LON * RAD);
		SendVesselMessage(Launcher.hObject, VM_SET_RANGE_ADJUSTMENT, RangeAdjustment);
		if (TargetName[0] == 0)
			strcpy(TargetName, "Other");
		SendVesselMessage(Launcher.hObject, VM_SET_TGT_LAT, TargetLat * RAD);
		SendVesselMessage(Launcher.hObject, VM_SET_TGT_LON, TargetLon * RAD);
	}

	// set orbit
	if (TrajectoryType == TRAJECTORY_ORBITAL)
	{
		// calculate trajectory
		Azimuth = DEG * asin(cos(Inc * RAD)/cos(LP_LAT * RAD));

		if (Inc < 0)
			Azimuth = -Azimuth;

		if (Azimuth < 0)
			Azimuth +=360;

		// convert to meters and radii and rad when sending to rocket
		SendVesselMessage(Launcher.hObject, VM_SET_APOGEE, Apogee*1000+6371000);
		SendVesselMessage(Launcher.hObject, VM_SET_PERIGEE, Perigee*1000+6371000);
		SendVesselMessage(Launcher.hObject, VM_SET_INCLINATION, Inc*RAD);
		SendVesselMessage(Launcher.hObject, VM_SET_AZIMUTH, Azimuth*RAD);
	}

	// if we got here, we have a good trajectory
	bTrajectoryIsSet = true;

	// restore original autopilot setting, whatever it was
	bAutopilot = bAutopilotCached;
}


void R7_launchpad::VerifyTargetingBallistic(char *Name, double Lat, double Lon)
{
	sScenarioWarning[0] = 0;

	// consider unknown target name without coordinates as error, with coordinates - use given name in kml
	if ( ( Name[0] != 0 ) && ( (Lat == 0) || (Lon == 0) ) )
		sprintf(sScenarioWarning, "Unknown target \"%s\" with no coordinates.", Name);

	// orbit specified for ballistic payload?
	else if ( (Apogee + Perigee + Inc) != 0) 
		sprintf(sScenarioWarning, "Payload not suitable for orbital trajectory.");

	// all required numbers set?
	else if (Lat == 0) 
		sprintf(sScenarioWarning, "Target latitude not specified.");
	else if (Lon == 0) 
		sprintf(sScenarioWarning, "Target longitude not specified.");

	// are they within input range?
	else if (fabs(Lat) > 90)
		sprintf(sScenarioWarning, "Target latitude %9.4lf is incorrect.", Lat);
	else if (fabs(Lon) > 180) 
		sprintf(sScenarioWarning, "Target longitude %9.4lf is incorrect.", Lon);

	// is flight range within the preset limits?
	else
	{
		double tempRange, tempAzimuth;
		DistanceAzimuthFromCoordPairKmDg(Lat*RAD, Lon*RAD, LP_LAT*RAD, LP_LON*RAD, &tempRange, &tempAzimuth);
		tempRange /= 1000;

		if (tempRange < 6000)
			sprintf(sScenarioWarning, "Target range of %6.0lf km is too short.", tempRange);
		else if (tempRange > 12000) 
			sprintf(sScenarioWarning, "Target range of %6.0lf km is too long.", tempRange);
	}
}


void R7_launchpad::VerifyTargetingOrbital(double perigee, double apogee, double inc)
{
	sScenarioWarning[0] = 0;

	// ballistic target specified for orbital payload?
	if ( (TargetLat+TargetLon) != 0)
		sprintf(sScenarioWarning, "Payload not suitable for ballistic trajectory.");

	// all required numbers set?
	else if (perigee <= 0)
		sprintf(sScenarioWarning, "Orbit perigee not specified.");
	else if (apogee <= 0)
		sprintf(sScenarioWarning, "Orbit apogee not specified.");
	else if (inc == 0) 
		sprintf(sScenarioWarning, "Orbit inclination not specified.");

	// are they within reacheable range?
	else if (apogee < perigee)
		sprintf(sScenarioWarning, "Orbit apogee %03.0f is lower than the perigee %03.0f.", apogee, perigee);
	else if (perigee < 190)
		sprintf(sScenarioWarning, "Orbit perigee %03.0f is set too low.", perigee);
	else if (fabs(inc) > 90)
		sprintf(sScenarioWarning, "Orbit inclination %05.1f is incorrect.", inc);
	else if (fabs(inc) < LP_LAT)
		sprintf(sScenarioWarning,  "Orbit inclination %05.1f is not reacheable.", inc);
}


void R7_launchpad::VerifyLaunchTime(double TMinus, double MJD)
{
	if (TMinus == TIMER_NOT_SET)
		sprintf(sTimeWarning, "Launch time is not set.");
	else if (TMinus < -24*3600)
		sprintf(sTimeWarning, "Launch time is set too late from current time.");
	
	// global timer is set to something, and we are initializing timer from original scenario
	else if ((TMinus != TIMER_NOT_SET) && (MJD != 0))
	{
		if (TMinus >= 0)
			sprintf(sTimeWarning, "Launch time is set in the past.");
		if (TMinus > -100)
			sprintf(sTimeWarning, "Launch time is set too close to current time.");
	}

	// update intermediate vars
	TimeSecondsToStr(TMinus, strTimer);
}


double R7_launchpad::TimerFromMJD()
{
	// no rocket? ignore timer/MJD settings
	if (Launcher.ObjectName[0] == 0)
		return TIMER_NOT_SET;

	// normally, Timer gets read directly from scenario
	if (Timer != TIMER_NOT_SET)
		return Timer;

	// override Timer from LaunchMJD if set, only use Timer from this point
	if (LaunchMJD != 0)
		return (oapiGetSimMJD()- LaunchMJD) * 24 * 3600; // converting MJD to seconds

	// return nothing
	return TIMER_NOT_SET;
}



int R7_launchpad::GetVesselMessage(IntVesselMessage VM, int Value)
{
	// it is not us (the launch pad) detaching from the ground :-).
	// it is a signal from the rocket that it is lifting off
	if (VM == VM_REQUEST_DETACH)
	{
		// enable kml  plotting, add launch event
		SendVesselMessage(kmlHandle, VM_KML_START_PLOTTING, (int)Launcher.hObject);
		kmlAddEvent(KML_LAUNCH, "Launch!");

		// properly detach rocket, keep its handle for some time more for smoke-sound exchange
		DetachChild(Launcher.hAttachment, 0);
		Launcher.ObjectName[0] = 0;
		Launcher.ObjectIntf2M->bAttached = false;

		// drop supports
		bSupportFalling = true;
		SupportDropOmega = SUPPORT_DROP_OMEGA_START;

		// drop cable mast(s) in case of premature separation
		bCableMastFalling = true;
		CableMastDropOmega = CABLE_MAST_DROP_OMEGA_START;
		SendVesselMessage(Launcher.hObject, VM_SET_PRE_LAUNCH_VENT, 0);

		// drop fueling mast(s) in case of premature separation
		if (FuelingMastMesh > 0)
		{
			bFuelingMastFalling = true;
			FuelingMastDropOmega = FUELING_MAST_DROP_OMEGA_START;
			SendVesselMessage(Launcher.hObject, VM_SET_PRE_LAUNCH_VENT_3ST, 0);
		}

		// stop any residual non-dropping animations
		bRingToAzimuth = false;
		bBoomsTurnDown = false;
		bPlatformRetract = false;

		// destroy rocket if booms are up
		if (BoomRotPercent > 0.6)
			SendVesselMessage(Launcher.hObject, VM_DESTROY, 0);

		// reset timer to prevent launch sequence
		Timer = 0;

		// ballistic? 
		if (TargetName[0] != 0)
		{
			// add target name, range and azimuth to kml
//			char TargetLabel[255];
//			sprintf(TargetLabel, "%s. R %.0f km. Az %.0f deg.",  TargetName, Range, Azimuth);
			kmlAddEvent(KML_TARGET, TargetName, TargetLat, TargetLon);
		}
	}

	else if (VM == VM_ABORT)
	{
		bAutopilot = false;
		bLaunchSequence = false;
		Timer = TIMER_NOT_SET;
		LastTE = TE_NONE;
		LaunchMJD = 0;
	}

	else if (VM == VM_DESTROYED)
	{
		Launcher.hObject = NULL;
		SmokeLevelMain = FlameLevel = SmokeLevelSides = 0;
		SetThrusterGroupLevel(THGROUP_MAIN, 0); 
		StopVesselWave3(SoundLibID, SND_ENGINE);
	}

	return  VESSEL2M::GetVesselMessage(VM, Value);

}




		// ========================== TIME MACHINE ========================



void R7_launchpad::clbkPostStep(double simt, double SimDT, double mjd)
{
	// common processing for all VESSEL2M-derived objects
	TimestepForVESSEL2M();

	// %%% check for inactivity to disable operations and save timesteps
	;

	// store time step
	SimDt = SimDT;

	// run animations
	RunAnimations();

	// timer-sequenscer 
	UpdateTimer();

	// update smokes and sounds
	UpdateSmokes();

	// are there any scenario warnings? one-time event
	if (bScenarioWarning)
	{
		sprintf(oapiDebugString(), sScenarioWarning);
		bScenarioWarning = false;
	}
}


void R7_launchpad::UpdateTimer()
{
	// update lights timer: check light conditions every 10 minutes
	LightsTimer += SimDt;
	if (LightsTimer > LIGHTS_TIMER)
	{
		LightsTimer = 0;
		VECTOR3 EPos, SPos;
		GetRelativePos(hEarth, EPos);
		GetGlobalPos(SPos);
		double dp = dotp(EPos, SPos) / (length(EPos)*length(SPos));
		double SunAngle = acos(dp) * DEG;

		if (fabs(SunAngle) < 90)
			SetThrusterGroupLevel(thg_lights, 1); 
		else
			SetThrusterGroupLevel(thg_lights, 0); 

		if (fabs(SunAngle) < 100)
		{
			for (int i = 0; i < BEACON_COUNT; i++)
			{
				if ((TableTopMeshName[0]!= 0) && (i==0) )
					continue;

				Beacons[i].active = true;
			}
		}
		else
		{
			for (int i = 0; i < BEACON_COUNT; i++)
				Beacons[i].active = false;
		}
	}

	// do we have to update timer at all?
	if (!bAutopilot)
		if (!bLaunchSequence)
			return;

	// update timer
	Timer += SimDt;

	// format timer into string
	TimeSecondsToStr(Timer, strTimer);

	// report MET
	sprintf(oapiDebugString(), "Time: T%s", strTimer);

	
	// debug output in every timestep
//	sprintf(oapiDebugString(), "Az: %.0f dg, %f rad", Azimuth, Azimuth*RAD);
//sprintf(oapiDebugString(), "V: %f , %f %f %f", length(VV), VV.x, VV.y, VV.z);
// sprintf(oapiDebugString(), "tp: %f ", tt);


	// walk timer events in reverse order

	if (Timer > 15)
		RunTimerEvent(TE_STOP_TIMER);

	else if (Timer > 0)
		RunTimerEvent(TE_MAIN);

	else if (Timer > -3)
		RunTimerEvent(TE_INTERMEDIATE);

	else if (Timer > -9)
		RunTimerEvent(TE_PRELIMINARY);

	else if (Timer > -12)
		RunTimerEvent(TE_IGNITION);

	else if (Timer > -18)
		RunTimerEvent(TE_CABLE_MAST_DROP);

	else if (Timer > -22)
		RunTimerEvent(TE_FUELING_MAST_DROP);

	else if (Timer > -25)
		RunTimerEvent(TE_VENT_STOP);

	else if (Timer > -60)
		RunTimerEvent(TE_BOOMS_DOWN);

	else if (Timer > -65)
		RunTimerEvent(TE_PLATFORM_RETRACT);

	else if (Timer > -100)
		RunTimerEvent(TE_TURN_TABLE);
}


void R7_launchpad::TimeSecondsToStr(double Seconds, char* strTime)
{
	if (Seconds == TIMER_NOT_SET)
		Seconds = 0;

	fabsTimer = fabs(Seconds);
	iTimerH = max((int)(fabsTimer/3600),0);
	iTimerM = max((int)(fabsTimer-iTimerH*3600)/60,0);
	iTimerS = max((int)((fabsTimer-iTimerH*3600-iTimerM*60)),0);

	if (Seconds < 0)
		sprintf(strTime, "-%02d:%02d:%02d",iTimerH, iTimerM, iTimerS);
	else
		sprintf(strTime, "%02d:%02d:%02d",iTimerH, iTimerM, iTimerS);
}


void R7_launchpad::RunTimerEvent(TimerEvents TE)
{
	// already done event? ignore
	if (TE == LastTE)
		return;

	// new event? reset!
	LastTE = TE;

	// run event once

	if (TE == TE_STOP_TIMER)
	{
		// stop timer and clean up
		bAutopilot = false; 
		bLaunchSequence = false;
		sprintf(oapiDebugString(), "");

		// switch to object
		if (bHaveFocus)
			if (oapiIsVessel (Launcher.hObject))			//%%% not enough, should better trace launcher failures
				oapiSetFocusObject(Launcher.hObject);
	}

	else if (TE == TE_MAIN)
		SendVesselMessage(Launcher.hObject, VM_SET_MAIN_ENGINE, 1);

	else if (TE == TE_INTERMEDIATE)
		SendVesselMessage(Launcher.hObject, VM_SET_MAIN_ENGINE, 0.5);

	else if (TE == TE_PRELIMINARY)
		SendVesselMessage(Launcher.hObject, VM_SET_MAIN_ENGINE, 0.25);

	else if (TE == TE_VENT_STOP)
	{
		SendVesselMessage(Launcher.hObject, VM_SET_PRE_LAUNCH_VENT, 0);
		SendVesselMessage(Launcher.hObject, VM_SET_PRE_LAUNCH_VENT_3ST, 0);
	}

	else if (TE == TE_CABLE_MAST_DROP)
		bCableMastFalling = true;

	else if (TE == TE_FUELING_MAST_DROP)
	{
		if (FuelingMastMesh > 0)
			bFuelingMastFalling = true;
	}

	else if (TE == TE_IGNITION)
		SendVesselMessage(Launcher.hObject, VM_SET_MAIN_ENGINE, 0.05);

	else if (TE == TE_BOOMS_DOWN)
		bBoomsTurnDown = true;

	else if (TE == TE_PLATFORM_RETRACT)
		bPlatformRetract = true;

	else if (TE == TE_TURN_TABLE)
	{
		bRingToAzimuth = true;
		if (oapiGetTimeAcceleration() > 10)
			oapiSetTimeAcceleration(10);
	}
}


// update additional smokes and engine sounds
void R7_launchpad::UpdateSmokes()
{
	// normal takeoff
	if (Launcher.hObject == NULL)
		return;

	// explosion, %%% not sure how reliable this check is...
	if (!oapiIsVessel(Launcher.hObject))
	{
		Launcher.hObject = NULL;
		SmokeLevelMain = FlameLevel = SmokeLevelSides = 0;
		SetThrusterGroupLevel(THGROUP_MAIN, 0); 
		return;
	}

	// get distance
	VECTOR3 R7Pos;
	GetRelativePos(Launcher.hObject, R7Pos);
	double R7Dist = length(R7Pos);

	// stop updating smokes when rocket got away by one km
	if (R7Dist > 1000)
	{
		// reset the rocket handle, otherwise the function may get re-activated 
		// if the rocket returns back during the failed launch...
		Launcher.hObject = NULL;
		SmokeLevelMain = FlameLevel = SmokeLevelSides = 0;
		SetThrusterGroupLevel(THGROUP_MAIN, 0); 
		return;
	}

	// get rocket engine level
	double R7Level = Launcher.ObjectIntf2M->GetVesselMessage(VM_GET_MAIN_ENGINE);

	// set smoke effects according to the rocket engine level and distance
	SmokeLevelMain = R7Level * (1-R7Dist/200);
	FlameLevel = SmokeLevelMain/2;
	SmokeLevelSides = SmokeLevelMain/2;
}




void R7_launchpad::CalcBallisticAzimuth()
{
	double LaunchLatR = LP_LAT * RAD;
	double LaunchLonR = LP_LON * RAD;
	double TargetLatR = TargetLat * RAD;
	double TargetLonR = TargetLon * RAD;

	// velocity of Earth rotation, Eastward, m/s
	double VE = RADIUS_E * OMEGA_E * sin(LaunchLatR);
	
	// ICBM velocity at MECO, m/s
	double Vk;

	// first iteration on azimuth: direct flight to target coords on non-rotating Earth
	double _targetLatR = TargetLatR;
	double _targetLonR = TargetLonR;

	// iteration vars
	double _pastTargetLonR = - TargetLonR;
	double _pastRange = 0;
	int i = 0;

	bool bAzimuthAjdusted = false;
	bool bRangeAjdusted = false;
	bool bBothParamsAjdusted = false;

	// empirical, good consistency
	double ThrowAngleR = 22*RAD;				// trajectory angle at the end of active flight?
	double ThrowAlt = 350*1000;				// Altitude at the engine cutoff, 300 km
	double ThrowRadius = RADIUS_E + ThrowAlt;

	//
	while ((fabs(_targetLonR -_pastTargetLonR) > 0.001*RAD))
	{
		// debug interations counter 
		i++;

		// save current iteration for next run check
		_pastTargetLonR = _targetLonR;

		// update ballistic range and zzinuth to the new iteration coords
		double AngRangeR;
		DistanceAzimuthFromCoordPairRad(LaunchLatR, LaunchLonR, _targetLatR, _targetLonR, &AngRangeR, &Azimuth);

		// store possible final values to be passed to ICBM
		Azimuth *= DEG;
		Range = AngRangeR * RADIUS_E;

		// adjust ballistic range to include only ballistic portion from MECO symmetrical around apogee
		AngRangeR -= 5*2.5*RAD; // empirical, from kml, double range of MECO
	
		// take half of remaining ballistic range, MECO to apogee
		double AngRangeApoR = AngRangeR / 2;

		// find Vk, from Feodosiev (7.29)
		double _vk = (tan(ThrowAngleR)+tan(AngRangeApoR)*tan(AngRangeApoR))/(tan(AngRangeApoR)+tan(ThrowAngleR));
		double Vcirc = sqrt(K_E/ThrowRadius);
		Vk = sqrt(_vk*Vcirc*Vcirc);

		// find the rest of the orbit params, from Feodosiev (7.26)
		double p = ThrowRadius * _vk * cos(ThrowAngleR)*cos(ThrowAngleR);
		double e = sqrt( 1 - (2 - _vk)*_vk * cos(ThrowAngleR)*cos(ThrowAngleR) );
		double a = p /(1 - e*e);
		double T = 2 * PI * sqrt(a*a*a) / sqrt(K_E);

		// find flight time of the symmetrical ballistic portion
		double TrueAnoThrowR = PI - AngRangeApoR;
		double TimePeriToThrow = TimeOfTrueAnomaly(TrueAnoThrowR, e, T);
		double TrueAnoApoR = PI;
		double TimePeriToApo = T / 2;
		double tp = (TimePeriToApo - TimePeriToThrow)*2;

		// add time for active and final legs, empirical, 250 and 150 s correspondingly
		tp += 400;			

		// find new target longitude 
		_targetLonR = TargetLonR + OMEGA_E * tp;

		//bBothParamsAjdusted = (fabs(_targetLonR -_pastTargetLonR) > 0.05*RAD);
	}

	// calculate AzimuthAdjustment and RangeAdjustment
	CalcAdjustments();

	double Vlon = Vk * sin(Azimuth*RAD);
	double Vlat = Vk * cos(Azimuth*RAD);
	Vlon -= (VE * AzimuthAdjustment); // 
	Vk = sqrt(Vlon*Vlon + Vlat*Vlat);
	Azimuth = asin(Vlon/Vk)*DEG;

	if ((Vlon<0) && (Vlat<0))
		Azimuth = (-180-Azimuth);

	if (Azimuth < 0)
		Azimuth +=360;
}

double KuraRanges[3] = {6000000, 6606000, 11000000};
double KuraAdjRng[3] = {2.4, 2.4, 2.3};
double KuraAdjAz[3] = {1.05, 1.05, 1.0};

double EastRanges[6] = {6000000, 9124000, 9419000, 9549000, 10507000, 11000000};
double EastAdjRng[6] = {2.4, 2.33, 2.45, 2.31, 2.2, 2.2};
double EastAdjAz[6] = {0.9, 0.9, 0.85, 0.85, 0.6, 0.6};

double WestRanges[3] = {6000000, 9179000, 11000000};
double WestAdjRng[3] = {2.4, 2.465, 2.52};
double WestAdjAz[3] = {0.9, 0.9, 0.45};

void R7_launchpad::GetAdjustments(int n, double Ranges[], double AdjRng[], double AdjAz[])
{
	if (Range < Ranges[0])
	{
		RangeAdjustment = AdjRng[0];
		AzimuthAdjustment = AdjAz[0];
		return;
	}

	if (Range > Ranges[n-1])
	{
		RangeAdjustment = AdjRng[n-1];
		AzimuthAdjustment = AdjAz[n-1];
		return;
	}

	int i1;
	for (int i=0; i < n-1; i++)
		if ((Range > Ranges[i]) && (Range < Ranges[i+1]))
			i1 = i;

	double R1 = Ranges[i1];
	double R2 = Ranges[i1+1];
	double r1 = AdjRng[i1];
	double r2 = AdjRng[i1+1];
	double a1 = AdjAz[i1];
	double a2 = AdjAz[i1+1];

	double x = Range - R1;
	double Lx = R2 - R1;
	double Lr = r2 - r1;
	double La = a2 - a1;
	
	double r = x * (Lr/Lx);
	double a = x * (La/Lx);

	RangeAdjustment = r1 + r;
	AzimuthAdjustment = a1 + a;
}


void R7_launchpad::CalcAdjustments()
{
	// already set for predefined target, skip the rest
	if ((RangeAdjustment != 0) || (AzimuthAdjustment != 0))
		return;

	// Kura, etc
	else if (Azimuth < 180)
		GetAdjustments(3, KuraRanges, KuraAdjRng, KuraAdjAz);

	// US east
	else if (Azimuth < 340)
		GetAdjustments(6, EastRanges, EastAdjRng, EastAdjAz);

	// US west
	else 
		GetAdjustments(3, WestRanges, WestAdjRng, WestAdjAz);
}











		// ======================= ANIMATIONS AND VISUALS ============================


void R7_launchpad::LoadMeshes()
{
	// flame trench and table 
	UINT TableBaseMesh = AddMesh(oapiLoadMeshGlobal("r7_S\\TableBase"), &TABLE_OFF); 

	// custom Soyuz meshes from scenario, only after 1967
	UINT TableTopMesh;
	if ((ScnYear >= 1967) && (TableTopMeshName[0] != 0))
		TableTopMesh = AddMesh(oapiLoadMeshGlobal(TableTopMeshName), &TABLE_OFF); 
	else
		TableTopMesh = AddMesh(oapiLoadMeshGlobal("r7_S\\TableTop1Early"), &TABLE_OFF); 

	UINT TrenchMesh = AddMesh(oapiLoadMeshGlobal("r7_S\\TrenchB1"), &TRENCH_OFF); 
	SetMeshVisibilityMode(TableBaseMesh, MESHVIS_ALWAYS | MESHVIS_EXTPASS);
	SetMeshVisibilityMode(TableTopMesh, MESHVIS_ALWAYS | MESHVIS_EXTPASS);
	SetMeshVisibilityMode(TrenchMesh, MESHVIS_ALWAYS | MESHVIS_EXTPASS);

	// supports 
	SupportMesh045 = AddMesh(oapiLoadMeshGlobal("r7_S\\Support045"), &SUPPORT_045_OFS);
	SupportMesh135 = AddMesh(oapiLoadMeshGlobal("r7_S\\Support135"), &SUPPORT_135_OFS);
	SupportMesh225 = AddMesh(oapiLoadMeshGlobal("r7_S\\Support225"), &SUPPORT_225_OFS);
	SupportMesh315 = AddMesh(oapiLoadMeshGlobal("r7_S\\Support315"), &SUPPORT_315_OFS);

	SetMeshVisibilityMode(SupportMesh045, MESHVIS_ALWAYS | MESHVIS_EXTPASS);
	SetMeshVisibilityMode(SupportMesh135, MESHVIS_ALWAYS | MESHVIS_EXTPASS);
	SetMeshVisibilityMode(SupportMesh225, MESHVIS_ALWAYS | MESHVIS_EXTPASS);
	SetMeshVisibilityMode(SupportMesh315, MESHVIS_ALWAYS | MESHVIS_EXTPASS);

	// platform
	PlatformMesh = AddMesh(oapiLoadMeshGlobal("r7_S\\Platform"), &PLATFORM_OFS);
	SetMeshVisibilityMode(PlatformMesh, MESHVIS_ALWAYS | MESHVIS_EXTPASS);

	PlatformRingMesh = AddMesh(oapiLoadMeshGlobal("r7_S\\PlatformRing"), &PLATFORM_OFS);
	SetMeshVisibilityMode(PlatformRingMesh, MESHVIS_ALWAYS | MESHVIS_EXTPASS);

	// original configurations for ICBM and satellites
	if (ScnYear <= 1958)
	{
		// ring 
		RingMesh = AddMesh(oapiLoadMeshGlobal("r7_S\\RingEarly"), &RING_OFF); 
		SetMeshVisibilityMode(RingMesh, MESHVIS_ALWAYS | MESHVIS_EXTPASS);

		// get payload type
		VM_PayloadTypes PayloadType = (VM_PayloadTypes)SendVesselMessage(Launcher.hObject, VM_PAYLOAD_TYPE);
		
		// booms with 1 service level, plus added ladder for servicing the sattelites
		if ( (PayloadType == PAYLOAD_SAT1) || (PayloadType == PAYLOAD_SAT2) || (PayloadType == PAYLOAD_SAT3) )
		{
			BoomMesh270 = AddMesh(oapiLoadMeshGlobal("r7_S\\BoomRSat270"), &BOOM_270_OFS);				
			BoomMesh090 = AddMesh(oapiLoadMeshGlobal("r7_S\\BoomRSat090"), &BOOM_090_OFS);				
		}
	
		// booms with 1 service level, no ladder
		else
		{
			BoomMesh270 = AddMesh(oapiLoadMeshGlobal("r7_S\\BoomR270"), &BOOM_270_OFS);				
			BoomMesh090 = AddMesh(oapiLoadMeshGlobal("r7_S\\BoomR090"), &BOOM_090_OFS);				
		}

		// two shorter cable masts
		CableMastMesh090 = AddMesh(oapiLoadMeshGlobal("r7_S\\CableMastShort090"), &CABLE_MAST_SHORT_090_OFS);
		CableMastMesh180 = AddMesh(oapiLoadMeshGlobal("r7_S\\CableMastShort180"), &CABLE_MAST_SHORT_180_OFS);
		SetMeshVisibilityMode(CableMastMesh090, MESHVIS_ALWAYS | MESHVIS_EXTPASS);
		SetMeshVisibilityMode(CableMastMesh180, MESHVIS_ALWAYS | MESHVIS_EXTPASS);
	}

	// after modifications for Vostok
	else
	{
		// ring 
		RingMesh = AddMesh(oapiLoadMeshGlobal("r7_S\\Ring"), &RING_OFF); 
		SetMeshVisibilityMode(RingMesh, MESHVIS_ALWAYS | MESHVIS_EXTPASS);

		// custom Soyuz meshes from scenario, only after 1967
		if ((ScnYear >= 1967) && (BoomsMeshName[0] != 0))
		{
			char FullMeshName[256];
			sprintf(FullMeshName, "%s%s", BoomsMeshName, "270");
			BoomMesh270 = AddMesh(oapiLoadMeshGlobal(FullMeshName), &BOOM_S_270_OFS);				
			sprintf(FullMeshName, "%s%s", BoomsMeshName, "090");
			BoomMesh090 = AddMesh(oapiLoadMeshGlobal(FullMeshName), &BOOM_S_090_OFS);				
		}

		// booms with 3 service levels 
		else
		{
			BoomMesh270 = AddMesh(oapiLoadMeshGlobal("r7_S\\BoomV270"), &BOOM_270_OFS);				
			BoomMesh090 = AddMesh(oapiLoadMeshGlobal("r7_S\\BoomV090"), &BOOM_090_OFS);				
		}

		// single cable mast 
		// custom Soyuz meshes from scenario, only after 1967
		if ((ScnYear >= 1967) && (CableMastMeshName[0] != 0))
			CableMastMesh = AddMesh(oapiLoadMeshGlobal(CableMastMeshName), &CABLE_MAST_OFS);
		else
			CableMastMesh = AddMesh(oapiLoadMeshGlobal("r7_S\\CableMast"), &CABLE_MAST_OFS);
		SetMeshVisibilityMode(CableMastMesh, MESHVIS_ALWAYS | MESHVIS_EXTPASS);

		// fueling mast on Vostoks and later
		FuelingMastMesh = AddMesh(oapiLoadMeshGlobal("r7_S\\FuelingMast"), &FUELING_MAST_OFS);
		SetMeshVisibilityMode(FuelingMastMesh, MESHVIS_ALWAYS | MESHVIS_EXTPASS);
	}

	SetMeshVisibilityMode(BoomMesh270, MESHVIS_ALWAYS | MESHVIS_EXTPASS);
	SetMeshVisibilityMode(BoomMesh090, MESHVIS_ALWAYS | MESHVIS_EXTPASS);

	//
}


void R7_launchpad::HideShadows()
{
	if (VisHandle == NULL)
		return;

	for (int j = 0; j<30; j++)
	{
		MESHHANDLE hMesh = GetMesh(VisHandle,0);
		if (hMesh == NULL) 
			return;

		// kill shadows 
		unsigned int i;
		for (i=0; i<oapiMeshGroupCount(hMesh); i++)
		{
			MESHGROUP *mg = oapiMeshGroup(hMesh, i);
			mg->UsrFlag=1; 
		}

		// cleanup
		hMesh = NULL;
	}
}


void R7_launchpad::clbkVisualCreated (VISHANDLE vis,int refcount)
{
	VisHandle = vis;
	HideShadows();
}




void R7_launchpad::DefineAnimations()
{
	// ring
	RingRot = new MGROUP_ROTATE(RingMesh, NULL, 0, _V(0,0,0), _V(0,-1,0), (float)(360*RAD));

	if (BoomsMeshName[0] != 0)
	{
		RingRotBoom090 = new MGROUP_ROTATE(BoomMesh090, NULL, 0, _V(-R_BOOM_S,0,0), _V(0,-1,0), (float)(360*RAD));
		RingRotBoom270 = new MGROUP_ROTATE(BoomMesh270, NULL, 0, _V( R_BOOM_S,0,0), _V(0,-1,0), (float)(360*RAD));
	}
	else
	{
		RingRotBoom090 = new MGROUP_ROTATE(BoomMesh090, NULL, 0, _V(-R_BOOM,0,0), _V(0,-1,0), (float)(360*RAD));
		RingRotBoom270 = new MGROUP_ROTATE(BoomMesh270, NULL, 0, _V( R_BOOM,0,0), _V(0,-1,0), (float)(360*RAD));
	}

	RingRotSupport045 = new MGROUP_ROTATE(SupportMesh045, NULL, 0, _V( L_SUPPORT,0, L_SUPPORT), _V(0,-1,0), (float)(360*RAD));
	RingRotSupport135 = new MGROUP_ROTATE(SupportMesh135, NULL, 0, _V(-L_SUPPORT,0, L_SUPPORT), _V(0,-1,0), (float)(360*RAD));
	RingRotSupport225 = new MGROUP_ROTATE(SupportMesh225, NULL, 0, _V(-L_SUPPORT,0,-L_SUPPORT), _V(0,-1,0), (float)(360*RAD));
	RingRotSupport315 = new MGROUP_ROTATE(SupportMesh315, NULL, 0, _V( L_SUPPORT,0,-L_SUPPORT), _V(0,-1,0), (float)(360*RAD));
	
	// original R7 configuration 
	if  (ScnYear <= 1958) 
	{
		RingRotCableMast090 = new MGROUP_ROTATE(CableMastMesh090, NULL, 0, _V( -R_CABLE_MAST_SHORT,0,0), _V(0,-1,0), (float)(360*RAD));
		RingRotCableMast180 = new MGROUP_ROTATE(CableMastMesh180, NULL, 0, _V( 0,0,-R_CABLE_MAST_SHORT), _V(0,-1,0), (float)(360*RAD));
	}

	// Vostok modification
	else
	{
		RingRotCableMast = new MGROUP_ROTATE(CableMastMesh, NULL, 0, _V( 0,0,-R_CABLE_MAST), _V(0,-1,0), (float)(360*RAD));
		RingRotFuelingMast = new MGROUP_ROTATE(FuelingMastMesh, NULL, 0, _V( 0,0,-R_FUELING_MAST), _V(0,-1,0), (float)(360*RAD));
	}

	RingRotAnim = CreateAnimation (0);

	AddAnimationComponent (RingRotAnim, 0, 1, RingRot);
	AddAnimationComponent (RingRotAnim, 0, 1, RingRotBoom090);
	AddAnimationComponent (RingRotAnim, 0, 1, RingRotBoom270);
	AddAnimationComponent (RingRotAnim, 0, 1, RingRotSupport045);
	AddAnimationComponent (RingRotAnim, 0, 1, RingRotSupport135);
	AddAnimationComponent (RingRotAnim, 0, 1, RingRotSupport225);
	AddAnimationComponent (RingRotAnim, 0, 1, RingRotSupport315);

	if  (ScnYear <= 1958) 
	{
		AddAnimationComponent (RingRotAnim, 0, 1, RingRotCableMast090);
		AddAnimationComponent (RingRotAnim, 0, 1, RingRotCableMast180);
	}

	else
	{
		AddAnimationComponent (RingRotAnim, 0, 1, RingRotCableMast);
		AddAnimationComponent (RingRotAnim, 0, 1, RingRotFuelingMast);
	}

	// booms

	if (BoomsMeshName[0] != 0)
	{
		BoomRot090 = new MGROUP_ROTATE(BoomMesh090, NULL, 0, _V(0,0,0), _V(0,0, 1),  (float)(90*RAD));
		BoomRot270 = new MGROUP_ROTATE(BoomMesh270, NULL, 0, _V(0,0,0), _V(0,0,-1),  (float)(90*RAD));
	}
	else
	{
		BoomRot090 = new MGROUP_ROTATE(BoomMesh090, NULL, 0, _V(0,0,0), _V(0,0, 1),  (float)(97*RAD));
		BoomRot270 = new MGROUP_ROTATE(BoomMesh270, NULL, 0, _V(0,0,0), _V(0,0,-1),  (float)(97*RAD));
	}

	BoomRotAnim = CreateAnimation (1);

	AddAnimationComponent (BoomRotAnim, 0, 1, BoomRot090);
	AddAnimationComponent (BoomRotAnim, 0, 1, BoomRot270);

	// supports
	SupportRot045 = new MGROUP_ROTATE(SupportMesh045, NULL, 0, _V(0,0,0), _V( COS45D, 0, -COS45D),  (float)(55*RAD));
	SupportRot135 = new MGROUP_ROTATE(SupportMesh135, NULL, 0, _V(0,0,0), _V( COS45D, 0,  COS45D),  (float)(55*RAD));
	SupportRot225 = new MGROUP_ROTATE(SupportMesh225, NULL, 0, _V(0,0,0), _V(-COS45D, 0,  COS45D),  (float)(55*RAD));
	SupportRot315 = new MGROUP_ROTATE(SupportMesh315, NULL, 0, _V(0,0,0), _V(-COS45D, 0, -COS45D),  (float)(55*RAD));

	SupportRotAnim = CreateAnimation (1);

	AddAnimationComponent (SupportRotAnim, 0, 1, SupportRot045);
	AddAnimationComponent (SupportRotAnim, 0, 1, SupportRot135);
	AddAnimationComponent (SupportRotAnim, 0, 1, SupportRot225);
	AddAnimationComponent (SupportRotAnim, 0, 1, SupportRot315);

	// platform
	PlatformTransl = new MGROUP_TRANSLATE(PlatformMesh, NULL, 0, PLATFORM_SHIFT);
	PlatformRingTransl = new MGROUP_TRANSLATE(PlatformRingMesh, NULL, 0, PLATFORM_SHIFT);

	PlatformTranslAnim = CreateAnimation (1);

	AddAnimationComponent (PlatformTranslAnim, 0, 1, PlatformTransl);
	AddAnimationComponent (PlatformTranslAnim, 0, 1, PlatformRingTransl);

	// cable mast: create animation, will add components downstream
	CableMastRotAnim = CreateAnimation (1);

	// original configuration, two shorter cable masts
	if  (ScnYear <= 1958) 
	{
		CableMastRot090 = new MGROUP_ROTATE(CableMastMesh090, NULL, 0, _V(0,0,0), _V(  0, 1, 0),  (float)(60*RAD));
		CableMastRot180 = new MGROUP_ROTATE(CableMastMesh180, NULL, 0, _V(0,0,0), _V( -1, 0, 0),  (float)(60*RAD));

		AddAnimationComponent (CableMastRotAnim, 0, 1, CableMastRot090);
		AddAnimationComponent (CableMastRotAnim, 0, 1, CableMastRot180);
	}

	// Vostok modification
	else
	{
		// single cable mast
		CableMastRot = new MGROUP_ROTATE(CableMastMesh, NULL, 0, _V(0,0,0), _V( -1, 0, 0),  (float)(30*RAD));
		AddAnimationComponent (CableMastRotAnim, 0, 1, CableMastRot);

		// fueling mast 
		FuelingMastRot = new MGROUP_ROTATE(FuelingMastMesh, NULL, 0, _V(0,0,0), _V( -1, 0, 0),  (float)(35*RAD));

		FuelingMastRotAnim = CreateAnimation (1);

		AddAnimationComponent (FuelingMastRotAnim, 0, 1, FuelingMastRot);

		SetAnimation (FuelingMastRotAnim, FuelingMastRotPercent);
	}

	// set animations as read from scenario
	SetAllRingAnimations();
	SetAnimation (SupportRotAnim, SupportRotPercent);
	SetAnimation (BoomRotAnim, BoomRotPercent);
	SetAnimation (CableMastRotAnim, CableMastRotPercent);
	SetAnimation (PlatformTranslAnim, PlatformTranslPercent);
}


// called from timestep
void R7_launchpad::RunAnimations()
{
		// ring block

	if (RingTurning != 0)
		AnimateRing();

	else if (bRingToAzimuth)
		AnimateRingToAzimuth();

	else 
		SendVesselWaveToFocused(SND_RING_MOVE, SND_STOP);

		// booms block

	if (BoomTurning != 0)
		AnimateBoom();
	
	else if (bBoomsTurnDown)
		AnimateBoomsAutoDown();

	else 
		SendVesselWaveToFocused(SND_BOOMS_MOVE, SND_STOP);

		// support block

	if (SupportTurning != 0)
		AnimateSupport();

	else if (bSupportFalling)
		AnimateSupportDrop();

	else 
		SendVesselWaveToFocused(SND_SUPPORTS_MOVE, SND_STOP);

		// platform block

	if (PlatformTranslating != 0)
		AnimatePlatform();

	else if (bPlatformRetract)
		AnimatePlatformAutoRetract();

	else 
		SendVesselWaveToFocused(SND_PLATFORM_MOVE, SND_STOP);

	// cable mast block

	if (CableMastTurning != 0)
		AnimateCableMast();

	else if (bCableMastFalling)
		AnimateCableMastDrop();

	else 
		SendVesselWaveToFocused(SND_CABLE_MAST_MOVE, SND_STOP);

	// fuel mast block

	if (FuelingMastTurning != 0)
		AnimateFuelingMast();

	else if (bFuelingMastFalling)
		AnimateFuelingMastDrop();

	else 
		SendVesselWaveToFocused(SND_FUELING_MAST_MOVE, SND_STOP);

	// reset directional animation flags, other functions will set them again as needed
	RingTurning = 0;
	BoomTurning = 0;
	SupportTurning = 0;
	PlatformTranslating = 0;
	CableMastTurning = 0;
	FuelingMastTurning = 0;
}


void R7_launchpad::AnimateRing(bool bRingStop)
{
	if (!bRingStop)
	{
		double dPercent = RING_ROT_OMEGA * RingTurning* SimDt;

		RingRotPercent += dPercent;

		if (RingRotPercent > 1.0)
			RingRotPercent = RingRotPercent - 1;
		else if (RingRotPercent < 0)
			RingRotPercent = RingRotPercent + 1;
	}

	else
	{
		bRingToAzimuth = false;
		RingRotPercent = TargetRingRotPercent;
	}

	SetAllRingAnimations();

	if (bHaveFocus)
	{
		double AzLocalAngle = (180 + LP_AZM -45) - RingRotPercent * 360;
		if (AzLocalAngle < 0)
			AzLocalAngle = 360 + AzLocalAngle;
		sprintf(oapiDebugString(), "Azimuth: %.2f", AzLocalAngle);
	}

	SendVesselWaveToFocused(SND_RING_MOVE, LOOP);
}


void R7_launchpad::SetAllRingAnimations()
{
	SetAnimation (RingRotAnim, RingRotPercent);

	// convert linear 0-1 rotation percent to angular measures 0-2pi (0-360 deg)
	double a = 2* PI * RingRotPercent;
	double sina = sin(a);
	double cosa = cos(a);
	
	// update boom animation coordinates and axis
	BoomRot090->axis = _V(-sina, 0,  cosa);
	BoomRot270->axis = _V( sina, 0, -cosa);

	if (BoomsMeshName[0] != 0)
	{
		BoomRot090->ref = _V(cosa*R_BOOM_S - R_BOOM_S, 0,  sina*R_BOOM_S);
		BoomRot270->ref = _V(-cosa*R_BOOM_S + R_BOOM_S, 0, -sina*R_BOOM_S);
	}
	else
	{
		BoomRot090->ref = _V(cosa*R_BOOM - R_BOOM, 0,  sina*R_BOOM);
		BoomRot270->ref = _V(-cosa*R_BOOM + R_BOOM, 0, -sina*R_BOOM);
	}

	// shift by 45 deg
	double b = a + PI/4;
	double sinb = sin(b);
	double cosb = cos(b);

	double c = (PI/4)-a;
	double sinc = sin(c);
	double cosc = cos(c);

	// update support animation coordinates and axis
	SupportRot045->axis = _V( sinb, 0, -cosb);
	SupportRot045->ref = _V(-sinc*R_SUPPORT + L_SUPPORT, 0,  -cosc*R_SUPPORT + L_SUPPORT );

	SupportRot135->axis = _V( cosb, 0, sinb);
	SupportRot135->ref = _V( cosc*R_SUPPORT - L_SUPPORT, 0,  -sinc*R_SUPPORT + L_SUPPORT);

	SupportRot225->axis = _V(-sinb, 0,  +cosb);
	SupportRot225->ref = _V(sinc*R_SUPPORT - L_SUPPORT, 0,  cosc*R_SUPPORT - L_SUPPORT );

	SupportRot315->axis = _V(-cosb, 0, -sinb);
	SupportRot315->ref = _V( -cosc*R_SUPPORT + L_SUPPORT, 0,  sinc*R_SUPPORT - L_SUPPORT);

	// original pad configuration
	if  (ScnYear <= 1958) 
	{
		CableMastRot090->axis = _V(-sina, 0,  cosa);
		CableMastRot090->ref = _V(cosa*R_CABLE_MAST_SHORT - R_CABLE_MAST_SHORT, 0, sina*R_CABLE_MAST_SHORT);

		CableMastRot180->axis = _V(-cosa, 0, -sina);
		CableMastRot180->ref = _V(-sina*R_CABLE_MAST_SHORT, 0, cosa*R_CABLE_MAST_SHORT - R_CABLE_MAST_SHORT);
	}

	// Vostok modification
	else
	{
		// cable mast
		CableMastRot->axis = _V(-cosa, 0, -sina);
		CableMastRot->ref = _V(-sina*R_CABLE_MAST, 0, cosa*R_CABLE_MAST - R_CABLE_MAST);

		// fueling mast
		FuelingMastRot->axis = _V(-cosa, 0, -sina);
		FuelingMastRot->ref = _V(-sina*R_FUELING_MAST, 0, cosa*R_FUELING_MAST - R_FUELING_MAST);
	}

	// update rocket attachment params
	if (Launcher.hObject != NULL)
	{
		Launcher.AttParamsOnUs.AttRot = _V(-sina, 0,  cosa); 

		SetAttachmentParams(	Launcher.hAttachment, 
								Launcher.AttParamsOnUs.AttPoint,
								Launcher.AttParamsOnUs.AttDir,
								Launcher.AttParamsOnUs.AttRot);
	}
}


void R7_launchpad::AnimateRingToAzimuth()
{
	// rotate target azimuth to local pad orientation angle, adjusting for 45 deg offset of the rocket plane
	double TargetAzLocalAngle = ((180 + LP_AZM - 45) - Azimuth);

	// normalize to 0..360 deg range
	if (TargetAzLocalAngle < 0)
		TargetAzLocalAngle += 360;
	if (TargetAzLocalAngle > 360)
		TargetAzLocalAngle -= 360;

	// normalize target angle sensor to 0..+1 range
	TargetRingRotPercent = TargetAzLocalAngle / 360;

	double dPercent = TargetRingRotPercent - RingRotPercent;
	double fdPercent = fabs(dPercent);
	RingTurning = (int)(dPercent / fdPercent);

	if (fdPercent > 0.5)
		RingTurning = -RingTurning;

	if (fdPercent < 0.001)
		AnimateRing(true);
	else
		AnimateRing();
}



void R7_launchpad::AnimateBoom()
{
	double dPercent = BOOM_ROT_OMEGA * BoomTurning* SimDt;

	double OldBoomRotPercent = BoomRotPercent;
	BoomRotPercent += dPercent;

	if (BoomRotPercent > 1)
	{
		BoomRotPercent = 1;

		if (OldBoomRotPercent < 1)
			SendVesselWaveToFocused(SND_CLANG_HEAVY);
	}

	else if (BoomRotPercent < 0)
	{
		BoomRotPercent = 0;

		if (OldBoomRotPercent > 0)
			SendVesselWaveToFocused(SND_CLANG_HEAVY);
	}

	SetAnimation (BoomRotAnim, BoomRotPercent);

	if (OldBoomRotPercent - BoomRotPercent != 0)
		SendVesselWaveToFocused(SND_BOOMS_MOVE, LOOP);
	else
		SendVesselWaveToFocused(SND_BOOMS_MOVE, SND_STOP);
}


void R7_launchpad::AnimateBoomsAutoDown()
{
	BoomTurning = -1;
	AnimateBoom();

	if (BoomRotPercent <= 0)
		bBoomsTurnDown = false;
}


void R7_launchpad::AnimateSupport()
{
	double dPercent = SUPPORT_ROT_OMEGA * SupportTurning* SimDt;

	double OldSupportRotPercent = SupportRotPercent;
	SupportRotPercent += dPercent;

	if (SupportRotPercent > 1)
	{
		SupportRotPercent = 1;

		if (OldSupportRotPercent < 1)
			SendVesselWaveToFocused(SND_CLANG_HEAVY);
	}

	else if (SupportRotPercent < 0)
	{
		SupportRotPercent = 0;

		if (OldSupportRotPercent > 0)
			SendVesselWaveToFocused(SND_CLANG_HEAVY);
	}

	SetAnimation (SupportRotAnim, SupportRotPercent);

	if (OldSupportRotPercent - SupportRotPercent != 0)
		SendVesselWaveToFocused(SND_SUPPORTS_MOVE, LOOP);
	else
		SendVesselWaveToFocused(SND_SUPPORTS_MOVE, SND_STOP);
}


void R7_launchpad::AnimateSupportDrop()
{
	SupportDropOmega += SUPPORT_DROP_OMEGA_INC* SimDt;
	double dPercent = SupportDropOmega * SimDt;

	SupportRotPercent -= dPercent;

	if (SupportRotPercent <= 0)
	{
		SupportRotPercent = 0;
		bSupportFalling = false;
		SendVesselWaveToFocused(SND_CLANG_HEAVY);
	}

	SetAnimation (SupportRotAnim, SupportRotPercent);
}


void R7_launchpad::AnimatePlatform()
{
	double dPercent = PLATFORM_TRANSL_VEL * PlatformTranslating* SimDt;

	double OldPlatformTranslPercent = PlatformTranslPercent;
	PlatformTranslPercent += dPercent;

	if (PlatformTranslPercent > 1)
	{
		PlatformTranslPercent = 1;

		if (OldPlatformTranslPercent < 1)
			SendVesselWaveToFocused(SND_CLANG_HEAVY);
	}

	else if (PlatformTranslPercent < 0)
	{
		PlatformTranslPercent = 0;

		if (OldPlatformTranslPercent > 0)
			SendVesselWaveToFocused(SND_CLANG_HEAVY);
	}

	SetAnimation (PlatformTranslAnim, PlatformTranslPercent);

	if ((OldPlatformTranslPercent - PlatformTranslPercent) != 0)
		SendVesselWaveToFocused(SND_PLATFORM_MOVE, LOOP);
	else
		SendVesselWaveToFocused(SND_PLATFORM_MOVE, SND_STOP);
}


void R7_launchpad::AnimatePlatformAutoRetract()
{
	PlatformTranslating = -1;
	AnimatePlatform();

	if (PlatformTranslPercent <= 0)
		bPlatformRetract = false;
}


void R7_launchpad::AnimateCableMast()
{
	double dPercent = CABLE_MAST_ROT_OMEGA * CableMastTurning* SimDt;

	double OldCableMastRotPercent = CableMastRotPercent;
	CableMastRotPercent += dPercent;

	if (CableMastRotPercent >= 1)
	{
		CableMastRotPercent = 1;
		SendVesselMessage(Launcher.hObject, VM_SET_PRE_LAUNCH_VENT, 1);
	}

	else if (CableMastRotPercent <= 0)
		CableMastRotPercent = 0;

	else
		SendVesselMessage(Launcher.hObject, VM_SET_PRE_LAUNCH_VENT, 0);

	SetAnimation (CableMastRotAnim, CableMastRotPercent);

	if (OldCableMastRotPercent - CableMastRotPercent != 0)
		SendVesselWaveToFocused(SND_CABLE_MAST_MOVE, LOOP);
	else
		SendVesselWaveToFocused(SND_CABLE_MAST_MOVE, SND_STOP);
}


void R7_launchpad::AnimateCableMastDrop()
{
	CableMastDropOmega += CABLE_MAST_DROP_OMEGA_INC* SimDt;
	double dPercent = CableMastDropOmega * SimDt;

	CableMastRotPercent -= dPercent;

	if (CableMastRotPercent <= 0)
	{
		CableMastRotPercent = 0;
		bCableMastFalling = false;
		SendVesselWaveToFocused(SND_CLANG_LIGHT);
	}

	SetAnimation (CableMastRotAnim, CableMastRotPercent);

	// synk fueling mast fall
	if ((FuelingMastRotPercent > 0) && (!bFuelingMastFalling))
	{
		bFuelingMastFalling = true;
		SendVesselMessage(Launcher.hObject, VM_SET_PRE_LAUNCH_VENT_3ST, 0);
	}
}


void R7_launchpad::AnimateFuelingMast()
{
	double dPercent = FUELING_MAST_ROT_OMEGA * FuelingMastTurning* SimDt;

	double OldFuelingMastRotPercent = FuelingMastRotPercent;
	FuelingMastRotPercent += dPercent;

	if (FuelingMastRotPercent >= 1)
	{
		FuelingMastRotPercent = 1;
		SendVesselMessage(Launcher.hObject, VM_SET_PRE_LAUNCH_VENT_3ST, 1);
	}

	else if (FuelingMastRotPercent <= 0)
		FuelingMastRotPercent = 0;

	else
		SendVesselMessage(Launcher.hObject, VM_SET_PRE_LAUNCH_VENT_3ST, 0);

	SetAnimation (FuelingMastRotAnim, FuelingMastRotPercent);

	if (OldFuelingMastRotPercent - FuelingMastRotPercent != 0)
		SendVesselWaveToFocused(SND_FUELING_MAST_MOVE, LOOP);
	else
		SendVesselWaveToFocused(SND_FUELING_MAST_MOVE, SND_STOP);
}


void R7_launchpad::AnimateFuelingMastDrop()
{
	FuelingMastDropOmega += FUELING_MAST_DROP_OMEGA_INC* SimDt;
	double dPercent = FuelingMastDropOmega * SimDt;

	FuelingMastRotPercent -= dPercent;

	if (FuelingMastRotPercent <= 0)
	{
		FuelingMastRotPercent = 0;
		bFuelingMastFalling = false;
		SendVesselWaveToFocused(SND_CLANG_LIGHT);
	}

	SetAnimation (FuelingMastRotAnim, FuelingMastRotPercent);
}


// for now, do nothing here
void R7_launchpad::Break()
{
}



void R7_launchpad::DebugSomething()
{
	sprintf(oapiDebugString(), "%f", oapiGetPlanetCurrentRotation(hEarth)*DEG);
//	pInput = &R7_launchpad::Input;
//	oapiOpenInputBox("Testing input box", &R7_launchpad::Input, 0, 25, 0);		
}





// ========================== PANELS =============================


bool R7_launchpad::clbkLoadPanel (int id)
{
	// no rocket?
	if (Launcher.ObjectName[0] == 0)
	{
		ShowAnnotationMessage("Targeting is disabled with no rocket installed.");
		return false;
	}

	// check autopilot
	if (bAutopilot)
	{
		ShowAnnotationMessage("Manual controls are disabled. To enable, use M key.");
		return false;
	}

	// check autopilot
	if (Timer > -100)
	{
		ShowAnnotationMessage("Re-targeting is disabled after the start of launch sequence.");
		return false;
	}

	// clear message display
	sprintf(oapiDebugString(), "");

	surfNumbers = oapiCreateSurface (LOADBMP ("Textures\\r7_S\\Numbers.bmp"));
	surfNumButtonUp = oapiCreateSurface (LOADBMP ("Textures\\r7_S\\NumberUp.bmp"));
	surfNumButtonDown = oapiCreateSurface (LOADBMP ("Textures\\r7_S\\NumberDown.bmp"));
	surfNumPlus = oapiCreateSurface (LOADBMP ("Textures\\r7_S\\NumberPlus.bmp"));
	surfNumMinus = oapiCreateSurface (LOADBMP ("Textures\\r7_S\\NumberMinus.bmp"));
	surfNumTransparent = oapiCreateSurface (LOADBMP ("Textures\\r7_S\\NumberTransparent.bmp"));
	surfNumDot = oapiCreateSurface (LOADBMP ("Textures\\r7_S\\NumberDot.bmp"));

	if (TrajectoryType == TRAJECTORY_ORBITAL)
	{
		// load panel background
		HBITMAP hBmp = LOADBMP ("Textures\\r7_S\\PanelOrbital.bmp");
		oapiRegisterPanelBackground (hBmp, PANEL_ATTACH_TOP, 0xFFFFFF);

		GetTargetOrbit();

		panCreateOrbParamControls();

	}
	else // if (TrajectoryType == TRAJECTORY_BALLISTIC)
	{
		// load panel background
		HBITMAP hBmp = LOADBMP ("Textures\\r7_S\\PanelBallistic.bmp");
		oapiRegisterPanelBackground (hBmp, PANEL_ATTACH_TOP, 0xFFFFFF);

		//
		NumInfos[AREA_BAL_TGT].offX = BAL_TGT_OFF_X;
		NumInfos[AREA_BAL_TGT].offY = BAL_TGT_OFF_Y;
		NumInfos[AREA_BAL_TGT].piVar = &PanBalTgtIdx;
		NumInfos[AREA_BAL_TGT].iVarCache = PanBalTgtIdx;
		NumInfos[AREA_BAL_TGT].iMin = 1;
		NumInfos[AREA_BAL_TGT].iMax = 7;
		NumInfos[AREA_BAL_TGT].sign = 0;
		NumInfos[AREA_BAL_TGT].bDot = false;
		NumInfos[AREA_BAL_TGT].bVisible = true;

		//
		panCreateNumberControl(AREA_BAL_TGT);
		panCreateBalTargetCoordinateControls();

		// var panels panel
		oapiRegisterPanelArea (AREA_BAL_DESCR, _R(BAL_DESCR_OFF_X, BAL_DESCR_OFF_Y,	BAL_DESCR_OFF_X+BAL_DESCR_W, BAL_DESCR_OFF_Y+BAL_DESCR_H), PANEL_REDRAW_USER, PANEL_MOUSE_IGNORE, PANEL_MAP_BACKGROUND);
	}

	panCreateTimeControls();

	// bitmaps
	surfIndValid = oapiCreateSurface (LOADBMP ("Textures\\r7_S\\IndicatorValid.bmp"));
	surfSetValid = oapiCreateSurface (LOADBMP ("Textures\\r7_S\\SetValid.bmp"));
	
		// var panels panel
	oapiRegisterPanelArea (	AREA_TGT_IND, _R(TGT_IND_OFF_X, TGT_IND_OFF_Y, TGT_IND_OFF_X+IND_W, TGT_IND_OFF_Y+IND_H), PANEL_REDRAW_USER, PANEL_MOUSE_IGNORE, PANEL_MAP_BACKGROUND);
	oapiRegisterPanelArea (	AREA_TIME_IND, _R(TIME_IND_OFF_X, TIME_IND_OFF_Y, TIME_IND_OFF_X+IND_W, TIME_IND_OFF_Y+IND_H), PANEL_REDRAW_USER, PANEL_MOUSE_IGNORE, PANEL_MAP_BACKGROUND);
	oapiRegisterPanelArea (	AREA_BTN_SET, _R(BTN_SET_OFF_X, BTN_SET_OFF_Y, BTN_SET_OFF_X+BTN_SET_W, BTN_SET_OFF_Y+BTN_SET_H), PANEL_REDRAW_USER, PANEL_MOUSE_DOWN, PANEL_MAP_BACKGROUND);
	oapiRegisterPanelArea (	AREA_TGT_STR, _R(STR_OFF_X, TGT_STR_OFF_Y, STR_OFF_X+STR_W, TGT_STR_OFF_Y+STR_H), PANEL_REDRAW_USER, PANEL_MOUSE_IGNORE, PANEL_MAP_BACKGROUND);
	oapiRegisterPanelArea (	AREA_TIME_STR, _R(STR_OFF_X, TIME_STR_OFF_Y, STR_OFF_X+STR_W, TIME_STR_OFF_Y+STR_H), PANEL_REDRAW_USER, PANEL_MOUSE_IGNORE, PANEL_MAP_BACKGROUND);

	// evaluate and redraw new panel
	panReevaluatePanel();

	// loaded!
	return true;
}


void R7_launchpad::panCreateNumberControl(int InfoIdx)
{
	NumberControlInfo Info = NumInfos[InfoIdx];

	oapiRegisterPanelArea (InfoIdx,				_R(Info.offX, Info.offY,			Info.offX+NC_W, Info.offY+NC_H				), PANEL_REDRAW_USER, PANEL_MOUSE_IGNORE, PANEL_MAP_BACKGROUND);
	oapiRegisterPanelArea (InfoIdx+AREAS_UP,	_R(Info.offX, Info.offY-2-NC_BTN_H, Info.offX+NC_W, Info.offY-2					), PANEL_REDRAW_USER, PANEL_MOUSE_DOWN, PANEL_MAP_BACKGROUND);
	oapiRegisterPanelArea (InfoIdx+AREAS_DOWN,	_R(Info.offX, Info.offY+2+NC_H,		Info.offX+NC_W, Info.offY+2+NC_H+NC_BTN_H	), PANEL_REDRAW_USER, PANEL_MOUSE_DOWN, PANEL_MAP_BACKGROUND);
}


void R7_launchpad::panCreateBalTargetCoordinateControls()
{
	BalTargetLatInfo.offX = BAL_LAT_OFF_X;
	BalTargetLatInfo.offY = BAL_LAT_OFF_Y;
	BalTargetLatInfo.pdVar = &TargetLat;
	strcpy(BalTargetLatInfo.Fmt, "%+09.4f");
	BalTargetLatInfo.length = 9;
	BalTargetLatInfo.FirstNumCtrlIdx = AREA_BAL_TGT_LAT;

	if (NumInfos[AREA_BAL_TGT].iVarCache < 7)
		BalTargetLatInfo.dVarCache = 0;
	else
		BalTargetLatInfo.dVarCache = TargetLat;

	panCreateDoubleValueControls(BalTargetLatInfo);

	BalTargetLonInfo.offX = BAL_LON_OFF_X;
	BalTargetLonInfo.offY = BAL_LON_OFF_Y;
	BalTargetLonInfo.pdVar = &TargetLon;
	strcpy(BalTargetLonInfo.Fmt, "%+09.4f");
	BalTargetLonInfo.length = 9;
	BalTargetLonInfo.FirstNumCtrlIdx = AREA_BAL_TGT_LON;

	if (NumInfos[AREA_BAL_TGT].iVarCache < 7)
		BalTargetLonInfo.dVarCache = 0;
	else
		BalTargetLonInfo.dVarCache = TargetLon;

	panCreateDoubleValueControls(BalTargetLonInfo);
}


void R7_launchpad::panCreateOrbParamControls()
{
	OrbPerInfo.offX = ORB_PER_OFF_X;
	OrbPerInfo.offY = ORB_PER_OFF_Y;
	OrbPerInfo.pdVar = &Perigee;
	strcpy(OrbPerInfo.Fmt, "%03.0f");
	OrbPerInfo.length = 3;
	OrbPerInfo.FirstNumCtrlIdx = AREA_ORB_PER;
	OrbPerInfo.dVarCache = Perigee;
	panCreateDoubleValueControls(OrbPerInfo);

	OrbApoInfo.offX = ORB_APO_OFF_X;
	OrbApoInfo.offY = ORB_APO_OFF_Y;
	OrbApoInfo.pdVar = &Apogee;
	strcpy(OrbApoInfo.Fmt, "%03.0f");
	OrbApoInfo.length = 3;
	OrbApoInfo.FirstNumCtrlIdx = AREA_ORB_APO;
	OrbApoInfo.dVarCache = Apogee;
	panCreateDoubleValueControls(OrbApoInfo);

	OrbIncInfo.offX = ORB_INC_OFF_X;
	OrbIncInfo.offY = ORB_INC_OFF_Y;
	OrbIncInfo.pdVar = &Inc;
	strcpy(OrbIncInfo.Fmt, "%+05.1f");
	OrbIncInfo.length = 5;
	OrbIncInfo.FirstNumCtrlIdx = AREA_ORB_INC;
	OrbIncInfo.dVarCache = Inc;
	panCreateDoubleValueControls(OrbIncInfo);
}


void R7_launchpad::panCreateTimeControls()
{
	TimeHInfo.offX = TIME_H_OFF_X;
	TimeHInfo.offY = TIME_H_OFF_Y;
	//TimeHInfo.pdVar = &iTimerH;
	strcpy(TimeHInfo.Fmt, "%02.0f");
	TimeHInfo.length = 2;
	TimeHInfo.FirstNumCtrlIdx = AREA_TIME_H;
	TimeHInfo.dVarCache = iTimerH;
	panCreateDoubleValueControls(TimeHInfo);

	TimeMnInfo.offX = TIME_MN_OFF_X;
	TimeMnInfo.offY = TIME_MN_OFF_Y;
	//TimeMnInfo.pdVar = &Perigee;
	strcpy(TimeMnInfo.Fmt, "%02.0f");
	TimeMnInfo.length = 2;
	TimeMnInfo.FirstNumCtrlIdx = AREA_TIME_MN;
	TimeMnInfo.dVarCache = iTimerM;
	panCreateDoubleValueControls(TimeMnInfo);

	TimeSInfo.offX = TIME_S_OFF_X;
	TimeSInfo.offY = TIME_S_OFF_Y;
	//TimeSInfo.pdVar = &Perigee;
	strcpy(TimeSInfo.Fmt, "%02.0f");
	TimeSInfo.length = 2;
	TimeSInfo.FirstNumCtrlIdx = AREA_TIME_S;
	TimeSInfo.dVarCache = iTimerS;
	panCreateDoubleValueControls(TimeSInfo);
}


void R7_launchpad::panCreateDoubleValueControls(DoubleValueInfo DInfo)
{
	sprintf(DInfo.str, DInfo.Fmt, DInfo.dVarCache);

	for (int i=0; i < DInfo.length; i++)
	{
		if ((DInfo.str[i] == '+') || (DInfo.str[i] == '-'))
		{
			if (DInfo.dVarCache < 0)
				NumInfos[DInfo.FirstNumCtrlIdx].sign = -1;
			else
				NumInfos[DInfo.FirstNumCtrlIdx].sign = 1;

			NumInfos[DInfo.FirstNumCtrlIdx].bDot = false;
			DInfo.ints[i] = 0;
		}

		else if (DInfo.str[i] == '.')
		{
			NumInfos[DInfo.FirstNumCtrlIdx+i].sign = 0;
			NumInfos[DInfo.FirstNumCtrlIdx+i].bDot = true;
			DInfo.ints[i] = 0;
		}

		else
		{
			NumInfos[DInfo.FirstNumCtrlIdx+i].sign = 0;
			NumInfos[DInfo.FirstNumCtrlIdx+i].bDot = false;
			DInfo.ints[i] = DInfo.str[i]-'0';
		}

		NumInfos[DInfo.FirstNumCtrlIdx+i].offX = DInfo.offX + i*2 + i*NC_W;
		NumInfos[DInfo.FirstNumCtrlIdx+i].offY = DInfo.offY;
		NumInfos[DInfo.FirstNumCtrlIdx+i].iMin = 0;
		NumInfos[DInfo.FirstNumCtrlIdx+i].iMax = 9;
		NumInfos[DInfo.FirstNumCtrlIdx+i].iVarCache = DInfo.ints[i];
		NumInfos[DInfo.FirstNumCtrlIdx+i].piVar = &DInfo.ints[i];

		if ((TrajectoryType == TRAJECTORY_BALLISTIC) && (DInfo.FirstNumCtrlIdx < AREA_TIME_H))
			NumInfos[DInfo.FirstNumCtrlIdx+i].bVisible = (PanBalTgtIdx == 7);
		else
			NumInfos[DInfo.FirstNumCtrlIdx+i].bVisible = true;

		panCreateNumberControl(DInfo.FirstNumCtrlIdx+i);
	}

}


bool R7_launchpad::clbkPanelRedrawEvent (int id, int event, SURFHANDLE surf)
{
	// debug only
//	if (surf == NULL)
//		return false;

	// number boxes
	if ((id >= AREAS_NUM) && (id < AREAS_UP))
	{
		if (!NumInfos[id].bVisible)
			oapiBlt (surf, surfNumTransparent, 0, 0, 0, 0, NC_W, NC_H, 0xFFFFFF);
		else 
			if (NumInfos[id].sign == -1)
			oapiBlt (surf, surfNumMinus, 0, 0, 0, 0, NC_W, NC_H);
		else if (NumInfos[id].sign == +1)
			oapiBlt (surf, surfNumPlus, 0, 0, 0, 0, NC_W, NC_H);
		else if (NumInfos[id].bDot)
			oapiBlt (surf, surfNumDot, 0, 0, 0, 0, NC_W, NC_H);
		else
			oapiBlt (surf, surfNumbers, 0, 0, NumInfos[id].iVarCache * NC_W, 0, NC_W, NC_H);

		return true;
	}

	// number up buttons
	if ((id >= AREAS_UP) && (id < AREAS_DOWN))
	{
		if ((NumInfos[id-AREAS_UP].bDot) || (!NumInfos[id-AREAS_UP].bVisible))
			oapiBlt (surf, surfNumTransparent, 0, 0, 0, 0, NC_W, NC_BTN_H, 0xFFFFFF);
		else
			oapiBlt (surf, surfNumButtonUp, 0, 0, 0, 0, NC_W, NC_BTN_H, 0xFFFFFF);
	
		return true;
	}

	// number down buttons
	if ((id >= AREAS_DOWN) && (id < AREAS_OTHER))
	{
		if ((NumInfos[id-AREAS_DOWN].bDot) || (!NumInfos[id-AREAS_DOWN].bVisible))
			oapiBlt (surf, surfNumTransparent, 0, 0, 0, 0, NC_W, NC_BTN_H, 0xFFFFFF);
		else
			oapiBlt (surf, surfNumButtonDown, 0, 0, 0, 0, NC_W, NC_BTN_H, 0xFFFFFF);

		return true;
	}

	// ballistic target description
	if (id == AREA_BAL_DESCR)
	{
		char tempStr[50];
		int TgtIdx = NumInfos[AREA_BAL_TGT].iVarCache-1;

		if (TgtIdx < 6)
			strcpy(tempStr, BalTargets[TgtIdx].NameCaps);
		else
			strcpy(tempStr, "OTHER");

		HDC DC = oapiGetDC(surf);
        SetTextColor(DC, 0x0000ff00); 
        SetBkColor(DC,  0x00000000); 
		TextOut(DC, 0, 0, tempStr, strlen(tempStr)); 
		oapiReleaseDC (surf, DC);
		return true;
	}

	// target error string
	if (id == AREA_TGT_STR)
	{
		HDC DC = oapiGetDC(surf);
        SetTextColor(DC, 0x000000ff); 
        SetBkColor(DC,  0x00000000); 
		TextOut(DC, 0, 0, sScenarioWarning, strlen(sScenarioWarning)); 
		oapiReleaseDC (surf, DC);
		return true;
	}

	// time error string
	if (id == AREA_TIME_STR)
	{
		HDC DC = oapiGetDC(surf);
        SetTextColor(DC, 0x000000ff); 
        SetBkColor(DC,  0x00000000); 
		TextOut(DC, 0, 0, sTimeWarning, strlen(sTimeWarning)); 
		oapiReleaseDC (surf, DC);
		return true;
	}

	// ballistic target indicator
	if (id == AREA_TGT_IND)
	{
		if (bTargetValid)
			oapiBlt (surf, surfIndValid, 0, 0, 0, 0, IND_W, IND_H, 0xFFFFFF);
		
		return true;
	}

	// ballistic target indicator
	if (id == AREA_TIME_IND)
	{
		if (bTimeValid)
			oapiBlt (surf, surfIndValid, 0, 0, 0, 0, IND_W, IND_H, 0xFFFFFF);
		
		return true;
	}

	// 'set flight program' button
	if (id == AREA_BTN_SET)
	{
		if (bTargetValid && bTimeValid)
			oapiBlt (surf, surfSetValid, 0, 0, 0, 0, BTN_SET_W, BTN_SET_H, 0xFFFFFF);
		
		return true;
	}

	return false;
}


bool R7_launchpad::clbkPanelMouseEvent (int id, int event, int mx, int my)
{
	// number up buttons
	if ((id >= AREAS_UP) && (id < AREAS_DOWN))
	{
		panClickNumberUp(id-AREAS_UP);
		return true;
	}

	// number up buttons
	if ((id >= AREAS_DOWN) && (id < AREAS_OTHER))
	{
		panClickNumberDown(id-AREAS_DOWN);
		return true;
	}

	if (id == AREA_BTN_SET)
	{
		panSetFlightProgram();
		return true;
	}

	return true;
}


void R7_launchpad::panClickNumberUp(int id)
{
	if (NumInfos[id].sign == 0)
	{
		NumInfos[id].iVarCache++;
		if (NumInfos[id].iVarCache > NumInfos[id].iMax)
			NumInfos[id].iVarCache = NumInfos[id].iMin;
	}
	else
		NumInfos[id].sign *= -1;
	
	oapiTriggerPanelRedrawArea (0, id);
	panReevaluatePanel();
	PlayVesselWave3(SoundLibID, SND_BUTTON);
}


void R7_launchpad::panClickNumberDown(int id)
{
	if (NumInfos[id].sign == 0)
	{
		NumInfos[id].iVarCache--;
		if (NumInfos[id].iVarCache < NumInfos[id].iMin)
			NumInfos[id].iVarCache = NumInfos[id].iMax;
	}
	else
		NumInfos[id].sign *= -1;
	
	oapiTriggerPanelRedrawArea (0, id);
	panReevaluatePanel();
	PlayVesselWave3(SoundLibID, SND_BUTTON);
}


void R7_launchpad::panReevaluatePanel()
{
	// reset scenario warning
	sScenarioWarning[0] = 0;
	sTimeWarning[0] = 0;

		// TARGET VALIDATION

	// reevaluate ballistic target validity...
	if (TrajectoryType == TRAJECTORY_ORBITAL)
		panValidateOrbTarget();

	// ...or, for orbit, more steps are needed
	else if (TrajectoryType == TRAJECTORY_BALLISTIC)
	{
		// redraw target description
		oapiTriggerPanelRedrawArea (0, AREA_BAL_DESCR);

		// show or hide custom lat-lon fields, depending on target selected
		panRedrawBalTargetCoordinates();

		// finally, reevaluate orbital target validity
		panValidateBalTarget();
	}

	// set flag for the target validity
	bTargetValid = (sScenarioWarning[0] == 0);

	// update target indicator
	oapiTriggerPanelRedrawArea (0, AREA_TGT_IND);
	oapiTriggerPanelRedrawArea (0, AREA_TGT_STR);

		// TIME VALIDATION

	// reevaluate launch time validity
	panValidateLaunchTime();

	// set flag for the launch time validity
	bTimeValid = (sTimeWarning[0] == 0);

	// update target indicator
	oapiTriggerPanelRedrawArea (0, AREA_TIME_IND);
	oapiTriggerPanelRedrawArea (0, AREA_TIME_STR);

		// FULL VALIDATION

	// redraw 'set flight program' button
	oapiTriggerPanelRedrawArea (0, AREA_BTN_SET);
}


void R7_launchpad::panRedrawBalTargetCoordinates()
{
	for (int i=AREA_BAL_TGT_LAT; i<AREA_BAL_TGT_LAT+9; i++)
	{
		NumInfos[i].bVisible = (NumInfos[AREA_BAL_TGT].iVarCache == 7);
		oapiTriggerPanelRedrawArea (0, i);
		oapiTriggerPanelRedrawArea (0, i+AREAS_UP);
		oapiTriggerPanelRedrawArea (0, i+AREAS_DOWN);
	}

	for (int i=AREA_BAL_TGT_LON; i<AREA_BAL_TGT_LON+9; i++)
	{
		NumInfos[i].bVisible = (NumInfos[AREA_BAL_TGT].iVarCache == 7);
		oapiTriggerPanelRedrawArea (0, i);
		oapiTriggerPanelRedrawArea (0, i+AREAS_UP);
		oapiTriggerPanelRedrawArea (0, i+AREAS_DOWN);
	}
}


void R7_launchpad::panValidateBalTarget()
{
	//
	if (NumInfos[AREA_BAL_TGT].iVarCache < 7)
		bTargetValid = true;
	else
	{
		// lat numbers to double
		BalTargetLatInfo.dVarCache = 0;
		BalTargetLatInfo.dVarCache += (NumInfos[AREA_BAL_TGT_LAT+1].iVarCache*100.);
		BalTargetLatInfo.dVarCache += (NumInfos[AREA_BAL_TGT_LAT+2].iVarCache*10.);
		BalTargetLatInfo.dVarCache += (NumInfos[AREA_BAL_TGT_LAT+3].iVarCache);
		BalTargetLatInfo.dVarCache += (NumInfos[AREA_BAL_TGT_LAT+5].iVarCache)/10.;
		BalTargetLatInfo.dVarCache += (NumInfos[AREA_BAL_TGT_LAT+6].iVarCache)/100.;
		BalTargetLatInfo.dVarCache += (NumInfos[AREA_BAL_TGT_LAT+7].iVarCache)/1000.;
		BalTargetLatInfo.dVarCache += (NumInfos[AREA_BAL_TGT_LAT+8].iVarCache)/10000.;
		BalTargetLatInfo.dVarCache *= NumInfos[AREA_BAL_TGT_LAT].sign;

		// lon numbers to double
		BalTargetLonInfo.dVarCache = 0;
		BalTargetLonInfo.dVarCache += (NumInfos[AREA_BAL_TGT_LON+1].iVarCache*100.);
		BalTargetLonInfo.dVarCache += (NumInfos[AREA_BAL_TGT_LON+2].iVarCache*10.);
		BalTargetLonInfo.dVarCache += (NumInfos[AREA_BAL_TGT_LON+3].iVarCache);
		BalTargetLonInfo.dVarCache += (NumInfos[AREA_BAL_TGT_LON+5].iVarCache)/10.;
		BalTargetLonInfo.dVarCache += (NumInfos[AREA_BAL_TGT_LON+6].iVarCache)/100.;
		BalTargetLonInfo.dVarCache += (NumInfos[AREA_BAL_TGT_LON+7].iVarCache)/1000.;
		BalTargetLonInfo.dVarCache += (NumInfos[AREA_BAL_TGT_LON+8].iVarCache)/10000.;
		BalTargetLonInfo.dVarCache *= NumInfos[AREA_BAL_TGT_LON].sign;

		//
		VerifyTargetingBallistic("", BalTargetLatInfo.dVarCache, BalTargetLonInfo.dVarCache);
	}
}


void R7_launchpad::panValidateOrbTarget()
{
	// per numbers to double
	OrbPerInfo.dVarCache = 0;
	OrbPerInfo.dVarCache += (NumInfos[AREA_ORB_PER].iVarCache*100.);
	OrbPerInfo.dVarCache += (NumInfos[AREA_ORB_PER+1].iVarCache*10.);
	OrbPerInfo.dVarCache += (NumInfos[AREA_ORB_PER+2].iVarCache);

	// apo numbers to double
	OrbApoInfo.dVarCache = 0;
	OrbApoInfo.dVarCache += (NumInfos[AREA_ORB_APO].iVarCache*100.);
	OrbApoInfo.dVarCache += (NumInfos[AREA_ORB_APO+1].iVarCache*10.);
	OrbApoInfo.dVarCache += (NumInfos[AREA_ORB_APO+2].iVarCache);

	// inc numbers to double
	OrbIncInfo.dVarCache = 0;
	OrbIncInfo.dVarCache += (NumInfos[AREA_ORB_INC+1].iVarCache*10.);
	OrbIncInfo.dVarCache += (NumInfos[AREA_ORB_INC+2].iVarCache);
	OrbIncInfo.dVarCache += (NumInfos[AREA_ORB_INC+4].iVarCache)/10.;
	OrbIncInfo.dVarCache *= NumInfos[AREA_ORB_INC].sign;

	//
	VerifyTargetingOrbital(OrbPerInfo.dVarCache, OrbApoInfo.dVarCache, OrbIncInfo.dVarCache);
}


void R7_launchpad::panValidateLaunchTime()
{
	// hours to double
	TimeHInfo.dVarCache = 0;
	TimeHInfo.dVarCache += (NumInfos[AREA_TIME_H].iVarCache*10.);
	TimeHInfo.dVarCache += (NumInfos[AREA_TIME_H+1].iVarCache);

	// minutes to double
	TimeMnInfo.dVarCache = 0;
	TimeMnInfo.dVarCache += (NumInfos[AREA_TIME_MN].iVarCache*10.);
	TimeMnInfo.dVarCache += (NumInfos[AREA_TIME_MN+1].iVarCache);

	// seconds to double
	TimeSInfo.dVarCache = 0;
	TimeSInfo.dVarCache += (NumInfos[AREA_TIME_S].iVarCache*10.);
	TimeSInfo.dVarCache += (NumInfos[AREA_TIME_S+1].iVarCache);

	// cached timer
	CachePanTimer = -(TimeSInfo.dVarCache + TimeMnInfo.dVarCache*60 + TimeHInfo.dVarCache*3600);
	if (CachePanTimer == 0)
		CachePanTimer = TIMER_NOT_SET;

	//
	VerifyLaunchTime(CachePanTimer, 1); 
}



void R7_launchpad::panSetFlightProgram()
{
	// don't do anything if conditions are not satisfactory
	if ((!bTargetValid) || (!bTimeValid))
		return;

	if (TrajectoryType == TRAJECTORY_ORBITAL)
	{
		Perigee = OrbPerInfo.dVarCache;
		Apogee = OrbApoInfo.dVarCache;
		Inc = OrbIncInfo.dVarCache;
	}
	else if (TrajectoryType == TRAJECTORY_BALLISTIC)
	{
		if (NumInfos[AREA_BAL_TGT].iVarCache == 7)
		{
			strcpy(TargetName, "");
			TargetLat = BalTargetLatInfo.dVarCache;
			TargetLon = BalTargetLonInfo.dVarCache;
		}
		else
			strcpy(TargetName, BalTargets[NumInfos[AREA_BAL_TGT].iVarCache-1].NameCamel);
	}

	// set timer to new or current value
	Timer = CachePanTimer;

	// press F8 to close targeting panel
	keybd_event(0, 0x42, KEYEVENTF_SCANCODE, 0);
	keybd_event(0, 0x42, KEYEVENTF_SCANCODE | KEYEVENTF_KEYUP, 0);

	// recalculate trajectory!
	VerifyTargeting();

	ShowAnnotationMessage("Flight program is set!");
	TrySetAutopilotControl();
	PlayVesselWave3(SoundLibID, SND_BUTTON);

	// return cable mast(s) to the rocket
	CableMastRotPercent = 1;
	SetAnimation (CableMastRotAnim, CableMastRotPercent);
}



void R7_launchpad::GetTargetOrbit()
{
	OBJHANDLE hTarget = oapiGetVesselByName ("Target");
	if (!oapiIsVessel(hTarget))
		return;

	// get target's orbit
	VESSEL *VIntf;
	VIntf = oapiGetVesselInterface(hTarget);
	ELEMENTS OrbEls;
	ORBITPARAM OrbParam;
	VIntf->GetElements(hEarth, OrbEls, &OrbParam, 0, FRAME_EQU);

	// get target's orbit inclination
	double OT_Inc;
	OT_Inc = OrbEls.i;
	double OT_IncD = OT_Inc*DEG;

	// get target's globe position
	double OT_Lat;
	double OT_Lon;
	double OT_R;
	VIntf->GetEquPos(OT_Lon, OT_Lat, OT_R);
	double OT_LatD = OT_Lat*DEG;
	double OT_LonD = OT_Lon*DEG;

	// get distance from Earth center to "launchpad circle" center
	double HLaunch = RADIUS_E * sin(LP_LAT*RAD);

	double HOrbitMax = RADIUS_E * sin(OT_Inc);
	double HOrbit = RADIUS_E * sin(OT_Lat);

	double LonOrbit = asin(HOrbit/HOrbitMax);
	double LonOrbitD = LonOrbit*DEG;

	double dLonOrbit = LonOrbit-OT_Lon;
	double dLonOrbitD = dLonOrbit*DEG;

	double LonLaunchOrbit = LP_LON*RAD + dLonOrbit; 
	double LonLaunchOrbitD = LonLaunchOrbit*DEG;

	double LonOrbitCross = asin(HLaunch/HOrbitMax);
	double LonOrbitCrossD = LonOrbitCross*DEG;

//	double LonCross = LonOrbitCross - dLonOrbit;
//	double LonCrossD = LonCross*DEG;


//	double dLon = LonLaunchOrbit-LonOrbitCross;
	double dLon = LonOrbitCross - LonLaunchOrbit;
	if (dLon < 0)
		dLon = 360 + dLon;
	double dLonD = dLon*DEG;

	double dt = dLon*DEG * 240;  // 1 deg = 240 time sec

/*
Inc = OT_IncD;

	Timer = dt;
	if (Timer > 0)
		Timer = -Timer;
	// update intermediate vars
	TimeSecondsToStr(Timer, strTimer);
*/
}


//=======================================================================

DLLCLBK VESSEL *ovcInit(OBJHANDLE hvessel, int flightmodel)
{
	return new R7_launchpad(hvessel, flightmodel);
}

DLLCLBK void ovcExit(VESSEL *vessel)
{
	if (vessel) delete (R7_launchpad*) vessel;
}


