#include	"Warhead.h"




Warhead::Warhead(OBJHANDLE hObj, int fmodel): VESSEL2M(hObj, fmodel)
{
	// VESSEL2M crash parameters
	VvCrash = DRand(5, 30); 
	VhCrash = DRand(20, 30);
	VMaxBlast = DRand(1000, 1); 
	QMax = DRand(350000, 50000);

	// VESSEL2M wrecks
	WreckIdxs[0] = 11;

	// default to test item
	MeAsPayloadType = PAYLOAD_WARHEAD_TEST;

	// nuke states and effects
	NukeState = N_NONE;
	Timer = -1;
	CloudLevel = 0;
	ColumnLevel = 0;
	WhiteLevel = 0;
	ColorLevel = 0;
	InitialBlastLevel = 0;

	// turn to nadir
	bTurnToNadir = false;
	hEarth = oapiGetGbodyByName("Earth");

	// glow effect
	hVisual = NULL;
	iMesh = -1;
	bAlphaNotBlended = true;

	// shine effect
	SURFHANDLE ExhTex = oapiRegisterExhaustTexture("r7_S\\Exhaust3");
	ShineTank = CreatePropellantResource(5);
	thShinePl = CreateThruster(_V(0, 0, 0), zplus, 200*G, ShineTank, 50000);
	thShineMn = CreateThruster(xyz0, zminus, 200*G, ShineTank, 50000);
	AddExhaust(thShinePl, 0, 1, ExhTex);

	// self-destructor
	bHaveSelfDestructor = true;
	
	// target
	PadLat = 0;
	PadLon = 0;
	TargetLat = 0;
	TargetLon = 0;

	// kml
	kmlPlotColor = KML_CLR_YELLOW;

	// mission outcome
	strcpy(Outcomes[OC_SUCCESS].OutcomeMessage, "Mission accomplished! Warhead reached the target area");
	strcpy(Outcomes[OC_PARTIAL].OutcomeMessage, "Mission partially successful! Warhead reached the target area, but did not survive reentry...");
	strcpy(Outcomes[OC_FAIL].OutcomeMessage, "Mission failed. Warhead did not reach the target area...");
}


void Warhead::clbkSetClassCaps(FILEHANDLE cfg)
{
	// physics and geometry
	SetCW (0.55, 0.7, 1.5, 1.5);							
	SetRotDrag (_V(0.7, 0.7, 0.1));				
	SetLiftCoeffFunc(0);
	SetPitchMomentScale(1e-4);
	SetBankMomentScale(1e-4);
	SetSurfaceFrictionCoeff(1e5, 1e5);
	SetTouchdownPoints (_V( 0, 0.4, -1), _V( 1, 0.7, 1), _V(-1, 0.7, 1));
}


void Warhead::clbkSaveState (FILEHANDLE scn)
{
	VESSEL2M::clbkSaveState(scn);
	oapiWriteScenario_string (scn, "=========== WARHEAD vars", "");

	if (NukeState > N_NONE)
		oapiWriteScenario_string (scn, "NUKE", "");

	if (NukeState > N_INSTALLED)
		oapiWriteScenario_int (scn, "N_STATE_1", NukeState);

	if (NukeState > N_DISTANTIANTING)
	{
		oapiWriteScenario_vec (scn, "N_STATE_2", _V(Timer, InitialBlastLevel, WhiteLevel));
		oapiWriteScenario_vec (scn, "N_STATE_3", _V(ColorLevel, ColumnLevel, CloudLevel));
	}

	if (PadLat+PadLon+TargetLat+TargetLon != 0)
	{
		oapiWriteScenario_vec (scn, "PAD_COORDS", _V(PadLat, PadLon, 0)*DEG);
		oapiWriteScenario_vec (scn, "TGT_COORDS", _V(TargetLat, TargetLon, 0)*DEG);
	}

}


void Warhead::ParseScenarioLineEx(char *line, void *status)
{
	if (!strnicmp (line, "NUKE", 4))
	{
		NukeState = N_INSTALLED;
		strcpy(Outcomes[OC_SUCCESS].OutcomeMessage, "Success - target destroyed!");
		strcpy(Outcomes[OC_PARTIAL].OutcomeMessage, "Partial success - target heavily damaged!");
		strcpy(Outcomes[OC_FAIL].OutcomeMessage, "Mission failure - target is intact!");
	}

	else if (!strnicmp (line, "N_STATE_1", 9))
		sscanf (line+9, "%d", &NukeState);

	else if (!strnicmp (line, "N_STATE_2", 9))
		sscanf (line+9, "%lf %lf %lf", &Timer, &InitialBlastLevel, &WhiteLevel);

	else if (!strnicmp (line, "N_STATE_3", 9))
		sscanf (line+9, "%lf %lf %lf", &ColorLevel, &ColumnLevel, &CloudLevel);

	else if (!strnicmp (line, "PAD_COORDS", 10))
	{
		sscanf (line+10, "%lf %lf", &PadLat, &PadLon);
		PadLat *= RAD;
		PadLon *= RAD;
	}

	else if (!strnicmp (line, "TGT_COORDS", 10))
	{
		sscanf (line+10, "%lf %lf", &TargetLat, &TargetLon);
		TargetLat *= RAD;
		TargetLon *= RAD;
	}

	else 
		VESSEL2M::ParseScenarioLineEx(line, status);
}


int Warhead::clbkConsumeBufferedKey(DWORD key, bool down, char *kstate)
{
	return VESSEL2M::clbkConsumeBufferedKey(key, down, kstate);
}


void Warhead::FinishConfiguration()
{
	// inherited;
	VESSEL2M::FinishConfiguration();

	// find out scenario date 
	SYSTEMTIME mjddate;
	MJDtoSystemTime(oapiGetSimMJD(),&mjddate);
	WORD ScnYear = mjddate.wYear;

	// define configuration
	if (ScnYear < 1958)
	{
		MeAsPayloadType = PAYLOAD_WARHEAD_TEST;
		hMesh = oapiLoadMeshGlobal("r7_S\\WarheadTest");
		SetEmptyMass(5500-5);
		SetSize(7.5);
		SetPMI(_V(1.62, 1.62, 0.41));
		SetCrossSections(_V(7.6, 7.6, 4.29));
		QMax = DRand(300000, 10000);
		NukeState = N_NONE;	// don't allow nuke in test warhead
	}

	else if (ScnYear < 1960)
	{
		MeAsPayloadType = PAYLOAD_WARHEAD_5T;
		hMesh = oapiLoadMeshGlobal("r7_S\\Warhead5t");
		SetEmptyMass(5500-5);
		SetSize(7.5);
		SetPMI(_V(1.58, 1.58, 0.42));
		SetCrossSections(_V(7.26, 7.26, 4.29));
		QMax = DRand(5000000, 50000);
	}
	
	else
	{
		MeAsPayloadType = PAYLOAD_WARHEAD_3T;
		hMesh = oapiLoadMeshGlobal("r7_S\\Warhead3t");
		SetEmptyMass(3700-5);
		SetSize(3.5);
		SetPMI(_V(0.77, 0.77, 0.22));
		SetCrossSections(_V(3.66, 3.66, 2.21));
		QMax = DRand(5000000, 50000);
	}

	// finish loading mesh
	iMesh = AddMesh(hMesh);	
	pMatTmpl = oapiMeshMaterial(hMesh, 1);

	// not nuke? hide radiation sign!
	if (NukeState == N_NONE)
	{
		// update instance's mesh "template" in case we'll need to re-create "visual" mesh
		SURFHANDLE hNewTexture = oapiLoadTexture("r7_S\\head_blank_wo_sign.dds", true);
		oapiSetTexture (hMesh, 2, hNewTexture);

		// update "visual" mesh, if any
		if (hVisual != NULL)
		{
			MESHHANDLE hMeshVis = GetMesh(hVisual, iMesh);
			oapiSetTexture (hMeshVis, 2, hNewTexture);
		}
	}

	// already blasting (restored from scenario)?
	else if (NukeState > N_DISTANTIANTING)
		ModifyWarheadToBlast();
}


void Warhead::clbkPostStep(double simt, double SimDT, double mjd)
{
	// common processing for all VESSEL2M-derived objects
	TimestepForVESSEL2M();

	// detect when we get separated from the rocket
	if (bAttached)
	{
		// this indicates our separation from the rocket
		if (GetAttachmentStatus(GetAttachmentHandle(TOPARENT, 0)) == NULL)
		{
			// sound broadcast does not work during focus switch, shoud duplicate here
			PlayVesselWave3(SoundLibID, SND_PYRO);
			bAttached = false;
		}
		return;
	}

	// update flight params
	double Alt = GetAltitude();
	double Vel = GetAirspeed();

	// don't shine at all if not moving really fast...
	if (Vel < 500)
		Vel = 0;

	// slow down time accel on reentry
	if ( (Alt < 200000) && (oapiGetTimeAcceleration() > 10) )
		oapiSetTimeAcceleration(10);
	if ( (Alt < 20000) && (oapiGetTimeAcceleration() > 1) )
		oapiSetTimeAcceleration(1);

	//
	double Kv = Vel / 12000;
	if (Kv > 1)
		Kv = 1;

		// shine effect on reentry

	double ShAltMax = Kv * 200000; // 200 km
	double ShAltMin = Kv * 100000; // 100 km

	double ShLevel;
	if (Alt > ShAltMax)
		ShLevel = 0;
	else if (Alt < ShAltMin)
		ShLevel = 1;
	else
		ShLevel = (ShAltMax - Alt) / (ShAltMax - ShAltMin);

	SetThrusterLevel(thShinePl, ShLevel);
	SetThrusterLevel(thShineMn, ShLevel);

		// glow effect on reentry

	double GlAltMax = Kv * 150000; // 150 km
	double GlAltMin = Kv * 50000; // 50 km

	double alpha;
	if (Alt > GlAltMax)
		alpha = 0;
	else if (Alt < GlAltMin)
		alpha = 1;
	else
		alpha = (GlAltMax - Alt) / (GlAltMax - GlAltMin);

	// invert aplha
	alpha = 1 - alpha;

	// do a one-time alpha blend operation
	if ( (alpha != 1) && (bAlphaNotBlended) )
	{
		bAlphaNotBlended = false;

		// blend alpha channels of texture and material
		// side effect: Sun disappears because of Orbiter bug
		oapiSetMeshProperty(hMesh, MESHPROPERTY_MODULATEMATALPHA, 1);

		if (hVisual != NULL)
		{
			MESHHANDLE hMeshVis = GetMesh(hVisual, iMesh);
			if (hMeshVis != NULL)
				oapiSetMeshProperty(hMeshVis, MESHPROPERTY_MODULATEMATALPHA, 1);
		}
	}
	
	//
	if (alpha != pMatTmpl->ambient.a) 
	{
		//
		pMatTmpl->ambient.a = (float)alpha;
		pMatTmpl->diffuse.a = (float)alpha;
		pMatTmpl->specular.a = (float)alpha;
		pMatTmpl->emissive.a = (float)alpha;

		//
		if (hVisual != NULL)
		{
			MESHHANDLE hMeshVis = GetMesh(hVisual, iMesh);

			if (hMeshVis != NULL)
			{
				MATERIAL* pMatVis = oapiMeshMaterial(hMeshVis, 1);

				pMatVis->ambient.a = (float)alpha;
				pMatVis->diffuse.a = (float)alpha;
				pMatVis->specular.a = (float)alpha;
				pMatVis->emissive.a = (float)alpha;
			}
		}
	}

	//  debug output
	//sprintf(oapiDebugString(), "H: %f      V: %f       ShLevel: %f", GetAltitude(), GetAirspeed(), ShLevel);

		// nuke effects

	// nothing more to do if no nuke
	if (NukeState == N_NONE)
		return;
	
		// process nuke activation steps

	// update timer if set
	if (Timer > 0)
		Timer += SimDT;


	if (NukeState == N_INSTALLED)
	{
		if (Alt > 200000)
			NukeState = N_ARMED;
	}

	else if (NukeState == N_ARMED)
	{			
		if (Alt < 5000)
			MoveAway();
	}

	else if (NukeState == N_DISTANTIANTING)
	{
		if (Alt < 1500)
			StartBlast();
	}

	else if (NukeState == N_DETONATING)
		UpdateBlast();

	if (bTurnToNadir)
	{
		GetNavmodeControls();
		SetGyroControls();
	}
}


void Warhead::MoveAway()
{
	// cancel time accel, if any
	if ( oapiGetTimeAcceleration() > 1)
		oapiSetTimeAcceleration(1);

	// move to next stage
	NukeState = N_DISTANTIANTING;

	if (!bHaveFocus)
		return;

	// force camera mode to external view
	double OriginalCameraMode = oapiCameraMode();

	// switch camera to outside view, global frame
	if (OriginalCameraMode == CAM_COCKPIT)
		oapiCameraAttach(GetHandle(), 1); 

	// move camera away
	double CurrDist = oapiCameraTargetDist();
	oapiCameraScaleDist(10000/CurrDist);

	sprintf(oapiDebugString(), "Put on eye protection!");
}


void Warhead::StartBlast()
{
	// stop blending alpha channels of texture and material
	// side effect: Sun becomes visible again
	oapiSetMeshProperty(hMesh, MESHPROPERTY_MODULATEMATALPHA, 0);

	// update "visual" mesh, if any
	if (hVisual != NULL)
	{
		MESHHANDLE hMeshVis = GetMesh(hVisual, iMesh);
		oapiSetMeshProperty(hMeshVis, MESHPROPERTY_MODULATEMATALPHA, 0);
	}

	sprintf(oapiDebugString(), "");

	ModifyWarheadToBlast();

	// set special effects timer
	Timer = 0.01;

	// write kml data, preventing nominal kmlFinish() to run
	kmlAddEvent(KML_NUKE);
	kmlAddEvent(KML_ARRIVAL, "Epicenter");

	// move to next stage
	NukeState = N_DETONATING;
}

void Warhead::ModifyWarheadToBlast()
{
	// create streams for visual effects
	CreateParticleStreams();

	// remove old warhead visuals
	ClearMeshes();

	// update physics and/or state: slow down, descend slowly, don't crash
	SetCW (10, 10, 10, 10);							
	SetEmptyMass(95);
	VvCrash = 100000; 
	VhCrash = 100000;
	VMaxBlast = 100000; 
	QMax = 10000000;

	// not quite a particle stream, but this is a good place to create it
	PROPELLANT_HANDLE UpTank = CreatePropellantResource(5);
	SetPropellantMass(UpTank, 5);
	thUp = CreateThruster(xyz0, -zplus, 200*G, UpTank, 50000);

	// disable self-destructor
	bHaveSelfDestructor = false;

	// stop plotting (or don't start it if just loaded)
	kmlNoPlotting = true;
	SendVesselMessage(kmlHandle, VM_KML_STOP_PLOTTING, (int)GetHandle());
}


void Warhead::UpdateBlast()
{
	// process nuke timeline
	if (Timer > 0)
	{
		// initial super-fast flash
		if (Timer < 1)
			InitialBlastLevel = 1;
		else
		{
			InitialBlastLevel = 0;
			bTurnToNadir = true;
		}

		if (Timer > 5)
			SetThruster();

		// bright white longer flash
		if (Timer < 10)
			WhiteLevel = 1;

		// gradually add color to white flash
		if ((Timer > 0) && (Timer < 10))
			ColorLevel = (Timer)/10;

		// sounds
		if ((Timer > 10)&& (Timer < 11))
			if (!IsPlaying3(SoundLibID, SND_BLAST1))
				SendVesselWaveToFocused(SND_BLAST1);

		if ((Timer > 13)&& (Timer < 14))
			if (!IsPlaying3(SoundLibID, SND_BLAST2))
				SendVesselWaveToFocused(SND_BLAST2);

		// gradually remove white glow, leaving only color
		if ((Timer > 10) && (Timer < 15))
			WhiteLevel = (15-Timer)/5;

		if (Timer > 15) 
			WhiteLevel = 0;

		// gradually remove color glow, leaving only smoke
		if ((Timer > 20) && (Timer < 25))
			ColorLevel = (25-Timer)/5;

		if (Timer > 25) 
			ColorLevel = 0;

		// start growing smoke cloud
		if ((Timer > 15) && (Timer < 20))
			CloudLevel = (Timer-15)/5;

		if ((Timer > 15) && (Timer < 15.05) && (!bOutcomeCriteriaSet))
			ShowHitOrNukeOutcome();

		// start growing smoke column
		if (Timer > 20) 
		{
			CloudLevel = 1;
			ColumnLevel = 1;
		}

		// done with all streams
		if (Timer > 120) 
		{
			CloudLevel = 0;
			ColumnLevel = 0;
			WhiteLevel = 0;
			ColorLevel = 0;
			InitialBlastLevel = 0;
		}
	}


	// landed? stop blast
	if (GroundContact())
	{
		// move to last step
		NukeState = N_DETONATED;

		// deactivate timer
		Timer = -1;
			
		// stop special effects
		InitialBlastLevel = 0;
		WhiteLevel = 0;
		ColorLevel = 0;
		ColumnLevel = 0;
		CloudLevel = 0;

		// any after effects on the ground?
		//%%%
	}
}


void Warhead::CreateParticleStreams()
{
	// create initial very quickly expanding but short-lived a very quickly expanding white "fireball"
	PARTICLESTREAMSPEC InitialBlast = {
		0, 
		50,								// srcsize, particle size at creation, m
		0.1,							// srcrate, average particle generation rate, Hz   
		0, 
		0, 
		4,								// lifetime, average particle lifetime [s]
		10000,							// growthrate, particle growth rate [m/s]
		0, 
		PARTICLESTREAMSPEC::EMISSIVE,
		PARTICLESTREAMSPEC::LVL_FLAT, 
		1, 
		1,
		PARTICLESTREAMSPEC::ATM_FLAT, 
		1, 
		1
	};

	AddParticleStream (&InitialBlast, _V(0,0,0), _V(0,1,0), &InitialBlastLevel);

	// create a very quickly expanding white "fireball"
	PARTICLESTREAMSPEC BlastWhite = {
		0, 
		0.5, 
		10, 
		0, 
		0, 
		4, 
		500, 
		0, 
		PARTICLESTREAMSPEC::EMISSIVE,
		PARTICLESTREAMSPEC::LVL_FLAT, 
		1, 
		1,
		PARTICLESTREAMSPEC::ATM_FLAT, 
		1, 
		1
	};

	AddParticleStream (&BlastWhite, _V(0,0,0), _V(0,1,0), &WhiteLevel);


	// create a very quickly expanding color "fireball" to gradually replace white fireball
	SURFHANDLE ptex = oapiRegisterParticleTexture("r7_S\\Contrail3");
	PARTICLESTREAMSPEC BlastColor = {
		0, 
		0.5, 
		1,										// srcrate, average particle generation rate, Hz   
		0, 
		0, 
		4, 
		500, 
		0, 
		PARTICLESTREAMSPEC::EMISSIVE,
		PARTICLESTREAMSPEC::LVL_FLAT, 
		1, 
		1,
		PARTICLESTREAMSPEC::ATM_FLAT, 
		1, 
		1,
		ptex 
	};

	AddParticleStream (&BlastColor, _V(0,0,0), _V(0,1,0), &ColorLevel);


	// create a smoke column to touch the ground
	PARTICLESTREAMSPEC SmokeColumn = {
		0,			// flags, reserved
		15,			// srcsize, particle size at creation, m
		10,			// srcrate, average particle generation rate, Hz   
		30,			// V0, average particle emission velocity, m/s
		0,			// srcspread, emission velocity distribution randomisation
		30,			// lifetime, average particle lifetime [s]
		12,			// growthrate, particle growth rate [m/s]
		0,			// atmslowdown, deceleration rate b in atmosphere, defined as v = v0 e- bdt 
		PARTICLESTREAMSPEC::DIFFUSE,
		PARTICLESTREAMSPEC::LVL_FLAT, 
		1, 
		1,
		PARTICLESTREAMSPEC::ATM_FLAT, 
		1, 
		1
	};

	AddParticleStream (&SmokeColumn, xyz0, zplus, &ColumnLevel);

	
	// create a smoke inside the fireball to gradually replace it
	PARTICLESTREAMSPEC SmokeCloud = {
		0,			// flags, reserved
		15,			// srcsize, particle size at creation, m
		5,			// srcrate, average particle generation rate, Hz   
		0,			// V0, average particle emission velocity, m/s
		0,			// srcspread, emission velocity distribution randomisation
		20,			// lifetime, average particle lifetime [s]
		35,			// growthrate, particle growth rate [m/s]
		100,		// atmslowdown, deceleration rate b in atmosphere, defined as v = v0 e- bdt 
		PARTICLESTREAMSPEC::DIFFUSE,
		PARTICLESTREAMSPEC::LVL_FLAT, 
		1, 
		1,
		PARTICLESTREAMSPEC::ATM_FLAT, 
		1, 
		1,
		ptex 
	};

	AddParticleStream (&SmokeCloud, xyz0, zminus, &CloudLevel);
}




int Warhead::GetVesselMessage(IntVesselMessage VM, int Value)
{
	if (VM == VM_PAYLOAD_TYPE)
		return MeAsPayloadType;

	return VESSEL2M::GetVesselMessage(VM, Value);
}

double Warhead::GetVesselMessage(DblVesselMessage VM, double Value)
{
	if (VM == VM_SET_LAT)
		PadLat = Value;

	if (VM == VM_SET_LON)
		PadLon = Value;

	else if (VM == VM_SET_TGT_LAT)
		TargetLat = Value;

	else if (VM == VM_SET_TGT_LON)
		TargetLon = Value;

	return 0;
}




void Warhead::GetNavmodeControls()
{
	VECTOR3 NavmodeControlLevel = _V(0,0,0);

	NavmodeControlLevel = GetWantedBodyTrackAttitude(hEarth);

	// useful debug BEFORE inputs will be replaced by signs or modified by stops
	VECTOR3 Om;
	GetAngularVel(Om);

	// set control levels from navmode levels
	ControlLevel.x = -NavmodeControlLevel.x;
	ControlLevel.y =  NavmodeControlLevel.y;
	ControlLevel.z = -NavmodeControlLevel.z;

	// filter input signals by proximity to the neutral point, modify as needed
	UpdateStopConditions(&ControlLevel);

	// one more filter: don't allow any roll movement until pitch and yaw are close to their nominals!
	UpdateRollCondition(&ControlLevel);

	// further fiter the signals: for gyro motors, we need just +1 or -1 signs, not actual deviations
	ReplaceControlInputsBySignsWithZero(&ControlLevel);
}


// this function returns the vector with all three attitide sensors deviation data normalized to -1 to +1 range
VECTOR3 Warhead::GetWantedBodyTrackAttitude(OBJHANDLE hBody)
{
	// get global direction for Z axis
	VECTOR3 DirGlobal;
	GetRelativePos(hBody, DirGlobal);
	
	// turn direction around so that Z axis will point TO the target, not AWAY from it 
	DirGlobal = -DirGlobal;

	// turn global direction to local
	VECTOR3 DirLocal; 
	LocalRot(DirGlobal, DirLocal);

	// get pich-yaw from body positional vector
	VECTOR3 AttCotrols = GetPitchYawSensorsDeviation(DirLocal);

	// get roll from horizon, (always current gravity body horizon regardless of target body)
	AttCotrols.z = -GetWantedRollFromHorizon();

	// done!
	return AttCotrols;
}


// this function returns the vector with pitch-yaw attitide sensors deviation data normalized to -1 to +1 range (roll is always 0)
VECTOR3 Warhead::GetPitchYawSensorsDeviation(VECTOR3 DirLocal)
{
	VECTOR3 CurrentOrientLevels = DirLocal;

	// normalize local direction to length of 1
	CurrentOrientLevels /= length(CurrentOrientLevels);

	// get pitch-yaw angles to the given direction
	double Pitch = atan2(-CurrentOrientLevels.y, CurrentOrientLevels.z);		// 0+-pi/2 always, regardless of yaw-roll
	double Yaw = atan2(-CurrentOrientLevels.x, CurrentOrientLevels.z);			// 0+-pi/2 always, regardless of pitch-roll

	// normalize yaw-pitch sensor signals to 0+-1 range
	double PitchLevel = Pitch / PI;
	double YawLevel = Yaw / PI;

	// set orientation vector (roll to 0, as we don't have this data...)
	CurrentOrientLevels.x = PitchLevel;
	CurrentOrientLevels.y = YawLevel;
	CurrentOrientLevels.z = 0; 
	
	// done!
	return CurrentOrientLevels;
}


// update roll attitude channel from local horizon angle
double Warhead::GetWantedRollFromHorizon()
{
	// get our pointing in the local horizon frame
	VECTOR3 DirNorthLocal;
	HorizonInvRot(zplus, DirNorthLocal);

	// get xy plane projection, eliminating z (thus we'll always get full 0-1 range regardless of projection length
	double xy = sqrt(DirNorthLocal.x*DirNorthLocal.x + DirNorthLocal.y*DirNorthLocal.y);

	// get initial roll signal level, relative to xy projection length, good to +y quadrants
	double RollLevel = (DirNorthLocal.x * 0.5) / xy;

	// update roll level for -y quadrants
	if (DirNorthLocal.y < 0) 
		RollLevel = signNoZero(DirNorthLocal.x) - RollLevel;

	// done!
	return RollLevel;
}


void Warhead::SetGyroControls()
{
	// get old omegas
	VECTOR3 omegas;
	GetAngularVel(omegas);

	// update omegas
	SetControllChannel(ControlLevel.x, omegas.x);
	SetControllChannel(ControlLevel.y, omegas.y);
	SetControllChannel(ControlLevel.z, omegas.z);

	// set new omegas
	SetAngularVel (omegas);
}

void Warhead::SetControllChannel(double ControlLevel, double &GyroLevel)
{
	if (ControlLevel != 0)
	{
		// spinning too fast? gradually slow down to nominal before doing anything else...
		if (fabs(GyroLevel) > OMEGA_NOM)
			GyroLevel = 0.99 * GyroLevel;

		// within nominal window and commanded to stop? do it...
		else if (ControlLevel == STOP_ROT)
			GyroLevel = 0;

		// set nominal spin rate in the specified direction
		else
			GyroLevel = ControlLevel * OMEGA_NOM;
	}
}


void Warhead::ReplaceControlInputsBySignsWithZero(VECTOR3* V)
{
	if (V->x != STOP_ROT)
		V->x = signWithZero(V->x);

	if (V->y != STOP_ROT)
		V->y = signWithZero(V->y);

	if (V->z != STOP_ROT)
		V->z = signWithZero(V->z);
}


void Warhead::UpdateStopConditions(VECTOR3* V)
{
	if (fabs(V->x) < EPS_STOP)
		V->x = STOP_ROT;

	if (fabs(V->y) < EPS_STOP)
		V->y = STOP_ROT;

	if (fabs(V->z) < EPS_STOP)
		V->z = STOP_ROT;
}


// control channels filter: don't allow any roll movement until pitch and yaw are close to their nominals!
void Warhead::UpdateRollCondition(VECTOR3* V)
{
	double x, y;

	if (V->x == STOP_ROT)
		x = 0;
	else
		x = V->x;

	if (V->y == STOP_ROT)
		y = 0;
	else
		y = V->x;

	if ( (fabs(x) > EPS_PITCH_YAW) || (fabs(y) > EPS_PITCH_YAW) )
		V->z = STOP_ROT;
}


const double VEL_V_NOM = 5;

void Warhead::SetThruster()
{
	double level = 1;

	SetThrusterLevel (thUp, level);
}


void Warhead::Burn()
{
	// stop blending alpha channels of texture and material
	// side effect: Sun becomes visible again
	oapiSetMeshProperty(hMesh, MESHPROPERTY_MODULATEMATALPHA, 0);

	// update "visual" mesh, if any
	if (hVisual != NULL)
	{
		MESHHANDLE hMeshVis = GetMesh(hVisual, iMesh);
		oapiSetMeshProperty(hMeshVis, MESHPROPERTY_MODULATEMATALPHA, 0);
	}

	GetEquPos(HitLon, HitLat, Radius);
	DistanceAzimuthFromCoordPairRad(TargetLat, TargetLon, HitLat, HitLon, &Distance, &Azimuth);
	Distance *= RADIUS_E;

	if (NukeState > N_NONE)
		OutcomeCriteria = OC_FAIL;

	else if (Distance < 30000)
	{
		OutcomeCriteria = OC_PARTIAL;
		strcpy(Outcomes[OC_PARTIAL].OutcomeMessage, "Mission partially successful! Warhead reached the target area with moderate accuracy...");
	}

	// inherited
	VESSEL2M::Burn();
}

void Warhead::Break()
{
	// write arrival event to the launchpad kmlVessel
	kmlAddEvent(KML_ARRIVAL, "Epicenter");

	GetEquPos(HitLon, HitLat, Radius);
	DistanceAzimuthFromCoordPairRad(TargetLat, TargetLon, HitLat, HitLon, &Distance, &Azimuth);
	Distance *= RADIUS_E;

	if (NukeState > N_NONE)
		OutcomeCriteria = OC_FAIL;

	else if (GroundContact())
	{

		if (Distance < 5000)
			OutcomeCriteria = OC_SUCCESS;

		else if (Distance < 30000)
		{
			OutcomeCriteria = OC_PARTIAL;
			strcpy(Outcomes[OC_PARTIAL].OutcomeMessage, "Mission partially successful! Warhead reached the target area with moderate accuracy...");
		}
	}
	else
	{
		if (Distance < 30000)
			OutcomeCriteria = OC_PARTIAL;
	}

	//
	bOutcomeCriteriaSet = true;

	// inherited
	VESSEL2M::Break();
}




void Warhead::clbkVisualCreated (VISHANDLE vis, int refcount)
{
	hVisual = vis;
}


void Warhead::clbkVisualDestroyed (VISHANDLE vis, int refcount)
{
	hVisual = NULL;
}


void Warhead::ShowHitOrNukeOutcome()
{
	GetEquPos(HitLon, HitLat, Radius);
	DistanceAzimuthFromCoordPairRad(TargetLat, TargetLon, HitLat, HitLon, &Distance, &Azimuth);
	Distance *= RADIUS_E;

	if (Distance < 5000)
		OutcomeCriteria = OC_SUCCESS;
	else if (Distance < 10000)
		OutcomeCriteria = OC_PARTIAL;

	bOutcomeCriteriaSet = true;
	OutcomeTimer =0.1;
}


// don't write intact warhead if blasted!
void Warhead::WriteLastKmlEntry()
{
	if (NukeState < N_DETONATING)
		VESSEL2M::WriteLastKmlEntry();
}



//=============================================================================
DLLCLBK VESSEL *ovcInit(OBJHANDLE hvessel, int flightmodel)
{
	return new Warhead(hvessel, flightmodel);
}

DLLCLBK void ovcExit(VESSEL *vessel)
{
	if (vessel) delete (Warhead*) vessel;
}