#include	"math.h"
#include	"orbitersdk.h"
#include	"stdio.h"
#include	"BlockBD.h"




BlockBD::BlockBD(OBJHANDLE hObj, int fmodel): VESSEL2M(hObj, fmodel)
{
	// VESSEL2M crash parameters
	VvCrash = DRand(1, 5); 
	VhCrash = DRand(10, 20);
	VMaxBlast = DRand(1000, 1); 
	QMax = DRand(30000, 10000);		// nominal at landing ~35 K?

	// VESSEL2M wrecks
	WreckSingleIdx = 0;
	WreckCount = 5;
	WreckIdxs[0] = 0;
	WreckIdxs[1] = 1;
	WreckIdxs[2] = 2;
	WreckIdxs[3] = 1;
	WreckIdxs[4] = 10;

	// running vars
	CERotationPercent = 0;
	SeparationTimer = 0;

	// VESSEL2M override
	bFocusOffByDefault = true;
	bFocus = false;

	// camera
	SetCameraData(0, _V(-1, 0, -3.5), xminus, -PI/2);

	// self-destructor
	bHaveSelfDestructor = true;

	// done?
	bSafed = false;

	// mesh
	BodyMeshName[0] = 0;

	// failures
	Reliability = 95;
	bFallout = false;
	FalloutTimer = -1;
	SavedThrust = 0;

	// fire location and direction of the flame,
	// unique for BlockBD
	Fire.Pos = _V(1, 1, -3);
	Fire.Dir = zplus;

	// kml
	kmlPlotColor = KML_CLR_BLUE;
}

void BlockBD::clbkSetClassCaps(FILEHANDLE cfg)
{
	// physics and geometry
	SetSize(19.0);
	SetEmptyMass(3915);
	SetCrossSections(_V(36.0, 36.0, 6.0));				// S=pi*D^2/4. m^2
	SetPMI(_V(24.5, 24.5, 0.7));
	SetCW (0.5, 1.0, 2.0, 2.0);							
	SetRotDrag (_V(6.7, 6.7, 0.1));				
	SetLiftCoeffFunc(0);
	SetPitchMomentScale(1e-4);
	SetBankMomentScale(1e-4);
	SetSurfaceFrictionCoeff(1e5, 1e5);
	SetTouchdownPoints (_V( 0, 0, 10), _V( 2, 1, -10), _V(-2, 2, -10));

	// fuel tank
	MainTank = CreatePropellantResource(38300);

	// textures and particles
	SURFHANDLE ExhTex = oapiRegisterExhaustTexture("Exhaust2");
	SURFHANDLE ptex = oapiRegisterParticleTexture("Contrail3");

	PARTICLESTREAMSPEC MainExhaustStream = { 0,			// flags, reserved
		1.5,			// srcsize, particle size at creation, m
		1000,			// srcrate, average particle generation rate, Hz   
		100,			// V0, average particle emission velocity, m/s
		0.01,			// srcspread, emission velocity distribution randomisation
		0.15,			// lifetime, average particle lifetime [s]
		5,				// growthrate, particle growth rate [m/s]
		0.1,			// atmslowdown, deceleration rate b in atmosphere, defined as v = v0 e- bdt 
		PARTICLESTREAMSPEC::EMISSIVE, PARTICLESTREAMSPEC::LVL_PSQRT, -1, 0.5, PARTICLESTREAMSPEC::ATM_PLIN, -1, 0.5, 
		ptex };			// texture

	PARTICLESTREAMSPEC ControlExhaustStream = { 0,			// flags, reserved
		0.05,			// srcsize, particle size at creation, m
		2000,			// srcrate, average particle generation rate, Hz   
		30,			// V0, average particle emission velocity, m/s
		0.01,			// srcspread, emission velocity distribution randomisation
		0.1,			// lifetime, average particle lifetime [s]
		5,				// growthrate, particle growth rate [m/s]
		0,			// atmslowdown, deceleration rate b in atmosphere, defined as v = v0 e- bdt 
		PARTICLESTREAMSPEC::EMISSIVE, PARTICLESTREAMSPEC::LVL_PSQRT, -1, 0.5, PARTICLESTREAMSPEC::ATM_PLIN, -1, 0.5, 
		ptex };			// texture

	PARTICLESTREAMSPEC VentStream = { 0,			// flags, reserved
		1.5,			// srcsize, particle size at creation, m
		10,			// srcrate, average particle generation rate, Hz   
		1,			// V0, average particle emission velocity, m/s
		0.01,			// srcspread, emission velocity distribution randomisation
		1.1,			// lifetime, average particle lifetime [s]
		5,				// growthrate, particle growth rate [m/s]
		0.1,			// atmslowdown, deceleration rate b in atmosphere, defined as v = v0 e- bdt 
		PARTICLESTREAMSPEC::EMISSIVE, PARTICLESTREAMSPEC::LVL_PSQRT, 0, 0.5, PARTICLESTREAMSPEC::ATM_PLIN, 0, 0.0};	

		// thrusters
	
	// main engines
	THRUSTER_HANDLE	th_main[6];	

	for (int i=0; i<4; i++)
	{
		th_main[i] = CreateThruster(TH_MAIN[i], zplus, 23100*G, MainTank, ISP_MAX, ISP_MIN);
		AddExhaust(th_main[i], EX_LEN, EX_WID, ExhTex);
		AddExhaustStream(th_main[i], TH_MAIN_STREAMS[i], &MainExhaustStream);
	}

	// control engines
	for (int i=0; i<2; i++)
	{
		th_ctrls[i] = CreateThruster(TH_CE[i], zplus, 3500*G, MainTank, ISP_MAX, ISP_MIN);
		th_main[i+4] = th_ctrls[i];
		AddExhaust(th_ctrls[i], EX_CE_LEN, EX_CE_WID, ExhTex);
		AddExhaustStream(th_ctrls[i], &ControlExhaustStream);
	}

	// main group combining all main and control engines
	thg_main = CreateThrusterGroup(th_main, 6, THGROUP_MAIN);

	// vent, does not need to be realistic as long as needed visual effect is achieved
	th_vent = CreateThruster(_V(-1, 0, 14.328), zplus, 5000*G, MainTank, 264.9);
	AddExhaustStream(th_vent, &VentStream);
}




void BlockBD::clbkSaveState(FILEHANDLE scn)
{
	// inherited
	VESSEL2M::clbkSaveState(scn);
	oapiWriteScenario_string (scn, "=========== BLOCK BD vars", "");

	// custom mesh
	if (BodyMeshName[0] != 0)
		oapiWriteScenario_string (scn, "BODY_MESHNAME", BodyMeshName);

	// failure flag(s)
	if (bFallout) 
		oapiWriteScenario_int (scn, "FLAGS_BI", bFallout);

	// failure timer
	if (FalloutTimer > 0) 
		oapiWriteScenario_float (scn, "FLAGS_BD_1", FalloutTimer);

	if ((Fire.FireLevel > 0) && (bAttached))
		oapiWriteScenario_float (scn, "FLAGS_BD_2", GetThrusterGroupLevel(THGROUP_MAIN));
}


void BlockBD::ParseScenarioLineEx(char *line, void *status)
{
	if (!strnicmp (line, "FLAGS_BI", 8))
		sscanf (line+8, "%d", &bFallout);

	else if (!strnicmp (line, "FLAGS_BD_1", 10))
		sscanf (line+10, "%lf", &FalloutTimer);

	else if (!strnicmp (line, "FLAGS_BD_2", 10))
		sscanf (line+10, "%lf", &SavedThrust);

	else if (!strnicmp (line, "BODY_MESHNAME", 13))
		sscanf (line+13, "%s", &BodyMeshName);

	// inherited processing
	else 
		VESSEL2M::ParseScenarioLineEx(line, status);
}

void BlockBD::FinishConfiguration()
{
	// meshes
	UINT BodyMesh;
	if (BodyMeshName[0] != 0)
		BodyMesh = AddMesh(oapiLoadMeshGlobal(BodyMeshName));
	else
		BodyMesh = AddMesh(oapiLoadMeshGlobal("r7_S\\BlockBD"));	
	SetMeshVisibilityMode(BodyMesh, MESHVIS_ALWAYS);

	UINT Meshes_CE[2];		
	for (int i=0; i<2; i++)
	{
		Meshes_CE[i] = AddMesh(oapiLoadMeshGlobal("r7_S\\ControlEngine"), &TH_CE[i]);
		SetMeshVisibilityMode(Meshes_CE[i], MESHVIS_ALWAYS);
	}

	// animations
	MGROUP_ROTATE *CE_MGrp[2];
	for (int i=0; i<2; i++)
	{
		CE_MGrp[i] = new MGROUP_ROTATE(Meshes_CE[i], NULL, 0, TH_CEM[i], xminus, (float)(90*RAD));
		CERotations[i] = CreateAnimation (CERotationPercent);
		AddAnimationComponent (CERotations[i], -1, 1, CE_MGrp[i]);
	}

	// inherited
	VESSEL2M::FinishConfiguration();
}


void BlockBD::clbkPostCreation()
{
	//VESSEL2M stuff
	VESSEL2M::clbkPostCreation();

	if (SavedThrust > 0)
		SetThrusterGroupLevel(THGROUP_MAIN, SavedThrust);
}

int BlockBD::clbkConsumeBufferedKey(DWORD key, bool down, char *kstate)
{
	// Control combinations
	if (KEYMOD_CONTROL (kstate))
	{
		// launch escape
		if (key == OAPI_KEY_A) 
			SendVesselMessage(GetAttachmentStatus(GetAttachmentHandle(TOPARENT, 0)), VM_LAUNCH_ESCAPE);
	}

	return VESSEL2M::clbkConsumeBufferedKey(key, down, kstate);
}


int BlockBD::clbkConsumeDirectKey(char *keystate)
{
	// disable manual attempts to interfere with the rocket
	if (
			KEYDOWN (keystate, OAPI_KEY_SUBTRACT) +
			KEYDOWN (keystate, OAPI_KEY_MULTIPLY) +
			KEYDOWN (keystate, OAPI_KEY_ADD)
		)
	{
		ShowAnnotationMessage("Blocks B-D cannot be controlled manually!");
		return 1;
	}

	return 0;
}



void BlockBD::clbkPreStep(double simt, double SimDT, double mjd)
{
	if (SeparationTimer > 0)
	{
		SeparationTimer -= (ForceImpuls * SimDT);
		AddForce(_V( ForceImpuls ,0,0), zplus);
	}
}


void BlockBD::clbkPostStep(double simt, double SimDT, double mjd)
{
	// common processing for all VESSEL2M-derived objects
	TimestepForVESSEL2M();

	//
	if (bSafed)
		return;

	// attached?
	if (bAttached)
	{
		SetControlEngineRotations();

		if (GetRemainingFuel() < 0.015)
			SendVesselMessage(GetAttachmentStatus(GetAttachmentHandle(TOPARENT, 0)), VM_SEPARATE_BLOCKS_BD);

		if (bFallout)
			if ((FalloutTimer -= SimDT) < 0 )
			{
				bFallout = false;
				SendVesselMessage(GetAttachmentStatus(GetAttachmentHandle(TOPARENT, 0)), VM_REQUEST_DETACH, (int)GetHandle());
				bDelayedExplode = true;
				DelayedExplodeTimer = DRand(4, 8);
			}
	}

	// separated?
	else
	{
		// still venting?
		if (GetRemainingFuel() > 0) 
			PlayVesselWave3(SoundLibID, SND_GAS, LOOP);

		else
		{
			// stop venting sound
			StopVesselWave3(SoundLibID, SND_GAS);
	
			// no engine fire? then we are done
			if (!(bFail && (Fire.FireLevel > 0.1)))
				bSafed = true;
		}
	}

//	sprintf(oapiDebugString(), "Q: %f      H: %f      V: %f", GetDynPressure(), GetAltitude(), GetAirspeed());
}



double BlockBD::GetRemainingFuel()
{
	double Mass = GetPropellantMass(MainTank);
	double MaxMass = GetPropellantMaxMass(MainTank);

	return Mass/MaxMass;
}




void BlockBD::SetControlEngineRotations()
{
	// move control engines to position (gets updated from block A)
	SetAnimation (CERotations[0], CERotationPercent);
	SetAnimation (CERotations[1], CERotationPercent);

	// rotate thrust vector for control engines to follow the animations
	double Angle = CERotationPercent * 45 * RAD;
	VECTOR3 Dir = _V(0, sin(Angle), cos(Angle));
	SetThrusterDir(th_ctrls[0], Dir);
	SetThrusterDir(th_ctrls[1], Dir);
}


int BlockBD::GetVesselMessage(IntVesselMessage VM, int Value)
{
	if (VM == VM_OPEN_VENT)
	{
		SeparationTimer = ForceImpuls;
		SetThrusterLevel(th_vent, 1);
		return 0;
	}

	if (VM == VM_SET_SEPARATION_TIMER)
	{
		bFallout = true;
		FalloutTimer = DRand(3,4);
		return 0;
	}

	return VESSEL2M::GetVesselMessage(VM, Value);
}


double BlockBD::GetVesselMessage(DblVesselMessage VM, double Value)
{
	if (VM == VM_SET_CTRL_ENGINE)
		CERotationPercent = Value;

	else if (VM == VM_SET_MAIN_ENGINE)
	{
		// ignore external value if we have engine fire
		if (bFail && (Fire.FireLevel > 0.1))
			return 0;

		// set thrust to external value
		SetThrusterGroupLevel(THGROUP_MAIN, Value);
	}

	else if (VM == VM_GET_MAIN_ENGINE_FORCE)
	{
		THRUSTER_HANDLE SingleMainThruster = GetThrusterHandleByIndex(0);
		double SingleMainThrusterLevel = GetThrusterLevel(SingleMainThruster);
		double SingleMainThrusterMaxForce = GetThrusterMax(SingleMainThruster);
		
		return 4 * SingleMainThrusterMaxForce * SingleMainThrusterLevel;
	}

	else if (VM == VM_GET_CTRL_ENGINE_FORCE)
	{
		THRUSTER_HANDLE SingleControlThruster = GetThrusterHandleByIndex(4);
		double SingleControlThrusterLevel = GetThrusterLevel(SingleControlThruster);
		double SingleControlThrusterMaxForce = GetThrusterMax(SingleControlThruster);

		return 2 * SingleControlThrusterMaxForce * SingleControlThrusterLevel;
	}

	return VESSEL2M::GetVesselMessage(VM);
}


		// =========== FAILURES


bool BlockBD::failInitialize()
{
	// no failure at all?
	if (VESSEL2M::failInitialize())
		return true;

	// for now, only engine fire
	// if (failType == F_FIRE)
		failTimer = DRand(0, 12+114);

	return false;
}


bool BlockBD::failUpdate()
{
	// inherited
	if (VESSEL2M::failUpdate())
		return true;

	// developing fire
//	if (failType == F_FIRE)
	{
		failUpdateFire();
	
		if (GetThrusterGroupLevel(THGROUP_MAIN) < 0.2)
			SendVesselMessage(GetAttachmentStatus(GetAttachmentHandle(TOPARENT, 0)), VM_REQUEST_DETACH, (int)GetHandle());
	}

	return false;
}


void BlockBD::failActivate()
{
	// inherited
	VESSEL2M::failActivate();

//	if (failType == F_FIRE)
		failStartFire();
}




//=============================================================================

DLLCLBK VESSEL *ovcInit(OBJHANDLE hvessel, int flightmodel)
{
	return new BlockBD(hvessel, flightmodel);
}

DLLCLBK void ovcExit(VESSEL *vessel)
{
	if (vessel) delete (BlockBD*) vessel;
}
