#include "stdio.h"
#include "kmlWriter.h"



kmlWriter::kmlWriter(OBJHANDLE hObj, int fmodel): VESSEL2M(hObj, fmodel)
{
	// VESSEL2M overrides
	bFocusOffByDefault = true;
	bFocus = false;
	bPermanentlyHidden = true;

	// local inits
	kmlTimer = TIMER_INTERVAL;
	idxService = -1;

	EvLaunch.Event = KML_NONE;
	EvTarget.Event = KML_NONE;
	EvArrive.Event = KML_NONE;

	// create service vessel
	kmlVessel kV;

	kV.Handle = (OBJHANDLE)-1;
	sprintf(kV.FolderName, "Flight");
	kV.bNoPlotting = true;
	kV.kmlPlotColor = KML_CLR_NONE;

	kmlVessels.push_back(kV);
	idxService = kmlVessels.size()-1;
}



void kmlWriter::clbkPostCreation()
{
	//VESSEL2M stuff - no need so far, only soundlib initialization there...
//	VESSEL2M::clbkPostCreation();

	// move to zero location via direct status update
	VESSELSTATUS vs;
	memset(&vs, 0, sizeof(vs)); // Orbiter bug workaround
	GetStatus(vs);

	vs.status = 1;
	vs.vdata[0] = xyz0;

	DefSetState(&vs);
}


void kmlWriter::clbkSaveState(FILEHANDLE scn)
{
	// do a small hack with bAttached to limit the amount of output by the parent function
	bAttached = true;
	VESSEL2M::clbkSaveState(scn);
	bAttached = false;

	if (EvLaunch.Event == KML_LAUNCH)
		oapiWriteScenario_vec (scn, "LAUNCH", _V(EvLaunch.Lat, EvLaunch.Lon, EvLaunch.Alt));

	if (EvTarget.Event == KML_TARGET)
	{
		oapiWriteScenario_vec (scn, "TARGET", _V(EvTarget.Lat, EvTarget.Lon, EvTarget.Alt));
		oapiWriteScenario_string (scn, "TGT_NAME", EvTarget.Label);
	}

	if (EvArrive.Event == KML_ARRIVAL)
		oapiWriteScenario_vec (scn, "EPICENTER", _V(EvArrive.Lat, EvArrive.Lon, EvArrive.Alt));

	// write kml file
	Write();
}


void kmlWriter::ParseScenarioLineEx(char *line, void *status)
{
	if (!strnicmp (line, "LAUNCH", 6))
	{
		sscanf (line+6, "%lf %lf %lf", &EvLaunch.Lat, &EvLaunch.Lon, &EvLaunch.Alt);
		EvLaunch.Event = KML_LAUNCH;
		strcpy(EvLaunch.Label, "Launch!");
		kmlVessels[idxService].Events.push_back(EvLaunch);
	}

	if (!strnicmp (line, "TGT_NAME", 8))
		sscanf (line+8, "%s", &EvTarget.Label);

	if (!strnicmp (line, "TARGET", 6))
	{
		sscanf (line+6, "%lf %lf %lf", &EvTarget.Lat, &EvTarget.Lon, &EvTarget.Alt);
		EvTarget.Event = KML_TARGET;
		kmlVessels[idxService].Events.push_back(EvTarget);
	}

	if (!strnicmp (line, "EPICENTER", 9))
	{
		sscanf (line+9, "%lf %lf %lf", &EvArrive.Lat, &EvArrive.Lon, &EvArrive.Alt);
		EvArrive.Event = KML_ARRIVAL;
		strcpy(EvLaunch.Label, "Epicenter");
		kmlVessels[idxService].Events.push_back(EvArrive);
	}

	// inherited processing
	else 
		VESSEL2M::ParseScenarioLineEx(line, status);
}


void kmlWriter::clbkPostStep(double simt, double SimDT, double mjd)
{
	// common processing for all VESSEL2M-derived objects - do we need it?
	// TimestepForVESSEL2M();

	// periodically plot all registered vessels
	if ((kmlTimer -= SimDT) <= 0)	
		AddPlots();
}


// cycle through all registered vessels and plot them as needed 
void kmlWriter::AddPlots()
{
	// iterate vessels
	for (unsigned i=0; i<kmlVessels.size(); i++)
	{
		// marked for no plottinig?
		if (kmlVessels[i].bNoPlotting)
			continue;

		// still valid for incremental processing?
		if (!oapiIsVessel(kmlVessels[i].Handle))
		{
			kmlVessels[i].bNoPlotting = true;
			continue;
		}

		// did we exceed the plot limit?
		if (kmlVessels[i].Plots.size() >= PLOT_MAX )
		{
			kmlVessels[i].bNoPlotting = true;
			continue;
		}

		// obtain plot
		KML_PLOT Plot;

		//
//		oapiGetGlobalPos(kmlVessels[i].Handle, &Plot.GlobalPos);
//		if (kmlVessels[i].Plots.size() > 0)
//			if (length(Plot.GlobalPos - kmlVessels[i].Plots[kmlVessels[i-1].Plots.size()-1].GlobalPos) < 5000)
//				continue;

		// get vessel coordinates
		double Radius;
		oapiGetEquPos(kmlVessels[i].Handle, &Plot.Lon, &Plot.Lat, &Radius);
		Plot.Lat *= DEG;
		Plot.Lon *= DEG;

		// get vessel altitude
		oapiGetAltitude(kmlVessels[i].Handle, &Plot.Alt);

		// get time
		Plot.Time = oapiGetSimTime();

		// stop plotting wreck fell to the ground
		if (kmlVessels[i].bWreck)
			if (Plot.Alt <= 5)
			{
				kmlVessels[i].bNoPlotting = true;

				KML_EVENT Event;
				Event.hOrigin = kmlVessels[i].Handle;
				sprintf(Event.Label, "");
				Event.Event = KML_WRECK;
				Event.Lat = Plot.Lat; 
				Event.Lon = Plot.Lon;
				Event.Alt = 0; 
				Event.Time = Plot.Time;

				kmlVessels[i].Events.push_back(Event);
				
				continue;
			}

		// get color
		Plot.Color = kmlVessels[i].kmlPlotColor;

		// save plot 
		kmlVessels[i].Plots.push_back(Plot);
	}

	// reset timer
	kmlTimer = TIMER_INTERVAL;
}


int kmlWriter::GetVesselMessage(IntVesselMessage VM, int Value)
{
	if (VM == VM_KML_REGISTER)
		return RegisterVessel((OBJHANDLE)Value);

	else if (VM == VM_KML_ADD_EVENT)
		return AddEvent((KML_EVENT*)Value);

	else if (VM == VM_KML_START_PLOTTING)
		return StartPlotting((OBJHANDLE)Value);

	else if (VM == VM_KML_STOP_PLOTTING)
		return StopPlotting((OBJHANDLE)Value);
		
	else if (VM == VM_KML_SET_COLOR)
		return SetColor((KML_COLOR_STRUCT*)Value);

	return VESSEL2M::GetVesselMessage(VM, Value);
}


// register new vessel object
int kmlWriter::RegisterVessel(OBJHANDLE Handle)
{
	kmlVessel kV;

	kV.Handle = Handle;

	VESSEL *Intf;
	VESSEL2M *Intf2M;
	R7_wreck *IntfW;

	Intf = oapiGetVesselInterface(Handle);

	if (IsVessel2M(Intf))
	{
		Intf2M = (VESSEL2M*)Intf;
		kV.bNoPlotting = Intf2M->kmlNoPlotting;
		kV.kmlPlotColor = Intf2M->kmlPlotColor;
		oapiGetObjectName(Handle, kV.FolderName, 20);
		kV.bWreck = false;
		kmlVessels.push_back(kV);
	}

	else if (IsWreck(Intf))
	{
		IntfW = (R7_wreck*)Intf;
		kV.bNoPlotting = false;
		kV.kmlPlotColor = KML_CLR_WHITE;
		sprintf(kV.FolderName, "wreck");
		kV.bWreck = true;
		kmlVessels.push_back(kV);
	}

	return (int)GetHandle();
}


int kmlWriter::AddEvent(KML_EVENT* pEvent)
{
	// set event variable
	KML_EVENT Event = *pEvent;

	// find vessel entry
	bool bFound = false;
	for (unsigned i=0; i<kmlVessels.size(); i++)
		if (kmlVessels[i].Handle == Event.hOrigin)
		{
			kmlVessels[i].Events.push_back(Event);
			bFound = true;
		}

	// if not found, add ot special 'service' folder
	if (!bFound)
		kmlVessels[idxService].Events.push_back(Event);

	// store "global" events for cross-event processing downstream
	if (Event.Event == KML_LAUNCH)
		EvLaunch = Event;
	else if (Event.Event == KML_TARGET)
		EvTarget = Event;
	else if (Event.Event == KML_ARRIVAL)
		EvArrive = Event;

	return 0;
}


int kmlWriter::SetColor(KML_COLOR_STRUCT* pColorStruct)
{
	for (unsigned i=0; i<kmlVessels.size(); i++)
		if (kmlVessels[i].Handle == pColorStruct->hOriginator)
			kmlVessels[i].kmlPlotColor = pColorStruct->kmlPlotColor;

	return 0;
}


int kmlWriter::StartPlotting(OBJHANDLE hOrigin)
{	
	for (unsigned i=0; i<kmlVessels.size(); i++)
		if (kmlVessels[i].Handle == hOrigin)
			kmlVessels[i].bNoPlotting = false;

	return 0;
}


int kmlWriter::StopPlotting(OBJHANDLE hOrigin)
{	
	for (unsigned i=0; i<kmlVessels.size(); i++)
		if (kmlVessels[i].Handle == hOrigin)
			kmlVessels[i].bNoPlotting = true;

	return 0;
}


// save kml file to disk, same location and name every time
void kmlWriter::Write()
{
	// force kml dirs creation
	CreateDirectory("GoogleEarth", NULL);
	CreateDirectory("GoogleEarth\\R-7", NULL);

	// create kml file
	FILE *kmlFile;
	kmlFile = fopen("GoogleEarth\\R-7\\(Current state).kml", "w");

    // write file header
	fprintf(kmlFile, FileStart, "(Current state).kml");

    // write kml styles
	fprintf(kmlFile, StyleLaunch);
	fprintf(kmlFile, StyleTarget);
	fprintf(kmlFile, StyleWreck);
	fprintf(kmlFile, StyleSeparation);
	fprintf(kmlFile, StyleFlight);
	fprintf(kmlFile, StyleNuke);

    // root folder start, no explicit end (will be part of file footer)
	fprintf(kmlFile, FolderStart, "(Current state).kml");

	// iterate kml vessels
	for (unsigned i=0; i<kmlVessels.size(); i++)
	{
		// get current kml vessel
		kmlVessel kV = kmlVessels[i];

		// start folder
		fprintf(kmlFile, FolderStart, kV.FolderName);

		// write events
		if (kmlVessels[i].Events.size() > 0)
			for (unsigned j=0; j<(kmlVessels[i].Events.size()); j++)
			{
				KML_EVENT Ev = kmlVessels[i].Events[j];

				// 'L' icon
				if (Ev.Event == KML_LAUNCH)
					fprintf(kmlFile, WptGround, Ev.Label, "", Ev.Lon, Ev.Lat, "stLaunch", Ev.Lon, Ev.Lat);

				// 'Flame' icon
				else if (Ev.Event == KML_WRECK)
				{
					if (Ev.Alt < 20)
						fprintf(kmlFile, WptGround, Ev.Label, "", Ev.Lon, Ev.Lat, "stWreck", Ev.Lon, Ev.Lat);
					else
						fprintf(kmlFile, WptAirborne, Ev.Label, "", Ev.Lon, Ev.Lat, "stWreck", Ev.Lon, Ev.Lat, Ev.Alt);
				}

				// 'red square' icon
				else if (Ev.Event == KML_SEPARATION)
				{
					// debug only: dump more data about event
					//char TargetLabel[255];
					//double Range, Azimuth;
					//DistanceAzimuthFromCoordPairKmDg(EvLaunch.Lat*RAD, EvLaunch.Lon*RAD, Ev.Lat*RAD, Ev.Lon*RAD, &Range, &Azimuth);
					//sprintf(TargetLabel, "%s. R %f km. R %f rad. R %f deg.",  Ev.Label, Range, Range/6371, DEG*Range/6371);
					//fprintf(kmlFile, WptAirborne, TargetLabel, Ev.Lon, Ev.Lat, "stSeparation", Ev.Lon, Ev.Lat, Ev.Alt);

					fprintf(kmlFile, WptAirborne, Ev.Label, "", Ev.Lon, Ev.Lat, "stSeparation", Ev.Lon, Ev.Lat, Ev.Alt);
				}

				// 'round' icon
				else if (Ev.Event == KML_FLIGHT)
					fprintf(kmlFile, WptAirborne, Ev.Label, "", Ev.Lon, Ev.Lat, "stFlight", Ev.Lon, Ev.Lat, Ev.Alt);

				// many objects here...
				else if (Ev.Event == KML_NUKE)
					WriteNukeObjects(kmlFile, Ev);

				// write the whole trajectory with the data from various sources events
				else if (Ev.Event == KML_ARRIVAL)
					WriteTrajectory(kmlFile, EvLaunch, EvTarget, EvArrive);

				// 'bull' eye' icon
				else if (Ev.Event == KML_TARGET)
				{
					// add range and azimuth to the target label
					//char TargetLabel[255];
					//double Range, Azimuth;
					//DistanceAzimuthFromCoordPairKmDg(EvLaunch.Lat*RAD, EvLaunch.Lon*RAD, Ev.Lat*RAD, Ev.Lon*RAD, &Range, &Azimuth);
					//sprintf(TargetLabel, "%s. R %.0f km. Az %.0f deg.",  Ev.Label, Range, Azimuth);
					//fprintf(kmlFile, WptGround, TargetLabel, Ev.Lon, Ev.Lat, "stTarget", Ev.Lon, Ev.Lat);

					// write target waypoint, target circle and arc from launch to target
					fprintf(kmlFile, WptGround, Ev.Label, "", Ev.Lon, Ev.Lat, "stTarget", Ev.Lon, Ev.Lat);
					WriteCircle(kmlFile, "Target area", Ev.Lon, Ev.Lat, 2.5, KML_CLR_WHITE);

					// draw white ground arc from launchpad to target
					if ((EvLaunch.Lon + EvLaunch.Lat) != 0)
						WriteGroundArc(kmlFile, "Launch to Target - Big Arc", EvLaunch.Lon, EvLaunch.Lat, Ev.Lon, Ev.Lat, KML_CLR_WHITE);
				}

			}

		// write object plots
		if (kmlVessels[i].Plots.size() > 0)
			for (unsigned j=0; j<(kmlVessels[i].Plots.size()-1); j++)
			{
				KML_PLOT Plt = kmlVessels[i].Plots[j];
				KML_PLOT PltNext = kmlVessels[i].Plots[j+1];

				// get colors and width
				int width;
				if ( (Plt.Color == KML_CLR_RED) || (Plt.Color == KML_CLR_WHITE) || (Plt.Color == KML_CLR_YELLOW) || (Plt.Color == KML_CLR_GREEN) )
					width = 4;
				else 
					width = 1;

				if (kmlVessels[i].bWreck)
					width = 1;

				// track line header, name, etc.
				fprintf(kmlFile, TrkPlacemarkStart, strColors[Plt.Color*2-1], j, strColors[Plt.Color*2], width);

				// track line coords
				char strCoords[256];
				sprintf(strCoords, TrkPlacemark, Plt.Lon, Plt.Lat, Plt.Alt, PltNext.Lon, PltNext.Lat, PltNext.Alt);
				fprintf(kmlFile, strCoords);

				// end track line
				fprintf(kmlFile, TrkPlacemarkEnd);
			}


		// end object folder
		fprintf(kmlFile, FolderEnd);
	}

/*	simtime = oapiGetSimTime();
	dy = (int)(simtime / 86400);
	hr = (int)((simtime - dy * 86400) / 3600);
	min = (int)((simtime - dy * 86400 - hr * 3600) / 60);
	sec = (int)(simtime - dy * 86400 - hr * 3600 - min * 60);
*/

	// folder end
	fprintf(kmlFile, FolderEnd);

	// file end
	fprintf(kmlFile, FileEnd);
	
	// close file
	fclose(kmlFile);
}



void kmlWriter::WriteNukeObjects(FILE *File, KML_EVENT Ev)
{
	// 'sun' icon
	fprintf(File, WptAirborne, Ev.Label, "", Ev.Lon, Ev.Lat, "stNuke", Ev.Lon, Ev.Lat, Ev.Alt);

	// destruction circles
	WriteCircle(File, "Total destruction", Ev.Lon, Ev.Lat, 5, KML_CLR_RED);
	WriteCircle(File, "Heavy destruction", Ev.Lon, Ev.Lat, 10, KML_CLR_YELLOW);
	WriteCircle(File, "Moderate destruction", Ev.Lon, Ev.Lat, 20, KML_CLR_GREEN);
}


void kmlWriter::WriteTrajectory(FILE *File, KML_EVENT EvLaunch, KML_EVENT EvTarget, KML_EVENT EvArrive)
{
	if ((EvTarget.Lon + EvTarget.Lat) == 0)
		strDescrTags[0] = 0;
	else
	{
		double Range, Azimuth;

		DistanceAzimuthFromCoordPairKmDg(EvLaunch.Lat*RAD, EvLaunch.Lon*RAD, EvTarget.Lat*RAD, EvTarget.Lon*RAD, &Range, &Azimuth);
		double TargetRange = Range/1000;
		double TargetAzimuth = Azimuth;

		DistanceAzimuthFromCoordPairKmDg(EvLaunch.Lat*RAD, EvLaunch.Lon*RAD, EvArrive.Lat*RAD, EvArrive.Lon*RAD, &Range, &Azimuth);
		double WarheadRange = Range/1000;
		double WarheadAzimuth = Azimuth;

		DistanceAzimuthFromCoordPairKmDg(EvTarget.Lat*RAD, EvTarget.Lon*RAD, EvArrive.Lat*RAD, EvArrive.Lon*RAD, &Range, &Azimuth);
		double Miss = Range/1000;

		sprintf(strDescrTags, TrajectoryDescription, TargetRange, TargetAzimuth, WarheadRange, WarheadAzimuth, Miss);
	}

	// 'bull's eye' icon, arrival waypoint
	fprintf(File, WptGround, EvArrive.Label, strDescrTags, EvArrive.Lon, EvArrive.Lat, "stTarget", EvArrive.Lon, EvArrive.Lat);

	// draw arc from target to actual arrival point, 
	if ((EvTarget.Lon + EvTarget.Lat) != 0)
		WriteGroundArc(File, "Launch to Impact - Big Arc", EvTarget.Lon, EvTarget.Lat, EvArrive.Lon, EvArrive.Lat, KML_CLR_WHITE);
}




void kmlWriter::WriteGroundArc(FILE *File, char* strLabel, double LonFromDeg, double LatFromDeg, double LonToDeg, double LatToDeg, kmlPlotColors Color)
{
	// arc start
	fprintf(File, GroundLineStart, strLabel, strColors[Color*2]);

	// two dots
	fprintf(File, "%.13f, %.13f, 0\n", LonFromDeg, LatFromDeg);
	fprintf(File, "%.13f, %.13f, 0\n", LonToDeg, LatToDeg);

	// arc end
	fprintf(File, GroundLineEnd);
}


void kmlWriter::WriteCircle(FILE *File, char* strLabel, double LonDeg, double LatDeg, double RadiusKm, kmlPlotColors Color)
{
	double PointsCount = 100;
	double dAngleDeg = 360.0 / PointsCount;
	double dAngleRad = dAngleDeg * RAD;
	double RadiusRad = RadiusKm/6371;
	double LatRad = LatDeg * RAD;
	double LonRad = LonDeg * RAD;
	double LatRadNew = 0;
	double LonRadNew = 0;
	double LatDegNew;
	double LonDegNew;

	// line start
	fprintf(File, GroundLineStart, strLabel, strColors[Color*2]);

	// first iteration of angle
	double angleRad = 0;

	for (int i = 0; i < PointsCount; i++)
	{
		// get current point
		CoordsFromDistanceAzimuthRad(LatRad, LonRad, RadiusRad, angleRad, &LatRadNew, &LonRadNew);

		// convert to degrees
		LatDegNew = LatRadNew * DEG;
        LonDegNew = LonRadNew * DEG;

		// write current dot
		fprintf(File, "%.13f, %.13f, 0\n", LonDegNew, LatDegNew);

		// update angle for the next iteration
        angleRad += dAngleRad;
	}

	// line end
	fprintf(File, GroundLineEnd);
}



//=============================================================================
DLLCLBK VESSEL *ovcInit(OBJHANDLE hvessel, int flightmodel)
{
	return new kmlWriter(hvessel, flightmodel);
}

DLLCLBK void ovcExit(VESSEL *vessel)
{
	if (vessel) delete (kmlWriter*) vessel;
}
