#include	"math.h"
#include	"orbitersdk.h"
#include	"stdio.h"
#include	"Sputnik2.h"



Sputnik2::Sputnik2(OBJHANDLE hObj, int fmodel): VESSEL2M(hObj, fmodel)
{
	// VESSEL2M crash parameters
	VvCrash = DRand(5, 10);					// 5-15 m/s
	VhCrash = DRand(20, 20);				// 20-40 m/s
	VMaxBlast = DRand(1000, 1);				// 1 km/s
	QMax = DRand(50000, 25000);				// 50k-75k

	// VESSEL2M wrecks
	WreckCount = 3;
	WreckIdxs[0] = 2;
	WreckIdxs[1] = 3;
	WreckIdxs[2] = 5;

	// radio
	RadioTimer = -1;
	DogTimer = -1;
	bPlayingSound = false;

	// camera
	double Y = 0.375212;
	double X = -sqrt(1-Y*Y);
	SetCameraData(0, _V(0.1565, 0.1811, 0.4567), _V(X,Y,0), -PI/2);

	// scenario only, make us active even when attached
	bActive = false;

	// mission outcome
	strcpy(Outcomes[OC_SUCCESS].OutcomeMessage, "Congratulations! You've just launched the very first living creature into Earth orbit!");
	strcpy(Outcomes[OC_FAIL].OutcomeMessage, "Mission not accomplished. The satellite has failed to reach the orbit...");
}


void Sputnik2::clbkSetClassCaps(FILEHANDLE cfg)
{
	// physics and geometry
	SetSize(3.5);										// D=2m, H=4m, cone
	SetEmptyMass(505.3);								// 508.3 kg
	SetCrossSections(_V(4, 4, 3.14));					// cone
	SetPMI(_V(1.18, 1.18, 0.53));						// from shipedit
	SetCW (0.1, 0.2, 0.4, 0.4);							// 
	SetRotDrag (_V(1, 1, 0.1));							// symmetrical. dOmega = -Omega * q * S i * RD i
	SetSurfaceFrictionCoeff(1e-4, 1e-4);
	SetTouchdownPoints (_V( 0, 0, -0.8), _V( 0.8, 0.4, 0.8), _V(-0.8, 0.8, 0.8));

	// meshes
	UINT MeshInside = AddMesh(oapiLoadMeshGlobal("r7_S\\Sputnik2Inside"));
	SetMeshVisibilityMode (MeshInside, MESHVIS_ALWAYS);
	UINT MeshOutside = AddMesh(oapiLoadMeshGlobal("r7_S\\Sputnik2"));
	SetMeshVisibilityMode (MeshOutside, MESHVIS_ALWAYS);
}


void Sputnik2::clbkSaveState (FILEHANDLE scn)
{
	// inherited
	VESSEL2M::clbkSaveState(scn);

	// separator 
	oapiWriteScenario_string (scn, "=========== SPUTNIK-2 vars", "");

	if (!bAttached)
		oapiWriteScenario_string (scn, "ACTIVE", "");

	if (bPlayingSound)
		oapiWriteScenario_string (scn, "PLAY_SOUND", "");

	if (RadioTimer > 0)
		oapiWriteScenario_float (scn, "RADIO_TIMER", RadioTimer);
}

void Sputnik2::ParseScenarioLineEx(char *line, void *status)
{
	if (!strnicmp (line, "ACTIVE", 6))
		bActive = true;

	else if (!strnicmp (line, "PLAY_SOUND", 10))
		bPlayingSound = true;

	else if (!strnicmp (line, "RADIO_TIMER", 11))
		sscanf (line+11, "%lf", &RadioTimer);

	else 
		VESSEL2M::ParseScenarioLineEx(line, status);
}


void Sputnik2::FinishConfiguration()
{
	if (bPlayingSound)
		StartSound();

	// in case we broke loose; block A should override it anyway, but we have a fix downstream
	bAttached = !bActive;

	//VESSEL2M stuff
	VESSEL2M::FinishConfiguration();
}


void Sputnik2::clbkPostStep(double simt, double SimDT, double mjd)
{
	// common processing for all VESSEL2M-derived objects
	TimestepForVESSEL2M();

	// play dog bark at random intervals
	if (bPlayingSound)
	{
		// maintain radio sound in the loop
		SendVesselWaveToFocused(SND_RADIO, LOOP);

		// update dog sound timer
		if (DogTimer > 0)
			DogTimer -= SimDT;

		// time for dog? (re)start dog sound and reset dog sound timer
		if (DogTimer <= 0)
		{
			SendVesselWaveToFocused(SND_DOG);
			DogTimer = DRand(15, 15);
		}
	}

	// detect if we get separated from the rocket - this may happen only by accident!
	if (bAttached)
	{
		if (GetAttachmentStatus(GetAttachmentHandle(TOPARENT, 0)) == NULL)
		{
			// start radio timer
			RadioTimer = 10;

			bAttached = false;

			// update mission criteria
			UpdateOrbitalMissionCriteria();
		}

		return;
	}

	// update radio timer
	if (RadioTimer > 0)
	{
		RadioTimer -= SimDT;
		if (RadioTimer <= 0)
			StartSound();
	}
}


int Sputnik2::clbkConsumeBufferedKey(DWORD key, bool down, char *kstate)
{
	// VESSEL2M stuff
	int result = VESSEL2M::clbkConsumeBufferedKey(key, down, kstate);
	if (result != 0)
		return result;

	// only process keydown events
	if (!down) 
		return 0; 

	// shift combinations are reserved
	if (KEYMOD_SHIFT (kstate)) 
		return 0; 

	// control combinations are not used
	if (KEYMOD_CONTROL (kstate)) 
		return 0; 

	// play radio and dog sound
	if (key == OAPI_KEY_B) 
	{
		if (bPlayingSound)
			StopSound();
		else
			StartSound();
	}
	
	return 0;
}


void Sputnik2::StartSound()
{
	bPlayingSound = true;
	SendVesselWaveToFocused(SND_RADIO, LOOP);
	DogTimer = DRand(3, 7);
}


void Sputnik2::StopSound()
{
	bPlayingSound = false;
	SendVesselWaveToFocused(SND_RADIO, SND_STOP);
	SendVesselWaveToFocused(SND_DOG, SND_STOP);
	DogTimer = -1;
}



int Sputnik2::GetVesselMessage(IntVesselMessage VM, int Value)
{
	// respond to request from the rocket to specify payload type
	if (VM == VM_PAYLOAD_TYPE)
		return PAYLOAD_SAT2;

	// not really a detach, a message to become active follwing rocket shutdown
	else if (VM == VM_DETACHED)
	{
		// start radio timer
		RadioTimer = 10;

		// update mission criteria
		UpdateOrbitalMissionCriteria();

		return 0;
	}

	// restore flag when when loaded from saved scenario and overridden by block A
	else if (VM == VM_UPDATE_ACTIVE)
	{
		bAttached = !bActive;

		return 0;
	}

	// unsupported message
	else 
		return VESSEL2M::GetVesselMessage(VM, Value);
}





// ========================================================================

DLLCLBK VESSEL *ovcInit(OBJHANDLE hvessel, int flightmodel)
{
	return new Sputnik2(hvessel, flightmodel);
}

DLLCLBK void ovcExit(VESSEL *vessel)
{
	if (vessel) delete (Sputnik2*) vessel;
}



