/*
	Intermediate vessel class for r7 vessel objects inheritance.
	Provides a lot of common additional functionality.
*/

#pragma once

// Orbiter infrastructure includes
#include "..\MyOrbiterLib_S\MyOrbiterLib.h"
#include "..\r7_wreck_S\wreck.h"
#include "OrbiterSoundSDK35.h"


	// kml generation (Google Earth)

// kml color indexes
enum kmlPlotColors {	KML_CLR_NONE,
						KML_CLR_RED,		// plot engine burns
						KML_CLR_WHITE,		// plot contrails; draw ground arcs
						KML_CLR_BLUE,		// plot passive trajectory of rocket parts
						KML_CLR_YELLOW,		// plot warheads
						KML_CLR_GREEN,		// plot orbital payloads
						KML_CLR_VIOLET		// plot fairing
					};

struct KML_COLOR_STRUCT{
	kmlPlotColors kmlPlotColor;
	OBJHANDLE hOriginator;
};


// kml event types
enum kmlEvents {	
	KML_NONE,
	KML_LAUNCH,		// ground location of the launch pad
	KML_TARGET,		// ground location of a target; belongs to launchpad, but specifies a remote location
	KML_ARRIVAL,	// belongs to launchpad, but links all three entries: launch point, target and actual arrival
	KML_NUKE,		// airborne location; belongs to warhead, used for special effects

	KML_FLIGHT,		// airborne location of some special in-flight; belongs to launchpad as it is a part of the trajectory
	KML_SEPARATION,	// airborne location of the separation point; belongs to launchpad as it is a part of the trajectory
	KML_WRECK		// airborne location of a wreck; belongs to vessel
};

struct KML_EVENT{
	OBJHANDLE hOrigin;
	char Label[50];
	kmlEvents Event;
	double Lat; // dg
	double Lon; // dg
	double Alt; // m
	double Time;
};


	// Attachments consolidation

// structure that holds together the three vectors responsible for the attachment geometry on the attached object
typedef struct {
	struct { VECTOR3 AttPoint, AttDir, AttRot ;};
} ATTACHMENT_PARAMS;

// one-line initializer of the above structure
inline ATTACHMENT_PARAMS _AP( VECTOR3 Point, VECTOR3 Dir, VECTOR3 Rot)
{
	ATTACHMENT_PARAMS ap = {Point.x, Point.y, Point.z, Dir.x, Dir.y, Dir.z, Rot.x, Rot.y, Rot.z}; 
	return ap;
}

// structure that holds the scenario variable names to be used to save attached object's attachment parameters in scenario
typedef struct{
	char ObjectName[50];		// variabie to use for the name of the attached object
	char PointName[50];			// variabie to use for the vector of the attachment point coordinates on the attached object
	char DirName[50];			// variabie to use for the vector of the attachment direction of the attached object
	char RotName[50];			// variabie to use for the vector of the attachment rotation of the attached object
}ATTACHMENT_SCN_VAR_NAMES;

// one-line initializer of the above structure
inline ATTACHMENT_SCN_VAR_NAMES _ASVN( char *ObjectName, char *PointName, char *DirName, char *RotName)
{
	ATTACHMENT_SCN_VAR_NAMES asvn;
	strcpy(asvn.ObjectName, ObjectName);
	strcpy(asvn.PointName, PointName);
	strcpy(asvn.DirName, DirName);
	strcpy(asvn.RotName, RotName); 
	return asvn;
}

// structure that holds all the needed data about the attached object 
typedef struct {
	char ObjectName[255];						// name of the attached object
	ATTACHMENTHANDLE hAttachment;				// attachment handle of the attached object on us
	OBJHANDLE hObject;							// object handle of the attached object 
	VESSEL2M* ObjectIntf2M;						// interface of the attached object 
	ATTACHMENT_PARAMS AttParamsOnUs;			// structure with the attachment geometry on us
	ATTACHMENT_PARAMS AttParamsOnObject;		// structure with the actual attachment geometry on the attached object 
	ATTACHMENT_PARAMS AttParamsOnObjectDefault;	// structure with the default attachment geometry on the attached object 
	ATTACHMENT_SCN_VAR_NAMES ScnVarNames;		// structure that holds the scenario variable names to be used to save attached object's attachment parameters in scenario
} ATTACHED_OBJECT;


// failures

struct FIRE{
	double tFireUp;
	double tFireSteady;
	double tThrustDown;
	VECTOR3 Pos;
	VECTOR3 Dir;
	double FireLevel;
};


	// VESSEL2M consts


// crash variations consts
const bool EXPLODE = true;
const bool NO_EXPLODE = false;

// on-screen message lifetime, sec
const double MSG_DISPLAY_TIME = 4;

// sound
const int SND_PYRO = 1;
const int SND_GAS = 2;

const int SND_RING_MOVE		= 3;
const int SND_BOOMS_MOVE	= 4;
const int SND_SUPPORTS_MOVE	= 5;
const int SND_PLATFORM_MOVE	= 6;		
const int SND_CABLE_MAST_MOVE = 7;
const int SND_CLANG_LIGHT	= 8;
const int SND_CLANG_HEAVY	= 9;

const int SND_BIP = 10;
const int SND_RADIO = 11;
const int SND_DOG = 12;

const int SND_CLANG = 13;

const int SND_BLAST1 = 14;  // nukes
const int SND_BLAST2 = 15;

const int SND_ENGINE = 16;

const int SND_ALARM = 17;

// defined in Wreck.h, don't override!
//const int SND_BLAST_LONG = 18;		// non-nukes
//const int SND_BLAST_MEDIUM = 19;
const int SND_FUELING_MAST_MOVE = 20;

// defined in Launchpad.h, don't override!
//const int SND_BUTTON = 19;

const int SND_STOP = 3;		// not a sound - a constant in addition to OrbiterSound's NOLOOP and LOOP



	


	// VESSEL2M class declaration

class VESSEL2M: public VESSEL2 {
public:
	// === this initial block of vars is shared with Wheck, so it should always stay in sync!!!
	char VesselSignature;			//+--
	bool bFocus;					//+++ 
	bool bFocusOffByDefault;		//+-- 
	bool bPermanentlyHidden;		//+--
	// === end of synced block of vars


	// constructor
	VESSEL2M (OBJHANDLE hObj, int fmodel);
	~VESSEL2M ();

	// structural speed and Q limits
	double VvCrash;											//+--
	double VhCrash;											//+--
	double VMaxBlast;										//+--
	double QMax;											//+--
	virtual void Burn();  // is overridden in Warhead to restore mesh alpha; can be commonolized
	virtual void Break(); // self-destruct or external destruct

	// wrecks
	int WreckSingleIdx;	//---	
	int WreckIdxs[10];	//---	reserve space for up tp to 10 separate wrecks per vessel, but usually should not need that many.
	int WreckCount;		//---	actual count of wrecks to be created

	// configuration
	bool bAttached;											//+-+

	// for complex objects, physics, scenario...
	double EmptyMassWithoutAttachments;						//+--
	void UpdateEmptyMass();
	void WritePropellantsToScenario(FILEHANDLE scn);
	void WriteStatusToScenario(FILEHANDLE scn);

	// VESSEL2 overrides
	virtual void clbkSaveState (FILEHANDLE scn);
	void clbkLoadStateEx (FILEHANDLE scn, void *status);
	virtual void ParseScenarioLineEx(char *line, void *status);
	virtual void FinishConfiguration();
	void clbkPostCreation (void);
	virtual int clbkConsumeBufferedKey(DWORD key, bool down, char *kstate);
	bool clbkLoadGenericCockpit();

	// attachments
	void AttachObject(ATTACHED_OBJECT* AttObject);
	void WriteAttachedObjectToScenario(FILEHANDLE scn, ATTACHED_OBJECT* AttObject);
	bool ReadAttachedObjectLineFromScenario(char* line, ATTACHED_OBJECT* AttObject);

	// time machine!
	void TimestepForVESSEL2M();

	// screen output
	void ShowAnnotationMessage (char* Message);

	// sending messages between VESSEL2M objects

	enum VM_PayloadTypes {
		PAYLOAD_UNKNOWN, 
		PAYLOAD_WARHEAD_TEST, 
		PAYLOAD_WARHEAD_5T, 
		PAYLOAD_WARHEAD_3T, 
		PAYLOAD_SAT1, 
		PAYLOAD_SAT2, 
		PAYLOAD_SAT3, 
		PAYLOAD_POLYOT,
		PAYLOAD_3ST
	}; 

	enum IntVesselMessage {
		VM_REQUEST_DETACH,	// ourselves, as a child, from the parent
		VM_DETACHED,		// child is told it is detached
		VM_PAYLOAD_TYPE, 
		VM_SEPARATE_BLOCKS_BD, 
		VM_SET_SEPARATION_TIMER,
		VM_OPEN_VENT, 
		VM_SET_AUTOPILOT,
		VM_DELAY_EXPLODE,
		VM_SHOW_OUTCOME,
		VM_UPDATE_ACTIVE,
		VM_ABORT,
		VM_LAUNCH_ESCAPE,
		VM_IGNITION,
		VM_DESTROY,
		VM_DESTROYED,
		VM_RESET_MET,
		VM_MASS_CHANGE,
		VM_KML_REGISTER, 
		VM_KML_ADD_EVENT,
		VM_KML_START_PLOTTING,
		VM_KML_STOP_PLOTTING,
		VM_KML_SET_COLOR
	};
	
	int SendVesselMessage(OBJHANDLE hTarget, IntVesselMessage VM, int Value = 0);
	virtual int GetVesselMessage(IntVesselMessage VM, int Value = 0);
	int BroadcastVesselMessage(IntVesselMessage VM, int Value = 0);

	enum DblVesselMessage {
		VM_GET_MAIN_ENGINE, 
		VM_SET_CTRL_ENGINE, 
		VM_SET_MAIN_ENGINE, 
		VM_GET_MAIN_ENGINE_FORCE, 
		VM_GET_CTRL_ENGINE_FORCE,
		VM_SET_APOGEE, 
		VM_SET_PERIGEE, 
		VM_SET_INCLINATION, 
		VM_SET_RANGE, 
		VM_SET_AZIMUTH, 
		VM_SET_LAT, 
		VM_SET_LON, 
		VM_SET_TGT_LAT,
		VM_SET_TGT_LON,
		VM_SET_RANGE_ADJUSTMENT,
		VM_SET_PRE_LAUNCH_VENT,
		VM_SET_PRE_LAUNCH_VENT_3ST
	};

	double SendVesselMessage(OBJHANDLE hTarget, DblVesselMessage VM, double Value);
    virtual double GetVesselMessage(DblVesselMessage VM, double Value = 0);

	// trajectories
	enum TrajectoryTypes {TRAJECTORY_UNKNOWN, TRAJECTORY_BALLISTIC, TRAJECTORY_ORBITAL}; 

	// kml
	kmlPlotColors kmlPlotColor;						//+-- to be overridden by vessel; default is green
	bool kmlNoPlotting;								//+-- to be overridden by vessel; default is yes
	OBJHANDLE kmlHandle;							//+-- handle the kmlWriter vessel
	void kmlAddEvent(kmlEvents kmlEvent, char* Label = "", double LatDeg = 0, double LonDeg = 0, OBJHANDLE hOrigin = NULL);
	void kmlSetColor(kmlPlotColors Color);
	virtual void WriteLastKmlEntry();

	// sound
	int SoundLibID;					//+--
	void SendVesselWaveToFocused(int SND_ID, int SND_MODE = NOLOOP, int Volume = 255);

	// cameras
	int CamerasCount;				//+--
	void SetCameraData(int Idx = 0, VECTOR3 Off = xyz0, VECTOR3 Dir = zplus, double Tilt = 0, bool bEnabled = true);

	// self-destructor
	bool bHaveSelfDestructor;
	bool bDelayedExplode;							//+--
	double DelayedExplodeTimer;						//+--

	// vessel-specific service functions
	void LocalRot(const VECTOR3 vecGlob, VECTOR3 &vecLoc);

	// fail-success criteria
	OUTCOME_OBJECT Outcomes[4];
	OUTCOME_CRITERIA OutcomeCriteria;
	boolean bOutcomeCriteriaSet;
	double OutcomeTimer;			//+--
	void UpdateOrbitalMissionCriteria();
	void ShowMissionOutcome(OBJHANDLE hVessel);

	// failures
	bool bFail;							//+??
	double Reliability;					//+-+ no need to save reliability - it is already "used up" if scenario was loaded at least once!
	double failTimer;					//-??
	virtual bool  failInitialize();
	virtual bool failUpdate();
	virtual void failActivate();

	FIRE Fire;							//-??
	void failStartFire();
	void failStartFireVisual();
	void failUpdateFire();


	// debug

private:
		// configuration
	// bLandedPermanently: 1. loaded as 'Landed' and not bAttached, 2. grounded when dust stops.
	bool bLandedPermanently;	//+-- 

	// landing-crashing-burning
	void TouchingTheGround();
    bool bFuelExplode();
	void Land();
	void Explode();
	void Crash();
	void ReplaceWithWreck(bool bExplode = TRUE);

	// dust cloud on low-speed impact, unlikely for R7 objects
	PARTICLESTREAMSPEC Dust;	//+--
	bool bGeneratingDust;		//+--
	double DustTimer;			//+--
	double DustLevel;			//+--
	void StartDust();
	void StopDust();

	// move focus manually between the vessels
	void SwitchFocus(focus_Dirs FocusDir);
	void ToggleEnableFocus();

	// screen output
	NOTEHANDLE MessagePort;							//+--
	double msgStartTime;							//+--
	bool bMessageDisplayed;							//+--

	// cameras
	int CameraID;									//+++
	CAMERA Cameras[5];								//+-- allow up to 5 static cameras
	void CycleOnboardCamera();
	void ActivateCurrentCamera();

	// debug
	double CockpitCameraTilt;						//+--
	bool bDebugInternalCameraMove;					//+--
	VECTOR3 CockpitCameraPos;						//---
	VECTOR3 CockpitCameraDir;						//---
	double DebugCockpitCameraTilt;					//---
	void DebugInternalCameraMove();
};



