#include	"math.h"
#include	"stdio.h"
#include	"BlockA.h"


BlockA::BlockA(OBJHANDLE hObj, int fmodel): VESSEL2M(hObj, fmodel)
{
	// VESSEL2M crash parameters
	VvCrash = DRand(1, 1); 
	VhCrash = DRand(3, 2);
	VMaxBlast = DRand(1000, 1); 
	QMax = DRand(100000, 15000);		// real max ~175 k?

	// VESSEL2M wrecks
	WreckSingleIdx = 8;
	WreckCount = 6;
	WreckIdxs[0] = 4;
	WreckIdxs[1] = 5;
	WreckIdxs[2] = 6;
	WreckIdxs[3] = 8;
	WreckIdxs[4] = 9;
	WreckIdxs[5] = 10;

	// VESSEL2M: store true individual empty mass for future recalculations
	EmptyMassWithoutAttachments = EMPTY_MASS;

	// default payload attachment params
	Payload.ObjectName[0] = 0;
	Payload.AttParamsOnUs = _AP(_V(0, 0, PAY_Z), zplus, xplus);
	Payload.AttParamsOnObjectDefault = _AP(xyz0, zminus, xplus);
	Payload.AttParamsOnObject = Payload.AttParamsOnObjectDefault;
	Payload.ScnVarNames = _ASVN("PAYLOAD", "PREF", "PDIR", "PROT");
	
	// default fairing1 attachment params
	Fairing1.ObjectName[0] = 0;
	Fairing1.AttParamsOnUs = _AP(_V(0, 0, PAY_Z-0.08), zplus, xplus);
	Fairing1.AttParamsOnObjectDefault = _AP(xyz0, zminus, xplus);
	Fairing1.AttParamsOnObject = Payload.AttParamsOnObjectDefault;
	Fairing1.ScnVarNames = _ASVN("FAIRING1", "FREF1", "FDIR1", "FROT1");

	// default fairing1 attachment params
	Fairing2.ObjectName[0] = 0;
	Fairing2.AttParamsOnUs = _AP(_V(0, 0, PAY_Z), zplus, xplus);
	Fairing2.AttParamsOnObjectDefault = _AP(xyz0, zminus, xplus);
	Fairing2.AttParamsOnObject = Payload.AttParamsOnObjectDefault;
	Fairing2.ScnVarNames = _ASVN("FAIRING2", "FREF2", "FDIR2", "FROT2");

	for (int i=0; i<4; i++)
	{
		BlocksBD[i].ObjectName[0] = 0;
		BlocksBD[i].AttParamsOnUs = _AP(_V(BD_X[i], BD_Y[i], BD_OFF_Z - MESH_OFF), zplus, BD_ROTS[i]);
		BlocksBD[i].AttParamsOnObjectDefault = _AP(BD_OBJ_POINT, zminus, yplus);
		BlocksBD[i].AttParamsOnObject = BlocksBD[i].AttParamsOnObjectDefault;
		BlocksBD[i].ScnVarNames = _ASVN(BD_NAMES[i], "UNDEFINED", "UNDEFINED", "UNDEFINED");
	}

	// autopilot default
	bAutopilot = true;

	// control engine animations: all start at neutral
	for (int i=0; i<4; i++)
		CERotationPercents[i] = 0;

	// running vars for blocks B-D
	for (int i=0; i<4; i++)
	{
		OmegaBlocks[i] = 0;
		AngleBlocks[i] = 0;
	}

	// flight phase
	bSafed = false;
	bFirstStage = false;
	bFirstStageSeparation = false;
	ContrailLevel = 0;
	Met = -1000;
	SeparationTimer = 0;
	CurrentTimerEvent = TE_DONE;
	storedCurrentTimerEvent = TE_NONE;

	// trajectory
	hEarth = oapiGetGbodyByName("Earth");
	Apogee = 0;							
	Perigee = 0;							
	Inc = 0;							
	Azimuth = 0;						
	Range = 0;
	RangeAdjustment = 0;
	bReachingPerigee = false;
	CurrentStage = SE_NOMINAL;
	YawSignalNormalizationRange = 5*RAD;
	PadLat = 0;
	PadLon = 0;
	GXplus = xplus;

	// default: switch focus to payload after its separation
	bSwitchFocus = true;

	// mesh configurations
	AdapterMeshName[0] = 0;
	BodyMeshName[0] = 0;
	PayloadType = PAYLOAD_SAT1; // %%% later change to Unknown
	hLaunchpad = NULL;

	// camera
	CamerasCount = 2;
	SetCameraData(0, _V(0, -1.35, 14.95), zminus);	// default camera is "down", rocketcam
	SetCameraData(1, _V(0, 1.36, 14.64));			// second camera is "up", payload

	// self-destructor
	bHaveSelfDestructor = true;

	// failures
	Reliability = 95;
	bYawFailed = bRollFailed = bPitchFailed = false;
	YawFixedR = RollFixedR = PitchFixedR = 0;
	SavedThrust = 0;
	NominalMainEngineLevel = 0;

	// fire location and direction of the flame,
	// unique for Block A
	Fire.Pos = _V(0.7, -0.7, EX_Z - MESH_OFF + 1.5);
	Fire.Dir = zplus;

	// kml
	kmlPlotColor = KML_CLR_BLUE;
}


void BlockA::clbkSetClassCaps(FILEHANDLE cfg)
{
	// physics and geometry
	SetSize(28.0);
	SetEmptyMass(EMPTY_MASS);
	SetCrossSections(_V(63.79, 63.91, 6.90));
	SetPMI(_V(71.48,71.48,0.74));
	SetCW (0.25, 0.5, 1.0, 1.0);							
	SetRotDrag (_V(6.7, 6.7, 0.1));				
	SetLiftCoeffFunc(0);
	SetPitchMomentScale(1e-4);
	SetBankMomentScale(1e-4);
	SetSurfaceFrictionCoeff(1e5, 1e5);
	SetTouchdownPoints (_V( 0, -1, -10), _V( 2, -1, 10), _V(-2, -1, 10));

	// fuel tank
	MainTank = CreatePropellantResource(TANK_CAPACITY);

	// textures and particles
	SURFHANDLE ExhTex = oapiRegisterExhaustTexture("r7_S\\Exhaust2");
	SURFHANDLE ptex = oapiRegisterParticleTexture("r7_S\\Contrail3");

	PARTICLESTREAMSPEC MainExhaustStream = { 0,			// flags, reserved
		1,			// srcsize, particle size at creation, m
		1000,			// srcrate, average particle generation rate, Hz   
		100,			// V0, average particle emission velocity, m/s
		0.01,			// srcspread, emission velocity distribution randomisation
		0.1,			// lifetime, average particle lifetime [s]
		5,				// growthrate, particle growth rate [m/s]
		0.1,			// atmslowdown, deceleration rate b in atmosphere, defined as v = v0 e- bdt 
		PARTICLESTREAMSPEC::EMISSIVE, PARTICLESTREAMSPEC::LVL_PSQRT, -1, 0.5, PARTICLESTREAMSPEC::ATM_PLIN, -1, 0.5, 
		ptex };			// texture

	PARTICLESTREAMSPEC ControlExhaustStream = { 0,			// flags, reserved
		0.05,			// srcsize, particle size at creation, m
		2000,			// srcrate, average particle generation rate, Hz   
		30,			// V0, average particle emission velocity, m/s
		0.01,			// srcspread, emission velocity distribution randomisation
		0.1,			// lifetime, average particle lifetime [s]
		5,				// growthrate, particle growth rate [m/s]
		0,			// atmslowdown, deceleration rate b in atmosphere, defined as v = v0 e- bdt 
		PARTICLESTREAMSPEC::EMISSIVE, PARTICLESTREAMSPEC::LVL_PSQRT, -1, 0.5, PARTICLESTREAMSPEC::ATM_PLIN, -1, 0.5, 
		ptex };			// texture

	PARTICLESTREAMSPEC psc1 = { 0,			// flags, reserved
		10,				// srcsize, particle size at creation, m
		1000,			// srcrate, average particle generation rate, Hz   
		200,			// V0, average particle emission velocity, m/s
		0.01,			// srcspread, emission velocity distribution randomisation
		2,				// lifetime, average particle lifetime [s]
		5,				// growthrate, particle growth rate [m/s]
		0.1,			// atmslowdown, deceleration rate b in atmosphere, defined as v = v0 e- bdt 
		PARTICLESTREAMSPEC::EMISSIVE, PARTICLESTREAMSPEC::LVL_PSQRT, 0, 0.5, PARTICLESTREAMSPEC::ATM_PLIN, 0, 0.5};	

	PARTICLESTREAMSPEC VentStream = { 0,			// flags, reserved
		1.5,			// srcsize, particle size at creation, m
		10,				// srcrate, average particle generation rate, Hz   
		5,				// V0, average particle emission velocity, m/s
		0.01,			// srcspread, emission velocity distribution randomisation
		1.1,			// lifetime, average particle lifetime [s]
		5,				// growthrate, particle growth rate [m/s]
		0.1,			// atmslowdown, deceleration rate b in atmosphere, defined as v = v0 e- bdt 
		PARTICLESTREAMSPEC::EMISSIVE, PARTICLESTREAMSPEC::LVL_PSQRT, 0, 0.5, PARTICLESTREAMSPEC::ATM_PLIN, 0, 0.0};	

	// main
	for (int i=0; i<4; i++)
	{
		th_main[i] = CreateThruster(TH_MAIN[i], zplus, MAX_MAIN_THRUST, MainTank, ISP_MAX, ISP_MIN);
		AddExhaust(th_main[i], EX_LEN, EX_WID, ExhTex);
		AddExhaustStream(th_main[i], TH_MAIN_STREAMS[i], &MainExhaustStream);
	}

	// control engines
	for (int i=0; i<4; i++)
	{
		th_ctrls[i] = CreateThruster(TH_CE[i], zplus, MAX_CE_THRUST, MainTank, ISP_MAX, ISP_MIN);
		th_main[i+4] = th_ctrls[i];
		AddExhaust(th_ctrls[i], EX_CE_LEN, EX_CE_WID, ExhTex);
		AddExhaustStream(th_ctrls[i], &ControlExhaustStream);
	}

	// main group, includes controls
	THGROUP_HANDLE thg_main = CreateThrusterGroup(th_main, 8, THGROUP_MAIN);

	// contrail
	PSTREAM_HANDLE ps = AddParticleStream(&psc1, _V(0, 0, -60), _V(0, 0, -1), &ContrailLevel);

	// end-of-mission vent
	th_vent1 = CreateThruster(_V( 0.77,  0.77, 15.388), _V(-COS45D, -COS45D, 0), 5000*G, MainTank, 264.9);
	th_vent2 = CreateThruster(_V(-0.77, -0.77, 15.388), _V( COS45D,  COS45D, 0), 5000*G, MainTank, 264.9);
	AddExhaustStream(th_vent1, &VentStream);
	AddExhaustStream(th_vent2, &VentStream);

	// meshes
	UINT Meshes_CE[4];		
	for (int i=0; i<4; i++)
	{
		Meshes_CE[i] = AddMesh(oapiLoadMeshGlobal("r7_S\\ControlEngine"), &TH_CE[i]);
		SetMeshVisibilityMode(Meshes_CE[i], MESHVIS_ALWAYS);
	}

	// animations
	MGROUP_ROTATE *CE_MGrp[4];
	for (int i=0; i<4; i++)
	{
		CE_MGrp[i] = new MGROUP_ROTATE(Meshes_CE[i], NULL, 0, TH_CEM[i], CE_AXIS[i], (float)CE_ANGLE);
		CERotations[i] = CreateAnimation (CERotationPercents[i]);
		AddAnimationComponent (CERotations[i], -1, 1, CE_MGrp[i]);
	}
}


void BlockA::clbkSaveState(FILEHANDLE scn)
{
	VESSEL2M::clbkSaveState(scn);
	oapiWriteScenario_string (scn, "=========== BLOCK A vars", "");

	if (!bAutopilot)
		oapiWriteScenario_string(scn, "MANUAL", "");

	// custom meshes
	if (AdapterMeshName[0] != 0)
		oapiWriteScenario_string (scn, "ADAPTER_MESHNAME", AdapterMeshName);
	if (BodyMeshName[0] != 0)
		oapiWriteScenario_string (scn, "BODY_MESHNAME", BodyMeshName);

	// separation steps
	if (SeparationTimer > 0)
		oapiWriteScenario_float (scn, "SEP_TIMER", SeparationTimer);
	if (CurrentTimerEvent > TE_PAYLOAD_SEP)
		oapiWriteScenario_int (scn, "SEP_PHASE", CurrentTimerEvent);

	// failure flag(s)
	if (bFail) 
		oapiWriteScenario_int (scn, "FLAGS_AI_1", failType);

	if (bPitchFailed + bYawFailed + bRollFailed) 
	{
		char strTemp[10];
		sprintf(strTemp, "%d %d %d", bPitchFailed, bYawFailed, bRollFailed);
		oapiWriteScenario_string (scn, "FLAGS_AI_2", strTemp);
	}

	if ((PitchFixedR + YawFixedR + RollFixedR)  != 0) 
		oapiWriteScenario_vec (scn, "FLAGS_AD_1", _V(PitchFixedR, YawFixedR, RollFixedR));

	if ((Fire.FireLevel > 0) && (GetThrusterGroupLevel(THGROUP_MAIN) > 0))
	{
		oapiWriteScenario_float (scn, "FLAGS_AD_2", GetThrusterGroupLevel(THGROUP_MAIN));
		oapiWriteScenario_float (scn, "FLAGS_AD_3", NominalMainEngineLevel);
	}

	// if there is no payload, store its type so that we'll jknow what adapter to restore later!
	if (Payload.ObjectName[0] == 0)
	{
		if (PayloadType != PAYLOAD_UNKNOWN)
			oapiWriteScenario_int (scn, "PL_TYPE", PayloadType);

		// if there is no payload, we are already done: 
		// don't write lots of stuff after this point!
		return;
	}

	// ========= The rest gets written only when we still have payload...

	// keep focus after payload separation?
	if (!bSwitchFocus)
		oapiWriteScenario_string (scn, "NO_FOCUS_SWITCH", "");

	// attachments
	WriteAttachedObjectToScenario(scn, &Payload);
	WriteAttachedObjectToScenario(scn, &Fairing1);
	WriteAttachedObjectToScenario(scn, &Fairing2);

	double dTempOmegas = 0;
	double dTempAngles = 0;
	char sTemp[256];

	// Blocks B-D
	for (int i=0; i<4; i++)
	{
		WriteAttachedObjectToScenario(scn, &BlocksBD[i]);
		dTempOmegas += OmegaBlocks[i];
		dTempAngles += AngleBlocks[i];
	}

	// block omegas
	if (dTempOmegas != 0)
	{
		sprintf(sTemp, "%lf %lf %lf %lf", OmegaBlocks[0], OmegaBlocks[1], OmegaBlocks[2], OmegaBlocks[3]);
		oapiWriteScenario_string (scn, "BLOCK_OMEGAS", sTemp);
	}

	// block angles
	if (dTempAngles != 0)
	{
		sprintf(sTemp, "%lf %lf %lf %lf", AngleBlocks[0], AngleBlocks[1], AngleBlocks[2], AngleBlocks[3]);
		oapiWriteScenario_string (scn, "BLOCK_ANGLES", sTemp);
	}

	// orbital trajectory
	if (Apogee + Perigee + Inc != 0)
		oapiWriteScenario_vec (scn, "TRJ_ORBIT", _V((Apogee-RADIUS_E)/1000, (Perigee-RADIUS_E)/1000, Inc*DEG));

	// ballistic trajectory
	if (Azimuth + Range + RangeAdjustment != 0)
		oapiWriteScenario_vec (scn, "TRJ_BALLISTIC", _V(Azimuth*DEG, Range/1000, RangeAdjustment*DEG));

	// pad coords
	if (PadLat+PadLon != 0)
		oapiWriteScenario_vec (scn, "PAD_COORDS", _V(PadLat, PadLon, 0)*DEG);

	// yaw control
	if (dist(GXplus, xplus) != 0)
		oapiWriteScenario_vec (scn, "GXPLUS", GXplus);

	// MET
	if (Met > 0)
		oapiWriteScenario_float (scn, "MET", Met);
}


void BlockA::ParseScenarioLineEx(char *line, void *status)
{
	if (!strnicmp (line, "MANUAL", 6))
		bAutopilot = false;

	// attachments
	else if (ReadAttachedObjectLineFromScenario(line, &Payload))
		;
	else if (ReadAttachedObjectLineFromScenario(line, &Fairing1))
		;
	else if (ReadAttachedObjectLineFromScenario(line, &Fairing2))
		;
	else if (ReadAttachedObjectLineFromScenario(line, &BlocksBD[0]))
		;
	else if (ReadAttachedObjectLineFromScenario(line, &BlocksBD[1]))
		;
	else if (ReadAttachedObjectLineFromScenario(line, &BlocksBD[2]))
		;
	else if (ReadAttachedObjectLineFromScenario(line, &BlocksBD[3]))
		;

	else if (!strnicmp (line, "NO_FOCUS_SWITCH", 15))
		bSwitchFocus = false;

	// custom meshes
	else if (!strnicmp (line, "ADAPTER_MESHNAME", 16))
		sscanf (line+16, "%s", &AdapterMeshName);
	else if (!strnicmp (line, "BODY_MESHNAME", 13))
		sscanf (line+13, "%s", &BodyMeshName);
	else if (!strnicmp (line, "PL_TYPE", 7))
		sscanf (line+7, "%d", &PayloadType);

	// blocks
	else if (!strnicmp (line, "BLOCK_OMEGAS", 12))
		sscanf (line+12, "%lf %lf %lf %lf", &OmegaBlocks[0], &OmegaBlocks[1], &OmegaBlocks[2], &OmegaBlocks[3]);
	else if (!strnicmp (line, "BLOCK_ANGLES", 12))
		sscanf (line+12, "%lf %lf %lf %lf", &AngleBlocks[0], &AngleBlocks[1], &AngleBlocks[2], &AngleBlocks[3]);

	// separation steps
	else if (!strnicmp (line, "SEP_TIMER", 9))
		sscanf (line+9, "%lf", &SeparationTimer);
	else if (!strnicmp (line, "SEP_PHASE", 9))
		sscanf (line+9, "%d", &storedCurrentTimerEvent);

	// orbital trajectory
	else if (!strnicmp (line, "TRJ_ORBIT", 9))
	{
		sscanf (line+9, "%lf %lf %lf", &Apogee, &Perigee, &Inc);
		Apogee = Apogee*1000+RADIUS_E;
		Perigee = Perigee*1000+RADIUS_E;
		Inc *= RAD;
		YawSignalNormalizationRange = 5*RAD;
	}

	// ballistic trajectory
	else if (!strnicmp (line, "TRJ_BALLISTIC", 13))
	{
		sscanf (line+13, "%lf %lf %lf", &Azimuth, &Range, &RangeAdjustment);
		Azimuth *= RAD;
		Range *= 1000;
		RangeAdjustment *= RAD;
		YawSignalNormalizationRange = 1*RAD;
	}

	// pad coords
	else if (!strnicmp (line, "PAD_COORDS", 10))
	{
		sscanf (line+10, "%lf %lf", &PadLat, &PadLon);
		PadLat *= RAD;
		PadLon *= RAD;
	}

	// yaw control
	else if (!strnicmp (line, "GXPLUS", 6))
		sscanf (line+6, "%lf %lf %lf", &GXplus.x, &GXplus.y, &GXplus.z);

	// MET
	else if (!strnicmp (line, "MET", 3))
		sscanf (line+3, "%lf", &Met);

	// failures
	else if (!strnicmp (line, "FLAGS_AI_1", 10))
		sscanf (line+10, "%d", &failType);
	else if (!strnicmp (line, "FLAGS_AI_2", 10))
		sscanf (line+10, "%1d %1d %1d", &bPitchFailed, &bYawFailed, &bRollFailed);
	else if (!strnicmp (line, "FLAGS_AD_1", 10))
		sscanf (line+10, "%lf %lf %lf", &PitchFixedR, &YawFixedR, &RollFixedR);
	else if (!strnicmp (line, "FLAGS_AD_2", 10))
		sscanf (line+10, "%lf", &SavedThrust);
	else if (!strnicmp (line, "FLAGS_AD_3", 10))
		sscanf (line+10, "%lf", &NominalMainEngineLevel);

	// inherited processing
	else 
		VESSEL2M::ParseScenarioLineEx(line, status);
}


void BlockA::clbkPostCreation()
{
	//VESSEL2M stuff
	VESSEL2M::clbkPostCreation();

	if (SavedThrust > 0)
		SetThrusterGroupLevel(THGROUP_MAIN, SavedThrust);

	if (AttachmentCount(TOPARENT) > 0)
		hLaunchpad = GetAttachmentStatus(GetAttachmentHandle(TOPARENT, 0));
}



void BlockA::FinishConfiguration()
{
	// finish attaching objects, update empty mass
	AttachObject(&Payload);
	AttachObject(&Fairing1);
	AttachObject(&Fairing2);

	// Blocks B-D
	for (int i=0; i<4; i++)
	{
		AttachObject(&BlocksBD[i]);
	
		if (BlocksBD[i].ObjectName[0] != 0)
			bFirstStage	= true;
	}

	// have payload? find out what payload type is
	if (Payload.ObjectName[0] != 0)
		PayloadType = (VM_PayloadTypes)SendVesselMessage(Payload.hObject, VM_PAYLOAD_TYPE);

	// special step for Sp2, needed when loading scenario with already active Sp2!
	if (PayloadType == PAYLOAD_SAT2)
		SendVesselMessage(Payload.hObject, VM_UPDATE_ACTIVE);

	UINT AdapterMesh;

	// custom adapter mesh from scenario?
	if (AdapterMeshName[0] != 0)
		AdapterMesh = AddMesh(oapiLoadMeshGlobal(AdapterMeshName), &ADAPTER_REF);
	else 
	{
		// if no payload [anymore], we still [might] remeber the old configuration
		if (PayloadType == PAYLOAD_WARHEAD_TEST)
			AdapterMesh = AddMesh(oapiLoadMeshGlobal("r7_S\\PA_Warhead5t"), &ADAPTER_REF);

		else if (PayloadType == PAYLOAD_WARHEAD_5T)
			AdapterMesh = AddMesh(oapiLoadMeshGlobal("r7_S\\PA_Warhead5t"), &ADAPTER_REF);

		else if (PayloadType == PAYLOAD_WARHEAD_3T)
			AdapterMesh = AddMesh(oapiLoadMeshGlobal("r7_S\\PA_Warhead3t"), &ADAPTER_REF);

		else if (PayloadType == PAYLOAD_SAT1)
			AdapterMesh = AddMesh(oapiLoadMeshGlobal("r7_S\\PA_Sp1"), &ADAPTER_REF);

		else if (PayloadType == PAYLOAD_SAT3)
			AdapterMesh = AddMesh(oapiLoadMeshGlobal("r7_S\\PA_Sp3"), &ADAPTER_REF);

		else if (PayloadType == PAYLOAD_POLYOT)
			AdapterMesh = AddMesh(oapiLoadMeshGlobal("r7_S\\PA_Polyot"), &ADAPTER_REF);
	}

	// have fairing? tell it what payload we have to load a proper mesh
	if (Fairing1.ObjectName[0] != 0)
		if (SendVesselMessage(Fairing1.hObject, VM_PAYLOAD_TYPE, PayloadType))
			UpdateEmptyMass();

	// set separation timer sequence
	if (storedCurrentTimerEvent != TE_NONE)
		CurrentTimerEvent = storedCurrentTimerEvent;
	else if (Fairing1.ObjectName[0] != 0)
		CurrentTimerEvent = TE_FAIRING_SEP;
	else if (Payload.ObjectName[0] != 0)
		CurrentTimerEvent = TE_PAYLOAD_SEP;

	// find out scenario date 
	SYSTEMTIME mjddate;
	MJDtoSystemTime(oapiGetSimMJD(),&mjddate);
	WORD ScnYear = mjddate.wYear;

	SetMeshVisibilityMode(AdapterMesh, MESHVIS_ALWAYS);

	UINT BodyMesh;

	// custom mesh from scenario?
	if (BodyMeshName[0] != 0)
		BodyMesh = AddMesh(oapiLoadMeshGlobal(BodyMeshName));

	// define main body mesh from scenario date 
	else
	{
		if ( (ScnYear < 1959) && (PayloadType <= PAYLOAD_WARHEAD_TEST) )
			BodyMesh = AddMesh(oapiLoadMeshGlobal("r7_S\\BlockA_Test"));	
		else
			BodyMesh = AddMesh(oapiLoadMeshGlobal("r7_S\\BlockA"));	
	}

	SetMeshVisibilityMode(BodyMesh, MESHVIS_ALWAYS);

	// add prelaunch vents
	// prelaunch vent vapours
	
	SURFHANDLE vent_Exhaust = oapiRegisterParticleTexture("r7_S\\Contrail");

	// high single vent
	PARTICLESTREAMSPEC VentStreamHigh = { 0,					// flags, reserved
		0.5,			// srcsize, particle size at creation, m
		35,				// srcrate, average particle generation rate, Hz   
		5,				// V0, average particle emission velocity, m/s
		-0.05,			// srcspread, emission velocity distribution randomisation
		1.0,			// lifetime, average particle lifetime [s]
		1.5,			// growthrate, particle growth rate [m/s]
		3.0,			// atmslowdown, deceleration rate b in atmosphere, defined as v = v0 e- bdt 
		PARTICLESTREAMSPEC::EMISSIVE, PARTICLESTREAMSPEC::LVL_PSQRT, 0, 2, PARTICLESTREAMSPEC::ATM_PLOG, 1e-4, 1, 
		vent_Exhaust
 	};

	// low double vent
	PARTICLESTREAMSPEC VentStreamLow = { 0,					// flags, reserved
		0.5,			// srcsize, particle size at creation, m
		35,				// srcrate, average particle generation rate, Hz   
		2,				// V0, average particle emission velocity, m/s
		-0.05,			// srcspread, emission velocity distribution randomisation
		0.7,			// lifetime, average particle lifetime [s]
		1.5,			// growthrate, particle growth rate [m/s]
		3.0,			// atmslowdown, deceleration rate b in atmosphere, defined as v = v0 e- bdt 
		PARTICLESTREAMSPEC::EMISSIVE, PARTICLESTREAMSPEC::LVL_PSQRT, 0, 2, PARTICLESTREAMSPEC::ATM_PLOG, 1e-4, 1, 
		vent_Exhaust
 	};


	PreLaunchVentLevel = 0;

	if (ScnYear < 1959) 
	{
		AddParticleStream (&VentStreamLow, _V(-1.0,  1.0, 2.8), zminus, &PreLaunchVentLevel);
		AddParticleStream (&VentStreamLow, _V(-1.0, -1.0, 2.8), zminus, &PreLaunchVentLevel);
	}
	else
		AddParticleStream (&VentStreamHigh, _V(-1.2, 1.2, 14.5), zminus, &PreLaunchVentLevel);

	// %%% This was supposed to be another realistic prelaunch vent stream under the launch table.
	// Unfortunately, creating this stream somehow slows down smoke particles from the main exhaust
	// (generated in Launchpad during launch). So, this stream is commented out for now,
	// until (or if) I find out how to use it without nterfering with launch.
	//AddParticleStream (&VentStream, _V(-0.8, 0.8, -11.5), zminus, &PreLaunchVentLevel);

	// autocalc bSafed so we don't have to read it from scn?
	bSafed = CurrentTimerEvent == TE_DONE;

	// 
	if (PayloadType == PAYLOAD_3ST)
		SetSoyuzMassesFuels();

	//VESSEL2M stuff
	VESSEL2M::FinishConfiguration();

	// special for Sputnik-2 when done
	if (	(PayloadType == PAYLOAD_SAT2) &&
			(Payload.ObjectName[0] != 0) &&
			bSafed
		)
		kmlNoPlotting = true;
}



int BlockA::GetVesselMessage(IntVesselMessage VM, int Value)
{
	if (VM == VM_SEPARATE_BLOCKS_BD)
	{
		bFirstStageSeparation = true;
		return 0;
	}

	else if (VM == VM_PAYLOAD_TYPE)
		return PayloadType;

	else if (VM == VM_SET_AUTOPILOT)
	{
		if ((int)bAutopilot != Value)
			ToggleAutopilot();
	}

	// some attachment requests a graceful detachment!
	else if (VM == VM_REQUEST_DETACH)
	{
		if ((int)Payload.hObject == Value)
		{
			SeparatePayload();

			if (Fairing1.ObjectName[0] != 0)
				SeparateFairing();
		}

		else if ((int)Fairing1.hObject == Value)
			SeparateFairing();

		else
			for (int i=0; i<4; i++)
				if ((int)BlocksBD[i].hObject == Value) 
					SeparateBlockBD(i);
	}

	else if (VM == VM_DESTROY)
		Break();

	else if (VM == VM_ABORT)
		SetThrusterGroupLevel(THGROUP_MAIN, 0);

	else if (VM == VM_LAUNCH_ESCAPE)
		SendVesselMessage(Payload.hObject, VM_LAUNCH_ESCAPE);

	else if (VM == VM_MASS_CHANGE)
		UpdateEmptyMass();

	// inherited
	return VESSEL2M::GetVesselMessage(VM, Value);
}


double BlockA::GetVesselMessage(DblVesselMessage VM, double Value)
{
	// this gets requested by Launchpad to adjust its flames-smokes
	if (VM == VM_GET_MAIN_ENGINE)
		return GetThrusterGroupLevel(THGROUP_MAIN);

	// this gets commanded from the Launcpad's launch sequence
	else if (VM == VM_SET_MAIN_ENGINE)
	{
		// cache the commanded value to be used on the B-D in case of A's fire
		NominalMainEngineLevel = Value;

		// ignore external value if we have engine fire in full thrust
		if (bFail && (Fire.FireLevel > 0.1) && (Value > 0.5))
			return 0;

		// set thrust to commanded value
		SetThrusterGroupLevel(THGROUP_MAIN, Value);
	}

	else if (VM == VM_SET_APOGEE)
	{
		Apogee = Value;
		YawSignalNormalizationRange = 5*RAD;
		SendVesselMessage(Payload.hObject, VM_SET_APOGEE, Value);
	}

	else if (VM == VM_SET_PERIGEE)
	{
		Perigee = Value;
		SendVesselMessage(Payload.hObject, VM_SET_PERIGEE, Value);
	}

	else if (VM == VM_SET_INCLINATION)
	{
		Inc = Value;
		SendVesselMessage(Payload.hObject, VM_SET_INCLINATION, Value);
	}

	else if (VM == VM_SET_LAT)
	{
		PadLat = Value;
		SendVesselMessage(Payload.hObject, VM_SET_LAT, Value);
	}

	else if (VM == VM_SET_LON)
	{
		PadLon = Value;
		SendVesselMessage(Payload.hObject, VM_SET_LON, Value);
	}

	else if (VM == VM_SET_TGT_LAT)
		SendVesselMessage(Payload.hObject, VM_SET_TGT_LAT, Value);

	else if (VM == VM_SET_TGT_LON)
		SendVesselMessage(Payload.hObject, VM_SET_TGT_LON, Value);

	else if (VM == VM_SET_RANGE)
	{
		Range = Value;
		YawSignalNormalizationRange = 1*RAD;
	}

	else if (VM == VM_SET_AZIMUTH)
	{
		Azimuth = Value;
		
		if (Azimuth > 180*RAD)
			Azimuth -= 360*RAD;
	}

	else if (VM == VM_SET_RANGE_ADJUSTMENT)
		RangeAdjustment = Value;

	else if (VM == VM_SET_PRE_LAUNCH_VENT)
		PreLaunchVentLevel = Value;

	else if (VM == VM_SET_PRE_LAUNCH_VENT_3ST)
		SendVesselMessage(Payload.hObject, VM_SET_PRE_LAUNCH_VENT_3ST, Value);

	return VESSEL2M::GetVesselMessage(VM, Value);
}



int BlockA::clbkConsumeBufferedKey(DWORD key, bool down, char *kstate)
{
	// VESSEL2M stuff
	int result = VESSEL2M::clbkConsumeBufferedKey(key, down, kstate);
	if (result != 0)
		return result;

	// only process keydown events
	if (!down) 
		return 0; 

	// shift combinations are reserved
	if (KEYMOD_SHIFT (kstate)) 
		return 0; 

	// Control combinations
	if (KEYMOD_CONTROL (kstate))
	{
		// switch focus to payload after its separation?
		if (key == OAPI_KEY_J) 
			ToggleSwitchFocus();

		// launch escape
		else if (key == OAPI_KEY_A) 
			SendVesselMessage(Payload.hObject, VM_LAUNCH_ESCAPE);

		return 0; 
	}

	// jetison next attached element
	if (key == OAPI_KEY_J)
		SeparateNextItem();

	// autopilot on or off
	if (key == OAPI_KEY_M)
		ToggleAutopilot();

	// manual debug
	if (key == OAPI_KEY_D)
		DebugSomething();

	return 0;
}


int BlockA::clbkConsumeDirectKey(char *keystate)
{
	// disable manual attempts in autopilot
	if (bAutopilot)
		if (
				KEYDOWN (keystate, OAPI_KEY_SUBTRACT) +
				KEYDOWN (keystate, OAPI_KEY_MULTIPLY) +
				KEYDOWN (keystate, OAPI_KEY_DIVIDE) +
				KEYDOWN (keystate, OAPI_KEY_ADD) + 
				KEYDOWN (keystate, OAPI_KEY_NUMPAD1) + 
				KEYDOWN (keystate, OAPI_KEY_NUMPAD3) + 
				KEYDOWN (keystate, OAPI_KEY_NUMPAD2) + 
				KEYDOWN (keystate, OAPI_KEY_NUMPAD8) + 
				KEYDOWN (keystate, OAPI_KEY_NUMPAD4) + 
				KEYDOWN (keystate, OAPI_KEY_NUMPAD6)
			)
		{
			ShowAnnotationMessage("Manual controls disabled. To enable, use M key.");
			return 1;
		}

	return 0;
}


void BlockA::clbkPostStep(double simt, double SimDT, double mjd)
{
	// common processing for all VESSEL2M-derived objects
	TimestepForVESSEL2M();

	// safed? Don't process anything down from here, save timesteps!
	if (bSafed)
		return;

	// store timestep, update met
	SimDt = SimDT;
	if (Met >=0 )
		Met += SimDT;

	// nominal screen output
	if (Met > 15)
		sprintf(oapiDebugString(), "Time: T+%.0f", Met);

	// store ground altitude
	Alt = GetAltitude();

	// contrail, plot: if we are within proper altitude range and the engine is on
	bTrusterOn = GetThrusterGroupLevel(THGROUP_MAIN) > 0;
	bool bContrailOn = (Alt > ALT_CONTRAIL_MIN) && (Alt < ALT_CONTRAIL_MAX) && (bTrusterOn);
	if (bContrailOn)
	{
		ContrailLevel = 1;
		kmlSetColor(KML_CLR_WHITE);
	}
	else
	{
		ContrailLevel = 0;

		if (bTrusterOn)
			kmlSetColor(KML_CLR_RED);
		else
			kmlSetColor(KML_CLR_BLUE);
	}

	// separation timer
	if (SeparationTimer > 0)
		UpdateSeparationTimer();


	// get "ideal" control inputs
	if (bAutopilot)
		GetAutopilotControls();
	else
	{
		GetManualControls();
		GetNavmodeControls();
	}

	// set physical controls
	SetControlEngineRotations();

	// exchange data with strap-ons...
	if (bFirstStage)
		SyncBlocksBD();
	
	// ... or check for MECO
	else
		CheckShutdownConditions();

	// sound broadcast
	double MainEngineLevel = GetThrusterGroupLevel(THGROUP_MAIN);
	if (MainEngineLevel > 0)
		SendVesselWaveToFocused(SND_ENGINE, LOOP, (int)(255*MainEngineLevel*1.6));
	else
		SendVesselWaveToFocused(SND_ENGINE, SND_STOP);


	// debug
//	if (CurrentStage == SE_MECO)
//		if (Alt < AltMECO)
//		{
//			kmlAddEvent(KML_SEPARATION, "MECO alt");
//			bSafed = true;
//		}


//	sprintf(oapiDebugString(), "Q: %f      H: %f      V: %f", GetDynPressure(), GetAltitude(), GetAirspeed());
}



void BlockA::clbkNavMode (int mode, bool active)
{
	// deactivate: allow silently
	if (!active) 
		return;

	// in manual mode, allow only killrot activation 
	if (!bAutopilot) 
	{
		if (mode != NAVMODE_KILLROT)
		{
			ShowAnnotationMessage("NavModes other than KILLROT are not implemented!");
			DeactivateNavmode(mode);
		}

		return;
	}

	// in autopilot: activate: override by immediately deactivating it, complain on-screen!
	DeactivateNavmode(mode);
	ShowAnnotationMessage("Manual controls disabled. To enable, use M key.");
}


void BlockA::GetManualControls()
{
	if (!bHaveFocus)
		return;

	double lpp = GetManualControlLevel(THGROUP_ATT_PITCHUP, MANCTRL_ATTMODE, MANCTRL_ANYDEVICE);
	double lpm = GetManualControlLevel(THGROUP_ATT_PITCHDOWN, MANCTRL_ATTMODE, MANCTRL_ANYDEVICE);
	double lp = lpp - lpm;

	double lyp = GetManualControlLevel(THGROUP_ATT_YAWLEFT, MANCTRL_ATTMODE, MANCTRL_ANYDEVICE);
	double lym = GetManualControlLevel(THGROUP_ATT_YAWRIGHT, MANCTRL_ATTMODE, MANCTRL_ANYDEVICE);
	double ly = lyp - lym;

	double lbp = GetManualControlLevel(THGROUP_ATT_BANKLEFT, MANCTRL_ATTMODE, MANCTRL_ANYDEVICE);
	double lbm = GetManualControlLevel(THGROUP_ATT_BANKRIGHT, MANCTRL_ATTMODE, MANCTRL_ANYDEVICE);
	double lb = lbp - lbm;

	// set control levels
	ControlLevel = _V(-lp, ly, lb);

	// any control inputs at all?
	AllControlLevels = fabs(ControlLevel.x) + fabs(ControlLevel.y) + fabs(ControlLevel.z);
}


void BlockA::GetNavmodeControls()
{
//	VECTOR3 ControlLevel1;
//	GetRelativeVel(hEarth, ControlLevel1);
//	ControlLevel1 = -GetAttitudeSensorDeviation(-ControlLevel1);


	// skip any navmode if we have manual input. don't disable, just skip temporarily.
	if (AllControlLevels > 0)
		return;

	if (GetNavmodeState(NAVMODE_KILLROT))
		KillRot();

/*	else if (GetNavmodeState(NAVMODE_RETROGRADE))
		ControlLevel = GetWantedNavmodeAttiude(NAVMODE_RETROGRADE);

	else if (GetNavmodeState(NAVMODE_PROGRADE))
		ControlLevel = GetWantedNavmodeAttiude(NAVMODE_PROGRADE);

	// useful debug BEFORE inputs will be replaced by signs or modified by stops
//	sprintf(oapiDebugString(), "x: %f    y: %f    z: %f", ControlLevel.x, ControlLevel.y, ControlLevel.z);

	// filter input signals by proximity to the neutral point, modify as needed
	UpdateStopConditions(&ControlLevel);
*/
}


void BlockA::GetAutopilotControls()
{
	// get omegas
	VECTOR3 Omegas;
	GetAngularVel(Omegas);

		// pitch

	// raw pitch angles and deviations
	double PitchWanted = GetPitchProg();

	// pitch channel failure?
	if (bPitchFailed)
		PitchWanted = PitchFixedR;

	double PitchCurrent = GetPitch();
	double dPitch = PitchWanted - PitchCurrent;
	
	// pitch signal normalized to to a range
	double PitchSignal = GetNormalizedValue(dPitch, 5*RAD);	// 0+-1, +-1 maps to +-5 deg deviation

	// omega signal normalized to to a range
	double PitchOmegaSignal = GetNormalizedValue(Omegas.x, 5*RAD);	//0+-1

	// pitch control signal as a combination of two signals, normalized to a control range
	double PitchControl = -GetNormalizedValue(PitchSignal - PitchOmegaSignal, 1); 

		// yaw
	VECTOR3 LZPlus, GZPlus;
	LZPlus = zplus;

	// yaw channel failure?
	if (bYawFailed)
	{
		LZPlus.z = cos(YawFixedR);
		LZPlus.x = sin(YawFixedR);
	}

	GlobalRot(LZPlus, GZPlus);
	double dp = dotp(GZPlus, GXplus);
	double angle = acos(dp);
	double dYaw = -(PI/2 - angle);

	//GetOrientErrors();
	//double AzimuthCurrent;
	//oapiGetHeading(GetHandle(), &AzimuthCurrent);
	//if (AzimuthCurrent > 180*RAD)
	//	AzimuthCurrent -= 360*RAD;
	//double dYaw = Azimuth - AzimuthCurrent;

	double YawSignal = GetNormalizedValue(dYaw, YawSignalNormalizationRange);	// 0+-1, +-1 maps to +-5 deg deviation
	double YawOmegaSignal = -GetNormalizedValue(Omegas.y, 5*RAD);	//0+-1
	double YawControl = -GetNormalizedValue(YawSignal - YawOmegaSignal, 1); 

	// don't touch yaw while we are vertical
	if (Met < 15)
		YawControl = 0;

		// roll

	double RollCurrent = GetBank();

	// roll channel failure?
	if (bRollFailed)
		RollCurrent += RollFixedR;

	double RollSignal = GetNormalizedValue(-RollCurrent, 5*RAD);	// 0+-1, +-1 maps to +-5 deg deviation
	double RollOmegaSignal = GetNormalizedValue(-Omegas.z, 5*RAD);	//0+-1
	double RollControl= GetNormalizedValue(RollSignal - RollOmegaSignal, 10); 

	if (Met < 15)
		RollControl = 0;


	// set control levels
	ControlLevel = _V(PitchControl, YawControl, RollControl);

//	sprintf(oapiDebugString(), "MET: %f   PitchCurrentDg : %f   PitchWantedDg : %f   PitchLevel: %f", Met, PitchCurrent*DEG, PitchWanted*DEG, PitchControl);
//	sprintf(oapiDebugString(), "MET: %f   YawSignal : %f   YawOmegaSignal : %f   YawControl: %f", Met, YawSignal, YawOmegaSignal, YawControl);
//	sprintf(oapiDebugString(), "MET: %f   RollSignal : %f   OmegaDg : %f   RollLevel: %f", Met, RollSignal, Omegas.z*DEG, RollControl);

//	sprintf(oapiDebugString(), "MET: %f   dYawDg : %f   YawLevel: %f", Met, dYaw*DEG, YawControl);
}


void BlockA::KillRot()
{
	// get Omega vector
	VECTOR3 Omega;
	GetAngularVel(Omega);

	// normalize Omega signal to +-1
	const double OMEGA_MAX = 10*RAD;	//deg/s
	Omega /= OMEGA_MAX;
	if (fabs(Omega.x) > 1)
		Omega.x = signWithZero(Omega.x);
	if (fabs(Omega.y) > 1)
		Omega.y = signWithZero(Omega.y);
	if (fabs(Omega.z) > 1)
		Omega.z = signWithZero(Omega.z);

	// populate control vector from normalized Omega vector
	ControlLevel.x = Omega.x;
	ControlLevel.y = -Omega.y;
	ControlLevel.z = Omega.z;
}




void BlockA::SetControlEngineRotations()
{
	// max distance the engine can traven in this timestep
	double CEVelDs = CE_VEL * SimDt;

	// in the code below, ControlLevel vector contains desired positions for pitch, yaw and roll.
	// Pitch is independent, so its engines (0 and 2) can be se tdirectly from the ControlLevel.x.
	// Yaw and Roll are interdependent - so their engines (1 and 3) move separately and in a more convoluted fashion.

		// PITCH

	// ...this is where two pitch engines (0 and 2) sit right now...
	double CurrentPitch = CERotationPercents[0];
	// ...this is fow far we want them to travel eventually from the current position...
	double PitchDiff = ControlLevel.x - CurrentPitch;
	// ...this is for far we'll actually travel from the current position in this timestep
	double dPitch;

	// check for physical limitations of the posssible engine move
	if (fabs(PitchDiff) < CEVelDs)
		dPitch = PitchDiff;
	else
		dPitch = CEVelDs * signNoZero(PitchDiff);

	//...this is the new position for pitch engines
	double NewPitch = CurrentPitch + dPitch;
	if (fabs(NewPitch) > 1)
		NewPitch = signNoZero(NewPitch);

	// move control engines 0 and 2 to the new position
	CERotationPercents[0] = NewPitch;
	CERotationPercents[2] = NewPitch;
	SetAnimation (CERotations[0], CERotationPercents[0]);
	SetAnimation (CERotations[2], CERotationPercents[2]);

	// rotate thrust vector for control engines 0 and 2 to follow the engines.
	PitchAngle = NewPitch * 45 * RAD;
	SetThrusterDir(th_ctrls[0], _V(0, sin(PitchAngle), cos(PitchAngle)));
	SetThrusterDir(th_ctrls[2], _V(0, sin(PitchAngle), cos(PitchAngle)));


	// calc yaw

	// current yaw "position" is the AVERAGE of two differently placed engines 1 and 3.
	double CurrentYaw = (CERotationPercents[1] + CERotationPercents[3]) / 2;
	// current roll "position" is the DIFFERENCE between two differently placed engines 1 and 3.
	double CurrentRoll = (CERotationPercents[1] - CERotationPercents[3]) / 2;

	// this is what we'd like to get - had the yaw-roll channels been independent...
	double WantedYaw = ControlLevel.y;
	double WantedRoll = ControlLevel.z;

	double WantedPos1, WantedPos3;

	// yaw priority
	if (fabs(WantedYaw) >= fabs(WantedRoll*5))
	{
		double WantedYawMax = WantedYaw + WantedRoll / 2;
		double WantedYawMin = WantedYaw - WantedRoll / 2;
		
		if (WantedYawMax > 1)
			WantedYawMax = 1;

		if (WantedYawMin < -1)
			WantedYawMax = -1;

		if (WantedRoll > 0)
		{
			WantedPos3 = WantedYawMin;
			WantedPos1 = WantedYawMax;
		}
		else
		{
			WantedPos3 = WantedYawMax;
			WantedPos1 = WantedYawMin;
		}
	}

	// roll priority
	else
	{
		double WantedRollMax =  fabs(WantedRoll) + WantedYaw / 2;
		double WantedRollMin = -fabs(WantedRoll) + WantedYaw / 2;
		
		if (WantedRollMax > 1)
			WantedRollMax = 1;

		if (WantedRollMin < -1)
			WantedRollMin = -1;

		if (WantedRoll < 0)
		{
			WantedPos3 = WantedRollMin;
			WantedPos1 = WantedRollMax;
		}
		else
		{
			WantedPos3 = WantedRollMax;
			WantedPos1 = WantedRollMin;
		}
	}

	double Diff1 = WantedPos1 - CERotationPercents[1];
	double Diff3 = WantedPos3 - CERotationPercents[3];
	double d1, d3;
	
	// check for physical limitations of the posssible engine move
	if (fabs(Diff1) < CEVelDs)
		d1 = Diff1;
	else
		d1 = CEVelDs * signNoZero(Diff1);

	if (fabs(Diff3) < CEVelDs)
		d3 = Diff3;
	else
		d3 = CEVelDs * signNoZero(Diff3);

	double New1 = CERotationPercents[1] + d1;
	double New3 = CERotationPercents[3] + d3;

	// move control engines 1, 3 
	CERotationPercents[1] = New1;
	CERotationPercents[3] = New3;
	SetAnimation (CERotations[1], CERotationPercents[1]);
	SetAnimation (CERotations[3], CERotationPercents[3]);

	// rotate thrust vector for control engines 1, 3
	Angle1 = New1 * 45 * RAD;
	Angle3 = New3 * 45 * RAD;
	SetThrusterDir(th_ctrls[1], _V(sin(Angle1), 0, cos(Angle1)));
	SetThrusterDir(th_ctrls[3], _V(sin(Angle3), 0, cos(Angle3)));

}


void BlockA::SyncBlocksBD()
{
	// rotate control engines
	if (BlocksBD[0].ObjectName[0] != 0)
		BlocksBD[0].ObjectIntf2M->GetVesselMessage(VM_SET_CTRL_ENGINE,  CERotationPercents[0]);
	if (BlocksBD[2].ObjectName[0] != 0)
		BlocksBD[2].ObjectIntf2M->GetVesselMessage(VM_SET_CTRL_ENGINE, -CERotationPercents[0]);
	if (BlocksBD[1].ObjectName[0] != 0)
		BlocksBD[1].ObjectIntf2M->GetVesselMessage(VM_SET_CTRL_ENGINE, -CERotationPercents[1]);
	if (BlocksBD[3].ObjectName[0] != 0)
		BlocksBD[3].ObjectIntf2M->GetVesselMessage(VM_SET_CTRL_ENGINE,  CERotationPercents[3]);

	double BlocksBDMainEngineLevel;

	// in process of separation?
	if (bFirstStageSeparation)
		BlocksBDMainEngineLevel = 0.8;

	// engine fire? set B-D to full thrust, don't trust the reduced core level!
	else if (bFail && (Fire.FireLevel > 0.1))
		BlocksBDMainEngineLevel = NominalMainEngineLevel;

	// otherwise, sync B-D trust to A thrust
	else 
		BlocksBDMainEngineLevel = GetThrusterGroupLevel(THGROUP_MAIN);

	// reset total thrust
	double TotalThrustForce = 0;

	// tolerances for breakups
	if (oapiGetTimeAcceleration() <= 1)
	{
		// spinnig too fast?
		VECTOR3 Omegas;
		GetAngularVel(Omegas);
		bool bFastSpin = (((abs(Omegas.x) + abs(Omegas.y) + abs(Omegas.z))*DEG) > 40);
		if (bFastSpin)
	//		for (int i=0; i<4; i++)
				SeparateBlockBD(0);

		// side Q is too strong?
		bool bStreamAngle = (((Alt < 35000) && (((abs(GetAOA())+abs(GetSlipAngle()))*DEG) > 20)));
		if (bStreamAngle)
	//		for (int i=0; i<4; i++)
				SeparateBlockBD(0);
	}

	// get and apply engines forces
	for (int i=0; i<4; i++)
	{
		if (BlocksBD[i].ObjectName[0] == 0)
			continue;

		// set main engines levels
		BlocksBD[i].ObjectIntf2M->GetVesselMessage(VM_SET_MAIN_ENGINE,  BlocksBDMainEngineLevel);

		// main engines force
		double MainEngineForce = BlocksBD[i].ObjectIntf2M->GetVesselMessage(VM_GET_MAIN_ENGINE_FORCE, 0);
		AddForce(_V(0, 0, MainEngineForce), BlocksBD[i].AttParamsOnUs.AttPoint);
		TotalThrustForce += MainEngineForce;

		// control engines force
		double ControlEngineForce = BlocksBD[i].ObjectIntf2M->GetVesselMessage(VM_GET_CTRL_ENGINE_FORCE, 0);
		TotalThrustForce += ControlEngineForce;

		// control engine force application point, calc from "local" control engine
		VECTOR3 CE_Ref = TH_CE[i] * 3;
		CE_Ref.z /= 3;

		// apply control engine force 
		if (i == 1)
			AddForce(_V(ControlEngineForce * sin(Angle1), 0, ControlEngineForce * cos(Angle1)), CE_Ref);
		else if (i == 3)
			AddForce(_V(ControlEngineForce * sin(Angle3), 0, ControlEngineForce * cos(Angle3)), CE_Ref);
		else
			AddForce(_V(0, ControlEngineForce * sin(PitchAngle), ControlEngineForce * cos(PitchAngle)), CE_Ref);

		// already separating!
		if (bFirstStageSeparation)
		{
			double		L = 15.14;	
			VECTOR3		BlockPMI;
			BlocksBD[i].ObjectIntf2M->GetPMI(BlockPMI);

			OmegaBlocks[i] += ( MainEngineForce*1.968 )*SimDt/(BlockPMI.x + L*L)/BlocksBD[i].ObjectIntf2M->GetMass();
			AngleBlocks[i] += OmegaBlocks[i]*SimDt;
			double SinAngle = sin(AngleBlocks[i]);

			if (i == 0)
				BlocksBD[i].AttParamsOnUs.AttDir.x = -SinAngle;
			else if (i == 1)
				BlocksBD[i].AttParamsOnUs.AttDir.y = -SinAngle;
			else if (i == 2)
				BlocksBD[i].AttParamsOnUs.AttDir.x = SinAngle;
			else //if (i == 3)
				BlocksBD[i].AttParamsOnUs.AttDir.y = SinAngle;

			BlocksBD[i].AttParamsOnUs.AttDir.z = sqrt(1 - SinAngle*SinAngle);

			SetAttachmentParams(	BlocksBD[i].hAttachment, 
									BlocksBD[i].AttParamsOnUs.AttPoint,
									BlocksBD[i].AttParamsOnUs.AttDir,
									BlocksBD[i].AttParamsOnUs.AttRot);
			
			// separate?
			if (AngleBlocks[i] >= ANGLE_BLOCKS_MAX)
				SeparateBlockBD(i);
		}
	}

	//
	UpdateEmptyMass();

	// add own thrust for pad separation
	if (bAttached)
	{
		THRUSTER_HANDLE SingleMainThruster = GetThrusterHandleByIndex(0);
		double SingleMainThrusterLevel = GetThrusterLevel(SingleMainThruster);
		double SingleMainThrusterMaxForce = GetThrusterMax(SingleMainThruster);
		
		TotalThrustForce += 4 * SingleMainThrusterMaxForce * SingleMainThrusterLevel;

		THRUSTER_HANDLE SingleControlThruster = GetThrusterHandleByIndex(4);
		double SingleControlThrusterLevel = GetThrusterLevel(SingleControlThruster);
		double SingleControlThrusterMaxForce = GetThrusterMax(SingleControlThruster);

		TotalThrustForce += 4 * SingleControlThrusterMaxForce * SingleControlThrusterLevel;

		// Liftoff!!
		if (TotalThrustForce/G >GetMass())
		{
			// request launchpad to gracefully separate us from the supports and do whatever else is needed on takeoff
			SendVesselMessage(GetAttachmentStatus(GetAttachmentHandle(TOPARENT, 0)), VM_REQUEST_DETACH);

			// reset MER
			Met = 0;

			// send MET moment to payloads/upper stages
			SendVesselMessage(Payload.hObject, VM_RESET_MET);

			// store global launch direction for yaw control
			GlobalRot(xplus, GXplus);
		}
	}
}


void BlockA::SeparateBlockBD(int idx)
{
	// already?
	if (BlocksBD[idx].ObjectName[0] == 0)
		return;
	
	//  get individual block name and store the event in kml
	char BlockStaging[256];
	sprintf(BlockStaging, "Staging %s", BlocksBD[idx].ObjectName);
	kmlAddEvent(KML_SEPARATION, BlockStaging);

	// separate
	double BlockVel = DRand(2, 3);
	DetachChild(BlocksBD[idx].hAttachment, BlockVel);
	BlocksBD[idx].ObjectName[0] = 0;
	BlocksBD[idx].ObjectIntf2M->SetThrusterGroupLevel(THGROUP_MAIN, 0);
	BlocksBD[idx].ObjectIntf2M->GetVesselMessage(VM_OPEN_VENT);
	BlocksBD[idx].ObjectIntf2M->bAttached = false;	

	// enable kml  plotting 
	SendVesselMessage(kmlHandle, VM_KML_START_PLOTTING, (int)BlocksBD[idx].hObject);

	// keep true until the last block is separated!
	bFirstStage = false;
	for (int i=0; i<4; i++)
		if (BlocksBD[i].ObjectName[0] != 0)
			bFirstStage = true;

	// become lighter!
	UpdateEmptyMass();

	// reset transient vars so that they don't get saved in scenario
	OmegaBlocks[idx] = 0;
	AngleBlocks[idx] = 0;

	// is it nominal separation? then do
	if (bFirstStageSeparation)
		return;

	// packet break-up? see if this is the first block out
	bool bFirstOut = true;
	for (int i=0; i<4; i++)
	{
		// skip ourselves
		if (i == idx)
			continue;

		// found another already out?
		if (BlocksBD[i].ObjectName[0] == 0)
			bFirstOut = false;
	}

	// is it first block out?
	if (bFirstOut)
	{
		// set up other side blocks for scattered fallouts
		for (int i=0; i<4; i++)
			if (i != idx)
				SendVesselMessage(BlocksBD[i].hObject, VM_SET_SEPARATION_TIMER);

		// start fire on ourselves
		if (Fire.FireLevel == 0)
		{
			bFail = true;
			failType = F_FIRE;
			failStartFire();
		}
	}
}


void BlockA::SetSoyuzMassesFuels()
{
	// mass
	EmptyMassWithoutAttachments = SOYUZ_EMPTY_MASS;
	SetEmptyMass(SOYUZ_EMPTY_MASS);

	// fuel
	SetPropellantMaxMass(MainTank, SOYUZ_TANK_CAPACITY);

	// engines
	for (int i=0; i<4; i++)
	{
		SetThrusterIsp(th_main[i], SOYUZ_ISP_MAX, SOYUZ_ISP_MIN);
		SetThrusterMax0(th_main[i], SOYUZ_MAX_MAIN_THRUST);

		SetThrusterIsp(th_ctrls[i], SOYUZ_ISP_MAX, SOYUZ_ISP_MIN);
		SetThrusterMax0(th_ctrls[i], SOYUZ_MAX_CE_THRUST);
	}
	
	for (int j=0; j<4; j++)
		if (BlocksBD[j].ObjectName[0] != 0)
		{
			BlocksBD[j].ObjectIntf2M->SetEmptyMass(SOYUZ_BD_EMPTY_MASS);
			BlocksBD[j].ObjectIntf2M->SetPropellantMaxMass(BlocksBD[j].ObjectIntf2M->GetPropellantHandleByIndex(0), SOYUZ_BD_TANK_CAPACITY);
		
			for (int i=0; i<4; i++)
			{
				BlocksBD[j].ObjectIntf2M->SetThrusterIsp(BlocksBD[j].ObjectIntf2M->GetThrusterHandleByIndex(i), SOYUZ_BD_ISP_MAX, SOYUZ_BD_ISP_MIN);
				BlocksBD[j].ObjectIntf2M->SetThrusterMax0(BlocksBD[j].ObjectIntf2M->GetThrusterHandleByIndex(i), SOYUZ_BD_MAX_MAIN_THRUST);

				if (i < 2)
				{
					BlocksBD[j].ObjectIntf2M->SetThrusterIsp(BlocksBD[j].ObjectIntf2M->GetThrusterHandleByIndex(i+4), SOYUZ_BD_ISP_MAX, SOYUZ_BD_ISP_MIN);
					BlocksBD[j].ObjectIntf2M->SetThrusterMax0(BlocksBD[j].ObjectIntf2M->GetThrusterHandleByIndex(i+4), SOYUZ_BD_MAX_CE_THRUST);
				}
			}
		}
	//
	UpdateEmptyMass();
}


void BlockA::ToggleAutopilot()
{
	// insert any checks for conditions that may prevent switching

	// toggle flag
	bAutopilot = !bAutopilot;

	// implement switching to the appropriate mode
	if (bAutopilot)
		SetAutopilotControl();
	else
		SetManualControl();
}


void BlockA::SetManualControl()
{
	ShowAnnotationMessage("Manual controls enabled. To return to autopilot, use M key");
	sprintf(oapiDebugString(), "");

	SendVesselMessage(Payload.hObject, VM_SET_AUTOPILOT, 0);
}


void BlockA::SetAutopilotControl()
{
	ShowAnnotationMessage("Manual controls disabled. To enable, use M key");

	// disable any possible residual navmodes
	DeactivateNavmode(NAVMODE_KILLROT);
	DeactivateNavmode(NAVMODE_HLEVEL);
	DeactivateNavmode(NAVMODE_PROGRADE);
	DeactivateNavmode(NAVMODE_RETROGRADE);
	DeactivateNavmode(NAVMODE_NORMAL);
	DeactivateNavmode(NAVMODE_ANTINORMAL);
	DeactivateNavmode(NAVMODE_HOLDALT);

	SendVesselMessage(Payload.hObject, VM_SET_AUTOPILOT, 1);

	// skip thrust update if we have fire
	if (bFail && (Fire.FireLevel > 0.1))
		return;
		
	// skip thrust update if we did not take off
	if (Met < 0)
		return;
	
	// skip thrust update if we did not take off
	if (CurrentStage == SE_MECO) 
		return;

	// reevaluate current stage
	CurrentStage = SE_NOMINAL;
	SetThrusterGroupLevel(THGROUP_MAIN, 1);
}


void BlockA::ToggleSwitchFocus()
{
	bSwitchFocus = !bSwitchFocus;

	if (bSwitchFocus)
		ShowAnnotationMessage("Switch focus to payload after separation? YES!");
	else
		ShowAnnotationMessage("Switch focus to payload after separation? NO!");
}



void BlockA::SeparateNextItem()
{
	if (bAutopilot)
		ShowAnnotationMessage("Manual controls disabled. To enable, use M key");

	else if ( (BlocksBD[0].ObjectName[0] != 0) || (BlocksBD[0].ObjectName[0] != 0) || 
		(BlocksBD[0].ObjectName[0] != 0) || (BlocksBD[0].ObjectName[0] != 0))
		SeparateBlockBD();

	else if (Fairing1.ObjectName[0] != 0)
		SeparateFairing();
	else if (Payload.ObjectName[0] != 0)
		SeparatePayload();
	else
		ShowAnnotationMessage("Nothing more to separate!");
}


void BlockA::SeparateBlockBD()
{
	if (Met < 5)
		ShowAnnotationMessage("Blocks B-D cannot be separated on the pad!");
	else
		bFirstStageSeparation = true;
}


void BlockA::SeparateFairing()
{
	// just in case...
	if (!Fairing1.ObjectIntf2M->bAttached)
		return;

	// get fairing name and store the event in kml
	char Staging[256];
	sprintf(Staging, "Separation: %s", Fairing1.ObjectName);
	kmlAddEvent(KML_SEPARATION, Staging);

	// enable kml plotting 
	SendVesselMessage(kmlHandle, VM_KML_START_PLOTTING, (int)Fairing1.hObject);

	double FairingVel = DRand(2, 1);
	DetachChild(Fairing1.hAttachment, FairingVel);
	Fairing1.ObjectName[0] = 0;
	UpdateEmptyMass();
	Fairing1.ObjectIntf2M->bAttached = false;	// by default, small service VESSEL2M items won't detect separation, we have to tell them.
	PlayVesselWave3(SoundLibID, SND_PYRO);		// don't broadcast
	CurrentTimerEvent = TE_PAYLOAD_SEP;
}


void BlockA::SeparatePayload()
{
	// just in case...
	if (!Payload.ObjectIntf2M->bAttached)
		return;

	// get payload name for the kml event
	char Staging[256];

	// enable kml plotting 
	SendVesselMessage(kmlHandle, VM_KML_START_PLOTTING, (int)Payload.hObject);

	// switch focus
	if ( (bSwitchFocus) && (bHaveFocus) )
		oapiSetFocusObject(Payload.hObject);

	double PayloadVel;

	if (PayloadType == PAYLOAD_SAT2)
	{
		sprintf(Staging, "Activation: %s", Payload.ObjectName);
		SendVesselMessage(kmlHandle, VM_KML_STOP_PLOTTING, (int)GetHandle());
		PayloadVel = 0;
		SendVesselMessage(Payload.hObject, VM_DETACHED);
		Payload.ObjectIntf2M->bAttached = false;	
	}
	else
	{
		sprintf(Staging, "Separation: %s", Payload.ObjectName);
		PayloadVel = DRand(1, 1);
		PlayVesselWave3(SoundLibID, SND_PYRO);			// don't broadcast
		DetachChild(Payload.hAttachment, PayloadVel);
		Payload.ObjectName[0] = 0;
		UpdateEmptyMass();
	}

	// store event the event in kml
	kmlAddEvent(KML_SEPARATION, Staging);

//	Payload.ObjectIntf2M->bAttached = false; :: Payloads should detect separation themselves, don't tell them!

	// engine shutdown, if not yet
	SetThrusterGroupLevel(THGROUP_MAIN, 0);
	SeparationTimer = 5;
	CurrentTimerEvent = TE_VENT_OPEN;

	// clear screen;
	Met = -1000;
	sprintf(oapiDebugString(), "");
}



double BlockA::GetPitchProg()
{
	if (Inc == 0)
		return GetPitchProgBallistic();
	else if (PayloadType != PAYLOAD_3ST)
		return GetPitchProgOrbital();
	else
		return GetPitchProg3St();
}


// return the required pitch angle for the current flight moment, in deg, relative to local horizon
double BlockA::GetPitchProgBallistic()
{
	// tuning
	const double MET_VERTICAL = 10;				// sec
	const double MET_ZERO_AOA_MIN = 40;			// sec
	const double MET_ZERO_AOA_MAX = 60;			// sec
	const double MET_MAX = 200;					// sec

	const double DEG_PITCHOVER = 70;			// deg
	const double DEG_PRG_START = 50;			// deg
	const double DEG_PRG_END = 16;				// deg
	// when still attached, GetPitch function is not active, so we supply 0 as a baseline
	if (bAttached)
		return 0;							
	
	// liftoff and vertical, baseline is 90 deg
	else if (Met <= MET_VERTICAL)
		return 90*RAD;							

	// between vertical and maxQ, do a pitchover
	else if (Met <= MET_ZERO_AOA_MIN)
		return DEG_PITCHOVER*RAD;							

	// max Q, fly at zero AOA
	else if (Met <= MET_ZERO_AOA_MAX)
	{
		VECTOR3 HorizonV;
		GetHorizonAirspeedVector(HorizonV);
		double Len = length(HorizonV);
		double SinP = HorizonV.y / Len;
		double P = asin(SinP);

		return P;
	}

	// this is the actual program, rotating pitch at constant omega in the specified range
	else if (Met < MET_MAX)
	{
		double dMet = Met-MET_ZERO_AOA_MAX;
		double dMetPercent = dMet / (MET_MAX - MET_ZERO_AOA_MAX);
		double dDegPercent = dMetPercent;
		double dDeg = dDegPercent * (DEG_PRG_START - DEG_PRG_END);
		double P = DEG_PRG_START - dDeg;

		return P * RAD;
	}

	// don't go lower than the lowest limit for pitch
	else
		return DEG_PRG_END * RAD;
}


// return the required pitch angle for the current flight moment, in deg, relative to local horizon
double BlockA::GetPitchProgOrbital()
{
	// tuning
	const double MET_VERTICAL = 10;				// sec
	const double MET_ZERO_AOA_MIN = 40;			// sec
	const double MET_PITCHOVER_2_START = 60;
	const double MET_PITCHOVER_2_STOP = 80;

	const double DEG_PITCHOVER_1 = 60;			// deg
	const double DEG_PITCHOVER_2= 45;			// deg

	// when still attached, GetPitch function is not active, so we supply 0 as a baseline
	if (bAttached)
		return 0;							
	
	// liftoff and vertical, baseline is 90 deg
	else if (Met <= MET_VERTICAL)
		return 90*RAD;							

	// between vertical and maxQ, do a pitchover
	else if (Met <= MET_ZERO_AOA_MIN)
		return DEG_PITCHOVER_1*RAD;							

	// between vertical and maxQ, do a pitchover
	else if ((Met > MET_PITCHOVER_2_START) && (Met <= MET_PITCHOVER_2_STOP))
		return DEG_PITCHOVER_2*RAD;							

	// max Q, fly at zero AOA
	else if (!bReachingPerigee)
	{
		VECTOR3 HorizonV;
		GetHorizonAirspeedVector(HorizonV);
		double Len = length(HorizonV);
		double SinP = HorizonV.y / Len;
		double P = asin(SinP);

		if (Met > MET_PITCHOVER_2_STOP)
			P -= 10*RAD;

		CheckReachingPerigee();

		return P;
	}

	// adjust pitch program to arrive at proper perigee
	else
	{
		// get velocity vector angle over horizon
		VECTOR3 HorizonV;
		GetHorizonAirspeedVector(HorizonV);
		double Len = length(HorizonV);
		double SinP = HorizonV.y / Len;
		double P = asin(SinP);

		if (fabs(P) < 3 * RAD)
			return -5*RAD;
		else
			return -30*RAD;
	}
}

// return the required pitch angle for the current flight moment, in deg, relative to local horizon
double BlockA::GetPitchProg3St()
{
	// tuning
	const double MET_VERTICAL = 10;				// sec
	const double MET_ZERO_AOA_MIN = 40;			// sec
	const double MET_ZERO_AOA_MAX = 100;			// sec
	const double MET_MAX = 175;					// sec 255

	const double DEG_PITCHOVER = 65;			// deg 75
	const double DEG_PRG_START = 25;			// deg 50
	const double DEG_PRG_END = 16;				// deg 8

	// when still attached, GetPitch function is not active, so we supply 0 as a baseline
	if (bAttached)
		return 0;							
	
	// liftoff and vertical, baseline is 90 deg
	else if (Met <= MET_VERTICAL)
		return 90*RAD;							

	// between vertical and maxQ, do a pitchover
	else if (Met <= MET_ZERO_AOA_MIN)
		return DEG_PITCHOVER*RAD;							

	// max Q, fly at zero AOA
	else if (Met <= MET_ZERO_AOA_MAX)
	{
		VECTOR3 HorizonV;
		GetHorizonAirspeedVector(HorizonV);
		double Len = length(HorizonV);
		double SinP = HorizonV.y / Len;
		double P = asin(SinP);

		return P;
	}

	// this is the actual program, rotating pitch at constant omega in the specified range
	else if (Met < MET_MAX)
	{
		double dMet = Met-MET_ZERO_AOA_MAX;
		double dMetPercent = dMet / (MET_MAX - MET_ZERO_AOA_MAX);
		double dDegPercent = dMetPercent;
		double dDeg = dDegPercent * (DEG_PRG_START - DEG_PRG_END);
		double P = DEG_PRG_START - dDeg;

		return P * RAD;
	}

	// don't go lower than the lowest limit for pitch
	else
		return DEG_PRG_END * RAD;
}



void BlockA::CheckReachingPerigee()
{
	// total radial acceleration from gravity and thrust with pitch of -20 deg
	VECTOR3 T;
	GetThrustVector(T);
	double mass = GetMass();
	double a = length(T)/mass;				
	double ar = G + a*sin(20 * RAD);			

	// current radial vel
	VECTOR3 V;
	GetHorizonAirspeedVector (V);
	double vr = V.y;

	// time needed to cancel out current radial vel with the above acceleration
	double t = vr/ar;	
	t -= 15;

	// vertical distance to be traveled during this cancelation
	double s = vr * t - (ar * t * t) / 2 ;

	// final altitude to be reached
	double FinalAlt = Alt + s + 5000;

	// set flag if it is time to do it
	bReachingPerigee = FinalAlt > (Perigee-RADIUS_E);
}


void BlockA::CheckShutdownConditions()
{
	UpdateCheckShutdownVars();

	if (CurrentStage == SE_NOMINAL)
		CheckForNoTimeAccel();

	else if (CurrentStage == SE_NO_TIME_ACCEL)
		CheckToReduceThrust();

	else if (CurrentStage == SE_REDUCED_THRUST)
		CheckForMECO();

/*
if (Apogee > 0)
	sprintf(oapiDebugString(), "A: %f      P: %f      PeArg: %f", ApDist/1000-6371, PeDist/1000-6371, PeArg*DEG);
else
	sprintf(oapiDebugString(), "currR: %f      tgtR: %f      AngRangeFree: %f", CurrRange/1000, Range/1000, AngRangeFree*DEG);
*/
}


void BlockA::UpdateCheckShutdownVars()
{
	// get remaining fuel time at current rate
	FuelMass = GetTotalPropellantMass();
	FlowRate = GetTotalPropellantFlowrate();
	FuelTime = FuelMass/FlowRate;

	// get current apogee and perigee
	if (Apogee > 0)
	{
		GetApDist(ApDist);
		GetPeDist(PeDist);
		GetArgPer(PeArg);
	}

	// get current range
	else
	{
		// get ballistic range (without Earth rotation)
		GetElements(hEarth, OrbEls, &OrbParam);
		AngRangeFree = 180*RAD - OrbParam.TrA;
		CurrRange = AngRangeFree * RADIUS_E * 2;

		// %%% need better way of integrating elapsed ground distance in launch reference, unsensitive to Earth rotation!
		double Lat;
		double Lon;
		double Radius;
		GetEquPos(Lon, Lat, Radius);

		double TraveledDist;
		double Azimuth;
		DistanceAzimuthFromCoordPairKmDg(Lat, Lon, PadLat, PadLon, &TraveledDist, &Azimuth);

		// adjustment for extnded final leg 
		CurrRange += (TraveledDist * RangeAdjustment); 

		// velocity of Earth rotation, Eastward, m/s
		double VE = RADIUS_E * OMEGA_E * sin(PadLat);

		// adjustment for Earth rotation assist/penalty
		double RangeEarthRot = VE * Met * sin(Azimuth);
		CurrRange -= RangeEarthRot;
	}
}


void BlockA::CheckForNoTimeAccel()
{
	// low fuel?
	if (
		(FuelTime < 5) 

		||
			// orbital
		(	
			(Apogee > 0) 
			&&
			// for Sputnik-2 and Spuntik-3, don't check trajectory; 
			// use low fuel condition as nominal MECO (this is historically true)
			(PayloadType != PAYLOAD_SAT2) && (PayloadType != PAYLOAD_SAT3) 
			&&
			// apogee approaching
			(PeArg < 180*RAD) && ((Apogee - ApDist) < 200000 )
		)
		||
			// ballistic
		(
			(Range > 0) 
			&&
			((Range-CurrRange) < 2000000)
		)

		)
	{
		// don't reduce time accel for 3-stage version staging
		if ( oapiGetTimeAcceleration() > 1)
			if (PayloadType != PAYLOAD_3ST)
				oapiSetTimeAcceleration(1);

		CurrentStage = SE_NO_TIME_ACCEL;
	}
}


void BlockA::CheckToReduceThrust()
{
// 2-stage version
if (PayloadType != PAYLOAD_3ST)
{
	// low fuel?
	if (
		(FuelTime < 0.3) // was 1

		|| 
		
			// orbital
		(	
			(Apogee > 0) 
			&&
			// for Sputnik-2 and Spuntik-3, don't check trajectory; 
			// use low fuel condition as nominal MECO (this is historically true)
			(PayloadType != PAYLOAD_SAT2) && (PayloadType != PAYLOAD_SAT3) 
			&&
			// apogee approaching
			(PeArg < 180*RAD) && ((Apogee - ApDist) < 100000 )
		)
		
		||
			// ballistic
		(
			(Range > 0) 
			&&
			((Range-CurrRange) < 500000)
		)
		
		)
	{
		SetThrusterGroupLevel(THGROUP_MAIN, 0.1);
		CurrentStage = SE_REDUCED_THRUST;
	}
}
// 3-rd stage
else
{
	// low fuel?
	if (FuelTime < 0.5)
	{
		SendVesselMessage(Payload.hObject, VM_IGNITION);
		SetThrusterGroupLevel(THGROUP_MAIN, 0.1);
		CurrentStage = SE_REDUCED_THRUST;
	}
}


}


void BlockA::CheckForMECO()
{
// 2-stage version
if (PayloadType != PAYLOAD_3ST)
{
	// low fuel?
	if (
		(FuelTime < 2)

		|| 
		
			// orbital
		(	
			(Apogee > 0) 
			&&
			// for Sputnik-2 and Spuntik-3, don't check trajectory; 
			// use low fuel condition as nominal MECO (this is historically true)
			(PayloadType != PAYLOAD_SAT2) && (PayloadType != PAYLOAD_SAT3) 
			&&
			// apogee reached
			(PeArg < 180*RAD) && ((Apogee - ApDist) < 0) 
			// perigee > 200 km
			&& (PeDist > 6371000+200000)
//			&& (PeDist > Perigee)
		)

		||
			// ballistic
		(
			(Range > 0) 
			&&
			(Range-CurrRange < 0)
		)
		
		)
	{
		//MECO, and leaving some fuel for vent
		SetThrusterGroupLevel(THGROUP_MAIN, 0);
		SeparationTimer = 5;
		kmlAddEvent(KML_SEPARATION, "MECO");
		CurrentStage = SE_MECO;
	}
}

// 3-rd stage
else
{
	// low fuel?
	if (FuelMass == 0)
	{
		SeparationTimer = 0.01;
		CurrentStage = SE_MECO;
	}
}
}







void BlockA::UpdateSeparationTimer()
{
	// update timer
	SeparationTimer -= SimDt;

	// process and reset next timer event
	if (SeparationTimer <= 0)
	{
		if (CurrentTimerEvent == TE_FAIRING_SEP)
		{
			SeparateFairing();
			SeparationTimer = 5;
		}

		else if (CurrentTimerEvent == TE_PAYLOAD_SEP)
			SeparatePayload();

		else if (CurrentTimerEvent == TE_VENT_OPEN)
			VentOpen();

		else if (CurrentTimerEvent == TE_VENT_CLOSE)
			VentClose();
	}
}


void BlockA::VentOpen()
{
	// start gas sound
	PlayVesselWave3(SoundLibID, SND_GAS);		// don't broadcast

	// activate vent thruster 
	SetThrusterLevel(th_vent1, 1);

	// for sputnik-2, activate two vent thrusters
	if (PayloadType == PAYLOAD_SAT2)
		SetThrusterLevel(th_vent2, 1);

	// next timer step
	CurrentTimerEvent = TE_VENT_CLOSE;
	SeparationTimer = 1;
}


void BlockA::VentClose()
{
	// close vents if not done already
	SetThrusterLevel(th_vent1, 0);
	SetThrusterLevel(th_vent2, 0);

	// if we had fire - let it burn to conclusion
	if (bFail && (Fire.FireLevel > 0.1))
		return;

	// only now we can say that we are completely done...
	bSafed = true;
	CurrentTimerEvent = TE_DONE;
	kmlPlotColor = KML_CLR_BLUE;
}


// customize breakup to properly separate all attached stuff
void BlockA::Break()
{
	for (int i=0; i<4; i++)
		if (BlocksBD[i].ObjectName[0] != 0)
			SeparateBlockBD(i);

	if (Fairing1.ObjectName[0] != 0)
		SeparateFairing();

	if (Payload.ObjectName[0] != 0)
		SeparatePayload();

	SendVesselMessage(hLaunchpad, VM_DESTROYED);
	SendVesselWaveToFocused(SND_ENGINE, SND_STOP);

	// 3-rd stage - unfocus wrecks
	if (PayloadType == PAYLOAD_3ST)
		if (bSafed)
			if (!bHaveFocus)
			{
				bFocusOffByDefault = true;
				bFocus = false;
			}

	// run inherited function
	VESSEL2M::Break();
}

















	// output and debugging

void BlockA::DebugSomething()
{
//	sprintf(oapiDebugString(), "o1 %f o2 %f i1 %d", dOutp1, dOutp2, iOutp1);
//	SendVesselWaveToFocused(SND_BLAST_LONG);
//	sprintf(oapiDebugString(), "o1 %f o2 %f i1 %d", dOutp1, dOutp2, iOutp1);
//	sprintf(oapiDebugString(), "class %d",	(Payload.ObjectIntf2M->SignVM == 'M'));
//	VentOpen();
//	Break();
}


void BlockA::GetOrientErrors()
{
	VECTOR3 GZPlus;
	GlobalRot(zplus, GZPlus);
	double dp = dotp(GZPlus, GXplus);
	double angle = acos(dp);
	double yaw = PI/2 - angle;

	sprintf(oapiDebugString(), "Y:%f", yaw*DEG);

/*
//	VECTOR3 GY = mul(TmLocToLaunch, yminus);

	VECTOR3 Y;
	LocalRot(GY, Y);

	double xy = sqrt(1 - Y.z*Y.z);
	double yaw = Y.x / xy;
	yaw = asin(yaw);
*/

/*
	VECTOR3	PK, ViP, ViPL, dir;
	double	dB, dP, dY;
	double	sin_dP, cos_dP;
	double	sin_dY, cos_dY;

	dir = GetLongAxisDir(Azimuth*RAD, Alpha);
	dir = dir*(1/length(dir));
	
	oapiGetGlobalPos(GetHandle(), &PK);
	ViP = PK + dir;
	Global2Local(ViP, ViPL);

	dB = 0;

	sin_dP = ViPL.y/length(ViPL);
	cos_dP = sqrt(1 - sin_dP*sin_dP);
	dP = atan2(sin_dP, cos_dP);

	sin_dY = ViPL.x/sqrt(ViPL.x*ViPL.x + ViPL.z*ViPL.z);
	cos_dY = ViPL.z/sqrt(ViPL.x*ViPL.x + ViPL.z*ViPL.z);
	dY = atan2(sin_dY, cos_dY);

	return _V(dB, dP, dY);*/
} 


		// =========== FAILURES


bool BlockA::failInitialize()
{
	// no failure at all?
	if (VESSEL2M::failInitialize())
		return true;

	// how? one of so many ways...
	failType = (FAIL_TYPE)IRand((int)F_FIRE, 3);
//%%% DEBUG!
//failType = F_CONTROL;

	// engine fire?
	if (failType == F_FIRE)
		failTimer = DRand(0, 12+255);

	// ignition failure?
	else if (failType == F_IGNITION)
		failTimer = DRand(0, 3.5);

	// control failure?
	else // if (failType == F_CONTROL)
		failTimer = DRand(5, 250);

	return false;
}


bool BlockA::failUpdate()
{
	// inherited
	if (VESSEL2M::failUpdate())
		return true;

	// developing fire
	if (failType == F_FIRE)
		failUpdateFire();

	return false;
}


void BlockA::failActivate()
{
	// inherited
	VESSEL2M::failActivate();

	if (failType == F_FIRE)
		failStartFire();

	else if (failType == F_IGNITION)
		failAbortLaunch();

	else //if (failType == F_CONTROL)
		failSetControlFailure();
}



void BlockA::failAbortLaunch()
{
	bAutopilot = false;
	ShowAnnotationMessage("Ignition failure detected, launch aborted!");
	sprintf(oapiDebugString(), "");

	SetThrusterGroupLevel(THGROUP_MAIN, 0);
	SyncBlocksBD();

	SendVesselMessage(GetAttachmentStatus(GetAttachmentHandle(TOPARENT, 0)), VM_ABORT);

	bFail = false;
}


void BlockA::failSetControlFailure()
{
	// yaw channel fail
	if (DRand(0, 30) < 10)
	{
		YawFixedR = DRand(-180, 360)*RAD;
		bYawFailed = true;
		ShowAnnotationMessage("Yaw control channel failure detected!");
		kmlAddEvent(KML_FLIGHT, "Yaw control channel failure detected!");
	}

	// roll channel fail
	else if (DRand(0, 30) > 20)
	{
		RollFixedR = DRand(-180, 360)*RAD;
		bRollFailed = true;
		ShowAnnotationMessage("Roll control channel failure detected!");
		kmlAddEvent(KML_FLIGHT, "Roll control channel failure detected!");
	}

	// pitch channel fail
	else
	{
		PitchFixedR = DRand(0, 90)*RAD;
		bPitchFailed = true;
		ShowAnnotationMessage("Pitch control channel failure detected!");
		kmlAddEvent(KML_FLIGHT, "Pitch control channel failure detected!");
	}

	// we don't need it, as control failure flag is split into separate channels
	bFail = false;
}





//=============================================================================
DLLCLBK VESSEL *ovcInit(OBJHANDLE hvessel, int flightmodel)
{
	return new BlockA(hvessel, flightmodel);
}

DLLCLBK void ovcExit(VESSEL *vessel)
{
	if (vessel) delete (BlockA*) vessel;
}
