#include	"math.h"
#include	"orbitersdk.h"
#include	"stdio.h"
#include	"Sputnik3.h"



Sputnik3::Sputnik3(OBJHANDLE hObj, int fmodel): VESSEL2M(hObj, fmodel)
{
	// VESSEL2M crash parameters
	VvCrash = DRand(5, 10);					// 5-15 m/s
	VhCrash = DRand(20, 20);				// 20-40 m/s
	VMaxBlast = DRand(1000, 1);				// 1 km/s
	QMax = DRand(50000, 25000);				// 50k-75k

	// VESSEL2M wrecks
	WreckIdxs[0] = 11;

	// radio 
	RadioTimer = -1;
	bPlayingSound = false;

	// define animation and its defaults
	AnimationTimer = 0;						// initial animations delay timer
	bAnimate = false;		
	AnimationPercent = 0;
	DefineAnimations();

	// mission outcome
	strcpy(Outcomes[OC_SUCCESS].OutcomeMessage, "Congratulations! You've just launched the first scientific payload into Earth orbit!");
	strcpy(Outcomes[OC_FAIL].OutcomeMessage, "Mission not accomplished. The satellite has failed to reach the orbit...");
}


void Sputnik3::clbkSetClassCaps(FILEHANDLE cfg)
{
	// physics and geometry
	SetSize(3.0);									// 1.73 x 3.57 cone
	SetEmptyMass(1327);								// 1327 kg
	SetCrossSections(_V(2.18, 2.18, 1.56));			// cone
	SetPMI(_V(0.36, 0.36, 0.16));						// from shipedit
	SetCW (0.1, 0.2, 0.4, 0.4);							// 
	SetRotDrag (_V(1, 1, 0.1));						// symmetrical. dOmega = -Omega * q * S i * RD i
	SetSurfaceFrictionCoeff(1e-4, 1e-4);
	SetTouchdownPoints (_V( 0, 0, -0.8), _V( 0.8, 0.4, 0.8), _V(-0.8, 0.8, 0.8));

	// mesh
	UINT Mesh = AddMesh(oapiLoadMeshGlobal("r7_S\\Sputnik3"));
	SetMeshVisibilityMode(Mesh, MESHVIS_ALWAYS);
}


void Sputnik3::clbkSaveState(FILEHANDLE scn)
{
	// do inherited VESSEL2M stuff before writing custom stuff
	VESSEL2M::clbkSaveState (scn);

	// separator 
	oapiWriteScenario_string (scn, "=========== SPUTNIK-3 vars", "");

	if (bPlayingSound)
		oapiWriteScenario_string (scn, "PLAY_SOUND", "");

	if (RadioTimer > 0)
		oapiWriteScenario_float (scn, "RADIO_TIMER", RadioTimer);

	if (AnimationTimer > 0)
		oapiWriteScenario_float(scn, "ANTENNAS_TIMER", AnimationTimer);

	if (AnimationPercent > 0)
		oapiWriteScenario_float (scn, "ANTENNAS_STATUS", AnimationPercent);
}


void Sputnik3::ParseScenarioLineEx(char *line, void *status)
{
	if (!strnicmp (line, "PLAY_SOUND", 10))
		bPlayingSound = true;

	else if (!strnicmp (line, "RADIO_TIMER", 11))
		sscanf (line+11, "%lf", &RadioTimer);

	else if (!strnicmp (line, "ANTENNAS_TIMER", 14))
		sscanf (line+14, "%lf", &AnimationTimer);

	else if (!strnicmp (line, "ANTENNAS_STATUS", 15))
		sscanf (line+15, "%lf", &AnimationPercent);

	else 
		VESSEL2M::ParseScenarioLineEx(line, status);
}


// called after all vessel in scn are created, but before first timestep
void Sputnik3::FinishConfiguration()
{
	// radio
	if (bPlayingSound)
		StartSound();

	// restore antennas location
	SetAnimation(AntennaRotation, AnimationPercent);

	// is animation still running?
	bAnimate = ( (AnimationPercent > 0) && (AnimationPercent < 1) );

	//VESSEL2M stuff
	VESSEL2M::FinishConfiguration();
}


void Sputnik3::clbkPostStep(double simt, double SimDT, double mjd)
{
	// common processing for all VESSEL2M-derived objects
	TimestepForVESSEL2M();

	// maintain radio sound in the loop
	if (bPlayingSound)
		SendVesselWaveToFocused(SND_RADIO, LOOP);

	// detect when we get separated from the rocket
	if (bAttached)
	{
		// this indicates our separation from the rocket
		if (GetAttachmentStatus(GetAttachmentHandle(TOPARENT, 0)) == NULL)
		{
			// sound broadcast does not work during focus switch, shoud duplicate here
			PlayVesselWave3(SoundLibID, SND_PYRO);

			// start timers
			AnimationTimer = 7;
			RadioTimer = 9;

			bAttached = false;

			// update mission criteria
			UpdateOrbitalMissionCriteria();
		}

		return;
	}

	SimDt = SimDT;

	// update radio timer
	if (RadioTimer > 0)
	{
		RadioTimer -= SimDT;
		if (RadioTimer <= 0)
			StartSound();
	}

	// update animation timer
	if (AnimationTimer > 0)
	{
		AnimationTimer -= SimDT;
		if (AnimationTimer <= 0)
			AnimationStart();
	}

	// run animation
	if (bAnimate)
		Animate();
}


int Sputnik3::clbkConsumeBufferedKey(DWORD key, bool down, char *kstate)
{
	// VESSEL2M stuff
	int result = VESSEL2M::clbkConsumeBufferedKey(key, down, kstate);
	if (result != 0)
		return result;

	// only process keydown events
	if (!down) 
		return 0; 

	// shift combinations are reserved
	if (KEYMOD_SHIFT (kstate)) 
		return 0; 

	// play radio sound
	if (key == OAPI_KEY_B) 
	{
		if (bPlayingSound)
			StopSound();
		else
			StartSound();
	}

	return 0;
}


void Sputnik3::StartSound()
{
	bPlayingSound = true;
	SendVesselWaveToFocused(SND_RADIO, LOOP);
}


void Sputnik3::StopSound()
{
	bPlayingSound = false;
	SendVesselWaveToFocused(SND_RADIO, SND_STOP);
}


int Sputnik3::GetVesselMessage(IntVesselMessage VM, int Value)
{
	if (VM == VM_PAYLOAD_TYPE)
		return PAYLOAD_SAT3;

	// unsupported message
	return VESSEL2M::GetVesselMessage(VM, Value);
}







	// ===== ANIMATIONS =====


void Sputnik3::DefineAnimations()
{
	// Antennas
	MGROUP_ROTATE *Antenna1, *Antenna2;

	static UINT grAntenna0[2] = {15, 16}; 
	static UINT grAntenna1[2] = {17, 18}; 

	Antenna1 = new MGROUP_ROTATE(0, grAntenna0, 2, ANTEN01_REF, xplus,  -(float)(90*RAD));
	Antenna2 = new MGROUP_ROTATE(0, grAntenna1, 2, ANTEN02_REF, xminus , -(float)(90*RAD));
	
	AntennaRotation = CreateAnimation (0);

	AddAnimationComponent (AntennaRotation, 0, 1, Antenna1);
	AddAnimationComponent (AntennaRotation, 0, 1, Antenna2);
}



void Sputnik3::Animate()
{
	double dPercent = ANTENNA_ROTATION_SPEED * SimDt;

	if (AnimationPercent < 1.0)
		AnimationPercent = min (1.0, AnimationPercent + dPercent);
	else
		AnimationEnd();

	SetAnimation (AntennaRotation, AnimationPercent);
}

void Sputnik3::AnimationStart()
{
	// ignore multipe restarts
	if (bAnimate)
		return;

	// pyro sound
	SendVesselWaveToFocused(SND_PYRO);

	// finally, allow animation
	bAnimate = true;
}


void Sputnik3::AnimationEnd()
{
	bAnimate = false;
	SendVesselWaveToFocused(SND_CLANG);
}







// ========================================================================

DLLCLBK VESSEL *ovcInit(OBJHANDLE hvessel, int flightmodel)
{
	return new Sputnik3(hvessel, flightmodel);
}

DLLCLBK void ovcExit(VESSEL *vessel)
{
	if (vessel) delete (Sputnik3*) vessel;
}



