#include	"math.h"
#include	"orbitersdk.h"
#include	"stdio.h"
#include	"Polyot.h"



Polyot::Polyot(OBJHANDLE hObj, int fmodel): VESSEL2M(hObj, fmodel)
{
	// VESSEL2M crash parameters
	VvCrash = DRand(5, 30); 
	VhCrash = DRand(20, 30);
	VMaxBlast = DRand(1000, 1); 
	QMax = DRand(3000, 10000);

	// VESSEL2M wrecks
	WreckCount = 6;
	WreckIdxs[0] = 1;
	WreckIdxs[1] = 2;
	WreckIdxs[2] = 3;
	WreckIdxs[3] = 1;
	WreckIdxs[4] = 2;
	WreckIdxs[5] = 3;

	// target
	TrackingState = TS_IDLE;
	hCurrTgt = NULL;
	MinDist = 0;

	// camera
	SetCameraData(0, _V(-0.549, 0.5856, 0.5423));

	// self-destructor
	bHaveSelfDestructor = true;

	// mission outcome
	strcpy(Outcomes[OC_SUCCESS].OutcomeMessage, "Mission accomplished! The first ever ASAT system has reached the orbit!");
	strcpy(Outcomes[OC_FAIL].OutcomeMessage, "Mission not accomplished. ASAT system has failed to reach the orbit...");
	bMissionHit = false;
}



void Polyot::clbkSetClassCaps(FILEHANDLE cfg)
{
	// physics and geometry
	SetSize(3.5);										// D=2m, H=4m, cone
	SetEmptyMass(400);									// %%%1400 kg full with fuel! // 1300?
	SetCrossSections(_V(1.87, 1.87, 1.584));			// cone
	SetPMI(_V(0.33, 0.33, 0.18));						// from shipedit
	SetCW (0.1, 0.2, 0.4, 0.4);							// 
	SetRotDrag (_V(1, 1, 0.1));							// symmetrical. dOmega = -Omega * q * S i * RD i
	SetSurfaceFrictionCoeff(1e-4, 1e-4);
	SetTouchdownPoints (_V( 0, -0.3, -0.8), _V( 0.8, -0.4, 0.8), _V(-0.8, -0.8, 0.8));

	// fuel tank
	MainTank = CreatePropellantResource(1000);

	// textures and particles
	SURFHANDLE ExhTex = oapiRegisterExhaustTexture("r7_S\\Exhaust2");
	SURFHANDLE ptex = oapiRegisterParticleTexture("r7_S\\Contrail3");

	PARTICLESTREAMSPEC BigExhaustStream = { 0,			// flags, reserved
		0.2,			// srcsize, particle size at creation, m
		0.1,			// srcrate, average particle generation rate, Hz   
		20,				// V0, average particle emission velocity, m/s
		-0.05,			// srcspread, emission velocity distribution randomisation
		0.05,			// lifetime, average particle lifetime [s]
		5,				// growthrate, particle growth rate [m/s]
		0.1,			// atmslowdown, deceleration rate b in atmosphere, defined as v = v0 e- bdt 
		PARTICLESTREAMSPEC::EMISSIVE, PARTICLESTREAMSPEC::LVL_PSQRT, 0, 0.5, PARTICLESTREAMSPEC::ATM_PLOG, 0, 0,
		ptex };			// texture

	PARTICLESTREAMSPEC MediumExhaustStream = { 0,			// flags, reserved
		0.05,			// srcsize, particle size at creation, m
		20.5,			// srcrate, average particle generation rate, Hz   
		5,				// V0, average particle emission velocity, m/s
		0.1,			// srcspread, emission velocity distribution randomisation
		0.05,			// lifetime, average particle lifetime [s]
		2,				// growthrate, particle growth rate [m/s]
		0.1,			// atmslowdown, deceleration rate b in atmosphere, defined as v = v0 e- bdt 
		PARTICLESTREAMSPEC::EMISSIVE, PARTICLESTREAMSPEC::LVL_PSQRT, -1, 0.5, PARTICLESTREAMSPEC::ATM_PLIN, -1, 0.5, 
		ptex };			// texture

	PARTICLESTREAMSPEC SmallExhaustStream = { 0,			// flags, reserved
		0.02,			// srcsize, particle size at creation, m
		20.5,			// srcrate, average particle generation rate, Hz   
		5,				// V0, average particle emission velocity, m/s
		0.1,			// srcspread, emission velocity distribution randomisation
		0.05,			// lifetime, average particle lifetime [s]
		2,				// growthrate, particle growth rate [m/s]
		0.1,			// atmslowdown, deceleration rate b in atmosphere, defined as v = v0 e- bdt 
		PARTICLESTREAMSPEC::EMISSIVE, PARTICLESTREAMSPEC::LVL_PSQRT, -1, 0.5, PARTICLESTREAMSPEC::ATM_PLIN, -1, 0.5, 
		ptex };			// texture

	// main
	THRUSTER_HANDLE	thrusters[22];	

	// rear bigs
	thrusters[0] = CreateThruster(_V(-RB_X,  RB_Y, RB_Z), zplus, TH_BIG, MainTank, ISP);
	thrusters[1] = CreateThruster(_V( RB_X, -RB_Y, RB_Z), zplus, TH_BIG, MainTank, ISP);

	// forward bigs
	thrusters[2] = CreateThruster(_V( FB_XY, 0, FB_Z), xminus,  TH_BIG, MainTank, ISP);
	thrusters[3] = CreateThruster(_V(-FB_XY, 0, FB_Z),  xplus, TH_BIG, MainTank, ISP);
	thrusters[4] = CreateThruster(_V(0,  FB_XY, FB_Z), yminus,  TH_BIG, MainTank, ISP);
	thrusters[5] = CreateThruster(_V(0, -FB_XY, FB_Z),  yplus, TH_BIG, MainTank, ISP);

	// all bigs
	for (int i=0; i<6; i++)
	{
		AddExhaust(thrusters[i], B_LEN, B_WID, ExhTex);
		AddExhaustStream(thrusters[i], TH_BIG_STREAMS[i], &BigExhaustStream);
	}

	// main group
	CreateThrusterGroup(thrusters, 2, THGROUP_MAIN);

	// translation groups
	thgFwd   = CreateThrusterGroup(thrusters, 2, THGROUP_ATT_FORWARD);
	thgLeft  = CreateThrusterGroup(&thrusters[2], 1, THGROUP_ATT_LEFT);
	thgRight = CreateThrusterGroup(&thrusters[3], 1, THGROUP_ATT_RIGHT);
	thgDown  = CreateThrusterGroup(&thrusters[4], 1, THGROUP_ATT_DOWN);
	thgUp    = CreateThrusterGroup(&thrusters[5], 1, THGROUP_ATT_UP);

	// pitch up med
	thrusters[6] = CreateThruster(_V( RM_XY_S,  RM_XY_L, RM_Z), yminus, TH_MED, MainTank, ISP);
	// pitch down med
	thrusters[8] = CreateThruster(_V(-RM_XY_S, -RM_XY_L, RM_Z),  yplus, TH_MED, MainTank, ISP);

	// pitch up small
	thrusters[7] =  CreateThruster(_V( RS_XY_S,  RS_XY_L, RS_Z), yminus, TH_SM, MainTank, ISP);
	// pitch down small
	thrusters[9] = CreateThruster(_V(-RS_XY_S, -RS_XY_L, RS_Z),  yplus, TH_SM, MainTank, ISP);

	// yaw right med
	thrusters[10] = CreateThruster(_V( RM_XY_L,  RM_XY_S, RM_Z), xminus, TH_MED, MainTank, ISP);
	// yaw left med
	thrusters[12] = CreateThruster(_V(-RM_XY_L, -RM_XY_S, RM_Z),  xplus, TH_MED, MainTank, ISP);

	// yaw right small
	thrusters[11] =  CreateThruster(_V( RS_XY_L,  RS_XY_S, RS_Z), xminus, TH_SM, MainTank, ISP);
	// yaw left small
	thrusters[13] =  CreateThruster(_V(-RS_XY_L, -RS_XY_S, RS_Z),  xplus, TH_SM, MainTank, ISP);

	for (int i=6; i<13; i+=2)
	{
		AddExhaust(thrusters[i], M_LEN, M_WID, ExhTex);
		AddExhaustStream(thrusters[i], &MediumExhaustStream);
	}

	for (int i=7; i<14; i+=2)
	{
		AddExhaust(thrusters[i], S_LEN, S_WID, ExhTex);
		AddExhaustStream(thrusters[i], &SmallExhaustStream);
	}

	const double FS_Z = 0.135;
	const double FS_XY_L = 0.565;
	const double FS_XY_S = 0.18;

	// roll left
	thrusters[14] =  CreateThruster(_V(-FS_XY_S,  FS_XY_L, FS_Z), xminus , TH_SM, MainTank, ISP);
	thrusters[15] =  CreateThruster(_V( FS_XY_L,  FS_XY_S, FS_Z), yplus, TH_SM, MainTank, ISP);
	thrusters[16] =  CreateThruster(_V( FS_XY_S, -FS_XY_L, FS_Z), xplus, TH_SM, MainTank, ISP);
	thrusters[17] =  CreateThruster(_V(-FS_XY_L, -FS_XY_S, FS_Z), yminus , TH_SM, MainTank, ISP);

	// roll right
	thrusters[18] =  CreateThruster(_V( FS_XY_S,  FS_XY_L, FS_Z), xplus, TH_SM, MainTank, ISP);
	thrusters[19] =  CreateThruster(_V( FS_XY_L, -FS_XY_S, FS_Z), yminus , TH_SM, MainTank, ISP);
	thrusters[20] =  CreateThruster(_V(-FS_XY_S, -FS_XY_L, FS_Z), xminus , TH_SM, MainTank, ISP);
	thrusters[21] =  CreateThruster(_V(-FS_XY_L,  FS_XY_S, FS_Z), yplus, TH_SM, MainTank, ISP);

	for (int i=14; i<22; i++)
	{
		AddExhaust(thrusters[i], M_LEN, M_WID, ExhTex);
		AddExhaustStream(thrusters[i], &MediumExhaustStream);
	}

	// rotation groups
	thgPitchUp   = CreateThrusterGroup(&thrusters[6],  2, THGROUP_ATT_PITCHUP);
	thgPitchDown = CreateThrusterGroup(&thrusters[8],  2, THGROUP_ATT_PITCHDOWN);
	thgYawRight  = CreateThrusterGroup(&thrusters[10], 2, THGROUP_ATT_YAWRIGHT);
	thgYawLeft   = CreateThrusterGroup(&thrusters[12], 2, THGROUP_ATT_YAWLEFT);
	thgRollLeft  = CreateThrusterGroup(&thrusters[14], 4, THGROUP_ATT_BANKLEFT);
	thgRollRight = CreateThrusterGroup(&thrusters[18], 4, THGROUP_ATT_BANKRIGHT);

	// mesh
	UINT Mesh = AddMesh(oapiLoadMeshGlobal("r7_S\\Polyot"));
	SetMeshVisibilityMode(Mesh, MESHVIS_ALWAYS);
}


void Polyot::clbkSaveState(FILEHANDLE scn)
{
	VESSEL2M::clbkSaveState(scn);
	oapiWriteScenario_string (scn, "=========== POLYOT vars", "");

	if (bMissionHit)
		oapiWriteScenario_string (scn, "MISSION_HIT", "");
	
	if (TrackingState != TS_IDLE)
		oapiWriteScenario_int (scn, "TRACKING_STATE", TrackingState);

	if (hCurrTgt != NULL)
		oapiWriteScenario_string (scn, "TARGET", TargetName);
}


void Polyot::ParseScenarioLineEx(char *line, void *status)
{
	if (!strnicmp (line, "MISSION_HIT", 11))
	{
		bMissionHit = true;
		strcpy(Outcomes[OC_SUCCESS].OutcomeMessage, "Consgratulations! The first ASAT system is successfully tested! Target is destroyed.");
		strcpy(Outcomes[OC_FAIL].OutcomeMessage, "ASAT testing has failed. ASAT spacecraft did not hit the target...");
	}

	else if (!strnicmp (line, "TRACKING_STATE", 14))
		sscanf (line+14, "%d", &TrackingState);

	else if (!strnicmp (line, "TARGET", 6))
	{
		sscanf (line+6, "%s", TargetName);
		hCurrTgt = oapiGetVesselByName(TargetName);
		if (hCurrTgt == NULL)
			TrackingState = TS_IDLE;
		else
			MinDist = (GetSize() + oapiGetSize(hCurrTgt)) / 2;
	}

	else 
		VESSEL2M::ParseScenarioLineEx(line, status);
}


int Polyot::clbkConsumeBufferedKey(DWORD key, bool down, char *kstate)
{
	// VESSEL2M stuff
	int result = VESSEL2M::clbkConsumeBufferedKey(key, down, kstate);
	if (result != 0)
		return result;

	// only process keydown events
	if (!down) 
		return 0; 

	// shift combinations are reserved
	if (KEYMOD_SHIFT (kstate)) 
		return 0; 

	// control combinations are reserved
	if (KEYMOD_CONTROL (kstate)) 
		return 0; 


	// L(ock): lock on selected target
	if (key == OAPI_KEY_L)
	{
		LockOnTarget();
		return 1;
	}

	// K(ill): kill selected target
	if (key == OAPI_KEY_K)
		KillTarget();

	// space: cycle-select to next target or "no target"
	if (key == OAPI_KEY_SPACE)
		SelectNextTarget();

	return 0;
}


int Polyot::clbkConsumeDirectKey(char *keystate)
{
	if (
			KEYDOWN (keystate, OAPI_KEY_SUBTRACT) +
			KEYDOWN (keystate, OAPI_KEY_MULTIPLY) +
			KEYDOWN (keystate, OAPI_KEY_DIVIDE) +
			KEYDOWN (keystate, OAPI_KEY_ADD) + 
			KEYDOWN (keystate, OAPI_KEY_NUMPAD1) + 
			KEYDOWN (keystate, OAPI_KEY_NUMPAD3) + 
			KEYDOWN (keystate, OAPI_KEY_NUMPAD2) + 
			KEYDOWN (keystate, OAPI_KEY_NUMPAD8) + 
			KEYDOWN (keystate, OAPI_KEY_NUMPAD4) + 
			KEYDOWN (keystate, OAPI_KEY_NUMPAD6)
		)
	{
		// disable manual attempts in auto modes
		if (TrackingState > TS_SELECTED)
		{
			ShowAnnotationMessage("Manual controls are disabled when homing on target!");
			return 1;
		}

		// disable manual attempts in auto modes
		if (bAttached)
		{
			ShowAnnotationMessage("Manual controls are disabled when attached to rocket!");
			return 1;
		}
	}

	return 0;
}


void Polyot::clbkNavMode (int mode, bool active)
{
	// deactivate: allow silently
	if (!active) 
		return;

	// only disable manual attempts in auto modes
	if (TrackingState < TS_LOCKING)
		return;

	ShowAnnotationMessage("Manual controls are disabled when homing on target!");
	DeactivateNavmode(mode);
}


void Polyot::clbkPostStep(double simt, double SimDT, double mjd)
{
	// common processing for all VESSEL2M-derived objects
	TimestepForVESSEL2M();

	// detect when we get separated from the rocket
	if (bAttached)
	{
		// this indicates our separation from the rocket
		if (GetAttachmentStatus(GetAttachmentHandle(TOPARENT, 0)) == NULL)
		{
			// pyro sound
			PlayVesselWave3(SoundLibID, SND_PYRO);
			
			bAttached = false;

			// update mission criteria
			if (!bMissionHit)
				UpdateOrbitalMissionCriteria();
		}

		return;
	}

	if (TrackingState > TS_IDLE)
	{
		UpdateTargetParams();

		if (TrackingState == TS_KILLING)
		{
			SlideToTarget();
			TurnToTarget();
		}

		else if (TrackingState >= TS_LOCKING)
			TurnToTarget();

		if (TrackingState > TS_IDLE) // 2nd check, could have changed above...
			ShowTargetParams();
	}

	if (GetThrusterGroupLevel(THGROUP_MAIN) > 0)
		kmlSetColor(KML_CLR_RED);
	else
		kmlSetColor(KML_CLR_GREEN);
}


// long function, lots of logic is involved in the selection...
void Polyot::SelectNextTarget()
{
	// still attached
	if (bAttached)
	{
		ShowAnnotationMessage("Targeting operations cannot start until separation from rocket.");
		return;
	}

	// set first candidate vessel to NULL
	OBJHANDLE hVesTgtCandidate = NULL;

	// if there is no currently selected target, 
	// we are ready to pick the next one
	bool bCanTakeNextVessel = (hCurrTgt == NULL);

	// start with "new selection not made"
	bool bSelected = false;

	if (GetTotalPropellantMass() == 0)
	{
		ShowAnnotationMessage("Out of fuel!");
		// %%% set mission failure flag here
		return;
	}

	//
	int VesCnt = oapiGetVesselCount();
	for (int i = 0; i < VesCnt; i++)
	{
		// get next vessel in the list
		OBJHANDLE hVes = oapiGetVesselByIndex(i);

		// skip ourselves
		if (hVes == GetHandle())
			continue;

		// get vessel interface
		VESSEL *Intf;
		Intf = oapiGetVesselInterface(hVes);

		// skip vessels with disabled focus
		if (!Intf->GetEnableFocus())
			continue;

		// skip vessels that are landed
		if ((Intf->GetFlightStatus() & 0x01))
			continue;

		// get distance to vessel
		double Dist = GetDistance(hVes);
		
		// skip vessels that are too far away (Moon, geostationary, etc)
		if (Dist > MIN_TGT_DIST)
			continue;

		// if we got there, we have candidate.
		// it this is our first candidate, store it temporarily
		if (hVesTgtCandidate == NULL)
			hVesTgtCandidate = hVes;

		// if we are not ready to pick the next vessel 
		// (did not reach the currently selected one),
		// skip until we find currently selected vessel to reset the flag.
		if (!bCanTakeNextVessel)
		{	
			// found currently selected vessel - _now_ we can select the next one
			if (hCurrTgt == hVes)
				bCanTakeNextVessel = true;

			// still skip current vessel: we need next one, not this one
			continue;
		}

		// select this target
		hCurrTgt = hVes;
		bSelected = true;
		break;
	}

	// if selection was not made for some reason...
	if (!bSelected)
	{
		// ...and there was no current selection...
		if (hCurrTgt == NULL)
		{
			// select first found candidate - if there was one
			if (hVesTgtCandidate != NULL)
				hCurrTgt = hVesTgtCandidate;
		}

		// ...and there was current selection - now select "none"
		else
			hCurrTgt = NULL;
	}

	// get target name and size...
	if (hCurrTgt != NULL)
	{
		VESSEL *IntfTgt;
		IntfTgt = oapiGetVesselInterface(hCurrTgt);
		sprintf(TargetName, IntfTgt->GetName());

		MinDist = (GetSize() + oapiGetSize(hCurrTgt)) / 2;
	}

	// ...or set it to NONE and 0
	else
	{
		sprintf(TargetName, "NONE");
		MinDist = 0;
	}

	// show on-screen message
	char Msg[256] ;
	sprintf(Msg, "Target selected: %s", TargetName);
	ShowAnnotationMessage(Msg);

	// get target size

	// do actual mode switch
	if (hCurrTgt != NULL)
		TrackingState = TS_SELECTED;
	else
		TrackingState = TS_IDLE;

	// clean up any residual output
	sprintf(oapiDebugString(), "");

	// freese the engines
	ShutDownEngines();
}


// for whatever resson...
void Polyot::TargetLost()
{
	// invalidate target handle
	hCurrTgt = NULL;

	// show onscreen message 
	ShowAnnotationMessage("Target lost!");

	// clean up any residual output
	sprintf(oapiDebugString(), "");

	// freese the engines
	ShutDownEngines();

	// switch mode
	TrackingState = TS_IDLE;

	// show mission failure
	if (bMissionHit)
	{
		bOutcomeCriteriaSet = true;
		OutcomeTimer = 7;
	}
}


// for whatever resson...
void Polyot::TargetUnlocked()
{
	if ( TrackingState == TS_KILLING) 
		kmlAddEvent(KML_SEPARATION, "Unlock");

	// show onscreen message 
	ShowAnnotationMessage("Target unlocked!");

	// clean up any residual output
	sprintf(oapiDebugString(), "");

	// freese the engines
	ShutDownEngines();

	// switch mode
	TrackingState = TS_SELECTED;
}


double Polyot::GetDistance(OBJHANDLE hVes)
{
	// if passed handle is invalid, simply return very big number
	if (!oapiIsVessel(hVes))
		return 100000000;

	// find distance
	VECTOR3 rPos;
	oapiGetRelativePos(GetHandle(), hVes, &rPos);
	double Dist = length(rPos);

	//
	return Dist;
}


void Polyot::UpdateTargetParams()
{
	// is target handle still valid?
	if (!oapiIsVessel(hCurrTgt))
	{
		TargetLost();
		return;
	}

	// update mutual position vector and linear distance
	oapiGetRelativePos(GetHandle(), hCurrTgt, &rPos);
	LocalRot(rPos, rPosLoc);
	Dist = length(rPos);

	// too far when not just selected? target lost!
	if ( (Dist > MIN_LOCK_DIST) && (TrackingState >= TS_LOCKING) )
	{
		TargetUnlocked();
		return;
	}

	// missed the target?
	if ( (TrackingState == TS_KILLING) && (rPosLoc.z > 0) )
	{
		TargetUnlocked();
		return;
	}

	// out of fuel?
	if (GetTotalPropellantMass() == 0)
	{
		TargetLost();
		return;
	}

	// too close? collision!
	if (Dist <MinDist) 
		MutualDestruction();

	// close enough for explosion? detonate!
	if ( (Dist <= 100) && (TrackingState == TS_KILLING) )
		MutualDestruction();
}


void Polyot::MutualDestruction()
{
	// before destruction, update mission success criteria, will be displayed in the wreck
	OutcomeCriteria = OC_SUCCESS;

	VESSEL *IntfTgt;
	IntfTgt = oapiGetVesselInterface(hCurrTgt);

	if (IsVessel2M(IntfTgt))
		((VESSEL2M*)IntfTgt)->Break();
	else
		oapiDeleteVessel (hCurrTgt, 0);

	Break();
}


void Polyot::ShowTargetParams()
{
	if (TrackingState == TS_KILLING)
		sprintf(oapiDebugString(),	"Homing on  target: %s:   Dist %.0f m   Vel %.0f m/s   Fuel %.0f %%", 
									TargetName, Dist, Vel, Fuel);
	// debug version, reporting more variables
//	sprintf(oapiDebugString(),	"Homing on  target: %s:   Dist %.0f m   Vel %.0f m/s      x %.0f  y %.0f  z %.0f ", 
//									TargetName, Dist, Vel, rPosLoc.x, rPosLoc.y, rPosLoc.z);

	else if (TrackingState == TS_LOCKED)
		sprintf(oapiDebugString(),	"Target locked: %s:   Distance %.0f m", 
									TargetName, Dist);

	else if (TrackingState == TS_LOCKING)
			sprintf(oapiDebugString(),	"Locking on target: %s:   Distance %.0f m   Pitch %.1f dg   Yaw %.1f dg", 
									TargetName, Dist, NavmodeControlLevel.x*180, NavmodeControlLevel.y*180);

	else
		sprintf(oapiDebugString(), "Target selected: %s:   Distance %.0f m", TargetName, Dist);
}


void Polyot::LockOnTarget()
{
	// still attached
	if (bAttached)
	{
		ShowAnnotationMessage("Targeting operations cannot start until separation from rocket.");
		return;
	}

	// no target
	if (hCurrTgt == NULL)
	{
		ShowAnnotationMessage("Cannot lock: no target. Select target by SPACE key first.");
		return;
	}

	// too far?
	if (Dist > MIN_LOCK_DIST)
	{
		ShowAnnotationMessage("Cannot lock: target is too far. Get to at least 50 km first.");
		return;
	}

	// already locked? unlock
	if (TrackingState >= TS_LOCKING)
		TargetUnlocked();

	// not locked? lock!
	else
	{
		ShowAnnotationMessage("Locking on target...");
		TrackingState = TS_LOCKING;

		// disable any possible residual navmodes
		DeactivateNavmode(NAVMODE_KILLROT);
		DeactivateNavmode(NAVMODE_HLEVEL);
		DeactivateNavmode(NAVMODE_RETROGRADE);
		DeactivateNavmode(NAVMODE_NORMAL);
		DeactivateNavmode(NAVMODE_ANTINORMAL);
		DeactivateNavmode(NAVMODE_HOLDALT);
		DeactivateNavmode(NAVMODE_PROGRADE);
	}
}


void Polyot::TurnToTarget()
{
	GetNavmodeControls();
	SetRotationControls();
}


void Polyot::GetNavmodeControls()
{
	NavmodeControlLevel = GetWantedBodyTrackAttitude(hCurrTgt);

	// useful debug BEFORE inputs will be replaced by signs or modified by stops
/*	VECTOR3 Om;
	GetAngularVel(Om);
	sprintf(oapiDebugString(), "x: %f    y: %f    z: %f    Om.x: %f    Om.y: %f    Om.z: %f", 
		NavmodeControlLevel.x, NavmodeControlLevel.y, NavmodeControlLevel.z,
		Om.x, Om.y, Om.z);
*/
	ControlLevel.x = -NavmodeControlLevel.x;
	ControlLevel.y =  NavmodeControlLevel.y;

	// filter input signals by proximity to the neutral point, modify as needed
	UpdateStopConditions(&ControlLevel);

	// further fiter the signals: for gyro motors, we need just +1 or -1 signs, not actual deviations
	ReplaceControlInputsBySignsWithZero(&ControlLevel);
}


// this function returns the vector with all three attitide sensors deviation data normalized to -1 to +1 range
VECTOR3 Polyot::GetWantedBodyTrackAttitude(OBJHANDLE hBody)
{
	// get global direction for Z axis
	VECTOR3 DirGlobal;
	GetRelativePos(hBody, DirGlobal);
	
	// turn direction around so that Z axis will point TO the target, not AWAY from it 
	DirGlobal = -DirGlobal;

	// turn global direction to local
	VECTOR3 DirLocal; 
	LocalRot(DirGlobal, DirLocal);

	// get pich-yaw from body positional vector
	VECTOR3 AttCotrols = GetPitchYawSensorsDeviation(DirLocal);

	// get roll from horizon, (always current gravity body horizon regardless of target body)
	AttCotrols.z = 0;

	// done!
	return AttCotrols;
}


// this function returns the vector with pitch-yaw attitide sensors deviation data normalized to -1 to +1 range (roll is always 0)
VECTOR3 Polyot::GetPitchYawSensorsDeviation(VECTOR3 DirLocal)
{
	VECTOR3 CurrentOrientLevels = DirLocal;

	// normalize local direction to length of 1
	CurrentOrientLevels /= length(CurrentOrientLevels);

	// get pitch-yaw angles to the given direction
	double Pitch = atan2(-CurrentOrientLevels.y, CurrentOrientLevels.z);		// 0+-pi/2 always, regardless of yaw-roll
	double Yaw = atan2(-CurrentOrientLevels.x, CurrentOrientLevels.z);			// 0+-pi/2 always, regardless of pitch-roll

	// normalize yaw-pitch sensor signals to 0+-1 range
	double PitchLevel = Pitch / PI;
	double YawLevel = Yaw / PI;

	// set orientation vector (roll to 0, as we don't have this data...)
	CurrentOrientLevels.x = PitchLevel;
	CurrentOrientLevels.y = YawLevel;
	CurrentOrientLevels.z = 0; 
	
	// done!
	return CurrentOrientLevels;
}


void Polyot::UpdateStopConditions(VECTOR3* V)
{
	double fx = fabs(V->x); 
	double fy = fabs(V->y); 

	//
	if ( (TrackingState == TS_LOCKING) && (fx < EPS_STOP*2) && (fy < EPS_STOP*2) )
		TrackingState = TS_LOCKED;

	else if ( (TrackingState == TS_LOCKED) && ((fx > EPS_STOP*2) || (fy > EPS_STOP*2)) )
		TrackingState = TS_LOCKING;


	if (fx < EPS_STOP)
		V->x = STOP_ROT;

	if (fy < EPS_STOP)
		V->y = STOP_ROT;
}


void Polyot::ReplaceControlInputsBySignsWithZero(VECTOR3* V)
{
	if (V->x != STOP_ROT)
		V->x = signWithZero(V->x);

	if (V->y != STOP_ROT)
		V->y = signWithZero(V->y);

	if (V->z != STOP_ROT)
		V->z = signWithZero(V->z);
}

void Polyot::SetRotationControls()
{
	// get old omegas
	VECTOR3 omegas;
	GetAngularVel(omegas);

	// update omegas
	SetControllChannel(ControlLevel.x, omegas.x, thgPitchDown, thgPitchUp);
	SetControllChannel(ControlLevel.y, omegas.y, thgYawRight, thgYawLeft);
	ControlLevel.z = STOP_ROT;
	SetControllChannel(ControlLevel.z, omegas.z, thgRollLeft, thgRollRight);
}


void Polyot::SetControllChannel(double &ControlLevel, double Omega, THGROUP_HANDLE thgNeg, THGROUP_HANDLE thgPos)
{
	// spinning too fast? gradually slow down to nominal before doing anything else...
	if (fabs(Omega) > OMEGA_NOM)
		ControlLevel = -signWithZero(Omega);

	// within nominal window and commanded to stop? do it...
	else if (ControlLevel == STOP_ROT)
		ControlLevel = -signWithZero(Omega);

	// set thrusters
	if (ControlLevel < 0)
	{
		SetThrusterGroupLevel(thgNeg, -ControlLevel);
		SetThrusterGroupLevel(thgPos, 0);
	}

	else if (ControlLevel > 0)
	{
		SetThrusterGroupLevel(thgNeg, 0);
		SetThrusterGroupLevel(thgPos, ControlLevel);
	}

	else
	{
		SetThrusterGroupLevel(thgNeg, 0);
		SetThrusterGroupLevel(thgPos, 0);
	}
}


void Polyot::ShutDownEngines()
{
	SetThrusterGroupLevel(thgPitchDown, 0);
	SetThrusterGroupLevel(thgPitchUp, 0);
	SetThrusterGroupLevel(thgYawLeft, 0);
	SetThrusterGroupLevel(thgYawRight, 0);
	SetThrusterGroupLevel(thgRollLeft, 0);
	SetThrusterGroupLevel(thgRollRight, 0);

	SetThrusterGroupLevel(thgFwd, 0);
	SetThrusterGroupLevel(thgUp, 0);
	SetThrusterGroupLevel(thgDown, 0);
	SetThrusterGroupLevel(thgLeft, 0);
	SetThrusterGroupLevel(thgRight, 0);
}



void Polyot::AbortKilling()
{
	TrackingState = TS_SELECTED;
	ShutDownEngines();
	ShowAnnotationMessage("Target destruction aborted.");
	kmlAddEvent(KML_SEPARATION, "Abort");
}


void Polyot::KillTarget()
{
	// already killing? abort!
	if (TrackingState == TS_KILLING)
		AbortKilling();

	// still attached
	else if (bAttached)
		ShowAnnotationMessage("Targeting operations cannot start until separation from rocket.");

	// no target?
	else if (TrackingState < TS_SELECTED)
		ShowAnnotationMessage("Cannot destroy: no target. Select target by SPACE key first.");

	// locked?
	else if (TrackingState < TS_LOCKING)
		ShowAnnotationMessage("Cannot destroy: target not locked. Lock target by L key first.");

	// really locked?
	else if (TrackingState < TS_LOCKED)
		ShowAnnotationMessage("Cannot destroy: wait for target lock to complete!");

	// OK!
	else
	{
		TrackingState = TS_KILLING;
		ShowAnnotationMessage("Target destruction mode activated!");
		kmlAddEvent(KML_SEPARATION, "Aquisition");

		ShutDownEngines();
		SetThrusterGroupLevel(thgFwd, 1);

		// disable any possible residual navmodes
		DeactivateNavmode(NAVMODE_KILLROT);
		DeactivateNavmode(NAVMODE_HLEVEL);
		DeactivateNavmode(NAVMODE_RETROGRADE);
		DeactivateNavmode(NAVMODE_NORMAL);
		DeactivateNavmode(NAVMODE_ANTINORMAL);
		DeactivateNavmode(NAVMODE_HOLDALT);
		DeactivateNavmode(NAVMODE_PROGRADE);
	}
}


void Polyot::SlideToTarget()
{
	// find velocity
	oapiGetRelativeVel(GetHandle(), hCurrTgt, &rVel);
	LocalRot(rVel, rVelLoc);
	Vel = length(rVel);

	// find fuel
	double Mass = GetPropellantMass(MainTank);
	double MaxMass = GetPropellantMaxMass(MainTank);
	Fuel = Mass/MaxMass*100;

	// don't approach too fast... 
	if ( (Vel > 500) && (rVelLoc.z > 0) )
		SetThrusterGroupLevel(thgFwd, 0);
	else
		SetThrusterGroupLevel(thgFwd, 1);

		// pitch

	if (rPosLoc.y < -5)
	{
		SetThrusterGroupLevel(thgUp, 1);
		SetThrusterGroupLevel(thgDown, 0);
	}

	else if (rPosLoc.y > 5)
	{
		SetThrusterGroupLevel(thgUp, 0);
		SetThrusterGroupLevel(thgDown, 1);
	}

	else
	{
		SetThrusterGroupLevel(thgUp, 0);
		SetThrusterGroupLevel(thgDown, 0);
	}


		// yaw

	if (rPosLoc.x > 5)
	{
		SetThrusterGroupLevel(thgLeft, 1);
		SetThrusterGroupLevel(thgRight, 0);
	}

	else if (rPosLoc.x < -5)
	{
		SetThrusterGroupLevel(thgLeft, 0);
		SetThrusterGroupLevel(thgRight, 1);
	}

	else
	{
		SetThrusterGroupLevel(thgLeft, 0);
		SetThrusterGroupLevel(thgRight, 0);
	}
}





int Polyot::GetVesselMessage(IntVesselMessage VM, int Value)
{
	if (VM == VM_PAYLOAD_TYPE)
		return PAYLOAD_POLYOT;

	// unsupported message
	return VESSEL2M::GetVesselMessage(VM, Value);
}


void Polyot::Break()
{
	if (bMissionHit)
		bOutcomeCriteriaSet = true;

	VESSEL2M::Break();
}



// ========================================================================

DLLCLBK VESSEL *ovcInit(OBJHANDLE hvessel, int flightmodel)
{
	return new Polyot(hvessel, flightmodel);
}

DLLCLBK void ovcExit(VESSEL *vessel)
{
	if (vessel)	delete (Polyot*) vessel;
}



