// Mini-library of useful contants, structures and functions. 
// Gets included in both VESSEL2M and Wreck headers.

#pragma warning(disable:4996)   // disable CRT string functions security warnings
#pragma once   

#include "orbitersdk.h"


// forward reference of used classes 
class VESSEL2M;
class kmlWriter;

// useful zero and directional unit vectors
const VECTOR3 xyz0 = {0, 0, 0};
const VECTOR3 xplus = {1, 0, 0};
const VECTOR3 yplus = {0, 1, 0};
const VECTOR3 zplus = {0, 0, 1};
const VECTOR3 xminus = {-1,  0,  0};
const VECTOR3 yminus = { 0, -1,  0};
const VECTOR3 zminus = { 0,  0, -1};

// useful constants for angles
const double COS45D = 0.707107;

// earth parameters
const double RADIUS_E = 6.371e6;				// Earth radius, m
const double MASS_E = 5.98e24;					// Earth mass, kg
const double K_E = MASS_E * GGRAV;				// Earth K, kg
const double OMEGA_E = 2*PI/3600/24;			// Earth Omega, rad/s

// meaningful named consts for attachment handling
const bool TOPARENT = true; //default
const bool TOCHILD = false;
const bool LOOSE = true; // deafult 
const bool NOTLOOSE = false; 


// ======= Common definitions, shared between Wreck and VESSEL2M =======
// focus switching direction in the vessels list, backward or forward
const static enum focus_Dirs {FOCUS_PREVIOUS, FOCUS_NEXT};

// macros: quckly check if VESSEL interface is Wreck or VESSEL2M 
#define IsWreck(Intf)	(((R7_wreck*)Intf)->VesselSignature == 'W')
#define IsVessel2M(Intf)	(((R7_wreck*)Intf)->VesselSignature == 'M')
#define IsWreckOrVessel2M(Intf)	(IsWreck(Intf) || IsVessel2M(Intf))
// ======= end of Common definitions, shared between Wreck and VESSEL2M =======


// structure to store camera information
typedef struct{
	VECTOR3 Off;
	VECTOR3 Dir;
	double Tilt;
	bool bEnabled;
}CAMERA;

// types of mission outcomes
enum OUTCOME_CRITERIA {
	OC_FOREIGN,			// use it to display an outcome passed from a different object
	OC_FAIL,
	OC_SUCCESS,
	OC_PARTIAL
}; 

// structure to keeep all the needed data about the mission outcome criteria. So far, only text message here...
typedef struct {
	char OutcomeMessage[255];
} OUTCOME_OBJECT;








	// Various helper functions


// ======================
// date-time functions section
const static long OCT5_1582  = 2299160L; // shifted 15-Oct-1582
const static long OCT14_1582 = 2299169L; // shifted 4-Oct-1582
const static long JAN1_1     = 1721423L;

const static long YEAR = 365;
const static long FOUR_YEARS = 1461;
const static long CENTURY = 36524L;
const static long FOUR_CENTURIES = 146097L;

static int DaysSoFar[][13] =
{
	{0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334, 365},
	{0, 31, 60, 91, 121, 152, 182, 213, 244, 274, 305, 335, 366}
};

static char *DayOfWeek[7] = { "Sun", "Mon", "Tue", "Wed", "Thu", "Fry", "Sat" };
static char *MonthOfYear[12] = { "Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };


static void MJDtoSystemTime(double MJD, LPSYSTEMTIME lpSystemTime)
{
	//Day starts at midnight
	MJD+=2400000;

	//Split days from fraction
	long julian=(long) MJD;
	double fraction=MJD-julian;

	//Calculate date
	long z,y;
	short m,d;
	int lp;

	z = julian+1;
	if (z >= OCT5_1582)
	{
		z -= JAN1_1;
		z = z + (z/CENTURY) - (z/FOUR_CENTURIES) -2;
		z += JAN1_1;
	}

	z = z - ((z-YEAR) / FOUR_YEARS); // Remove leap years before current year
	y = z / YEAR;
	d = (short) (z - (y * YEAR));
	y = y - 4712; // our base year in 4713BC

	if (y < 1) 
		y--;

	lp = !(y & 3); // lp = 1 if this is a leap year.
	
	if (d==0)
	{
		y--;
		d = (short) (YEAR + lp);
	}

	m = (short) (d/30); // guess at month
	while (DaysSoFar[lp][m] >=d) m--; // correct guess
	lpSystemTime->wDay = (unsigned short) (d - DaysSoFar[lp][m]);
	lpSystemTime->wMonth = (unsigned short) m;
	lpSystemTime->wYear = (unsigned short) y;

	//Calculate time
	lpSystemTime->wHour=(unsigned short)(fraction*=24);
	fraction-=lpSystemTime->wHour;
	lpSystemTime->wMinute=(unsigned short)(fraction*=60);
	fraction-=lpSystemTime->wMinute;
	lpSystemTime->wSecond=(unsigned short)(fraction*=60);
	fraction-=lpSystemTime->wSecond;
	lpSystemTime->wMilliseconds=(unsigned short)(fraction*=1000);

	//Calculate day of week
	lpSystemTime->wDayOfWeek=(unsigned short)( (julian+2) % 7 );
}


static double SystemTimeToMJD(LPSYSTEMTIME lpSystemTime)
{
	int a;
	int work_year=lpSystemTime->wYear;
	long j;
	int lp;
	double julian;

	if (work_year < 0) 
		work_year++; // correct for negative year (-1 = 1BC = year 0)

	lp = !(work_year & 3); // lp = 1 if this is a leap year.
	j = ((work_year-1) / 4) + // ALL leap years
		DaysSoFar[lp][lpSystemTime->wMonth] + // days in this year
		lpSystemTime->wDay + // day in this month
		(work_year * 365L) + // days in years
		JAN1_1 + // first january in year 1
		-366; // adjustments

	// deal with Gregorian calendar
	if (j >= OCT14_1582)
	{
		a = (int)(work_year/100);
		j = j+ 2 - a + a/4; // skip days which didn't exist.
	}

	julian=j-2400000.5; // convert to modified julian days
	
	return julian + // return julian days + time fraction
		(((lpSystemTime->wMilliseconds/1000 +
		lpSystemTime->wSecond)/60 +
		lpSystemTime->wMinute)/60 +
		lpSystemTime->wHour)/24;
}

//
inline int GetScenarioYear()
{
	SYSTEMTIME mjddate;
	MJDtoSystemTime(oapiGetSimMJD(),&mjddate);
	return mjddate.wYear;
}

// end of date-time functions section
// ======================


// =============== Geodetic section =================

// rad, rad, rad, rad. rad, rad
// azimuth 0 to 2pi
inline void DistanceAzimuthFromCoordPairRad(double Lat1, double Lon1, double Lat2, double Lon2, double* pDistance, double* pAzimuth)
{
	double t = sin(Lat1)*sin(Lat2) + cos(Lat1)*cos(Lat2)*cos(Lon2-Lon1);
	if (t > 1)
		t = 1;
	else if (t < -1)
		t = -1;

	*pDistance = acos(t);
	
	t = (sin(Lat2) - sin(Lat1)*cos(*pDistance))/(sin(*pDistance)*cos(Lat1));
	if (t > 1)
		t = 1;
	else if (t < -1)
		t = -1;

	*pAzimuth = acos(t);
	
	if (Lon2 < Lon1) 
		*pAzimuth = 2*PI - *pAzimuth ;
}


// rad, rad, rad, rad. m, dg
inline void DistanceAzimuthFromCoordPairKmDg(double Lat1, double Lon1, double Lat2, double Lon2, double* pDistance, double* pAzimuth)
{
	DistanceAzimuthFromCoordPairRad(Lat1, Lon1, Lat2, Lon2, pDistance, pAzimuth);
	*pDistance *= RADIUS_E;
	*pAzimuth = *pAzimuth*DEG;
}



// rad, rad, rad, rad. rad, rad
inline void CoordsFromDistanceAzimuthRad(double Lat1, double Lon1, double Distance, double Azimuth, double* pLat2, double* pLon2)
{
	double t = (cos(Azimuth)*sin(Distance)*cos(Lat1)) + sin(Lat1)*cos(Distance);
	if (t > 1)
		t = 1;
	else if (t < -1)
		t = -1;

	*pLat2 = asin(t);

	t = (cos(Distance) - sin(Lat1)*sin(*pLat2)) / (cos(Lat1)*cos(*pLat2));
	if (t > 1)
		t = 1;
	else if (t < -1)
		t = -1;

	double DeltaLon = acos(t);

	if (Azimuth < (PI))
	  *pLon2 = Lon1 + DeltaLon;
	else
	  *pLon2 = Lon1 - DeltaLon;
}


// ============== Helper section =============

inline MATRIX3 MakeTranslationMatrix( const VECTOR3 &X, const VECTOR3 &Y,const VECTOR3 &Z )
{
	MATRIX3 TM;

	TM.m11 = X.x;
	TM.m12 = X.y;
	TM.m13 = X.z;
	
	TM.m21 = Y.x;
	TM.m22 = Y.y;
	TM.m23 = Y.z;

	TM.m31 = Z.x;
	TM.m32 = Z.y;
	TM.m33 = Z.z;

	return TM;
}

//
inline bool VectorEqual(VECTOR3 V1, VECTOR3 V2)
{
	return ( (V1.x == V2.x) && (V1.y == V2.y) && (V1.z == V2.z) );
}

//
#define bHaveFocus (GetHandle() == oapiGetFocusObject())

//
inline double DRand(double from, double  span)
{
	return ((double)rand() / (double)RAND_MAX) * span + from;
}

//
inline int IRand(int from, int  span)
{
	return (int)( (double)rand() / (double)RAND_MAX * (double)span ) + from;
}



// returns -1, 0 or 1
inline int	signWithZero(double x)
{
	if (x > 0)
		return 1;
	else if (x < 0)
		return -1;
	else
		return 0;
}


// returns -1 or 1 only; 0 input returns 1
inline int	signNoZero(double x)
{
	if (x < 0)
		return -1;
	else
		return 1;
}


// input: +-0..180 deg, returns 0+-1
inline double AngleToSignal(double Deg)
{
	if (Deg > 180)
		Deg -= 360;
	if (Deg < -180)
		Deg += 360;
	return Deg/180;
}


// map any value to 0+-MaxValue range 
inline double GetNormalizedValue(double inValue, double MaxValue)
{
	if (inValue > MaxValue)
		inValue = MaxValue;
	else if (inValue < -MaxValue)
		inValue = -MaxValue;
	return inValue/MaxValue;
}


// ============ Orbital section ================

// eccentricity vector, AKA Laplace vector normalized by Mu, points to pericenter
inline VECTOR3 VEccentricity(double Mu, const VECTOR3 VPos, const VECTOR3 VVel)
{
	VECTOR3 VE, VPart1, VPart2, M;
	double Pos;

	M = crossp(VPos, VVel);
	VPart1 = crossp(VVel, M) / Mu;

	Pos = length(VPos);
	VPart2 = VPos / Pos; 

	VE = VPart1 - VPart2;

	return VE;
}


inline double TrueAnomaly(VECTOR3 &VE, VECTOR3 &VPos, VECTOR3 &VVel)
{
	double E = length(VE);
	double Pos = length(VPos);

	double f = acos((VPos.x * VE.x + VPos.y * VE.y + VPos.z * VE.z) / (Pos * E));

	if ( dotp(VPos, VVel) < 0)
		f = 2*PI - f;

	return f;
}


inline double ArgOfPeri(VECTOR3 &VNormal, VECTOR3 &VE)
{
	double Normal = length(VNormal);
	double E = length(VE);

	double omega = acos((VNormal.x * VE.x + VNormal.y * VE.y + VNormal.z * VE.z) / (Normal * E));

	if (VE.z < 0) 
		omega = 2*PI - omega;

	return omega;
}


inline double EccentricAnomaly(double E, double f)
{
	double ea = 2 * atan(sqrt((1-E)/(1+E)) * tan(f/2));
	
	if (ea < 0)
		ea = 2*PI + ea;

	return ea;
}


inline double MeanAnomaly(double EccentricAnomaly, double E)
{
	return EccentricAnomaly - E * sin(EccentricAnomaly);
}


inline double TimeSincePeriapsis(double MeanAno, double T)
{
	return MeanAno / (2*PI) * T;
}



inline double TimeOfTrueAnomaly(double TrueAno, double E, double T)
{
	double ea = EccentricAnomaly(E, TrueAno);
	double ma = MeanAnomaly(ea, E);
	double tm = TimeSincePeriapsis(ma, T);
	return tm;
}


