// ==============================================================
// ORBITER vessel module: LOK Lunar Orbiter
// ==============================================================
// Smoke particle effects and C++ class conversion by Dave Rowbotham
//
#include "Orbitersdk.h"
#include "cstdio"

#include "lok_zond.h"

const VECTOR3 OFS_SERVICE = { -0.15-0.15,0.25,0};
const VECTOR3 OFS_CHUTE = {0.15-0.15,0,-4.75-2.0};

 

const double MAX_ATT_LOK = 1800;
const double MAX_ATT_RCS = 67;
const double THRUST_LOK_VAC   = 3388*9.81;
const double THRUST_RETRO_VAC   = 1388*9.81;
const double ISP_LOK_VAC  = 314*9.81;
const double ISP_RETRO_VAC  = 314*9.81;
const double LOK_PROP_MASS = 18300;
const double RCS_PROP_MASS = 30;



double LiftCoeff (double aoa)
{
const int nlift = 9;
static const double AOA[nlift] =
{-180*RAD,-155*RAD,-120*RAD,-20*RAD,0*RAD,20*RAD,120*RAD,155*RAD,180*RAD};
static const double CL[nlift]  = {0.12,  0.21,       0,      0,    0,     0,      0,      0.21,0.12};
static const double SCL[nlift] = {(CL[1]-CL[0])/(AOA[1]-AOA[0]),
(CL[2]-CL[1])/(AOA[2]-AOA[1]),
                              (CL[3]-CL[2])/(AOA[3]-AOA[2]),
(CL[4]-CL[3])/(AOA[4]-AOA[3]),
  (CL[5]-CL[4])/(AOA[5]-AOA[4]), (CL[6]-CL[5])/(AOA[6]-AOA[5]),
  (CL[7]-CL[6])/(AOA[7]-AOA[6]), (CL[8]-CL[7])/(AOA[8]-AOA[7])};
int i;
for (i = 0; i < nlift-1 && AOA[i+1] < aoa; i++);
return CL[i] + (aoa-AOA[i])*SCL[i];
}

void VLiftCoeff (double aoa, double M, double Re, double *cl, double *cm, double *cd)
{
	int i;
	const int nabsc = 9;
//	static const double AOA[nabsc] = {-180*RAD,-155*RAD,-120*RAD, -20*RAD, 0*RAD,20*RAD,120*RAD,155*RAD,180*RAD};
//	static const double CL[nabsc]  = {0.12,  0.21,       0,      0,    0,     0,      0,      0.21,0.12};
	static const double AOA[nabsc] = {-180*RAD,-135*RAD,-90*RAD, -45*RAD, 0*RAD,45*RAD,90*RAD,135*RAD,180*RAD};
	static const double CL[nabsc]  = {       0,    -0.3,       0,   -0.2,     0,  +0.2,     0,   +0.3,      0};
	for (i = 0; i < nabsc-1 && AOA[i+1] < aoa; i++);
	double f = (aoa-AOA[i]) / (AOA[i+1]-AOA[i]);
	*cl = CL[i] + (CL[i+1]-CL[i]) * f;  // aoa-dependent lift coefficient
	*cm = 0;
	double s = cos(aoa);
	*cd = 0.015 + 0.4*s*s + oapiGetInducedDrag (*cl, 1.5, 0.7) + oapiGetWaveDrag (M, 0.75, 1.0, 1.1, 0.04);
}

// 2. horizontal lift component (vertical stabilisers and body)

void HLiftCoeff (double beta, double M, double Re, double *cl, double *cm, double *cd)
{
	int i;
	const int nabsc = 9;
	static const double BETA[nabsc] = {-180*RAD,-135*RAD,-90*RAD,-45*RAD, 0, 45*RAD,90*RAD,135*RAD,180*RAD};
	static const double CL[nabsc]   = {       0,    +0.3,      0,   -0.2, 0,  +0.2,     0,   -0.3,      0};
	for (i = 0; i < nabsc-1 && BETA[i+1] < beta; i++);
	*cl = CL[i] + (CL[i+1]-CL[i]) * (beta-BETA[i]) / (BETA[i+1]-BETA[i]);
	*cm = 0.0;
	double s = cos(beta);
	*cd = 0.015 + 0.4*s*s + oapiGetInducedDrag (*cl, 1.5, 0.6) + oapiGetWaveDrag (M, 0.75, 1.0, 1.1, 0.04);
//	sprintf(oapiDebugString(), "beta = %.2f   cl = %.2f", beta*DEG, *cl );
}


LOK_ZOND::LOK_ZOND( OBJHANDLE hObj, int fmodel ) : 
   VESSEL( hObj, fmodel )
{

	bManualSeparate = false;
	stage_sep = 0;
	stage = 0;
	status = 0;

	refcount = 0;
	hLOK2 = oapiLoadMeshGlobal ("n1lok2");
	hLOK3 = oapiLoadMeshGlobal ("n1lok3");
	hLOKchute = oapiLoadMeshGlobal ("n1chute");

	ph_lok = NULL;
	ph_rcs = NULL;
	ph_srm = NULL;
}


LOK_ZOND::~LOK_ZOND()
{

}

void LOK_ZOND::SetAttControlsLOK(VESSEL *vessel)
{
	THRUSTER_HANDLE th_rot[2];
	th_rot[0] = vessel->CreateThruster (_V(0,-0.2,-2.0), _V(0,-1,0), MAX_ATT_LOK , ph_lok, ISP_LOK_VAC);
	th_rot[1] = vessel->CreateThruster (_V(0,0.75,5.25), _V(0,1,0), MAX_ATT_LOK , ph_lok, ISP_LOK_VAC);
	vessel->AddExhaust (th_rot[0], 1.5, 0.05, _V( 0,  0.75,-1.0  ), _V(0, 1.0,0));
	vessel->AddExhaust (th_rot[1], 1.5, 0.05, _V( 0,  -0.5, 1.75  ), _V(0,-1.0,0));
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_PITCHUP);

	th_rot[0] = vessel->CreateThruster (_V(0,-0.2,-5.25), _V(0,1,0), MAX_ATT_LOK , ph_lok, ISP_LOK_VAC);
	th_rot[1] = vessel->CreateThruster (_V(0,-0.75,5.25), _V(0,1,0), MAX_ATT_LOK , ph_lok, ISP_LOK_VAC);
	vessel->AddExhaust (th_rot[0], 1.75, 0.05, _V( 0,  -0.6,-1.0 ), _V(0, -1.0,0));
	vessel->AddExhaust (th_rot[1], 1.5, 0.05, _V( 0,  -0.5, 1.75  ), _V(0,-1.0,0));
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_UP);

	th_rot[0] = vessel->CreateThruster (_V(0,-0.75,5.25), _V(0,-1,0), MAX_ATT_LOK , ph_lok, ISP_LOK_VAC);
	th_rot[1] = vessel->CreateThruster (_V(0,0.35,-2.0), _V(0, 1,0), MAX_ATT_LOK , ph_lok, ISP_LOK_VAC);
	vessel->AddExhaust (th_rot[1], 1.5, 0.05, _V( 0,  -0.6,-1.0  ), _V(0, -1.0,0));
	vessel->AddExhaust (th_rot[0], 1.5, 0.05, _V( 0,  0.5, 1.75  ), _V(0,1.0,0));
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_PITCHDOWN);

	th_rot[0] = vessel->CreateThruster (_V(0,-0.2,-5.25), _V(0,-1,0), MAX_ATT_LOK , ph_lok, ISP_LOK_VAC);
	th_rot[1] = vessel->CreateThruster (_V(0,-0.75,5.25), _V(0,-1,0), MAX_ATT_LOK , ph_lok, ISP_LOK_VAC);
	vessel->AddExhaust (th_rot[0], 1.75, 0.05, _V( 0,  0.75,-1.0 ), _V(0, 1.0,0));
	vessel->AddExhaust (th_rot[1], 1.5, 0.05, _V( 0,  0.6, 1.75  ), _V(0,1.0,0));
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_DOWN);

	th_rot[0] = vessel->CreateThruster (_V(-2,1.25,-0.3), _V(0,-1,0), MAX_ATT_LOK , ph_lok, ISP_LOK_VAC);
	th_rot[1] = vessel->CreateThruster (_V(2, 1.25,-0.3), _V(0,1,0), MAX_ATT_LOK , ph_lok, ISP_LOK_VAC);
	vessel->AddExhaust (th_rot[1], 1.5, 0.05, _V( -1.15, 0.0,0.20  ), _V(0, 1.0,0));
	vessel->AddExhaust (th_rot[0], 1.5, 0.05, _V(  1.15, 0.0,0.20  ), _V(0,-1.0,0));
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_BANKLEFT);

	th_rot[0] = vessel->CreateThruster (_V(2,0.0,0.0), _V(-1,0,0), MAX_ATT_LOK , ph_lok, ISP_LOK_VAC);
	vessel->AddExhaust (th_rot[0], 1.5, 0.05, _V(  1.15, 0.0,0.20 ), _V(1,0,0));
	vessel->CreateThrusterGroup (th_rot, 1, THGROUP_ATT_LEFT);

	th_rot[0] = vessel->CreateThruster (_V(-2,1.25,-0.3), _V(0, 1,0), MAX_ATT_LOK , ph_lok, ISP_LOK_VAC);
	th_rot[1] = vessel->CreateThruster (_V(2,1.25,-0.3), _V(0,-1,0), MAX_ATT_LOK , ph_lok, ISP_LOK_VAC);
	vessel->AddExhaust (th_rot[1], 1.5, 0.05, _V( -1.15, 0.0,0.20  ), _V(0,-1.0,0));
	vessel->AddExhaust (th_rot[0], 1.5, 0.05, _V(  1.15, 0.0,0.20  ), _V(0, 1.0,0));
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_BANKRIGHT);

	th_rot[0] = vessel->CreateThruster (_V(2,0.0,0.0), _V(1,0,0), MAX_ATT_LOK , ph_lok, ISP_LOK_VAC);
	vessel->AddExhaust (th_rot[0], 1.5, 0.05, _V(  -1.15, 0.0,0.20 ), _V(-1,0,0));
	vessel->CreateThrusterGroup (th_rot, 1, THGROUP_ATT_RIGHT);

	th_rot[0] = vessel->CreateThruster (_V(0,0, 4), _V( 1,0,0), MAX_ATT_LOK , ph_lok, ISP_LOK_VAC);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-4), _V(-1,0,0), MAX_ATT_LOK , ph_lok, ISP_LOK_VAC);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_YAWRIGHT);
	th_rot[0] = vessel->CreateThruster (_V(0,0, 4), _V(-1,0,0), MAX_ATT_LOK , ph_lok, ISP_LOK_VAC);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-4), _V( 1,0,0), MAX_ATT_LOK , ph_lok, ISP_LOK_VAC);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_YAWLEFT);

	th_rot[0] = vessel->CreateThruster (_V(0.0,0.0,-4.0), _V(0.0,0,1.0), MAX_ATT_LOK , ph_lok, ISP_LOK_VAC);
	vessel->CreateThrusterGroup (th_rot, 1, THGROUP_ATT_FORWARD);

	th_rot[0] = vessel->CreateThruster (_V(0.0,0.0,4.0), _V(0.0,0,-1.0), MAX_ATT_LOK , ph_lok, ISP_LOK_VAC);
	vessel->CreateThrusterGroup (th_rot, 1, THGROUP_ATT_BACK);
}

void LOK_ZOND::SetAttControlsRCS(VESSEL *vessel)
{
	THRUSTER_HANDLE th_rot[2];
	th_rot[0] = vessel->CreateThruster (_V(0,0, 2), _V(0, 1,0), MAX_ATT_RCS , ph_rcs, ISP_LOK_VAC);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-2), _V(0,-1,0), MAX_ATT_RCS , ph_rcs, ISP_LOK_VAC);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_PITCHUP);
	th_rot[0] = vessel->CreateThruster (_V(0,0, 2), _V(0,-1,0), MAX_ATT_RCS , ph_rcs, ISP_LOK_VAC);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-2), _V(0, 1,0), MAX_ATT_RCS , ph_rcs, ISP_LOK_VAC);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_PITCHDOWN);
	th_rot[0] = vessel->CreateThruster (_V( 1.5,0,0), _V(0, 1,0), MAX_ATT_RCS , ph_rcs, ISP_LOK_VAC);
	th_rot[1] = vessel->CreateThruster (_V(-1.5,0,0), _V(0,-1,0), MAX_ATT_RCS , ph_rcs, ISP_LOK_VAC);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_BANKLEFT);
	th_rot[0] = vessel->CreateThruster (_V( 1.5,0,0), _V(0,-1,0), MAX_ATT_RCS , ph_rcs, ISP_LOK_VAC);
	th_rot[1] = vessel->CreateThruster (_V(-1.5,0,0), _V(0, 1,0), MAX_ATT_RCS , ph_rcs, ISP_LOK_VAC);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_BANKRIGHT);
	th_rot[0] = vessel->CreateThruster (_V(0,0, 1), _V( 1,0,0), MAX_ATT_RCS , ph_rcs, ISP_LOK_VAC);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-1), _V(-1,0,0), MAX_ATT_RCS , ph_rcs, ISP_LOK_VAC);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_YAWRIGHT);
	th_rot[0] = vessel->CreateThruster (_V(0,0, 1), _V(-1,0,0), MAX_ATT_RCS , ph_rcs, ISP_LOK_VAC);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-1), _V( 1,0,0), MAX_ATT_RCS , ph_rcs, ISP_LOK_VAC);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_YAWLEFT);
}


void LOK_ZOND::SetZondStage(VESSEL *vessel)
{
	PARTICLESTREAMSPEC contrail = {
		0, 8.0, 5, 150, 0.3, 4.0, 4, 3.0, PARTICLESTREAMSPEC::DIFFUSE,
		PARTICLESTREAMSPEC::LVL_SQRT, 0, 1,
		PARTICLESTREAMSPEC::ATM_PLOG, 1e-4, 1
	};
	PARTICLESTREAMSPEC exhaust_main = {
		0, 2.0, 20, 150, 0.1, 0.2, 16, 1.0, PARTICLESTREAMSPEC::EMISSIVE,
		PARTICLESTREAMSPEC::LVL_SQRT, 0, 1,
		PARTICLESTREAMSPEC::ATM_PLOG, 1e-5, 0.1
	};
	vessel->SetSize (7);
	vessel->SetCOG_elev (3.5);
	vessel->SetEmptyMass (5500.0);
//	vessel->ShiftCentreOfMass(_V(0,0,5));
	vessel->SetPMI (_V(5.5,5.5,0.75));
	vessel->SetCrossSections (_V(16,15,5.75));
	vessel->SetCW (0.1, 0.3, 1.4, 1.4);
	vessel->SetRotDrag (_V(0.7,0.7,1.2));
	vessel->SetPitchMomentScale (0);
	vessel->SetBankMomentScale (0);
	vessel->SetLiftCoeffFunc (0);
    vessel->ClearMeshes();
//	vessel->SetDockParams (_V(0.2,0,2.55), _V(0,0,1), _V(0,1,0));
	VECTOR3 mesh_dir=OFS_SERVICE;
	vessel->AddMesh (hLOK3, &mesh_dir);	
	mesh_dir=_V(0.175-0.15,-0.11,1.775);
	vessel->AddMesh (hLOK2, &mesh_dir);
	if (!ph_lok) ph_lok  = vessel->CreatePropellantResource (LOK_PROP_MASS); 
	vessel->SetDefaultPropellantResource (ph_lok); 
	vessel->ClearThrusterDefinitions();
	th_lok[0] = vessel->CreateThruster (_V(0,0,-1.05), _V(0,0,1), THRUST_LOK_VAC, ph_lok, ISP_LOK_VAC);
	vessel->AddExhaustStream (th_lok[0], _V(0,0,-1.05), &contrail);
	vessel->AddExhaustStream (th_lok[0], _V(0,0,-1.05), &exhaust_main);
	thg_lok = vessel->CreateThrusterGroup (th_lok, 1, THGROUP_MAIN);
	th_retro[0] = vessel->CreateThruster (_V(0,0,1.05), _V(0,0,-1), THRUST_RETRO_VAC, ph_lok, ISP_RETRO_VAC);
	thg_retro = vessel->CreateThrusterGroup (th_retro, 1, THGROUP_RETRO);
	vessel->SetThrusterGroupLevel (thg_lok,0.0);
	vessel->AddExhaust (th_lok[0], 2.0, 0.25, _V(0.175-0.15,0.05,-1.05), _V(0,0,-1));
	SetAttControlsLOK(vessel);
	vessel->EnableTransponder(true);
	status=0;
}

void LOK_ZOND::SetReentryStage(VESSEL *vessel)
{
/*
	vessel->SetSize (2);
	vessel->SetCOG_elev (3.5);
	vessel->SetEmptyMass (2500.0);
	vessel->SetPMI (_V(2.5,2.5,1.25));
	vessel->SetCrossSections (_V(6,6,2));
	vessel->SetCW (0.1, 0.75, 0.3, 0.3);
	vessel->SetRotDrag (_V(0.7,0.7,1.2));
	vessel->SetPitchMomentScale (0);
	vessel->SetBankMomentScale (0);
	vessel->SetLiftCoeffFunc (LiftCoeff); 
*/
	vessel->SetSize (2);
	vessel->SetEmptyMass (2500.0);
//	vessel->SetPMI (_V(2,2,0.55);
	vessel->SetPMI (_V(0.35,0.35,0.3));
//	vessel->SetCrossSections (_V(8.75,8.75,7.79));
	vessel->SetCrossSections (_V(2.5,2.5,2.2));
	vessel->SetCW (0.1, 0.75, 0.3, 0.3);
	vessel->SetRotDrag (_V(0.05,0.05,0.05));
	if (vessel->GetFlightModel() >= 1) 
	{
//		vessel->SetPitchMomentScale (-5e-5);
//		vessel->SetBankMomentScale (-5e-5);
		vessel->CreateAirfoil(LIFT_VERTICAL, _V(0,0,0), VLiftCoeff, 1, 2.2, 1);
		vessel->CreateAirfoil(LIFT_HORIZONTAL, _V(0,0,0), HLiftCoeff, 1, 2.2, 1);
//		vessel->SetLiftCoeffFunc (0);
    }
	vessel->ClearMeshes();
	vessel->ShiftCentreOfMass (_V(0,0.0,2.0));
	vessel->SetTouchdownPoints (_V(-1,-1,-1.5-2.0),_V(0,1,-1.5-2.0),_V(1,-1,-1.5-2.0));
	VECTOR3 mesh_dir=_V(0.175-0.15,-0.11,1.775-2.0);
	vessel->AddMesh (hLOK2, &mesh_dir);
	vessel->DelPropellantResource (ph_lok);
	if (!ph_rcs) ph_rcs  = vessel->CreatePropellantResource (RCS_PROP_MASS);
	vessel->SetDefaultPropellantResource (ph_rcs); 
	vessel->ClearThrusterDefinitions();
	SetAttControlsRCS(vessel);
	status=1;
}

void LOK_ZOND::DeployMain(VESSEL *vessel)
{

	PARTICLESTREAMSPEC oms_exhaust = {
	0, 8.0, 50, 150, 0.3, 4.0, 4, 3.0, PARTICLESTREAMSPEC::DIFFUSE,
		PARTICLESTREAMSPEC::LVL_SQRT, 0, 1,
		PARTICLESTREAMSPEC::ATM_PLOG, 1e-4, 1
	};

//	PARTICLESTREAMSPEC oms_exhaust = {
//	0, 0.4, 3000, 150, 0.05, 0.5, 5, 0.0, PARTICLESTREAMSPEC::EMISSIVE,
//	PARTICLESTREAMSPEC::LVL_PLIN, 0, 2,
//	PARTICLESTREAMSPEC::ATM_PLOG, 0.0, 0.0
//	};

/*
	vessel->SetSize (12);
	vessel->SetCOG_elev (1);
	vessel->SetEmptyMass (2500);
	vessel->SetPMI (_V(260,260,10));
	vessel->SetCrossSections (_V(5,5,80));
	vessel->SetCW (2.0, 1.5, 1.4, 1.4);
	vessel->SetRotDrag (_V(0.7,0.7,1.2));
	if (vessel->GetFlightModel() >= 1) 
	{
		vessel->SetPitchMomentScale (-5e-4);
		vessel->SetBankMomentScale (-5e-4);
	}
	vessel->SetLiftCoeffFunc (0);
    vessel->ClearMeshes();
	vessel->ShiftCentreOfMass (_V(0,0,6.25));
	vessel->SetTouchdownPoints (_V(-1,-1,-7.35),_V(0,1,-7.35),_V(1,-1,-7.35));
*/
	vessel->SetSize (10);
	vessel->SetEmptyMass (2500);
	vessel->SetPMI (_V(2.0,2.0,0.55));
	vessel->SetCrossSections (_V(5,5,500));
	vessel->SetCW (1.1, 2.2, 3.5, 3.5);
	vessel->SetRotDrag (_V(0.05,0.05,0.05));
	vessel->ClearAirfoilDefinitions();
	if (vessel->GetFlightModel() >= 1) 
	{
		vessel->SetPitchMomentScale (-5e-5);
		vessel->SetBankMomentScale (-5e-5);
	}
	vessel->SetLiftCoeffFunc (0);
    vessel->ClearMeshes();
	vessel->ShiftCentreOfMass (_V(0,0,6.25));
	vessel->SetTouchdownPoints (_V(-1,-1,-7.35),_V(0,1,-7.35),_V(1,-1,-7.35));

	VECTOR3 mesh_dir=_V(0.175,-0.11,1.775-2.0-6.25);
	vessel->AddMesh (hLOK2, &mesh_dir);
	mesh_dir=OFS_CHUTE;
	vessel->AddMesh (hLOKchute, &mesh_dir);
	if (ph_rcs) vessel->DelPropellantResource (ph_rcs);
	vessel->ClearThrusterDefinitions();
	vessel->ClearPropellantResources();
	vessel->ClearExhaustRefs();
	if(!ph_srm) ph_srm = vessel->CreatePropellantResource(17.0);
	vessel->SetDefaultPropellantResource(ph_srm);
	th_srm[0] = vessel->CreateThruster (_V(0, 0, -1), _V(0,0,1), 140000.0, ph_srm, 2000.0, 0.0);
	vessel->AddExhaustStream(th_srm[0], _V(0,1,-7), &oms_exhaust);
	vessel->AddExhaustStream(th_srm[0], _V(-0.7,-0.7,-7), &oms_exhaust);
	vessel->AddExhaustStream(th_srm[0], _V(0.7,-0.7,-7), &oms_exhaust);
	thg_srm = vessel->CreateThrusterGroup (th_srm, 1, THGROUP_USER);
	vessel->SetThrusterGroupLevel(thg_srm, 0.0);
	status=2;
}

void LOK_ZOND::SeparateStage (VESSEL *vessel, UINT stage)
{
	VESSELSTATUS vs;

	VECTOR3 ofs = _V(0,0,0);
	VECTOR3 vel = _V(0,0,0);
	VECTOR3 rofs, rvel;


	if (stage == 0)
	{
		ofs = OFS_SERVICE;
		vel = _V(0,0, -0.5);
	}
	else if (stage == 1)
	{
		ofs = OFS_CHUTE;	
		vel = _V(0,0,-2.0);
	}
	
	vessel->GetStatus (vs);
	vs.eng_main = vs.eng_hovr = 0.0;
	
	if (stage == 0)
	{
		rvel = _V(vs.rvel.x, vs.rvel.y, vs.rvel.z);
		vessel->Local2Rel (ofs, vs.rpos);
		vessel->GlobalRot (vel, rofs);
		vs.rvel.x = rvel.x+rofs.x;
		vs.rvel.y = rvel.y+rofs.y;
		vs.rvel.z = rvel.z+rofs.z;
		vs.vrot.x = 0.0;
		vs.vrot.y = 0.0;
		vs.vrot.z = 0.0;
        oapiCreateVessel ("LOK Service Module", "n1lok3", vs);
		SetReentryStage (vessel);	
		stage = 1;
	}		
    else if (stage == 1)
	{
		rvel = _V(vs.rvel.x, vs.rvel.y, vs.rvel.z);
		vessel->Local2Rel (ofs, vs.rpos);
		vessel->GlobalRot (vel, rofs);
		vs.rvel.x = rvel.x+rofs.x;
		vs.rvel.y = rvel.y+rofs.y;
		vs.rvel.z = rvel.z+rofs.z;
		DeployMain(vessel);
		stage = 2;
	}

}


void LOK_ZOND::SetClassCaps( FILEHANDLE cfg )
{

	ph_lok			 = NULL;
	ph_rcs			 = NULL;
	thg_lok	         = NULL;
	thg_retro        = NULL;

	SetZondStage( this );


}

void LOK_ZOND::LoadState( VESSEL *vessel, FILEHANDLE scn , void *vs)
{

    char *line;

	while (oapiReadScenario_nextline (scn, line)) {
        if (!_strnicmp (line, "CONFIGURATION_ZOND", 18)) {
            sscanf_s (line+18, "%d", &status);
		} 
		else {
            vessel->ParseScenarioLineEx (line, vs);
			// unrecognised option - pass to Orbiter's generic parser
        }
    }
	switch (status) {
	case 0:
		stage=0;
		SetZondStage(vessel);
		break;
	case 1:
		stage=1;
		SetReentryStage(vessel);
		break;
	case 2:
		stage=2;
		DeployMain(vessel);
		break;
	}

}

void LOK_ZOND::SaveState( FILEHANDLE scn )
{
	SaveDefaultState (scn);
	oapiWriteScenario_int (scn, "CONFIGURATION_ZOND", status);
}


void LOK_ZOND::TimeStep( double simt )
{
	if (stage == 0 && bManualSeparate)
	{
		bManualSeparate=false;
		SeparateStage (this, stage);
		stage = 1;		
	}
	else if (stage == 1 && (GetAtmPressure() > 52000 || bManualSeparate) )
	{
			stage_sep = simt;
			bManualSeparate=false;
			SeparateStage (this, stage);
			stage = 2;		
	}
	else if (stage == 2)
	{
		if (GetAltitude() < 100.0 && oapiGetTimeAcceleration() > 1.0) oapiSetTimeAcceleration(1.0);	
	    if (GetAltitude() < 9.0 ) SetThrusterGroupLevel(thg_srm, 1.0);
	}
	if (stage == 2 && GetAltitude() < 10.0 && GetAltitude() > 7.5)
	{
		sprintf_s(oapiDebugString(), 40, "Touchdown speed = %.3f m/s", GetAirspeed() );
	}
}


// ==============================================================
// API interface
// ==============================================================

// Initialisation
DLLCLBK VESSEL *ovcInit (OBJHANDLE hvessel, int flightmodel)
{
	LOK_ZOND* lok_zond = new LOK_ZOND( hvessel, flightmodel );

	return lok_zond;
}

// Cleanup
DLLCLBK void ovcExit (VESSEL *vessel)
{
	if (vessel) delete vessel;
}

// Keyboard interface handler
DLLCLBK int ovcConsumeKey (VESSEL *vessel, const char *keystate)
{
	LOK_ZOND* lok_zond = (LOK_ZOND*)vessel;

	if (KEYMOD_SHIFT (keystate)) 
	{
		return 0; // shift combinations are reserved
	}
	else if (KEYMOD_CONTROL (keystate)) 
	{
		// insert ctrl key combinations here
	}
	else 
	{ // unmodified keys
		if (KEYDOWN (keystate, OAPI_KEY_J)) { // "Jettison Stage"
			if (oapiAcceptDelayedKey (OAPI_KEY_J, 1.0))
				lok_zond->bManualSeparate = true;
			return 1;
		}
		
	}
	return 0;
}

// Set the capabilities of the vessel class
DLLCLBK void ovcSetClassCaps (VESSEL *vessel, FILEHANDLE cfg)
{
	LOK_ZOND* lok_zond = (LOK_ZOND*)vessel;
	lok_zond->SetClassCaps( cfg );
}

DLLCLBK void ovcTimestep (VESSEL *vessel, double simt)
{

	LOK_ZOND* lok_zond = (LOK_ZOND*)vessel;
	lok_zond->TimeStep( simt );
}

// Read status from scenario file

DLLCLBK void ovcLoadStateEx (VESSEL *vessel, FILEHANDLE scn, void *vs)

{

	LOK_ZOND* lok_zond = (LOK_ZOND*)vessel;
	lok_zond->LoadState( lok_zond, scn , vs);

}

// Save status to scenario file

DLLCLBK void ovcSaveState (VESSEL *vessel, FILEHANDLE scn)
{
	LOK_ZOND* lok_zond = (LOK_ZOND*)vessel;
	// vessel parameters
	lok_zond->SaveState (scn);
}

DLLCLBK void ovcPostCreation (VESSEL *vessel)
{
//	vessel->SetNavRecv(0,(DWORD)588);
//	vessel->SetNavRecv(1,(DWORD)598);
}