// ==============================================================
// ORBITER vessel module: LK Lunar Lander
// ==============================================================
// Smoke particle effects and C++ class conversion by Dave Rowbotham
//

#include "lk.h"


const VECTOR3 OFS_LK2 = { 0.0,0.0,-2.0};


LK::LK (OBJHANDLE hObj, int fmodel)
: VESSEL (hObj, fmodel)
{
	bManualSeparate = false;
	stage_sep = 0;
	stage = 0;
	status = 0;
	gear_stat = 0;


	hLK1     = oapiLoadMeshGlobal ("n1lk1");
	hLK2     = oapiLoadMeshGlobal ("n1lk2");
	hCM1     = NULL;
	hLKChute = NULL;

	


}

LK::~LK()
{

	refcount--;	
}

void LK::DefineAnimations (VESSEL *vessel)
{

	static UINT groups1[3] = {0,1,2};
	static ANIMCOMP left_gear1 = { 
		groups1,3,
		0.0,1.0,
		1, 1, 0,					// rotation reference
		1,-1,0,                      // rotation axis
		(float)(-0.15*PI),	        // rotation range
		0,                          // mesh no.
		0,                          // not used
		MESHGROUP_TRANSFORM::ROTATE // transform type
	};
	

	static UINT groups2[3] = {3,4,5};
	static ANIMCOMP right_gear1 = { 
		groups2,3,
		0.0,1.0,
		-1, 1, 0,					// rotation reference
		-1,-1,0,                    // rotation axis
		(float)(0.15*PI),			// rotation range
		0,                          // mesh no.
		0,                          // not used
		MESHGROUP_TRANSFORM::ROTATE // transform type
	};


	static UINT groups3[3] = {6,7,8};
	static ANIMCOMP left_gear2 = { 
		groups3,3,
		0.0,1.0,
		-1, -1, 0,					// rotation reference
		-1,1,0,                     // rotation axis
		(float)(-0.15*PI),			// rotation range
		0,                          // mesh no.
		0,                          // not used
		MESHGROUP_TRANSFORM::ROTATE // transform type
	};

	static UINT groups4[3] = {9,10,11};
	static ANIMCOMP right_gear2 = { 
		groups4,3,
		0.0,1.0,
		1, -1, 0,					// rotation reference
		1,1,0,                      // rotation axis
		(float)(0.15*PI),			// rotation range
		0,                          // mesh no.
		0,                          // not used
		MESHGROUP_TRANSFORM::ROTATE // transform type
	};

		anim_gear = vessel->RegisterAnimSequence (0.0); 
		vessel->AddAnimComp (anim_gear, &left_gear1); 
		vessel->AddAnimComp (anim_gear, &right_gear1);   
		vessel->AddAnimComp (anim_gear, &left_gear2);
		vessel->AddAnimComp (anim_gear, &right_gear2);

}

void LK::ActivateGears (GearStatus action)
{
	gear_status = action;
}

void LK::OperateGears ()
{
	ActivateGears (gear_status == GEAR_CLOSED || gear_status == GEAR_CLOSING ?
						 GEAR_OPENING : GEAR_CLOSING);
}

void LK::SetAttControlsLK(VESSEL *vessel)
{

	THRUSTER_HANDLE th_rot[2];
	th_rot[0] = vessel->CreateThruster (_V(0,-0.2,-2.0), _V(0,-1,0), MAX_ATT_LK , ph_lk, ISP_LK_VAC);
	th_rot[1] = vessel->CreateThruster (_V(0,0.75,5.25), _V(0,1,0), MAX_ATT_LK , ph_lk, ISP_LK_VAC);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_PITCHUP);

	th_rot[0] = vessel->CreateThruster (_V(0,-0.2,-5.25), _V(0,1,0), MAX_ATT_LK , ph_lk, ISP_LK_VAC);
	th_rot[1] = vessel->CreateThruster (_V(0,-0.75,5.25), _V(0,1,0), MAX_ATT_LK , ph_lk, ISP_LK_VAC);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_UP);

	th_rot[0] = vessel->CreateThruster (_V(0,-0.75,5.25), _V(0,-1,0), MAX_ATT_LK , ph_lk, ISP_LK_VAC);
	th_rot[1] = vessel->CreateThruster (_V(0,0.35,-2.0), _V(0, 1,0), MAX_ATT_LK , ph_lk, ISP_LK_VAC);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_PITCHDOWN);

	th_rot[0] = vessel->CreateThruster (_V(0,-0.2,-5.25), _V(0,-1,0), MAX_ATT_LK , ph_lk, ISP_LK_VAC);
	th_rot[1] = vessel->CreateThruster (_V(0,-0.75,5.25), _V(0,-1,0), MAX_ATT_LK , ph_lk, ISP_LK_VAC);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_DOWN);

	th_rot[0] = vessel->CreateThruster (_V(-2,1.25,-0.3), _V(0,-1,0), MAX_ATT_LK , ph_lk, ISP_LK_VAC);
	th_rot[1] = vessel->CreateThruster (_V(2, 1.25,-0.3), _V(0,1,0), MAX_ATT_LK , ph_lk, ISP_LK_VAC);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_BANKLEFT);

	th_rot[0] = vessel->CreateThruster (_V(2,0.0,0.0), _V(-1,0,0), MAX_ATT_LK , ph_lk, ISP_LK_VAC);
	vessel->AddExhaust (th_rot[0], 1.5, 0.05, _V(  1.25, 0.0,0.20 ), _V(1,0,0));
	vessel->CreateThrusterGroup (th_rot, 1, THGROUP_ATT_LEFT);

	th_rot[0] = vessel->CreateThruster (_V(-2,1.25,-0.3), _V(0, 1,0), MAX_ATT_LK , ph_lk, ISP_LK_VAC);
	th_rot[1] = vessel->CreateThruster (_V(2,1.25,-0.3), _V(0,-1,0), MAX_ATT_LK , ph_lk, ISP_LK_VAC);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_BANKRIGHT);

	th_rot[0] = vessel->CreateThruster (_V(2,0.0,0.0), _V(1,0,0), MAX_ATT_LK , ph_lk, ISP_LK_VAC);
	vessel->AddExhaust (th_rot[0], 1.5, 0.05, _V(  -1.25, 0.0,0.20 ), _V(-1,0,0));
	vessel->CreateThrusterGroup (th_rot, 1, THGROUP_ATT_RIGHT);

	th_rot[0] = vessel->CreateThruster (_V(0,0, 4), _V( 1,0,0), MAX_ATT_LK , ph_lk, ISP_LK_VAC);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-4), _V(-1,0,0), MAX_ATT_LK , ph_lk, ISP_LK_VAC);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_YAWRIGHT);
	th_rot[0] = vessel->CreateThruster (_V(0,0, 4), _V(-1,0,0), MAX_ATT_LK , ph_lk, ISP_LK_VAC);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-4), _V( 1,0,0), MAX_ATT_LK , ph_lk, ISP_LK_VAC);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_YAWLEFT);

	th_rot[0] = vessel->CreateThruster (_V(0.0,0.0,-4.0), _V(0.0,0,1.0), MAX_ATT_LK , ph_lk, ISP_LK_VAC);
	vessel->CreateThrusterGroup (th_rot, 1, THGROUP_ATT_FORWARD);

	th_rot[0] = vessel->CreateThruster (_V(0.0,0.0,4.0), _V(0.0,0,-1.0), MAX_ATT_LK , ph_lk, ISP_LK_VAC);
	vessel->CreateThrusterGroup (th_rot, 1, THGROUP_ATT_BACK);

}

void LK::SetLKStage(VESSEL *vessel)
{

	PARTICLESTREAMSPEC contrail = {
		0, 8.0, 5, 150, 0.3, 4.0, 4, 3.0, PARTICLESTREAMSPEC::DIFFUSE,
		PARTICLESTREAMSPEC::LVL_SQRT, 0, 1,
		PARTICLESTREAMSPEC::ATM_PLOG, 1e-4, 1
	};

	PARTICLESTREAMSPEC exhaust_main = {
		0, 2.0, 20, 150, 0.1, 0.2, 16, 1.0, PARTICLESTREAMSPEC::EMISSIVE,
		PARTICLESTREAMSPEC::LVL_SQRT, 0, 1,
		PARTICLESTREAMSPEC::ATM_PLOG, 1e-5, 0.1
	};

	exhaust_main.tex = oapiRegisterParticleTexture ("Contrail2");

	vessel->SetSize (4);
//	vessel->SetCOG_elev (2.5);
	vessel->SetSurfaceFrictionCoeff(0.5, 0.5);
	vessel->SetEmptyMass (3500.0);
	vessel->SetPMI (_V(6.5,6.5,0.75));
	vessel->SetCrossSections (_V(16,15,5.75));
	vessel->SetCW (0.1, 0.3, 1.4, 1.4);
	vessel->SetRotDrag (_V(0.7,0.7,1.2));
	vessel->SetPitchMomentScale (0);
	vessel->SetBankMomentScale (0);
	vessel->SetLiftCoeffFunc (0); 
    vessel->ClearMeshes();
	vessel->SetTouchdownPoints (_V(0,-2.6,-2.75), _V(-1.84,1.84,-2.75), _V(1.84,1.84,-2.75));
	vessel->SetDockParams (_V(0.15,0,1.95), _V(0,0,1), _V(0,1,0));
	VECTOR3	mesh_dir=_V(0.175,0.0,-2.0);
	vessel->AddMesh (hLK2, &mesh_dir);
	mesh_dir=_V(0.0,0.0,0.0);
	vessel->AddMesh (hLK1, &mesh_dir);
	if (!ph_lk) ph_lk  = vessel->CreatePropellantResource (LK_PROP_MASS);
	if (!ph_down) ph_down  = vessel->CreatePropellantResource (DOWN_PROP_MASS);
	vessel->SetDefaultPropellantResource (ph_lk);
	vessel->ClearThrusterDefinitions();

	th_lk[0] = vessel->CreateThruster (_V(0,0,-2.25), _V(0,0,1), THRUST_LK_VAC, ph_lk, ISP_LK_VAC);
	vessel->AddExhaustStream (th_lk[0], _V(0,0,-2.25), &exhaust_main);
	SURFHANDLE tex = oapiRegisterExhaustTexture ("Exhaust");
	vessel->AddExhaust (th_lk[0], 2.0, 0.15, _V(0.085,0,-2.05), _V(0,0,-1), tex);
	thg_lk = vessel->CreateThrusterGroup (th_lk, 1, THGROUP_MAIN);

	SURFHANDLE down_tex = oapiRegisterExhaustTexture ("Exhaust2");

	int i;
	double pos;
	VECTOR3 th_vect, pos_vect;


	pos = 0.92;
	i = 0;
	pos_vect = _V(pos + 0.35, pos, -0.67);
	th_vect = _V(0, -2.0, 2.75);
	th_down[i] = vessel->CreateThruster (th_vect, _V(0,0,-1), THRUST_DOWN_VAC, ph_down, ISP_LK_VAC);
	vessel->AddExhaustStream (th_down[i], pos_vect, &exhaust_main);
	vessel->AddExhaust (th_down[i], 0.5, 0.05, pos_vect, _V(0,0,1), down_tex);

	i = 1;
	pos_vect = _V(pos + 0.35, -pos, -0.67);
	th_vect = _V(0, 2.0, 2.75);
	th_down[i] = vessel->CreateThruster (th_vect, _V(0,0,-1), THRUST_DOWN_VAC, ph_down, ISP_LK_VAC);
	vessel->AddExhaustStream (th_down[i], pos_vect, &exhaust_main);
	vessel->AddExhaust (th_down[i], 0.5, 0.05, pos_vect, _V(0,0,1), down_tex);

	i = 2;
	pos_vect = _V(-pos, pos, -0.67);
	th_vect = _V(-2.0, 0, 2.75);
	th_down[i] = vessel->CreateThruster (th_vect, _V(0,0,-1), THRUST_DOWN_VAC, ph_down, ISP_LK_VAC);
	vessel->AddExhaustStream (th_down[i], pos_vect, &exhaust_main);
	vessel->AddExhaust (th_down[i], 0.5, 0.05, pos_vect, _V(0,0,1), down_tex);


	i = 3;
	pos_vect = _V(-pos, -pos, -0.67);
	th_vect = _V(2.0, 0, 2.75);
	th_down[i] = vessel->CreateThruster (th_vect, _V(0,0,-1), THRUST_DOWN_VAC, ph_down, ISP_LK_VAC);
	vessel->AddExhaustStream (th_down[i], pos_vect, &exhaust_main);
	vessel->AddExhaust (th_down[i], 0.5, 0.05, pos_vect, _V(0,0,1), down_tex);

	thg_down = vessel->CreateThrusterGroup (th_down, 4, THGROUP_USER);

	vessel->SetThrusterGroupLevel (thg_lk, 0.0);
	vessel->SetThrusterGroupLevel (thg_down, 0.0);
    
	SetAttControlsLK(vessel);

	vessel->EnableTransponder(true);
	status=0;

	if (gear_stat == 0)
	{
		gear_status = GEAR_CLOSED;
		gear_proc = 0;
	}
	else {
		gear_status = GEAR_OPEN;
		gear_proc = 1;
	}

}

void LK::SetAscentStage(VESSEL *vessel)
{

	PARTICLESTREAMSPEC contrail = {
		0, 8.0, 5, 150, 0.3, 4.0, 4, 3.0, PARTICLESTREAMSPEC::DIFFUSE,
		PARTICLESTREAMSPEC::LVL_SQRT, 0, 1,
		PARTICLESTREAMSPEC::ATM_PLOG, 1e-4, 1
	};

	PARTICLESTREAMSPEC exhaust_main = {
		0, 2.0, 20, 150, 0.1, 0.2, 16, 1.0, PARTICLESTREAMSPEC::EMISSIVE,
		PARTICLESTREAMSPEC::LVL_SQRT, 0, 1,
		PARTICLESTREAMSPEC::ATM_PLOG, 1e-5, 0.1
	};

	vessel->SetSize (3);
	vessel->SetCOG_elev (1.5);
	vessel->SetEmptyMass (2850.0);
	vessel->SetPMI (_V(6.5,6.5,0.75));
	vessel->SetCrossSections (_V(16,15,5.75));
	vessel->SetCW (0.1, 0.3, 1.4, 1.4);
	vessel->SetRotDrag (_V(0.7,0.7,1.2));
	vessel->SetPitchMomentScale (0);
	vessel->SetBankMomentScale (0);
	vessel->SetLiftCoeffFunc (0); 
    vessel->ClearMeshes();
	vessel->SetDockParams (_V(0.15,0,1.95), _V(0,0,1), _V(0,1,0));
	VECTOR3	mesh_dir=_V(0,0.0,0.0);
	vessel->AddMesh (hLK1, &mesh_dir);
	if (!ph_lk) ph_lk  = vessel->CreatePropellantResource (LK_PROP_MASS); 
	vessel->SetDefaultPropellantResource (ph_lk); 
	vessel->ClearThrusterDefinitions();
	th_lk[0] = vessel->CreateThruster ( _V(0,0,-2.25), _V(0,0,1), THRUST_LK_VAC, ph_lk, ISP_LK_VAC);
	vessel->AddExhaustStream (th_lk[0], _V(0,0,-2.25), &exhaust_main);

	thg_lk = vessel->CreateThrusterGroup (th_lk, 1, THGROUP_MAIN);
	vessel->SetThrusterGroupLevel(thg_lk, 1.0);
	SURFHANDLE tex = oapiRegisterExhaustTexture ("Exhaust");
	vessel->AddExhaust (th_lk[0], 2.0, 0.15, _V(0.085,0,-2.05), _V(0,0,-1), tex);
	SetAttControlsLK(vessel);
	status=1;
}

void LK::SeparateStage (VESSEL *vessel, UINT stage)
{
	VESSELSTATUS vs1;
	VECTOR3 ofs1 = _V(0,0,0);
	VECTOR3 vel1 = _V(0,0,0);
	vessel->GetStatus (vs1);

	if (stage == 0)
	{
		ofs1 = OFS_LK2;
		vel1 = _V(0,0,0);
	}
	
	VECTOR3 rofs1, rvel1 = {vs1.rvel.x, vs1.rvel.y, vs1.rvel.z};
	vessel->Local2Rel (ofs1, vs1.rpos);
	vessel->GlobalRot (vel1, rofs1);
	vs1.rvel.x = rvel1.x+rofs1.x;
	vs1.rvel.y = rvel1.y+rofs1.y;
	vs1.rvel.z = rvel1.z+rofs1.z;
	
	if (stage == 0)
	{
		vs1.vrot.x = 0.0;
		vs1.vrot.y = 0.0;
		vs1.vrot.z = 0.0;		
        oapiCreateVessel ("LK Land Structure", "n1lk2", vs1);
		SetAscentStage (vessel);	
		stage = 1;
	}		
}

void LK::LoadStateEx (VESSEL *vessel, FILEHANDLE scn, void *vs)
{
	char *line;
	status = 0;
	stage  = 0;
	gear_stat = 0;

	while (oapiReadScenario_nextline (scn, line)) 
	{
        if (!_strnicmp (line, "CONFIGURATION_LK", 16)) 
		{
            sscanf_s (line+16, "%d", &status);
		}
		else if (!_strnicmp (line, "GEAR",4)) 
		{
            sscanf_s (line+4, "%d", &gear_stat);
		}
//		else if (!_strnicmp (line, "STAGE", 5)) 
//		{
//			sscanf_s (line+5, "%i", &stage);
//		}
		else 
		{
            vessel->ParseScenarioLineEx (line, vs);
			// unrecognised option - pass to Orbiter's generic parser
        }
    }

	/**  Set the stage according to the status ( seemingly ).
	*/
	switch (status) 
	{
	case 0:
		stage=0;
		SetLKStage(vessel);
		break;
	case 1:
		stage=1;
		SetAscentStage(vessel);
		break;
	}

}

void LK::SaveState (VESSEL *vessel, FILEHANDLE scn)
{
	// default vessel parameters
	vessel->SaveDefaultState (scn);

	// custom parameters
	oapiWriteScenario_int (scn, "CONFIGURATION_LK", status);
	oapiWriteScenario_int (scn, "GEAR", gear_stat);

}

void LK::SetClassCaps(VESSEL *vessel, FILEHANDLE cfg)
{

	// animation
	gear_status    = GEAR_CLOSED;
	gear_moving    =false;
	gear_endanim   =false;
	gear_proc      =0;
	vgear_proc     =0;
	stage          =0;
	status         =0;

	DefineAnimations( vessel );
	
	ph_lk			 = NULL;
	ph_down			 = NULL;
	thg_lk	         = NULL;
	thg_down		 = NULL;

	if (status==0) SetLKStage(vessel);
		else SetAscentStage(vessel);

}


void LK::Timestep( double simt )
{


//	VECTOR3 h_vect;
//	double VesselAlt, VesselPitch;
	bool bGroundCont;
//	char *GroundCont;
	static double last_time;
	double dt=time-last_time;
	last_time=time;



	if (stage == 0)
	{  		
		if (gear_status == GEAR_CLOSING || gear_status == GEAR_OPENING)
		{
			double da = oapiGetSimStep() * GEAR_OPERATING_SPEED;
			if (gear_status == GEAR_CLOSING) 
			{
				if (gear_proc > 0.0)
					gear_proc = max (0.0, gear_proc-da);
				else {
					gear_status = GEAR_CLOSED;
					gear_stat = 0;
				}
				
			}
			else  
				{ // gear opening
				if (gear_proc < 1.0)
					gear_proc = min (1.0, gear_proc+da);
				else {
					gear_status = GEAR_OPEN;
					gear_stat = 1;
				}
			}

		}
		SetAnimState (anim_gear, gear_proc);


		bGroundCont = GroundContact();
		if (GroundContact()) sprintf_s(oapiDebugString(), 21, "We are on the Moon!!");

//		sprintf_s(oapiDebugString(), 70, "vx=%.1f vy=%.1f vz=%.1f  v_hor=%.1f  p=%.1f %s",
//		h_vect.x,h_vect.y,h_vect.z, sqrt(h_vect.x*h_vect.x+h_vect.y*h_vect.y),
//		GetPitch()*DEG, GroundCont);

		if (bGroundCont) {
			if (GetPropellantMass(ph_down) > 0.2) SetThrusterGroupLevel(thg_down, 1.0);
			else SetThrusterGroupLevel(thg_down, 0.0);
		}

		if (bManualSeparate)
		{
			SeparateStage (this, stage);
			bManualSeparate=false;
			stage = 1;
		}

	}


}

// ==============================================================
// API interface
// ==============================================================

// Initialisation
DLLCLBK VESSEL *ovcInit (OBJHANDLE hvessel, int flightmodel)
{
	LK* lk = new LK( hvessel, flightmodel );
	return lk;
}

// Cleanup
DLLCLBK void ovcExit (VESSEL *vessel)
{
	if(vessel) delete vessel;
}

// Keyboard interface handler
DLLCLBK int ovcConsumeKey (VESSEL *vessel, const char *keystate)
{
	LK *lk = (LK*)vessel;

	if (KEYMOD_SHIFT (keystate)) 
	{
		return 0; // shift combinations are reserved
	}
	else if (KEYMOD_CONTROL (keystate)) 
	{
		// insert ctrl key combinations here
	}
	else 
	{ // unmodified keys
		if (KEYDOWN (keystate, OAPI_KEY_J)) { // "Jettison Stage"
			if (oapiAcceptDelayedKey (OAPI_KEY_J, 1.0))
				lk->ManualSeparate(true);
			return 1;
		}
		if (KEYDOWN (keystate, OAPI_KEY_K)) { 
			if (oapiAcceptDelayedKey (OAPI_KEY_K, 1.0))
			{
				lk->OperateGears();
			}
			return 1;
		}
		
	}
	return 0;
}

// Set the capabilities of the vessel class
DLLCLBK void ovcSetClassCaps (VESSEL *vessel, FILEHANDLE cfg)
{
	LK* lk = (LK*)vessel;
	lk->SetClassCaps( lk, cfg);
}

DLLCLBK void ovcTimestep (VESSEL *vessel, double simt)
{
	LK* lk = (LK*)vessel;
	lk->Timestep( simt );
	
}

// Read status from scenario file

DLLCLBK void ovcLoadStateEx (VESSEL *vessel, FILEHANDLE scn, void *vs)
{ 
	LK* lk = (LK*)vessel;
	lk->LoadStateEx( lk, scn, vs );
}

// Save status to scenario file

DLLCLBK void ovcSaveState (VESSEL *vessel, FILEHANDLE scn)
{
	LK* lk = (LK*)vessel;
	lk->SaveState( lk, scn );
}

DLLCLBK void ovcPostCreation (VESSEL *vessel)
{
	LK* lk = (LK*)vessel;
	lk->SetNavRecv(0,(DWORD)588);
	lk->SetNavRecv(1,(DWORD)598);
}




