// ==============================================================
// ORBITER vessel module: N1 Module
// Portions taken from Rob Conley's code:
// GetCPitch()
// Smoke particle effects and C++ class conversion by Dave Rowbotham
//
// Thanks to:
// Rob Conley (misc functions), 
// Vinka (autopilot)
//
//
// Credits:
// Core module by Manuel Amorim (McDope)
// Guidance algorithm by Richard Croy (rjcroy)
// ==============================================================


#include "n1.h"


#define LOADBMP(id) (LoadBitmap (g_Param.hDLL, MAKEINTRESOURCE (id)))
GDIParams g_Param;


const VECTOR3 OFS_STAGE1	=  { 0, 0, -35};
const VECTOR3 OFS_STAGE2	=  { 0, 0, -10-15};
const VECTOR3 OFS_STAGE3	=  { 0, 0, 6-9.5-15};
const VECTOR3 OFS_SHROUD1	=  { 0, 1.7, 28-9.5-15};
const VECTOR3 OFS_SHROUD2	=  { 0, -1.7, 28-9.5-15};
const VECTOR3 OFS_TOWER		=  { 0, 0, 49-15};
const VECTOR3 OFS_BLOCKG	= {0,0,12-15-9.5+6.5};
const VECTOR3 OFS_LOK		= { 0.0,0.0,21.75-15-9.5+6.5};
//const VECTOR3 OFS_LOK		= { 0.0,0.0,22.425-15-9.5+3.5-2.75};
const VECTOR3 OFS_LK		= {0-0.15,0,18.85-15-9.5+6.5+1.0};

const double ISP_FIRST_SL		= 284*9.81;
const double ISP_FIRST_VAC		= 330*9.81;
const double ISP_SECOND_SL		= 296*9.81;
const double ISP_SECOND_VAC		= 346*9.81;
const double ISP_THIRD_SL_NUC	= 800*9.81;
const double ISP_THIRD_VAC_NUC	= 855*9.81;
const double ISP_THIRD_SL_CHEM	= 300*9.81;
const double ISP_THIRD_VAC_CHEM	= 355*9.81;
const double ISP_BLOCKG_VAC		= 295*9.81;
const double ISP_BLOCKD_VAC		= 390*9.81;
const double ISP_LK_VAC			= 365*9.81;
const double THRUST_FIRST_VAC   = 5130000*9.81/30.0;
const double THRUST_SECOND_VAC  = 1431680*9.81/6.0;
const double THRUST_THIRD_VAC   = 164000*9.81/6.0;
const double THRUST_BLOCKG_VAC  = 45480*9.81;
const double THRUST_BLOCKD_VAC  = 10500*9.81;
const double THRUST_LK_VAC		= 2050*9.81;

const double MAX_ATT_FIRST		= 135000;
const double MAX_ATT_SECOND		= 120000;
const double MAX_ATT_THIRD		= 50000;
const double MAX_ATT_BLOCKG		= 8000;
const double MAX_ATT_BLOCKD		= 6000;
const double MAX_ATT_LK			= 1000;

const double FIRST_PROP_MASS	= 1750000;
const double SECOND_PROP_MASS	= 505000;
const double THIRD_PROP_MASS	= 175000;
const double BLOCKG_PROP_MASS	= 55800;
const double BLOCKD_PROP_MASS	= 29000;
const double LK_PROP_MASS		= 4500;

N1::N1 (OBJHANDLE hObj, int fmodel)
: VESSEL (hObj, fmodel)
{
	stage				= 0;
	stage_sep			= 0;
	bManualSeparate		= false;

	third_stage_isp_vac = (int)ISP_THIRD_VAC_CHEM;
	third_stage_isp_sl  = (int)ISP_THIRD_SL_CHEM;
	ph_first			= NULL;
	ph_second			= NULL;
	ph_third			= NULL;
	ph_blockg			= NULL;
	ph_blockd			= NULL;
	ph_lk				= NULL;
	thg_first			= NULL;
	thg_second			= NULL;
	thg_third			= NULL;
	thg_blockg			= NULL;
	thg_blockd			= NULL;
	thg_lk				= NULL;
	hN1stg1				= oapiLoadMeshGlobal ("n1stg1");
	hN1stg2				= oapiLoadMeshGlobal ("n1stg2");
	hN1stg3				= oapiLoadMeshGlobal ("n1stg3");
	hN1shroud1			= oapiLoadMeshGlobal ("n1shroud1");
	hN1shroud2			= oapiLoadMeshGlobal ("n1shroud2");
	hN1tower			= oapiLoadMeshGlobal ("n1tower");
	hN1blockg			= oapiLoadMeshGlobal ("n1blockg");
	hN1blockd			= oapiLoadMeshGlobal ("n1blockd");
	hN1mylk1			= oapiLoadMeshGlobal ("n1lk1");
	hN1mylk2			= oapiLoadMeshGlobal ("n1lk2");
	hN1mylok1			= oapiLoadMeshGlobal ("n1lok1");
	hN1mylok2			= oapiLoadMeshGlobal ("n1lok2");
	hN1mylok3			= oapiLoadMeshGlobal ("n1lok3");
	
	InitialiseGuidance();

}

N1::~N1()
{

	delete cpitch;
	delete met;
	delete Planet_Name;
	ReleaseSurfaces();
}

void N1::ReleaseSurfaces ()
{
	int i;
	for (i = 0; i < nsurf; i++)
		if (srf[i]) {
			oapiDestroySurface (srf[i]);
			srf[i] = 0;
		}
}

void N1::InitialiseGuidance()
{
	N						= 11;
	bManualSeparate			= false;
	bRadio					= false;
	stage					= 0;
	status					= 0;
	Set_First_Stage_Thlevel	= false;
	First_Stage_Thlevel		= 0;
	Third_Stage_Propultion	= 0; // 0 - chemical, 1 - nuclear

	// rjcroy Guidance variables

	INCLINATION				= 51.63 ;// degrees

	guidance				= false;	// guidance switch
	Alt_Flag				= false;	// True if max launch altitude reached
	MECO_Cond				= false; // True if MECO conditions met
	First_Guidance_Pass		= false;
	T_minus_10				= true;	// True at T minus 10 seconds.
	TZero					= -1000;		// Time Zero - Launch time
	Tcorr					= 0;
	PC_D					= 0.9;		// Pitch control coefficient D for d_theta /d_t
	PC_K					= 0.6;		//    "    "         "       K  "  theta
	YC_D					= 0.5	;		// Yaw Control
	YC_K					= 0.3;

	PC_D_FINAL				= 0.9;  // Pitch control coefficients for final section.
	PC_K_FINAL				= 0.6; 
	D_FINAL					= 0.8;	// Vertical velocity control coefficients
	K_FINAL					= 0.8;
	ENG_THRST_2				= 1608900 ;// Thrust rating of Stage 2 engine. (N)  

	i_last					= 0;			// value of incliniation in previous iteration.

//	char Planet_Name[6] = "Earth";
	Planet_Name				= new char[6];
	strcpy_s( Planet_Name, 6, "Earth");

	R_EARTH					= 6371000; // radius of the earth (m)
	U_EARTH					= 3.986e5 * 1000 * 1000 * 1000; //( m^3/sec^2)
	Ob_EARTH				= 23.45 * RAD ; // Obliquity of the Earth (rad)
//	APO_ALT					= 380 ;     // Altitude of apoapsis
	APO_ALT					= 275 ;     // Altitude of apoapsis

	ICF_K					= 62.008 * INCLINATION*INCLINATION - 6848.7*INCLINATION + 189807;	//  inclination correction factor constant. (3580)

	DisplayIncSP			= false;  // toggle for displaying the inclination setpoint.
	offDisplayIncSP			= false;  // toggle for displaying the inclination setpoint.
	PadHeading				= 243.17; // Heading of the Soyuz on the pad (deg)
							// Default heading is for inclination 51.63 deg.
	NewIncSP				= false ;		// True if a user has entered a new inclination SP
	PostMECO				= false ;		// switch for cleaning up after MECO

	BadHeading				= false;	// true if soyuz is at an azimuth not suitable for the G&CS

	met						= new double[11];
	cpitch					= new double[11];
/*
	met[0]					= 0.0;
	met[1]					= 30.0;
	met[2]					= 45.0;
	met[3]					= 60.0;
	met[4]					= 75.0;
	met[5]					= 90.0;
	met[6]					= 110.0;
	met[7]					= 140.0;
	met[8]					= 200.0;
	met[9]					= 250.0;
	met[10]					= 400.0;
*/
	met[0]					= 0.0;
	met[1]					= 0.075  * 400 - 15;
	met[2]					= 0.1125 * 400 - 15;
	met[3]					= 0.150 * 400 - 15;
	met[4]					= 0.1875 * 400 - 15;
	met[5]					= 0.225 * 400 - 15;
	met[6]					= 0.275 * 400 - 15;
	met[7]					= 0.350 * 400 - 15;
	met[8]					= 0.500 * 400 - 15;
	met[9]					= 0.625 * 400 - 15;
	met[10]					= 1.000 * 400 - 15;

	cpitch[0]				= 90.0;
	cpitch[1]				= 80.0;
	cpitch[2]				= 70.0;
	cpitch[3]				= 60.0;
	cpitch[4]				= 50.0;
	cpitch[5]				= 40.0;
	cpitch[6]				= 30.0;
	cpitch[7]				= 20.0;
	cpitch[8]				= 10.0;
	cpitch[9]				= 5.0;
	cpitch[10]				= 0.0;

}

double N1::GetCPitch(double t)
{
	int i;
	if (t>met[N-1]) return cpitch[N-1];
	i=1;
	while (met[i]<t) i++;
	return cpitch[i-1]+(cpitch[i]-cpitch[i-1])/(met[i]-met[i-1])*(t-met[i-1]);
}

// rjcroy Lateral Guidance System ( called by Guidance_Routine() ).
void N1::Lateral_Guidance(VESSEL *vessel, double SimT, double SimT_Last)
{
	// Lateral Guidance System 
	
	// Only use heading control after T + 15 seconds.
	if (MET > 15 ) 
	{
		// Get state data
		oapiGetAirspeedVector(this->GetHandle(), &Airspeed_Vector);
		// Get current velocity
		GetRelativeVel (hEarth, Rel_Vel);
		Velocity = sqrt((Rel_Vel.x*Rel_Vel.x) + (Rel_Vel.y*Rel_Vel.y) + (Rel_Vel.z*Rel_Vel.z));
			
		GetEquPos( Longitude, Latitude, Radius );
		// Calculate earth rotation speed.
		Earth_x_Vel = (2*PI*(cos(fabs(Latitude)) * R_EARTH)) / 86164.09;
		Airspeed_Vector.x = Airspeed_Vector.x + Earth_x_Vel ;
		
		// Soyuz true course
		True_Course = atan(Airspeed_Vector.x / Airspeed_Vector.z) * DEG;
		if (True_Course < 0) True_Course = 180 + True_Course;
		
		//  Find the true course SP
		oapiGetHeading( this->GetHandle(), &Heading ) ;
		Clairaut_Constant = sin ( 90 * RAD - INCLINATION * RAD) ;

		double Clairaut_Lat = Clairaut_Constant / cos(Latitude);
		if (Clairaut_Lat > 1.0) Clairaut_Lat = 1;
		if (True_Course < 90) 
		{
			True_Course_SP = (asin(Clairaut_Lat)) * DEG ;
		}
		else 
		{
			True_Course_SP = -((asin(Clairaut_Lat)) * DEG) + 180 ;
		}
		
		// Calculate the current orbital inclination
		GetRelativeVel( hEarth, Rel_Vel);
		GetRelativePos( hEarth, Rel_Pos);

		// Change ecliptic rel. vectors to equatorial vectors.
 		temp.x = Rel_Vel.x;
		temp.y = Rel_Vel.z;
		temp.z = Rel_Vel.y;
		veccpy( Rel_Vel, temp);
		temp.x = Rel_Pos.x;
		temp.y = Rel_Pos.z;
		temp.z = Rel_Pos.y;
		veccpy( Rel_Pos, temp);
		temp.x = Rel_Pos.x ;
		temp.y = cos(-Ob_EARTH)* Rel_Pos.y + sin(-Ob_EARTH)*Rel_Pos.z;
		temp.z =  -sin(-Ob_EARTH)* Rel_Pos.y + cos(-Ob_EARTH)*Rel_Pos.z;
		veccpy( Rel_Pos, temp);
		temp.x = Rel_Vel.x ;
		temp.y = cos(-Ob_EARTH)* Rel_Vel.y + sin(-Ob_EARTH)*Rel_Vel.z;
		temp.z =  -sin(-Ob_EARTH)* Rel_Vel.y + cos(-Ob_EARTH)*Rel_Vel.z;
		veccpy( Rel_Vel, temp);

		h_vector = crossp(Rel_Vel, Rel_Pos);
		h_abs = sqrt((h_vector.x*h_vector.x) + (h_vector.y*h_vector.y) + (h_vector.z*h_vector.z));
		i = 180 - acos(h_vector.z / h_abs) * DEG;
		i_dot = (i - i_last) / (MET - SimT_Last) ;
		i_last = i;
		ICF = Velocity / ICF_K;

		// Calculated error in inclination as output signal
		i_err = (i - INCLINATION) * (150 * ICF) + 250 * ICF * i_dot;
		if (i_err > ICF )
		{	
			i_err = ICF;
		}
		else if ( i_err < -ICF) 
		{
			i_err = -ICF;
		}
			
		// Feed the inclination error into the control loop after T + 25s 
		if (MET > 25) {
			if (True_Course < 90) True_Course_SP = True_Course_SP + i_err ;
			else True_Course_SP = True_Course_SP - i_err ;
		}		
		
		True_Course_Err = True_Course - True_Course_SP; 

		// True course error rate
		TC_Error_Dot = (True_Course_Err - TC_Error_Last) / (MET - SimT_Last);
		TC_Error_Last = True_Course_Err;
			
		// Adjusted heading setpoint
//		Heading_SP_Ajd = True_Course_Err * 4 + TC_Error_Dot * 8 ;
		Heading_SP_Ajd = True_Course_Err * 2 + TC_Error_Dot * 4 ;
			
		if (Heading_SP_Ajd > 2 )
		{	
			Heading_SP_Ajd = 2;
		}
		else if (Heading_SP_Ajd < -2) 
		{
			Heading_SP_Ajd = -2;
		}
		
		//	  Compare with the actual heading
		double Heading_Act = Heading * DEG;
		Error_Course = Heading_Act - ( True_Course_SP - Heading_SP_Ajd); 
		Yaw_Error_Dot =  - Angular_Velocity.y * DEG ; //get the yaw rate of change
			
//		sprintf_s(oapiDebugString(), 170, "MET: %f, TrueCourse: %f, Heading: %f, Pitch: %f, Pitch SP: %f", MET, True_Course, Heading * DEG, Pitch, Pitch_SP);
			
		//  Yaw control mechanism
		Yaw_Cntrl_Sig = - YC_D * Yaw_Error_Dot + YC_K * Error_Course;
	
//		sprintf_s(oapiDebugString(), 170, "MET %.0f, Soyuz TC %.3f, TrueCourseSP, %.3f, TrueCrsError, %.3f, YawDot %.3f, YawSig %.3f, Inc, %.3f, i_err*4, %.3f, idot, %.3f", MET, True_Course, True_Course_SP, True_Course_Err, Yaw_Error_Dot, Yaw_Cntrl_Sig, i, i_err, i_dot);
		
		if (Yaw_Cntrl_Sig > 1 )
		{
			vessel->SetAttitudeRotLevel( 1, -1); // 0 is y-axis
		}
		else if (Yaw_Cntrl_Sig < -1) 
		{
			vessel->SetAttitudeRotLevel( 1, 1);
		}
		else {
			vessel->SetAttitudeRotLevel( 1 , -Yaw_Cntrl_Sig);
		}
	}
		
	// sprintf(oapiDebugString(), "Time: %f, Pitch_SP: %f, Pitch: %f, PitchErrorDot: %f, Output: %f",MET, Pitch_SP, Pitch, Pitch_Error_Dot, Pitch_Cntrl_Sig);
		
	if ( (i - INCLINATION) < 0.006 && MET > 150)
	{
		sprintf_s(oapiDebugString(), 31, "T + %.0f s  Guidance Converged",MET);
	}
	else 
	{
		sprintf_s(oapiDebugString(), 11, "T + %.0f s",MET);
	}

}

// rjcroy Guidance & Control System
void N1::Guidance_Routine(VESSEL *vessel, double SimT, double SimT_Last)
{

	hVessel = vessel->GetHandle();
	hEarth = oapiGetGbodyByName ( Planet_Name );
	
	if (T_minus_10)
	{
		// Record launch time
		if (TZero == 0){
			TZero = SimT + 11;
		}

		if (MET > -5 && MET < -3) {
			oapiSetEngineLevel(hVessel, ENGINE_MAIN, 0.3);
		}
		else if (MET > -3 && MET < 0) {
			oapiSetEngineLevel(hVessel, ENGINE_MAIN, (0.2*MET + 1));
		}
		else if (MET > 0)
		{
			T_minus_10 = false;
			First_Guidance_Pass = true;
		}
		sprintf_s(oapiDebugString(), 9, "T - %.1f", fabs(MET));
	}		

	else if (First_Guidance_Pass) 
	{
		
		// Start the main engines
		oapiSetEngineLevel(hVessel, ENGINE_MAIN, 1);

		// Unset the first pass flag
		First_Guidance_Pass = false;
	}
	else if (!Alt_Flag) 
	{
		// Get correct pitch from table
		Pitch_SP=GetCPitch(MET);
			
		// Control to pitch setpoint
		Pitch = GetPitch(  );
		Pitch = Pitch * DEG;
		Pitch_Error = Pitch - Pitch_SP;

		vessel->GetAngularVel( Angular_Velocity ); 
		Pitch_Error_Dot = Angular_Velocity.x * DEG; // get pitch change rate in degrees/s

		Pitch_Cntrl_Sig =  - PC_D * Pitch_Error_Dot + PC_K * Pitch_Error;
		
		if (Pitch_Cntrl_Sig > 1 )
		{
			vessel->SetAttitudeRotLevel( 0, 1); // 0 is y-axis
		}
		else if (Pitch_Cntrl_Sig < -1) 
		{
			vessel->SetAttitudeRotLevel( 0, -1);
		}
		else {
			vessel->SetAttitudeRotLevel( 0 , Pitch_Cntrl_Sig);
		}
		
		// Heading control
		Lateral_Guidance(vessel, SimT, SimT_Last);
		
		// Turn off warp for final section of orbital insertion.
		if (MET > 459 )  {
//		if (MET > 529 )  {
			oapiSetTimeAcceleration (1 );
		}
		
		//Timeout on Alt_Flag
		
		if (MET > 460) {
//		if (MET > 530) {
			Alt_Flag = true;
			// Record the current vertical speed and time for next section.
			vessel->GetHorizonAirspeedVector( Surface_Vel );
			Vert_Vel_Last = Surface_Vel.y ;
//			SimT_Last = SimT;
		}
		
	}
	else if (!MECO_Cond) // Final ~ 30 secs of launch
	{

		//Calculate if MECO velocity has been reached
		Altitude = GetAltitude();
		a = ((APO_ALT * 1000 + R_EARTH) + (Altitude + R_EARTH)) / 2;
		e = ((APO_ALT * 1000 + R_EARTH) / a ) - 1;
		h = sqrt(U_EARTH * a * ( 1 - (e * e)));
		V_periapsis = h / (Altitude + R_EARTH);
		
		// Get current velocity
		GetRelativeVel (hEarth, Rel_Vel);
		Velocity = sqrt((Rel_Vel.x*Rel_Vel.x) + (Rel_Vel.y*Rel_Vel.y) + (Rel_Vel.z*Rel_Vel.z));
		
		if (Velocity > V_periapsis) 
		{
			MECO_Cond = true;
			TimeMECO = SimT;
		}

		// Get vertical velocity and vertical acceleration
		vessel->GetHorizonAirspeedVector( Surface_Vel );
		Vert_Acc = (Surface_Vel.y - Vert_Vel_Last) / (MET - SimT_Last);
		Vert_Vel_Last = Surface_Vel.y;
		
		// Engine Throttling
		// Restricts acceleration in final stages to about 2.5G
		oapiSetEngineLevel(hVessel, ENGINE_MAIN, ( oapiGetMass(hVessel) * 25 / ENG_THRST_2 ));
	
		// Pitch Control
		Pitch_SP = - D_FINAL * Vert_Acc - K_FINAL * Surface_Vel.y;

		Pitch=GetPitch(  );
		Pitch = Pitch * DEG;
		Pitch_Error = Pitch - Pitch_SP;

		vessel->GetAngularVel( Angular_Velocity ); 
		Pitch_Error_Dot = Angular_Velocity.x * DEG; // get pitch change rate in degrees/s

		Pitch_Cntrl_Sig = - PC_D_FINAL * Pitch_Error_Dot + PC_K_FINAL * Pitch_Error;
		
		if (Pitch_Cntrl_Sig > 1 )
		{
			vessel->SetAttitudeRotLevel( 0, 1); // 0 is y-axis
		}
		else if (Pitch_Cntrl_Sig < -1) 
		{
			vessel->SetAttitudeRotLevel( 0, -1);
		}
		else {
			vessel->SetAttitudeRotLevel( 0 , Pitch_Cntrl_Sig);
		}
		
		// Heading control
		Lateral_Guidance(vessel, SimT, SimT_Last);


//		 sprintf_s(oapiDebugString(), 150, "MET: %f, Surface_Vel.y, %f, Vert Acc: %f, PitchSP, %f, PC_DOut: %f, PC_KOut: %f", MET, Surface_Vel.y, Vert_Acc, Pitch_SP, Pitch_Error_Dot, Pitch_Error );
	}
	else if (MECO_Cond && !PostMECO)
	{
		// MECO.  Shutoff main engine, neutralize thrusters, turn off guidance
		oapiSetEngineLevel(hVessel, ENGINE_MAIN, 0);
		// Turn off pitch thrusters
		vessel->SetAttitudeRotLevel( 0 , 0);
		// Turn off the yaw thrusters
		vessel->SetAttitudeRotLevel( 1, 0);
		vessel->ActivateNavmode(NAVMODE_KILLROT);

		// retrieve the final orbital elements
		MJD_Time = oapiGetSimMJD();
		vessel->GetElements( elements, MJD_Time );
		Alt_Apoapsis = (elements.a * ( 1 + elements.e )/1000) - 6371; 
		Alt_Periapsis = (elements.a * ( 1 - elements.e )/1000) - 6371; 
		
		// Inclination (equatorial)
		oapiGetRelativeVel(hVessel, hEarth, &Rel_Vel);
		oapiGetRelativePos(hVessel, hEarth, &Rel_Pos);
		
		// Change ecliptic rel. vectors to equatorial vectors.
 		temp.x = Rel_Vel.x;
		temp.y = Rel_Vel.z;
		temp.z = Rel_Vel.y;
		veccpy( Rel_Vel, temp);
		temp.x = Rel_Pos.x;
		temp.y = Rel_Pos.z;
		temp.z = Rel_Pos.y;
		veccpy( Rel_Pos, temp);
		temp.x = Rel_Pos.x ;
		temp.y = cos(-Ob_EARTH)* Rel_Pos.y + sin(-Ob_EARTH)*Rel_Pos.z;
		temp.z =  -sin(-Ob_EARTH)* Rel_Pos.y + cos(-Ob_EARTH)*Rel_Pos.z;
		veccpy( Rel_Pos, temp);
		temp.x = Rel_Vel.x ;
		temp.y = cos(-Ob_EARTH)* Rel_Vel.y + sin(-Ob_EARTH)*Rel_Vel.z;
		temp.z =  -sin(-Ob_EARTH)* Rel_Vel.y + cos(-Ob_EARTH)*Rel_Vel.z;
		veccpy( Rel_Vel, temp);

		h_vector = crossp(Rel_Vel, Rel_Pos);
		h_abs = sqrt((h_vector.x*h_vector.x) + (h_vector.y*h_vector.y) + (h_vector.z*h_vector.z));
		i = 180 - acos(h_vector.z / h_abs) * DEG;
		
		PostMECO = true;

	}
	else if (MECO_Cond && PostMECO) 
	{
		if ((SimT - TimeMECO) < 10) {
			sprintf_s(oapiDebugString(), 48, "Guidance OFF.  Inclination (Equatorial): %.3f", i); 
		}	
		else {
			guidance = false;
			sprintf_s(oapiDebugString(), 1, "");
		}
	}
}
// END Guidance_Routine()

// Guidance user interface

void N1::GetInclinationSP()
{
	// Gets a new Inclination setting from the pilot
	bool IncInput (void *id, char *str, void *data);

	oapiOpenInputBox ("Inclination Setpoint, e.g. 5163 = 51.63 deg (47 to 56)", IncInput, 0, 25, (void*)this);
}

bool IncInput (void *id, char *str, void *data) 
{ 
    return ((N1*)data)->IncInputIn(str);

} 

bool N1::IncInputIn (char *rstr)
{
	int setINCLINATION;
	VESSELSTATUS2 vstatus;
	double c2c;

	sscanf_s(rstr, "%4d", &setINCLINATION);
	if (setINCLINATION > 4700. && setINCLINATION < 5600. ) {
		INCLINATION = (double) setINCLINATION / 100.;
		NewIncSP = true;
		vstatus.version = 2; 
        vstatus.flag = 0;
		this->GetStatusEx(&vstatus); 
         if(vstatus.status==1) //landed
		 {
			 this->GetEquPos( Longitude, Latitude, Radius );
			 c2c = cos(INCLINATION*RAD)/cos(Latitude);
			 if (c2c < 1.)
			 {
			   vstatus.surf_hdg = asin(c2c)+180*RAD;
			   sprintf_s(oapiDebugString(), 51, "New heading = %.2f deg", asin(c2c)*DEG+180.);
			   this->DefSetStateEx(&vstatus);
			 }

		 }
	}
	return 1 ;
}

// END rjcroy Guidance System

void N1::SetAttControlsFirst(VESSEL *vessel)
{
	THRUSTER_HANDLE th_rot[2];
	th_rot[0] = vessel->CreateThruster (_V(0,0, 25), _V(0, 1,0), MAX_ATT_FIRST, ph_first, ISP_FIRST_SL );
	th_rot[1] = vessel->CreateThruster (_V(0,0,-25), _V(0,-1,0), MAX_ATT_FIRST, ph_first, ISP_FIRST_SL);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_PITCHUP);
	th_rot[0] = vessel->CreateThruster (_V(0,0, 25), _V(0,-1,0), MAX_ATT_FIRST, ph_first, ISP_FIRST_SL);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-25), _V(0, 1,0), MAX_ATT_FIRST, ph_first, ISP_FIRST_SL);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_PITCHDOWN);
	th_rot[0] = vessel->CreateThruster (_V( 8,0,0), _V(0, 1,0), MAX_ATT_FIRST, ph_first, ISP_FIRST_SL);
	th_rot[1] = vessel->CreateThruster (_V(-8,0,0), _V(0,-1,0), MAX_ATT_FIRST, ph_first, ISP_FIRST_SL);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_BANKLEFT);
	th_rot[0] = vessel->CreateThruster (_V( 8,0,0), _V(0,-1,0), MAX_ATT_FIRST, ph_first, ISP_FIRST_SL);
	th_rot[1] = vessel->CreateThruster (_V(-8,0,0), _V(0, 1,0), MAX_ATT_FIRST, ph_first, ISP_FIRST_SL);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_BANKRIGHT);
	th_rot[0] = vessel->CreateThruster (_V(0,0, 10), _V( 1,0,0), MAX_ATT_FIRST, ph_first, ISP_FIRST_SL);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-10), _V(-1,0,0), MAX_ATT_FIRST, ph_first, ISP_FIRST_SL);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_YAWRIGHT);
	th_rot[0] = vessel->CreateThruster (_V(0,0, 10), _V(-1,0,0), MAX_ATT_FIRST, ph_first, ISP_FIRST_SL);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-10), _V( 1,0,0), MAX_ATT_FIRST, ph_first, ISP_FIRST_SL);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_YAWLEFT);
}

void N1::SetAttControlsSecond(VESSEL *vessel)
{
	THRUSTER_HANDLE th_rot[2];
	th_rot[0] = vessel->CreateThruster (_V(0,0, 15), _V(0, 1,0), MAX_ATT_SECOND, ph_second, ISP_SECOND_SL);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-15), _V(0,-1,0), MAX_ATT_SECOND, ph_second, ISP_SECOND_SL);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_PITCHUP);
	th_rot[0] = vessel->CreateThruster (_V(0,0, 15), _V(0,-1,0), MAX_ATT_SECOND, ph_second, ISP_SECOND_SL);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-15), _V(0, 1,0), MAX_ATT_SECOND, ph_second, ISP_SECOND_SL);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_PITCHDOWN);
	th_rot[0] = vessel->CreateThruster (_V( 8,0,0), _V(0, 1,0), MAX_ATT_SECOND, ph_second, ISP_SECOND_SL);
	th_rot[1] = vessel->CreateThruster (_V(-8,0,0), _V(0,-1,0), MAX_ATT_SECOND, ph_second, ISP_SECOND_SL);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_BANKLEFT);
	th_rot[0] = vessel->CreateThruster (_V( 8,0,0), _V(0,-1,0), MAX_ATT_SECOND, ph_second, ISP_SECOND_SL);
	th_rot[1] = vessel->CreateThruster (_V(-8,0,0), _V(0, 1,0), MAX_ATT_SECOND, ph_second, ISP_SECOND_SL);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_BANKRIGHT);
	th_rot[0] = vessel->CreateThruster (_V(0,0, 8), _V( 1,0,0), MAX_ATT_SECOND, ph_second, ISP_SECOND_SL);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-8), _V(-1,0,0), MAX_ATT_SECOND, ph_second, ISP_SECOND_SL);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_YAWRIGHT);
	th_rot[0] = vessel->CreateThruster (_V(0,0, 8), _V(-1,0,0), MAX_ATT_SECOND, ph_second, ISP_SECOND_SL);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-8), _V( 1,0,0), MAX_ATT_SECOND, ph_second, ISP_SECOND_SL);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_YAWLEFT);
}

void N1::SetAttControlsThird(VESSEL *vessel)
{
	THRUSTER_HANDLE th_rot[2];
	th_rot[0] = vessel->CreateThruster (_V(0,0, 5), _V(0, 1,0), MAX_ATT_THIRD, ph_third, third_stage_isp_sl);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-5), _V(0,-1,0), MAX_ATT_THIRD, ph_third, third_stage_isp_sl);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_PITCHUP);
	th_rot[0] = vessel->CreateThruster (_V(0,0, 5), _V(0,-1,0), MAX_ATT_THIRD, ph_third, third_stage_isp_sl);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-5), _V(0, 1,0), MAX_ATT_THIRD, ph_third, third_stage_isp_sl);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_PITCHDOWN);
	th_rot[0] = vessel->CreateThruster (_V( 3,0,0), _V(0, 1,0), MAX_ATT_THIRD, ph_third, third_stage_isp_sl);
	th_rot[1] = vessel->CreateThruster (_V(-3,0,0), _V(0,-1,0), MAX_ATT_THIRD, ph_third, third_stage_isp_sl);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_BANKLEFT);
	th_rot[0] = vessel->CreateThruster (_V( 3,0,0), _V(0,-1,0), MAX_ATT_THIRD, ph_third, third_stage_isp_sl);
	th_rot[1] = vessel->CreateThruster (_V(-3,0,0), _V(0, 1,0), MAX_ATT_THIRD, ph_third, third_stage_isp_sl);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_BANKRIGHT);
	th_rot[0] = vessel->CreateThruster (_V(0,0, 4), _V( 1,0,0), MAX_ATT_THIRD, ph_third, third_stage_isp_sl);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-4), _V(-1,0,0), MAX_ATT_THIRD, ph_third, third_stage_isp_sl);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_YAWRIGHT);
	th_rot[0] = vessel->CreateThruster (_V(0,0, 4), _V(-1,0,0), MAX_ATT_THIRD, ph_third, third_stage_isp_sl);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-4), _V( 1,0,0), MAX_ATT_THIRD, ph_third, third_stage_isp_sl);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_YAWLEFT);
}

void N1::SetAttControlsBlockG(VESSEL *vessel)
{
	THRUSTER_HANDLE th_rot[2];
	th_rot[0] = vessel->CreateThruster (_V(0,0, 5), _V(0, 1,0), MAX_ATT_BLOCKG, ph_blockg, ISP_BLOCKG_VAC);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-5), _V(0,-1,0), MAX_ATT_BLOCKG, ph_blockg, ISP_BLOCKG_VAC);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_PITCHUP);
	th_rot[0] = vessel->CreateThruster (_V(0,0, 5), _V(0,-1,0), MAX_ATT_BLOCKG, ph_blockg, ISP_BLOCKG_VAC);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-5), _V(0, 1,0), MAX_ATT_BLOCKG, ph_blockg, ISP_BLOCKG_VAC);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_PITCHDOWN);
	th_rot[0] = vessel->CreateThruster (_V( 3,0,0), _V(0, 1,0), MAX_ATT_BLOCKG, ph_blockg, ISP_BLOCKG_VAC);
	th_rot[1] = vessel->CreateThruster (_V(-3,0,0), _V(0,-1,0), MAX_ATT_BLOCKG, ph_blockg, ISP_BLOCKG_VAC);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_BANKLEFT);
	th_rot[0] = vessel->CreateThruster (_V( 3,0,0), _V(0,-1,0), MAX_ATT_BLOCKG, ph_blockg, ISP_BLOCKG_VAC);
	th_rot[1] = vessel->CreateThruster (_V(-3,0,0), _V(0, 1,0), MAX_ATT_BLOCKG, ph_blockg, ISP_BLOCKG_VAC);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_BANKRIGHT);
	th_rot[0] = vessel->CreateThruster (_V(0,0, 4), _V( 1,0,0), MAX_ATT_BLOCKG, ph_blockg, ISP_BLOCKG_VAC);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-4), _V(-1,0,0), MAX_ATT_BLOCKG, ph_blockg, ISP_BLOCKG_VAC);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_YAWRIGHT);
	th_rot[0] = vessel->CreateThruster (_V(0,0, 4), _V(-1,0,0), MAX_ATT_BLOCKG, ph_blockg, ISP_BLOCKG_VAC);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-4), _V( 1,0,0), MAX_ATT_BLOCKG, ph_blockg, ISP_BLOCKG_VAC);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_YAWLEFT);
}

void N1::SetAttControlsBlockD(VESSEL *vessel)
{
	THRUSTER_HANDLE th_rot[2];
	th_rot[0] = vessel->CreateThruster (_V(0,0, 5), _V(0, 1,0), MAX_ATT_BLOCKD, ph_blockd, ISP_BLOCKD_VAC);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-5), _V(0,-1,0), MAX_ATT_BLOCKD, ph_blockd, ISP_BLOCKD_VAC);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_PITCHUP);
	th_rot[0] = vessel->CreateThruster (_V(0,0, 5), _V(0,-1,0), MAX_ATT_BLOCKD, ph_blockd, ISP_BLOCKD_VAC);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-5), _V(0, 1,0), MAX_ATT_BLOCKD, ph_blockd, ISP_BLOCKD_VAC);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_PITCHDOWN);
	th_rot[0] = vessel->CreateThruster (_V( 3,0,0), _V(0, 1,0), MAX_ATT_BLOCKD, ph_blockd, ISP_BLOCKD_VAC);
	th_rot[1] = vessel->CreateThruster (_V(-3,0,0), _V(0,-1,0), MAX_ATT_BLOCKD, ph_blockd, ISP_BLOCKD_VAC);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_BANKLEFT);
	th_rot[0] = vessel->CreateThruster (_V( 3,0,0), _V(0,-1,0), MAX_ATT_BLOCKD, ph_blockd, ISP_BLOCKD_VAC);
	th_rot[1] = vessel->CreateThruster (_V(-3,0,0), _V(0, 1,0), MAX_ATT_BLOCKD, ph_blockd, ISP_BLOCKD_VAC);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_BANKRIGHT);
	th_rot[0] = vessel->CreateThruster (_V(0,0, 4), _V( 1,0,0), MAX_ATT_BLOCKD, ph_blockd, ISP_BLOCKD_VAC);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-4), _V(-1,0,0), MAX_ATT_BLOCKD, ph_blockd, ISP_BLOCKD_VAC);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_YAWRIGHT);
	th_rot[0] = vessel->CreateThruster (_V(0,0, 4), _V(-1,0,0), MAX_ATT_BLOCKD, ph_blockd, ISP_BLOCKD_VAC);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-4), _V( 1,0,0), MAX_ATT_BLOCKD, ph_blockd, ISP_BLOCKD_VAC);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_YAWLEFT);
}

void N1::SetAttControlsLK(VESSEL *vessel)
{
	THRUSTER_HANDLE th_rot[2];
	th_rot[0] = vessel->CreateThruster (_V(0,0, 5), _V(0, 1,0), MAX_ATT_LK, ph_lk, ISP_LK_VAC);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-5), _V(0,-1,0), MAX_ATT_LK, ph_lk, ISP_LK_VAC);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_PITCHUP);
	th_rot[0] = vessel->CreateThruster (_V(0,0, 5), _V(0,-1,0), MAX_ATT_LK, ph_lk, ISP_LK_VAC);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-5), _V(0, 1,0), MAX_ATT_LK, ph_lk, ISP_LK_VAC);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_PITCHDOWN);
	th_rot[0] = vessel->CreateThruster (_V( 3,0,0), _V(0, 1,0), MAX_ATT_LK, ph_lk, ISP_LK_VAC);
	th_rot[1] = vessel->CreateThruster (_V(-3,0,0), _V(0,-1,0), MAX_ATT_LK, ph_lk, ISP_LK_VAC);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_BANKLEFT);
	th_rot[0] = vessel->CreateThruster (_V( 3,0,0), _V(0,-1,0), MAX_ATT_LK, ph_lk, ISP_LK_VAC);
	th_rot[1] = vessel->CreateThruster (_V(-3,0,0), _V(0, 1,0), MAX_ATT_LK, ph_lk, ISP_LK_VAC);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_BANKRIGHT);
	th_rot[0] = vessel->CreateThruster (_V(0,0, 4), _V( 1,0,0), MAX_ATT_LK, ph_lk, ISP_LK_VAC);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-4), _V(-1,0,0), MAX_ATT_LK, ph_lk, ISP_LK_VAC);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_YAWRIGHT);
	th_rot[0] = vessel->CreateThruster (_V(0,0, 4), _V(-1,0,0), MAX_ATT_LK, ph_lk, ISP_LK_VAC);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-4), _V( 1,0,0), MAX_ATT_LK, ph_lk, ISP_LK_VAC);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_YAWLEFT);
}

void N1::SetFirstStage (VESSEL *vessel)
{
	vessel->SetSize (55);
	vessel->SetCOG_elev (50);
	vessel->SetEmptyMass (985000.0);
	vessel->SetPMI (_V(140,145,28));
	vessel->SetCrossSections (_V(395,380,115));
	vessel->SetCW (0.1, 0.3, 1.4, 1.4);
	vessel->SetRotDrag (_V(0.7,0.7,1.2));
	vessel->SetPitchMomentScale (1e-6);
	vessel->SetBankMomentScale (1e-6);
	vessel->SetLiftCoeffFunc (0); 
	vessel->ClearMeshes();
	vessel->SetTouchdownPoints (_V(0,-1.0,-50), _V(-.7,.7,-50), _V(.7,.7,-50));
	VECTOR3 mesh_dir=_V(0,0,-35);
	vessel->AddMesh (hN1stg1, &mesh_dir);
    mesh_dir=_V(0,0,-10);
    vessel->AddMesh (hN1stg2, &mesh_dir);
	mesh_dir=_V(0,0,6);
    vessel->AddMesh (hN1stg3, &mesh_dir);
	mesh_dir=_V(0,1.7,28);
    vessel->AddMesh (hN1shroud1, &mesh_dir);
	mesh_dir=_V(0,-1.7,28);
    vessel->AddMesh (hN1shroud2, &mesh_dir);
	mesh_dir=_V(0,0,49);
    vessel->AddMesh (hN1tower, &mesh_dir);
	if (!ph_first)  ph_first  = vessel->CreatePropellantResource (FIRST_PROP_MASS); 
	vessel->SetDefaultPropellantResource (ph_first); 
	vessel->ClearThrusterDefinitions();
	PARTICLESTREAMSPEC contrail = {
		0, 5.0, 5, 150, 0.3, 4.0, 4, 3.0, PARTICLESTREAMSPEC::DIFFUSE,
		PARTICLESTREAMSPEC::LVL_SQRT, 0, 1,
		PARTICLESTREAMSPEC::ATM_PLOG, 1e-4, 1
	};
	PARTICLESTREAMSPEC exhaust_main = {
		0, 0.8, 20, 150, 0.1, 0.2, 16, 1.0, PARTICLESTREAMSPEC::EMISSIVE,
		PARTICLESTREAMSPEC::LVL_SQRT, 0, 1,
		PARTICLESTREAMSPEC::ATM_PLOG, 1e-5, 0.1
	};
	
	exhaust_main.tex = oapiRegisterParticleTexture ("Contrail2");

	int i;
	double th_x,th_y;
	for (i = 0; i < 24; i++) {
		th_x = sin (2 * PI / 24.0 * double (i)) * 7.2;
		th_y = cos (2 * PI / 24.0 * double (i)) * 7.2;
		th_first[i] = vessel->CreateThruster (_V(th_x,th_y,-51), _V(0,0,1), THRUST_FIRST_VAC, ph_first, ISP_FIRST_VAC, ISP_FIRST_SL);
		//vessel->AddExhaustStream (th_first[i], _V(th_x,th_y,-58), &contrail);
		vessel->AddExhaustStream (th_first[i], _V(th_x,th_y,-54), &exhaust_main);
	}
    SURFHANDLE tex = oapiRegisterExhaustTexture ("Exhaust2");
	for (i = 0; i < 6; i++) {
		th_x = sin (2 * PI / 6.0 * double (i)) * 2.4;
		th_y = cos (2 * PI / 6.0 * double (i)) * 2.4;
		th_first[i+24] = vessel->CreateThruster (_V(th_x,th_y,-51), _V(0,0,1), THRUST_FIRST_VAC, ph_first, ISP_FIRST_VAC, ISP_FIRST_SL);
//		vessel->AddExhaustStream (th_first[i+24], _V(th_x,th_y,-58), &contrail);
		vessel->AddExhaustStream (th_first[i+24], _V(th_x,th_y,-54), &exhaust_main);
//		vessel->AddExhaust (th_first[0], 20.0, 0.5, _V(th_x,th_y,-54), _V (0,0,1), tex);
	}

	thg_first = vessel->CreateThrusterGroup (th_first, 30, THGROUP_MAIN);
	for (int i = 0; i < 30; i++) vessel->AddExhaust (th_first[i], 20.0, 0.5, tex);
	SetAttControlsFirst(vessel);
	status = 0;
}

void N1::SetSecondStage (VESSEL *vessel)
{
		PARTICLESTREAMSPEC contrail = {
		0, 8.0, 5, 150, 0.3, 4.0, 4, 3.0, PARTICLESTREAMSPEC::DIFFUSE,
		PARTICLESTREAMSPEC::LVL_SQRT, 0, 1,
		PARTICLESTREAMSPEC::ATM_PLOG, 1e-4, 1
	};
	PARTICLESTREAMSPEC exhaust_main = {
		0, 2.0, 20, 150, 0.1, 0.2, 16, 1.0, PARTICLESTREAMSPEC::EMISSIVE,
		PARTICLESTREAMSPEC::LVL_SQRT, 0, 1,
		PARTICLESTREAMSPEC::ATM_PLOG, 1e-5, 0.1
	};

	vessel->SetSize (37.5);
	vessel->SetCOG_elev (35);
	vessel->SetEmptyMass (350000.0);
	vessel->SetPMI (_V(94,94,20));
	vessel->SetCrossSections (_V(267,267,97));
	vessel->SetCW (0.1, 0.3, 1.4, 1.4);
	vessel->SetRotDrag (_V(0.7,0.7,1.2));
	vessel->SetPitchMomentScale (1e-6);
	vessel->SetBankMomentScale (1e-6);
	vessel->SetLiftCoeffFunc (0); 
    vessel->ClearMeshes();
	vessel->ShiftCentreOfMass (_V(0,0,15));
	vessel->SetDockParams (_V(0.15-0.15,0,60.30), _V(0,0,1), _V(0,1,0));
	VECTOR3 mesh_dir=_V(0,0,-10-15);
    vessel->AddMesh (hN1stg2, &mesh_dir);
	mesh_dir=_V(0,0,6-15);
    vessel->AddMesh (hN1stg3, &mesh_dir);
	mesh_dir=_V(0,1.7,28-15);
    vessel->AddMesh (hN1shroud1, &mesh_dir);
	mesh_dir=_V(0,-1.7,28-15);
    vessel->AddMesh (hN1shroud2, &mesh_dir);
	mesh_dir=_V(0,0,49-15);
    vessel->AddMesh (hN1tower, &mesh_dir);
	if (ph_first) vessel->DelPropellantResource (ph_first);
	if (!ph_second) ph_second  = vessel->CreatePropellantResource (SECOND_PROP_MASS); 
	vessel->SetDefaultPropellantResource (ph_second); 
	vessel->ClearThrusterDefinitions();

	int i;
	double th_x,th_y;
	for (i = 0; i < 6; i++) {
		th_x = sin (2 * PI / 6.0 * double (i)) * 2.4;
		th_y = cos (2 * PI / 6.0 * double (i)) * 2.4;
		th_second[i] = vessel->CreateThruster (_V(th_x,th_y,-17.5-18.75), _V(0,0,1), THRUST_SECOND_VAC, ph_second, ISP_SECOND_VAC, ISP_SECOND_SL);
		vessel->AddExhaustStream (th_second[i], _V(th_x,th_y,-17.5-20.75), &contrail);
		vessel->AddExhaustStream (th_second[i], _V(th_x,th_y,-17.5-22.75), &exhaust_main);
	}

	thg_second = vessel->CreateThrusterGroup (th_second, 6, THGROUP_MAIN);
	SURFHANDLE tex = oapiRegisterExhaustTexture ("Exhaust2");
	for (int i = 0; i < 6; i++) vessel->AddExhaust (th_second[i], 10.0, 0.65,tex);
	SetAttControlsSecond(vessel);
	vessel->SetThrusterGroupLevel(thg_second, 1.0);
	status = 1;
}

void N1::SetThirdStage (VESSEL *vessel)
{
	PARTICLESTREAMSPEC contrail = {
		0, 8.0, 5, 150, 0.3, 4.0, 4, 3.0, PARTICLESTREAMSPEC::DIFFUSE,
		PARTICLESTREAMSPEC::LVL_SQRT, 0, 1,
		PARTICLESTREAMSPEC::ATM_PLOG, 1e-4, 1
	};
	PARTICLESTREAMSPEC exhaust_main = {
		0, 2.0, 20, 150, 0.1, 0.2, 16, 1.0, PARTICLESTREAMSPEC::EMISSIVE,
		PARTICLESTREAMSPEC::LVL_SQRT, 0, 1,
		PARTICLESTREAMSPEC::ATM_PLOG, 1e-5, 0.1
	};

	vessel->SetSize (27.15);
	vessel->SetCOG_elev (25);
	vessel->SetEmptyMass (119300.0);
	vessel->SetPMI (_V(69,69,6));
	vessel->SetCrossSections (_V(138,139,35));
	vessel->SetCW (0.1, 0.3, 1.4, 1.4);
	vessel->SetRotDrag (_V(0.7,0.7,1.2));
	vessel->SetPitchMomentScale (1e-6);
	vessel->SetBankMomentScale (1e-6);
	vessel->SetLiftCoeffFunc (0); 
    vessel->ClearMeshes();
	vessel->SetDockParams (_V(0.15-0.15,0,13.30), _V(0,0,1), _V(0,1,0));
	vessel->ShiftCentreOfMass (_V(0,0,9.5));
	VECTOR3	mesh_dir=_V(0,0,6-15-9.5);
    vessel->AddMesh (hN1stg3, &mesh_dir);
	mesh_dir=_V(0,1.7,28-15-9.5);
    vessel->AddMesh (hN1shroud1, &mesh_dir);
	mesh_dir=_V(0,-1.7,28-15-9.5);
    vessel->AddMesh (hN1shroud2, &mesh_dir);
	if (ph_second) vessel->DelPropellantResource (ph_second);
	if (!ph_third) ph_third  = vessel->CreatePropellantResource (THIRD_PROP_MASS); 
	vessel->SetDefaultPropellantResource (ph_third); 
	vessel->ClearThrusterDefinitions();

	if( Third_Stage_Propultion == 0 )
	{
		third_stage_isp_vac = ISP_THIRD_VAC_CHEM;
		third_stage_isp_sl  = ISP_THIRD_SL_CHEM;
	}
	else 
	{
		third_stage_isp_vac = ISP_THIRD_VAC_NUC;
		third_stage_isp_sl  = ISP_THIRD_SL_NUC;
	}
	int i;
	double th_x,th_y;
	for (i = 0; i < 6; i++) {
		th_x = sin (2 * PI / 6.0 * double (i)) * 1.0;
		th_y = cos (2 * PI / 6.0 * double (i)) * 1.0;
		th_third[i] = vessel->CreateThruster (_V(th_x,th_y,-25.5), _V(0,0,1), THRUST_THIRD_VAC, ph_third, third_stage_isp_vac, third_stage_isp_sl);
		vessel->AddExhaustStream (th_third[i], _V(th_x,th_y,-27.5), &contrail);
		vessel->AddExhaustStream (th_third[i], _V(th_x,th_y,-29.5), &exhaust_main);
	}

	thg_third = vessel->CreateThrusterGroup (th_third, 6, THGROUP_MAIN);
	SURFHANDLE tex = oapiRegisterExhaustTexture ("Exhaust");
	for (int i = 0; i < 6; i++) vessel->AddExhaust (th_third[i], 10.0, 0.45, tex);
	SetAttControlsThird(vessel);
	vessel->SetThrusterGroupLevel (thg_third,1.0);
	status = 2;
}

void N1::SetBlockGStage (VESSEL *vessel)
{
	PARTICLESTREAMSPEC contrail = {
		0, 8.0, 5, 150, 0.3, 4.0, 4, 3.0, PARTICLESTREAMSPEC::DIFFUSE,
		PARTICLESTREAMSPEC::LVL_SQRT, 0, 1,
		PARTICLESTREAMSPEC::ATM_PLOG, 1e-4, 1
	};
	PARTICLESTREAMSPEC exhaust_main = {
		0, 2.0, 20, 150, 0.1, 0.2, 16, 1.0, PARTICLESTREAMSPEC::EMISSIVE,
		PARTICLESTREAMSPEC::LVL_SQRT, 0, 1,
		PARTICLESTREAMSPEC::ATM_PLOG, 1e-5, 0.1
	};

	vessel->SetSize (7);
	vessel->SetCOG_elev (3.5);
	vessel->SetEmptyMass (59800.0);
	vessel->SetPMI (_V(39,39,8));
	vessel->SetCrossSections (_V(68,69,25));
	vessel->SetCW (0.1, 0.3, 1.4, 1.4);
	vessel->SetRotDrag (_V(0.7,0.7,1.2));
	vessel->SetPitchMomentScale (1e-6);
	vessel->SetBankMomentScale (1e-6);
	vessel->SetLiftCoeffFunc (0); 
    vessel->ClearMeshes();
	vessel->SetDockParams (_V(0.15-0.15,0,7.30), _V(0,0,1), _V(0,1,0));
	vessel->ShiftCentreOfMass (_V(0,0,-6.5));
	VECTOR3 mesh_dir=_V(0,0,12-15-9.5+6.5);
	vessel->AddMesh (hN1blockg, &mesh_dir);
	mesh_dir=_V(0,0,15-15-9.5+6.5);
	vessel->AddMesh (hN1blockd, &mesh_dir);
	mesh_dir=_V(0-0.15,0,18.85-15-9.5+6.5);
	vessel->AddMesh (hN1mylk1, &mesh_dir);
	mesh_dir=_V(0-0.15,0,16.85-15-9.5+6.5);
	vessel->AddMesh (hN1mylk2, &mesh_dir);
	mesh_dir=_V(0.15-0.15,0.05,25.35-15-9.5+6.5);
	vessel->AddMesh (hN1mylok1, &mesh_dir);
	mesh_dir=_V(0.175-0.15,-0.11,23.45-15-9.5+6.5);
	vessel->AddMesh (hN1mylok2, &mesh_dir);
	mesh_dir=_V(-0.15-0.15,0.245,21.675-15-9.5+6.5);
	vessel->AddMesh (hN1mylok3, &mesh_dir);
	if (ph_third) vessel->DelPropellantResource (ph_third);
	if (!ph_blockg)  ph_blockg  = vessel->CreatePropellantResource (BLOCKG_PROP_MASS); 
	vessel->SetDefaultPropellantResource (ph_blockg); 
	vessel->ClearThrusterDefinitions();
	th_blockg[0] = vessel->CreateThruster (_V(0,0,-9.75), _V(0,0,1), THRUST_BLOCKG_VAC, ph_blockg, ISP_BLOCKG_VAC);
	vessel->AddExhaustStream (th_blockg[0], _V(0,0,-9.75), &contrail);
	vessel->AddExhaustStream (th_blockg[0], _V(0,0,-9.75), &exhaust_main);

	thg_blockg = vessel->CreateThrusterGroup (th_blockg, 1, THGROUP_MAIN);
	vessel->SetThrusterGroupLevel (thg_blockg,1.0);
	SURFHANDLE tex = oapiRegisterExhaustTexture ("Exhaust");
	vessel->AddExhaust (th_blockg[0], 6.0, 0.5,tex);
	SetAttControlsBlockG(vessel);
	status = 3;
}

void N1::SetStackStage (VESSEL *vessel)
{
		PARTICLESTREAMSPEC contrail = {
		0, 8.0, 5, 150, 0.3, 4.0, 4, 3.0, PARTICLESTREAMSPEC::DIFFUSE,
		PARTICLESTREAMSPEC::LVL_SQRT, 0, 1,
		PARTICLESTREAMSPEC::ATM_PLOG, 1e-4, 1
	};
	PARTICLESTREAMSPEC exhaust_main = {
		0, 2.0, 20, 150, 0.1, 0.2, 16, 1.0, PARTICLESTREAMSPEC::EMISSIVE,
		PARTICLESTREAMSPEC::LVL_SQRT, 0, 1,
		PARTICLESTREAMSPEC::ATM_PLOG, 1e-5, 0.1
	};

	vessel->SetSize (7);
	vessel->SetCOG_elev (3.5);
	vessel->SetEmptyMass (25000.0);
	vessel->SetPMI (_V(5.35,5.35,0.85));
	vessel->SetCrossSections (_V(14.75,16.25,6.75));
	vessel->SetCW (0.1, 0.3, 1.4, 1.4);
	vessel->SetRotDrag (_V(0.7,0.7,1.2));
	vessel->SetPitchMomentScale (1e-6);
	vessel->SetBankMomentScale (1e-6);
	vessel->SetLiftCoeffFunc (0); 
	vessel->SetDockParams (_V(0.15-0.15,0,7.30), _V(0,0,1), _V(0,1,0));
    vessel->ClearMeshes();
	VECTOR3 mesh_dir=_V(0,0,15-15-9.5+6.5);
	vessel->AddMesh (hN1blockd, &mesh_dir);
	mesh_dir=_V(0-0.15,0,18.85-15-9.5+6.5);
	vessel->AddMesh (hN1mylk1, &mesh_dir);
	mesh_dir=_V(0-0.15,0,16.85-15-9.5+6.5);
	vessel->AddMesh (hN1mylk2, &mesh_dir);
	mesh_dir=_V(0.15-0.15,0.05,25.35-15-9.5+6.5);
	vessel->AddMesh (hN1mylok1, &mesh_dir);
	mesh_dir=_V(0.175-0.15,-0.11,23.45-15-9.5+6.5);
	vessel->AddMesh (hN1mylok2, &mesh_dir);
	mesh_dir=_V(-0.15-0.15,0.245,21.675-15-9.5+6.5);
	vessel->AddMesh (hN1mylok3, &mesh_dir);
	if (ph_blockg) vessel->DelPropellantResource (ph_blockg);
	if (!ph_blockd) ph_blockd  = vessel->CreatePropellantResource (BLOCKD_PROP_MASS); 
	vessel->SetDefaultPropellantResource (ph_blockd); 
	vessel->ClearThrusterDefinitions();
	th_blockd[0] = vessel->CreateThruster (_V(0,0,-5), _V(0,0,1), THRUST_BLOCKD_VAC, ph_blockd, ISP_BLOCKD_VAC);
	vessel->AddExhaustStream (th_blockd[0], _V(0,0,-5), &contrail);
	vessel->AddExhaustStream (th_blockd[0], _V(0,0,-5), &exhaust_main);

	thg_blockd = vessel->CreateThrusterGroup (th_blockd, 1, THGROUP_MAIN);
	vessel->SetThrusterGroupLevel (thg_blockd,1.0);
	SURFHANDLE tex = oapiRegisterExhaustTexture ("Exhaust");
	vessel->AddExhaust (th_blockd[0], 6.0, 0.5,tex);
	SetAttControlsBlockD(vessel);
	status = 4;
}

void N1::SetDescentStage (VESSEL *vessel)
{
		PARTICLESTREAMSPEC contrail = {
		0, 8.0, 5, 150, 0.3, 4.0, 4, 3.0, PARTICLESTREAMSPEC::DIFFUSE,
		PARTICLESTREAMSPEC::LVL_SQRT, 0, 1,
		PARTICLESTREAMSPEC::ATM_PLOG, 1e-4, 1
	};
	PARTICLESTREAMSPEC exhaust_main = {
		0, 2.0, 20, 150, 0.1, 0.2, 16, 1.0, PARTICLESTREAMSPEC::EMISSIVE,
		PARTICLESTREAMSPEC::LVL_SQRT, 0, 1,
		PARTICLESTREAMSPEC::ATM_PLOG, 1e-5, 0.1
	};

	vessel->SetSize (4);
	vessel->SetCOG_elev (3.5);
	vessel->SetEmptyMass (5560.0+3500);
	vessel->SetPMI (_V(13,15,3));
	vessel->SetCrossSections (_V(28,28,8));
	vessel->SetCW (0.1, 0.3, 1.4, 1.4);
	vessel->SetRotDrag (_V(0.7,0.7,1.2));
	vessel->SetPitchMomentScale (0);
	vessel->SetBankMomentScale (0);
	vessel->SetLiftCoeffFunc (0); 
    vessel->ClearMeshes();
	vessel->SetDockParams (_V(0.15-0.15,0,3.8), _V(0,0,1), _V(0,1,0));
//	vessel->SetDockParams (_V(0.15,0,3.30), _V(0,0,1), _V(0,1,0));
	vessel->ShiftCentreOfMass (_V(0,0,-1.0));
	VECTOR3 mesh_dir=_V(0,0,15-15-9.5+6.5+1.0);
	vessel->AddMesh (hN1blockd, &mesh_dir);
	mesh_dir=_V(0-0.15,0,18.85-15-9.5+6.5+1.0);
	vessel->AddMesh (hN1mylk1, &mesh_dir);
	mesh_dir=_V(0-0.15,0,16.85-15-9.5+6.5+1.0);
	vessel->AddMesh (hN1mylk2, &mesh_dir);
	if (!ph_blockd)  ph_blockd  = vessel->CreatePropellantResource (BLOCKD_PROP_MASS); 
	vessel->SetDefaultPropellantResource (ph_blockd); 
	vessel->ClearThrusterDefinitions();
	th_blockd[0] = vessel->CreateThruster (_V(0,0,-4.25), _V(0,0,1), THRUST_BLOCKD_VAC, ph_blockd, ISP_BLOCKD_VAC);
	vessel->AddExhaustStream (th_blockd[0], _V(0,0,-5), &contrail);
	vessel->AddExhaustStream (th_blockd[0], _V(0,0,-5), &exhaust_main);
	
	thg_blockd = vessel->CreateThrusterGroup (th_blockd, 1, THGROUP_MAIN);
	vessel->SetThrusterGroupLevel (thg_blockd,0.0);
	SURFHANDLE tex = oapiRegisterExhaustTexture ("Exhaust");
	vessel->AddExhaust (th_blockd[0], 6.0, 0.5,tex);
	SetAttControlsBlockD(vessel);
	status = 5;
}

void N1::SetDummyStage (VESSEL *vessel)
{
		PARTICLESTREAMSPEC contrail = {
		0, 8.0, 5, 150, 0.3, 4.0, 4, 3.0, PARTICLESTREAMSPEC::DIFFUSE,
		PARTICLESTREAMSPEC::LVL_SQRT, 0, 1,
		PARTICLESTREAMSPEC::ATM_PLOG, 1e-4, 1
	};
	PARTICLESTREAMSPEC exhaust_main = {
		0, 2.0, 20, 150, 0.1, 0.2, 16, 1.0, PARTICLESTREAMSPEC::EMISSIVE,
		PARTICLESTREAMSPEC::LVL_SQRT, 0, 1,
		PARTICLESTREAMSPEC::ATM_PLOG, 1e-5, 0.1
	};

	vessel->SetSize (3);
	vessel->SetCOG_elev (1.5);
	vessel->SetEmptyMass (2560);
	vessel->SetPMI (_V(9,9,3));
	vessel->SetCrossSections (_V(12,12,5));
	vessel->SetCW (0.1, 0.3, 1.4, 1.4);
	vessel->SetRotDrag (_V(0.7,0.7,1.2));
	vessel->SetPitchMomentScale (0);
	vessel->SetBankMomentScale (0);
	vessel->SetLiftCoeffFunc (0); 
    vessel->ClearMeshes();
	vessel->ShiftCentreOfMass (_V(0,0,-2.15));
	VECTOR3 mesh_dir=_V(0,0,15-15-9.5+6.5+1.0+2.15);
	vessel->AddMesh (hN1blockd, &mesh_dir);
	if (ph_blockd) vessel->DelPropellantResource (ph_blockd);
	vessel->ClearThrusterDefinitions();
	status = 6;
}

void N1::SeparateStage (VESSEL *vessel, UINT stage)
{
	VESSELSTATUS vs1;
	VESSELSTATUS vs2;
	VESSELSTATUS vs3;
	VECTOR3 ofs1 = _V(0,0,0);
	VECTOR3 vel1 = _V(0,0,0);
	VECTOR3 ofs2 = _V(0,0,0);
	VECTOR3 vel2 = _V(0,0,0);
	VECTOR3 ofs3 = _V(0,0,0);
	VECTOR3 vel3 = _V(0,0,0);
    
	DOCKHANDLE hDock = NULL;

	vessel->GetStatus (vs1);
	vessel->GetStatus (vs2);
	vessel->GetStatus (vs3);

	//undock before separate if docked
	hDock = vessel->GetDockHandle(0);
	if (hDock != NULL && vessel->GetDockStatus(hDock)) vessel->Undock(0);

	if (stage == 0)
	{	    
		ofs1 = OFS_STAGE1;		
		vel1 = _V(0,0,-8.0);	 
	}
	
	else if (stage == 1)
	{
	 	ofs1 = OFS_STAGE2;		 
		vel1 = _V(0,0,-6.0);	
		ofs2 = OFS_TOWER;		 
		vel2 = _V(4.0,4.0,12.0);	
	}
	else if (stage == 2)
	{
		ofs1 = OFS_STAGE3;		 
		vel1 = _V(0.0,0.0,-2.0);	
		ofs2 = OFS_SHROUD1;		
		vel2 = _V(0.0,4.0,-2.0);	
		ofs3 = OFS_SHROUD2;		 
		vel3 = _V(0.0,-4.0,-2.0);	
	}
	else if (stage == 3)
	{
		ofs1 = OFS_BLOCKG;		
		vel1 = _V(0.0,0.0,-2.0);	
	}
	else if (stage == 4)
	{
		ofs1 = OFS_LOK;		
		vel1 = _V(0.0,0.0,0.05);	 
	}
	else if (stage == 5)
	{
		ofs1 = OFS_LK;		
		vel1 = _V(0.0,0.0,0.05);
	}

	VECTOR3 rofs1, rvel1 = {vs1.rvel.x, vs1.rvel.y, vs1.rvel.z};
	VECTOR3 rofs2, rvel2 = {vs2.rvel.x, vs2.rvel.y, vs2.rvel.z};
	VECTOR3 rofs3, rvel3 = {vs3.rvel.x, vs3.rvel.y, vs3.rvel.z};
	vessel->Local2Rel (ofs1, vs1.rpos);
	vessel->Local2Rel (ofs2, vs2.rpos);
	vessel->Local2Rel (ofs3, vs3.rpos);
	vessel->GlobalRot (vel1, rofs1);
	vessel->GlobalRot (vel2, rofs2);
	vessel->GlobalRot (vel3, rofs3);
	vs1.rvel.x = rvel1.x+rofs1.x;
	vs1.rvel.y = rvel1.y+rofs1.y;
	vs1.rvel.z = rvel1.z+rofs1.z;
	vs2.rvel.x = rvel2.x+rofs2.x;
	vs2.rvel.y = rvel2.y+rofs2.y;
	vs2.rvel.z = rvel2.z+rofs2.z;
	vs3.rvel.x = rvel3.x+rofs3.x;
	vs3.rvel.y = rvel3.y+rofs3.y;
	vs3.rvel.z = rvel3.z+rofs3.z;

    if (stage == 0)
	{
	    vs1.vrot.x = 0.025;	
		vs1.vrot.y = 0.025;
		vs1.vrot.z = 0.0;
		oapiCreateVessel("N1 First Stage", "n1stg1", vs1);
		
		vessel->DelPropellantResource (ph_first);

		vessel->DelThrusterGroup (thg_first, THGROUP_MAIN, true);

		vessel->DelThrusterGroup (THGROUP_ATT_PITCHUP, true);
		vessel->DelThrusterGroup (THGROUP_ATT_PITCHDOWN, true);
		vessel->DelThrusterGroup (THGROUP_ATT_BANKLEFT, true);
		vessel->DelThrusterGroup (THGROUP_ATT_BANKRIGHT, true);
		vessel->DelThrusterGroup (THGROUP_ATT_YAWLEFT, true);
		vessel->DelThrusterGroup (THGROUP_ATT_YAWRIGHT, true);

		SetSecondStage (vessel);	
	}
	else if (stage == 1)
	{
		vs1.vrot.x = 0.05;	
		vs1.vrot.y = 0.05;	
		vs1.vrot.z = 0.0;
		vs2.vrot.x = 0.0;	
		vs2.vrot.y = 0.0;	
		vs2.vrot.z = 0.0;
		oapiCreateVessel("N1 Second Stage", "n1stg2", vs1);
		oapiCreateVessel("N1 Escape Tower", "n1tower", vs2);

		vessel->DelPropellantResource (ph_second);

		vessel->DelThrusterGroup (thg_second, THGROUP_MAIN, true);

		vessel->DelThrusterGroup (THGROUP_ATT_PITCHUP, true);
		vessel->DelThrusterGroup (THGROUP_ATT_PITCHDOWN, true);
		vessel->DelThrusterGroup (THGROUP_ATT_BANKLEFT, true);
		vessel->DelThrusterGroup (THGROUP_ATT_BANKRIGHT, true);
		vessel->DelThrusterGroup (THGROUP_ATT_YAWLEFT, true);
		vessel->DelThrusterGroup (THGROUP_ATT_YAWRIGHT, true);

		vessel->ClearThrusterDefinitions();

		SetThirdStage (vessel);
	}
	else if (stage == 2)
	{
		vs1.vrot.x = 0.0;
		vs1.vrot.y = 0.0;
		vs1.vrot.z = 0.0;
		vs2.vrot.x = 0.12;
		vs2.vrot.y = 0.0;
		vs2.vrot.z = 0.0;
		vs3.vrot.x = -0.12;
		vs3.vrot.y = 0.0;
		vs3.vrot.z = 0.0;
		oapiCreateVessel("N1 Third Stage", "n1stg3", vs1);
		oapiCreateVessel("N1 Shroud Left", "n1shroud1", vs2);
		oapiCreateVessel("N1 Shroud Right", "n1shroud2", vs3);
		
		vessel->DelPropellantResource (ph_third);

		vessel->DelThrusterGroup (thg_third, THGROUP_MAIN, true);

		vessel->DelThrusterGroup (THGROUP_ATT_PITCHUP, true);
		vessel->DelThrusterGroup (THGROUP_ATT_PITCHDOWN, true);
		vessel->DelThrusterGroup (THGROUP_ATT_BANKLEFT, true);
		vessel->DelThrusterGroup (THGROUP_ATT_BANKRIGHT, true);
		vessel->DelThrusterGroup (THGROUP_ATT_YAWLEFT, true);
		vessel->DelThrusterGroup (THGROUP_ATT_YAWRIGHT, true);

		vessel->ClearThrusterDefinitions();

		SetBlockGStage (vessel);
	}
	else if (stage == 3)
	{
		vs1.vrot.x = 0.0;
		vs1.vrot.y = 0.0;
		vs1.vrot.z = 0.0;
		oapiCreateVessel("N1 Block G", "n1blockg", vs1);

		vessel->DelPropellantResource (ph_blockg);

		vessel->DelThrusterGroup (thg_blockg, THGROUP_MAIN, true);

		vessel->DelThrusterGroup (THGROUP_ATT_PITCHUP, true);
		vessel->DelThrusterGroup (THGROUP_ATT_PITCHDOWN, true);
		vessel->DelThrusterGroup (THGROUP_ATT_BANKLEFT, true);
		vessel->DelThrusterGroup (THGROUP_ATT_BANKRIGHT, true);
		vessel->DelThrusterGroup (THGROUP_ATT_YAWLEFT, true);
		vessel->DelThrusterGroup (THGROUP_ATT_YAWRIGHT, true);

		vessel->ClearThrusterDefinitions();

		SetStackStage (vessel);
	}
	else if (stage == 4)
	{
		vs1.vrot.x = 0.0;
		vs1.vrot.y = 0.0;
		vs1.vrot.z = 0.0;
		vs1.fuel=1.0;
		oapiCreateVessel("LOK", "n1lok", vs1);

		SetDescentStage (vessel);
	}
	else if (stage == 5)
	{
		vs1.vrot.x = 0.0;
		vs1.vrot.y = 0.0;
		vs1.vrot.z = 0.0;
		vs1.fuel=1.0;
		oapiCreateVessel("LK", "n1lk", vs1);	

		vessel->DelPropellantResource (ph_blockd);

		vessel->DelThrusterGroup (thg_blockd, THGROUP_MAIN, true);

		vessel->DelThrusterGroup (THGROUP_ATT_PITCHUP, true);
		vessel->DelThrusterGroup (THGROUP_ATT_PITCHDOWN, true);
		vessel->DelThrusterGroup (THGROUP_ATT_BANKLEFT, true);
		vessel->DelThrusterGroup (THGROUP_ATT_BANKRIGHT, true);
		vessel->DelThrusterGroup (THGROUP_ATT_YAWLEFT, true);
		vessel->DelThrusterGroup (THGROUP_ATT_YAWRIGHT, true);

		vessel->ClearThrusterDefinitions();

		SetDummyStage (vessel);
	}
}
	
void N1::TimeStep( double simt )
{
//	VESSELSTATUS2 vstatus;

	if (Set_First_Stage_Thlevel)
	{
		this->SetThrusterGroupLevel (THGROUP_MAIN, First_Stage_Thlevel);
		Set_First_Stage_Thlevel = false;

	}
	if (!guidance && !MECO_Cond ) {
		if(!NewIncSP) {
			// Set the Inclination setpoint for the guidance computer
			this->GetEquPos( Longitude, Latitude, Radius );
			if (PadHeading < 258.40 && PadHeading > 233.44) {
				Clairaut_Constant = sin((PadHeading - 180)*RAD) * cos(Latitude);
				INCLINATION = acos(Clairaut_Constant)*DEG;
			} else {
				BadHeading = true ;
			}		
		}
//		else {
//			vstatus.version =2; 
//          vstatus.flag =0;
//			this->GetStatusEx(&vstatus);
//			sprintf_s(oapiDebugString(), 51, "Heading = %.2f", vstatus.surf_hdg);
//			vstatus.surf_hdg = 60.;
//			this->DefSetStateEx(&vstatus);
//		}
	}

		
	//Check if guidance is on and if so, call
	if (guidance)
	{
	    DisplayIncSP = false; //switch off the display of INCLINATION
		if (BadHeading) {
			sprintf_s(oapiDebugString(), 51, "Guidance system cannot launch from this heading.");
			guidance = false;
		} else if (stage < 4) {

			// Calculate MET
			if (TZero == -1000) {
				MET = -10 ;
				TZero = 0;
			}
			else {
				MET = simt - TZero;
			}

			// Call the guidance routine
			Guidance_Routine(this, simt, SimT_Last);
			SimT_Last = MET;
		}
	}

	// Display the inclination setpoint if the switch is on.
	if (!DisplayIncSP) {
		if (offDisplayIncSP) {
			sprintf_s(oapiDebugString(), 1, "");
			offDisplayIncSP = false;
		}
	}
	else {
			sprintf_s(oapiDebugString(), 40, "Inclination Setpoint: %.2f deg", INCLINATION);
			offDisplayIncSP = true;
	}


	if (stage == 0)
	{

        if (this->GetPropellantMass(ph_first) == 0 || bManualSeparate)
		{
			stage_sep = simt;
			SeparateStage (this, stage);
			bManualSeparate=false;
			stage = 1;
		} 
	}
	
	else if (stage == 1)
	{
		if (this->GetPropellantMass(ph_second) == 0 || bManualSeparate)
		{
			stage_sep = simt;
			SeparateStage (this, stage);
			bManualSeparate=false;
			stage = 2;
		} 
		
	}
	else if (stage == 2)
	{
		if (this->GetPropellantMass(ph_third) == 0 || bManualSeparate)
		{
			stage_sep = simt;
			SeparateStage (this, stage);
			stage = 3;
			if (bManualSeparate) this->SetThrusterGroupLevel (thg_blockg,0.0);
			bManualSeparate=false;
		}
	}
	else if (stage == 3)
	{
		if (this->GetPropellantMass(ph_blockg) == 0 || bManualSeparate)
		{
			stage_sep = simt;
			bManualSeparate=false;
			SeparateStage (this, stage);
			stage = 4;
		}
	}

	else if (stage == 4)
	{
		if (bManualSeparate)
		{
			stage_sep = simt;
			bManualSeparate=false;
			SeparateStage (this, stage);
			stage = 5;
		}
	}

	else if (stage == 5)
	{
		if (bManualSeparate)
		{
			stage_sep = simt;
			bManualSeparate=false;
			SeparateStage (this, stage);
			stage = 6;
		}
	}
}
	
void N1::SetClassCaps (VESSEL *vessel, FILEHANDLE cfg)
{

}
	
void N1::LoadStateEx (VESSEL *vessel, FILEHANDLE scn, void *vs)
{
	char *line;

	while (oapiReadScenario_nextline (scn, line)) 
	{
        if (!_strnicmp (line, "CONFIGURATION", 13)) 
		{
            sscanf_s (line+13, "%d", &status);
		} 
		else if (!_strnicmp (line, "HEADING", 7)) 
		{
			sscanf_s(line+7, "%lf", &PadHeading);
			vessel->ParseScenarioLineEx (line, vs);
		} 
		else if (!_strnicmp (line, "THIRD_STAGE_PROPULSION", 22) )
		{
			int tmp = 0;
			sscanf_s(line+22, "%ld", &tmp);
			Third_Stage_Propultion = tmp;
		}
		else if (!_strnicmp (line, "FIRST_STAGE_THLEVEL", 19) )
		{
			float f_tmp = 0;
			sscanf_s(line+19, "%f", &f_tmp);
			First_Stage_Thlevel = (double)f_tmp;
			Set_First_Stage_Thlevel = true;
		}
		else if (!_strnicmp (line, "GUIDANCE", 8) )
		{
			guidance = true;
		}

		else if (!_strnicmp (line, "T_minus_10", 10) )
		{
			int tmp = 0;
			sscanf_s(line+10, "%ld", &tmp);
			if (tmp == 0) T_minus_10 = false;
			else T_minus_10 = true;
			sprintf_s(oapiDebugString(), 40, "T_minus_10 = %d", tmp);
		}

		else if (!_strnicmp (line, "Alt_Flag", 8) )
		{
			int tmp = 0;
			sscanf_s(line+8, "%ld", &tmp);
			if (tmp == 0) Alt_Flag = false;
			else Alt_Flag = true;
		}

		else if (!_strnicmp (line, "MECO_Cond", 9) )
		{
			int tmp = 0;
			sscanf_s(line+9, "%ld", &tmp);
			if (tmp == 0) MECO_Cond = false;
			else MECO_Cond = true;
		}
	
		else if (!_strnicmp (line, "First_Guidance_Pass", 19) )
		{
			int tmp = 0;
			sscanf_s(line+19, "%ld", &tmp);
			if (tmp == 0) First_Guidance_Pass = false;
			else First_Guidance_Pass = true;
		}
		else if (!_strnicmp (line, "PostMECO", 8) )
		{
			int tmp = 0;
			sscanf_s(line+8, "%ld", &tmp);
			if (tmp == 0) PostMECO = false;
			else PostMECO = true;
		}

		else if (!_strnicmp (line, "MET", 3) )
		{
			float f_tmp = 0;
			sscanf_s(line+3, "%f", &f_tmp);
			TZero = -(double)f_tmp;
			SimT_Last = -TZero;
			Tcorr = -TZero;

		}

		else if (!_strnicmp (line, "INCLINATION", 11) )
		{
			float f_tmp = 0;
			sscanf_s(line+11, "%f", &f_tmp);
			INCLINATION = (double)f_tmp;
		}

		else if (!_strnicmp (line, "i_last", 6) )
		{
			float f_tmp = 0;
			sscanf_s(line+6, "%f", &f_tmp);
			i_last = (double)f_tmp;
		}
		else 
		{
            vessel->ParseScenarioLineEx (line, vs);
			// unrecognised option - pass to Orbiter's generic parser
        }
    }
	switch (status) {
	case 0:
		stage=0;
		SetFirstStage(vessel);
		break;
	case 1:
		stage=1;
		SetSecondStage(vessel);
		break;
	case 2:
		stage=2;
		SetThirdStage(vessel);
		break;
	case 3:
		stage=3;
		SetBlockGStage(vessel);
		break;
	case 4:
		stage=4;
		SetStackStage(vessel);
		break;
	case 5:
		stage=5;
		SetDescentStage(vessel);
		break;
	case 6:
		stage=6;
		SetDummyStage(vessel);
		break;

	}
}
	
void N1::SaveState (VESSEL *vessel, FILEHANDLE scn)
{
	// default vessel parameters
	vessel->SaveDefaultState (scn);

	// custom parameters
	oapiWriteScenario_int (scn, "CONFIGURATION", status);
	oapiWriteScenario_int (scn, "THIRD_STAGE_PROPULSION", status);

	if (stage == 0) oapiWriteScenario_float (scn, "FIRST_STAGE_THLEVEL", (float)vessel->GetThrusterGroupLevel(thg_first));
	if (guidance) 
	{
		oapiWriteScenario_string (scn, "GUIDANCE", "ON");

		if (T_minus_10) oapiWriteScenario_int (scn, "T_minus_10", 1);
		else oapiWriteScenario_int (scn, "T_minus_10", 0);

		if (Alt_Flag) oapiWriteScenario_int (scn, "Alt_Flag", 1);
		else oapiWriteScenario_int (scn, "Alt_Flag", 0);

		if (MECO_Cond) oapiWriteScenario_int (scn, "MECO_Cond", 1);
		else oapiWriteScenario_int (scn, "MECO_Cond", 0);

		if (First_Guidance_Pass) oapiWriteScenario_int (scn, "First_Guidance_Pass", 1);
		else oapiWriteScenario_int (scn, "First_Guidance_Pass", 0);

		if (PostMECO) oapiWriteScenario_int (scn, "PostMECO", 1);
		else oapiWriteScenario_int (scn, "PostMECO", 0);

		oapiWriteScenario_float (scn, "MET", (float)(MET));
		oapiWriteScenario_float (scn, "INCLINATION", (float)INCLINATION);
		oapiWriteScenario_float (scn, "i_last", (float)i_last);

	}
}


void N1::InitPanel( int panel )
{


	switch (panel) {
	case 0: // main panel
		srf[0] = oapiCreateSurface (LOADBMP (IDB_PANEL1));
		break;
	}
}

BOOL WINAPI DllMain (HINSTANCE hModule,
					 DWORD ul_reason_for_call,
					 LPVOID lpReserved)
{
	int i;

	switch (ul_reason_for_call) {
	case DLL_PROCESS_ATTACH:
		g_Param.hDLL = hModule;
		// allocate GDI resources
		g_Param.font[0]  = CreateFont (-13, 0, 0, 0, 700, 0, 0, 0, 0, 0, 0, 0, 0, "Arial");
		g_Param.font[1]  = CreateFont (-10, 0, 0, 0, 400, 0, 0, 0, 0, 0, 0, 0, 0, "Arial");
//		g_Param.brush[0] = CreateSolidBrush (RGB(0,255,0));    // green
//		g_Param.brush[1] = CreateSolidBrush (RGB(255,0,0));    // red
//		g_Param.brush[2] = CreateSolidBrush (RGB(80,80,224));  // blue
//		g_Param.brush[3] = CreateSolidBrush (RGB(160,120,64));  // brown
//		g_Param.brush[4] = CreateSolidBrush (RGB(210,210,210)); // white
		g_Param.pen[0] = CreatePen (PS_SOLID, 1, RGB(224,224,224));
		g_Param.pen[1] = CreatePen (PS_SOLID, 3, RGB(164,164,164));
		break;
	case DLL_PROCESS_DETACH:
		// deallocate GDI resources
		for (i = 0; i < 2; i++) DeleteObject (g_Param.font[i]);
//		for (i = 0; i < 5; i++) DeleteObject (g_Param.brush[i]);
		for (i = 0; i < 2; i++) DeleteObject (g_Param.pen[i]);
		break;
	}
	return TRUE;
}
// ==============================================================
// API interface
// ==============================================================

// Initialisation
DLLCLBK VESSEL *ovcInit (OBJHANDLE hvessel, int flightmodel)
{
	N1* n1 = new N1( hvessel, flightmodel );
	for (int i = 0; i < n1->nsurf; i++) n1->srf[i] = 0;
	return n1;

}

// Cleanup
DLLCLBK void ovcExit (VESSEL *vessel)
{

	if (vessel) delete vessel;

}

// Keyboard interface handler
DLLCLBK int ovcConsumeKey (VESSEL *vessel, const char *keystate)
{
	N1 *n1 = (N1*)vessel;

	if (KEYMOD_SHIFT (keystate)) 
	{
		return 0; // shift combinations are reserved
	}
	else if (KEYMOD_CONTROL (keystate)) 
	{
		// insert ctrl key combinations here
		if (KEYDOWN (keystate, OAPI_KEY_U))
		{ 
			if (oapiAcceptDelayedKey (OAPI_KEY_U, 1.0))
				// Report the inclination setpoint
				if(n1->DisplayIncSP) {
					n1->DisplayIncSP = false;
				}
				else {
					n1->DisplayIncSP = true;
				}
																
			return 1;
		}

	}
	else 
	{ // unmodified keys
		if (KEYDOWN (keystate, OAPI_KEY_J)) 
		{ // "Jettison Stage"
			if (oapiAcceptDelayedKey (OAPI_KEY_J, 1.0))
				n1->bManualSeparate = true;
			return 1;
		}
		if (KEYDOWN (keystate, OAPI_KEY_O))
		{ 
			if (oapiAcceptDelayedKey (OAPI_KEY_O, 1.0))
				if (!n1->guidance) { 
					n1->guidance=true;
					//sprintf(oapiDebugString(),"Guidance ON");
				}
				else {
					n1->guidance = false;
					//sprintf(oapiDebugString(),"Guidance OFF");
				}
			return 1;
		}
		if (KEYDOWN (keystate, OAPI_KEY_U))
		{ 
			if (oapiAcceptDelayedKey (OAPI_KEY_U, 1.0))
				// Get a new Inclination setpoint from pilot.
				if(!n1->guidance) {
					n1->GetInclinationSP() ;
				}
				
			return 1;
		}
	}
return 0;
}

DLLCLBK void ovcSetClassCaps (VESSEL *vessel, FILEHANDLE cfg)
{
	N1* n1 = (N1*)vessel;

	n1->SetClassCaps( n1, cfg );
}


DLLCLBK void ovcTimestep (VESSEL *vessel, double simt)
{
	N1* n1 = (N1*)vessel;
	n1->TimeStep( simt );
}

// Read status from scenario file
DLLCLBK void ovcLoadStateEx (VESSEL *vessel, FILEHANDLE scn, void *vs)
{
	N1* n1 = (N1*)vessel;
	n1->LoadStateEx( n1, scn, vs);
}

// Save status to scenario file
DLLCLBK void ovcSaveState (VESSEL *vessel, FILEHANDLE scn)
{
	N1* n1 = (N1*)vessel;
	n1->SaveState( n1, scn );
}

DLLCLBK bool ovcLoadPanel (VESSEL *vessel, int id)
{

	N1 *n1 = (N1*)vessel;
	n1->ReleaseSurfaces();

	HBITMAP hBmp = LoadBitmap ( g_Param.hDLL, MAKEINTRESOURCE (IDB_PANEL1));

	switch (id) {
	case 0: // main panel
		oapiRegisterPanelBackground (hBmp, PANEL_ATTACH_BOTTOM, 0xffffff);
		oapiSetPanelNeighbours (-1,-1,-1,-1);
		
	}

	n1->InitPanel (id);
	return hBmp != NULL;
}