// ==============================================================
// ORBITER vessel module: LOK Lunar Orbiter
// ==============================================================
// Smoke particle effects and C++ class conversion by Dave Rowbotham
//
#include "Orbitersdk.h"
#include "cstdio"

#include "lok.h"

const VECTOR3 OFS_ORBITAL = { 0.18-0.15,0.05,3.675};
const VECTOR3 OFS_SERVICE = { -0.15-0.15,0.25,0};
const VECTOR3 OFS_CHUTE = {0.15-0.15,0,-4.75-2.0};

 

const double MAX_ATT_LOK = 1800;
const double MAX_ATT_RCS = 67;
const double THRUST_LOK_VAC   = 3388*9.81;
const double THRUST_RETRO_VAC   = 1388*9.81;
const double ISP_LOK_VAC  = 314*9.81;
const double ISP_RETRO_VAC  = 314*9.81;
const double LOK_PROP_MASS = 18300;
const double RCS_PROP_MASS = 30;



double LiftCoeff (double aoa)
{
const int nlift = 9;
static const double AOA[nlift] =
{-180*RAD,-155*RAD,-120*RAD,-20*RAD,0*RAD,20*RAD,120*RAD,155*RAD,180*RAD};
static const double CL[nlift]  = {0.12,  0.21,       0,      0,    0,     0,      0,      0.21,0.12};
static const double SCL[nlift] = {(CL[1]-CL[0])/(AOA[1]-AOA[0]),
(CL[2]-CL[1])/(AOA[2]-AOA[1]),
                              (CL[3]-CL[2])/(AOA[3]-AOA[2]),
(CL[4]-CL[3])/(AOA[4]-AOA[3]),
  (CL[5]-CL[4])/(AOA[5]-AOA[4]), (CL[6]-CL[5])/(AOA[6]-AOA[5]),
  (CL[7]-CL[6])/(AOA[7]-AOA[6]), (CL[8]-CL[7])/(AOA[8]-AOA[7])};
int i;
for (i = 0; i < nlift-1 && AOA[i+1] < aoa; i++);
return CL[i] + (aoa-AOA[i])*SCL[i];
}



LOK::LOK( OBJHANDLE hObj, int fmodel ) : 
   VESSEL( hObj, fmodel )
{

	bManualSeparate = false;
	stage_sep = 0;
	stage = 0;
	status = 0;

		refcount = 0;
		hLOK1 = oapiLoadMeshGlobal ("n1lok1");
		hLOK2 = oapiLoadMeshGlobal ("n1lok2");
		hLOK3 = oapiLoadMeshGlobal ("n1lok3");
		hLOKchute = oapiLoadMeshGlobal ("n1chute");
}


LOK::~LOK()
{

}

void LOK::SetAttControlsLOK(VESSEL *vessel)
{
	THRUSTER_HANDLE th_rot[2];
	th_rot[0] = vessel->CreateThruster (_V(0,-0.2,-2.0), _V(0,-1,0), MAX_ATT_LOK , ph_lok, ISP_LOK_VAC);
	th_rot[1] = vessel->CreateThruster (_V(0,0.75,5.25), _V(0,1,0), MAX_ATT_LOK , ph_lok, ISP_LOK_VAC);
	vessel->AddExhaust (th_rot[0], 1.5, 0.05, _V( 0,  0.75,-1.0  ), _V(0, 1.0,0));
	vessel->AddExhaust (th_rot[1], 1.5, 0.05, _V( 0,  -0.5, 3.75  ), _V(0,-1.0,0));
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_PITCHUP);

	th_rot[0] = vessel->CreateThruster (_V(0,-0.2,-5.25), _V(0,1,0), MAX_ATT_LOK , ph_lok, ISP_LOK_VAC);
	th_rot[1] = vessel->CreateThruster (_V(0,-0.75,5.25), _V(0,1,0), MAX_ATT_LOK , ph_lok, ISP_LOK_VAC);
	vessel->AddExhaust (th_rot[0], 1.75, 0.05, _V( 0,  -0.6,-1.0 ), _V(0, -1.0,0));
	vessel->AddExhaust (th_rot[1], 1.5, 0.05, _V( 0,  -0.5, 3.75  ), _V(0,-1.0,0));
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_UP);

	th_rot[0] = vessel->CreateThruster (_V(0,-0.75,5.25), _V(0,-1,0), MAX_ATT_LOK , ph_lok, ISP_LOK_VAC);
	th_rot[1] = vessel->CreateThruster (_V(0,0.35,-2.0), _V(0, 1,0), MAX_ATT_LOK , ph_lok, ISP_LOK_VAC);
	vessel->AddExhaust (th_rot[1], 1.5, 0.05, _V( 0,  -0.6,-1.0  ), _V(0, -1.0,0));
	vessel->AddExhaust (th_rot[0], 1.5, 0.05, _V( 0,  0.5, 3.75  ), _V(0,1.0,0));
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_PITCHDOWN);

	th_rot[0] = vessel->CreateThruster (_V(0,-0.2,-5.25), _V(0,-1,0), MAX_ATT_LOK , ph_lok, ISP_LOK_VAC);
	th_rot[1] = vessel->CreateThruster (_V(0,-0.75,5.25), _V(0,-1,0), MAX_ATT_LOK , ph_lok, ISP_LOK_VAC);
	vessel->AddExhaust (th_rot[0], 1.75, 0.05, _V( 0,  0.75,-1.0 ), _V(0, 1.0,0));
	vessel->AddExhaust (th_rot[1], 1.5, 0.05, _V( 0,  0.6, 3.75  ), _V(0,1.0,0));
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_DOWN);

	th_rot[0] = vessel->CreateThruster (_V(-2,1.25,-0.3), _V(0,-1,0), MAX_ATT_LOK , ph_lok, ISP_LOK_VAC);
	th_rot[1] = vessel->CreateThruster (_V(2, 1.25,-0.3), _V(0,1,0), MAX_ATT_LOK , ph_lok, ISP_LOK_VAC);
	vessel->AddExhaust (th_rot[1], 1.5, 0.05, _V( -1.15, 0.0,0.20  ), _V(0, 1.0,0));
	vessel->AddExhaust (th_rot[0], 1.5, 0.05, _V(  1.15, 0.0,0.20  ), _V(0,-1.0,0));
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_BANKLEFT);

	th_rot[0] = vessel->CreateThruster (_V(2,0.0,0.0), _V(-1,0,0), MAX_ATT_LOK , ph_lok, ISP_LOK_VAC);
	vessel->AddExhaust (th_rot[0], 1.5, 0.05, _V(  1.15, 0.0,0.20 ), _V(1,0,0));
	vessel->CreateThrusterGroup (th_rot, 1, THGROUP_ATT_LEFT);

	th_rot[0] = vessel->CreateThruster (_V(-2,1.25,-0.3), _V(0, 1,0), MAX_ATT_LOK , ph_lok, ISP_LOK_VAC);
	th_rot[1] = vessel->CreateThruster (_V(2,1.25,-0.3), _V(0,-1,0), MAX_ATT_LOK , ph_lok, ISP_LOK_VAC);
	vessel->AddExhaust (th_rot[1], 1.5, 0.05, _V( -1.15, 0.0,0.20  ), _V(0,-1.0,0));
	vessel->AddExhaust (th_rot[0], 1.5, 0.05, _V(  1.15, 0.0,0.20  ), _V(0, 1.0,0));
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_BANKRIGHT);

	th_rot[0] = vessel->CreateThruster (_V(2,0.0,0.0), _V(1,0,0), MAX_ATT_LOK , ph_lok, ISP_LOK_VAC);
	vessel->AddExhaust (th_rot[0], 1.5, 0.05, _V(  -1.15, 0.0,0.20 ), _V(-1,0,0));
	vessel->CreateThrusterGroup (th_rot, 1, THGROUP_ATT_RIGHT);

	th_rot[0] = vessel->CreateThruster (_V(0,0, 4), _V( 1,0,0), MAX_ATT_LOK , ph_lok, ISP_LOK_VAC);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-4), _V(-1,0,0), MAX_ATT_LOK , ph_lok, ISP_LOK_VAC);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_YAWRIGHT);
	th_rot[0] = vessel->CreateThruster (_V(0,0, 4), _V(-1,0,0), MAX_ATT_LOK , ph_lok, ISP_LOK_VAC);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-4), _V( 1,0,0), MAX_ATT_LOK , ph_lok, ISP_LOK_VAC);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_YAWLEFT);

	th_rot[0] = vessel->CreateThruster (_V(0.0,0.0,-4.0), _V(0.0,0,1.0), MAX_ATT_LOK , ph_lok, ISP_LOK_VAC);
	vessel->CreateThrusterGroup (th_rot, 1, THGROUP_ATT_FORWARD);

	th_rot[0] = vessel->CreateThruster (_V(0.0,0.0,4.0), _V(0.0,0,-1.0), MAX_ATT_LOK , ph_lok, ISP_LOK_VAC);
	vessel->CreateThrusterGroup (th_rot, 1, THGROUP_ATT_BACK);
}

void LOK::SetAttControlsRCS(VESSEL *vessel)
{
	THRUSTER_HANDLE th_rot[2];
	th_rot[0] = vessel->CreateThruster (_V(0,0, 2), _V(0, 1,0), MAX_ATT_RCS , ph_rcs, ISP_LOK_VAC);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-2), _V(0,-1,0), MAX_ATT_RCS , ph_rcs, ISP_LOK_VAC);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_PITCHUP);
	th_rot[0] = vessel->CreateThruster (_V(0,0, 2), _V(0,-1,0), MAX_ATT_RCS , ph_rcs, ISP_LOK_VAC);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-2), _V(0, 1,0), MAX_ATT_RCS , ph_rcs, ISP_LOK_VAC);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_PITCHDOWN);
	th_rot[0] = vessel->CreateThruster (_V( 1.5,0,0), _V(0, 1,0), MAX_ATT_RCS , ph_rcs, ISP_LOK_VAC);
	th_rot[1] = vessel->CreateThruster (_V(-1.5,0,0), _V(0,-1,0), MAX_ATT_RCS , ph_rcs, ISP_LOK_VAC);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_BANKLEFT);
	th_rot[0] = vessel->CreateThruster (_V( 1.5,0,0), _V(0,-1,0), MAX_ATT_RCS , ph_rcs, ISP_LOK_VAC);
	th_rot[1] = vessel->CreateThruster (_V(-1.5,0,0), _V(0, 1,0), MAX_ATT_RCS , ph_rcs, ISP_LOK_VAC);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_BANKRIGHT);
	th_rot[0] = vessel->CreateThruster (_V(0,0, 1), _V( 1,0,0), MAX_ATT_RCS , ph_rcs, ISP_LOK_VAC);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-1), _V(-1,0,0), MAX_ATT_RCS , ph_rcs, ISP_LOK_VAC);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_YAWRIGHT);
	th_rot[0] = vessel->CreateThruster (_V(0,0, 1), _V(-1,0,0), MAX_ATT_RCS , ph_rcs, ISP_LOK_VAC);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-1), _V( 1,0,0), MAX_ATT_RCS , ph_rcs, ISP_LOK_VAC);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_YAWLEFT);
}

void LOK::SetLOKStage(VESSEL *vessel)
{
	PARTICLESTREAMSPEC contrail = {
		0, 8.0, 5, 150, 0.3, 4.0, 4, 3.0, PARTICLESTREAMSPEC::DIFFUSE,
		PARTICLESTREAMSPEC::LVL_SQRT, 0, 1,
		PARTICLESTREAMSPEC::ATM_PLOG, 1e-4, 1
	};
	PARTICLESTREAMSPEC exhaust_main = {
		0, 2.0, 20, 150, 0.1, 0.2, 16, 1.0, PARTICLESTREAMSPEC::EMISSIVE,
		PARTICLESTREAMSPEC::LVL_SQRT, 0, 1,
		PARTICLESTREAMSPEC::ATM_PLOG, 1e-5, 0.1
	};
	vessel->SetSize (7);
	vessel->SetCOG_elev (3.5);
	vessel->SetEmptyMass (5500.0);
	vessel->SetPMI (_V(5.5,5.5,0.75));
	vessel->SetCrossSections (_V(16,15,5.75));
	vessel->SetCW (0.1, 0.3, 1.4, 1.4);
	vessel->SetRotDrag (_V(0.7,0.7,1.2));
	vessel->SetPitchMomentScale (0);
	vessel->SetBankMomentScale (0);
	vessel->SetLiftCoeffFunc (0); 
    vessel->ClearMeshes();
	vessel->SetDockParams (_V(0.0,0,4.55), _V(0,0,1), _V(0,1,0));
//	vessel->SetDockParams (_V(0.2,0,-4.55), _V(0,0,-1), _V(0,1,0));
	VECTOR3 mesh_dir=OFS_SERVICE;
	vessel->AddMesh (hLOK3, &mesh_dir);	
	mesh_dir=_V(0.175-0.15,-0.11,1.775);
	vessel->AddMesh (hLOK2, &mesh_dir);
	mesh_dir=OFS_ORBITAL;
	vessel->AddMesh (hLOK1, &mesh_dir);
	if (!ph_lok) ph_lok  = vessel->CreatePropellantResource (LOK_PROP_MASS); 
	vessel->SetDefaultPropellantResource (ph_lok); 
	vessel->ClearThrusterDefinitions();
	th_lok[0] = vessel->CreateThruster (_V(0,0,-1.05), _V(0,0,1), THRUST_LOK_VAC, ph_lok, ISP_LOK_VAC);
		vessel->AddExhaustStream (th_lok[0], _V(0,0,-1.05), &contrail);
	vessel->AddExhaustStream (th_lok[0], _V(0,0,-1.05), &exhaust_main);
	thg_lok = vessel->CreateThrusterGroup (th_lok, 1, THGROUP_MAIN);
	th_retro[0] = vessel->CreateThruster (_V(0,0,1.05), _V(0,0,-1), THRUST_RETRO_VAC, ph_lok, ISP_RETRO_VAC);
	thg_retro = vessel->CreateThrusterGroup (th_retro, 1, THGROUP_RETRO);
	vessel->SetThrusterGroupLevel (thg_lok,0.0);
	vessel->AddExhaust (th_lok[0], 2.0, 0.25, _V(0.175-0.15,0.05,-1.05), _V(0,0,-1));
	SetAttControlsLOK(vessel);
	vessel->EnableTransponder(true);
	status=0;
}



void LOK::SetLOSStage(VESSEL *vessel)
{
	vessel->SetSize (1.65);
	vessel->SetCOG_elev (1.0);
	vessel->SetEmptyMass (1810.0);
	vessel->SetPMI (_V(1.3,0.6,1.5));
	vessel->SetCrossSections (_V(3.5,3.5,3.75));
	vessel->SetCW (0.1, 0.3, 1.4, 1.4);
	vessel->SetRotDrag (_V(0.7,0.7,0.3));
	vessel->SetPitchMomentScale (0);
	vessel->SetBankMomentScale (0);
	vessel->SetLiftCoeffFunc (0);
    vessel->ClearMeshes();
	vessel->ShiftCentreOfMass (_V(0.18-0.15,0.05,3.675));
	VECTOR3 mesh_dir=_V(0,0,0);
	vessel->AddMesh (hLOK1, &mesh_dir);	
	vessel->SetDockParams (_V(0.02,-0.05,0.9), _V(0,0,1), _V(0,1,0));
	vessel->ClearPropellantResources();
	vessel->ClearThrusterDefinitions();
	vessel->EnableTransponder(true);
	status=1;
}

void LOK::SeparateStage (VESSEL *vessel, UINT stage)
{
	VESSELSTATUS vs;
	OBJHANDLE hNewVessel;
	double dist;
	VECTOR3 ofs = _V(0,0,0);
	VECTOR3 vel = _V(0,0,0);
	VECTOR3 rofs, rvel;


	if (stage == 0)
	{
//		ofs = _V(0.175,-0.11,1.775);
		vel = _V(0,0, -0.5);
	}
	
	vessel->GetStatus (vs);
	vs.eng_main = vs.eng_hovr = 0.0;
	
	if (stage == 0)
	{
		rvel = _V(vs.rvel.x, vs.rvel.y, vs.rvel.z);
//		vessel->Local2Rel (ofs, vs.rpos);
		vessel->GlobalRot (vel, rofs);
		vs.rvel.x = rvel.x+rofs.x;
		vs.rvel.y = rvel.y+rofs.y;
		vs.rvel.z = rvel.z+rofs.z;
		vs.vrot.x = 0.0;
		vs.vrot.y = 0.0;
		vs.vrot.z = 0.0;
        hNewVessel = oapiCreateVessel ("LOK ZOND", "n1lok_zond", vs);
		SetLOSStage (vessel);
		dist = oapiCameraTargetDist();
		oapiSetFocusObject(hNewVessel);
		oapiCameraScaleDist( dist / oapiCameraTargetDist() );

		stage = 1;
	}

}


void LOK::SetClassCaps( FILEHANDLE cfg )
{

	ph_lok			 = NULL;
	ph_rcs			 = NULL;
	thg_lok	         = NULL;
	thg_retro        = NULL;

	SetLOKStage( this );


}

void LOK::LoadState( VESSEL *vessel, FILEHANDLE scn , void *vs)
{

    char *line;

	while (oapiReadScenario_nextline (scn, line)) {
        if (!_strnicmp (line, "CONFIGURATION_LOK", 17)) {
            sscanf_s (line+17, "%d", &status);
		} 
		else {
            vessel->ParseScenarioLineEx (line, vs);
			// unrecognised option - pass to Orbiter's generic parser
        }
    }
	switch (status) {
	case 0:
		stage=0;
		SetLOKStage(vessel);
		break;
	case 1:
		stage=1;
		SetLOSStage(vessel);
		break;
	}

}

void LOK::SaveState( FILEHANDLE scn )
{
	SaveDefaultState (scn);
	oapiWriteScenario_int (scn, "CONFIGURATION_LOK", status);
}


void LOK::TimeStep( double simt )
{
	if (stage == 0)
	{      
		if (bManualSeparate)
		{
			SeparateStage (this, stage);
			bManualSeparate=false;
			stage = 1;
		} 
	}

}


// ==============================================================
// API interface
// ==============================================================

// Initialisation
DLLCLBK VESSEL *ovcInit (OBJHANDLE hvessel, int flightmodel)
{
	LOK* lok = new LOK( hvessel, flightmodel );

	return lok;
}

// Cleanup
DLLCLBK void ovcExit (VESSEL *vessel)
{
	if (vessel) delete vessel;
}

// Keyboard interface handler
DLLCLBK int ovcConsumeKey (VESSEL *vessel, const char *keystate)
{
	LOK* lok = (LOK*)vessel;

	if (KEYMOD_SHIFT (keystate)) 
	{
		return 0; // shift combinations are reserved
	}
	else if (KEYMOD_CONTROL (keystate)) 
	{
		// insert ctrl key combinations here
	}
	else 
	{ // unmodified keys
		if (KEYDOWN (keystate, OAPI_KEY_J)) { // "Jettison Stage"
			if (oapiAcceptDelayedKey (OAPI_KEY_J, 1.0))
				lok->bManualSeparate = true;
			return 1;
		}
		
	}
	return 0;
}

// Set the capabilities of the vessel class
DLLCLBK void ovcSetClassCaps (VESSEL *vessel, FILEHANDLE cfg)
{
	LOK* lok = (LOK*)vessel;
	lok->SetClassCaps( cfg );
}

DLLCLBK void ovcTimestep (VESSEL *vessel, double simt)
{

	LOK* lok = (LOK*)vessel;
	lok->TimeStep( simt );
}

// Read status from scenario file

DLLCLBK void ovcLoadStateEx (VESSEL *vessel, FILEHANDLE scn, void *vs)

{

	LOK* lok = (LOK*)vessel;
	lok->LoadState( lok, scn , vs);

}

// Save status to scenario file

DLLCLBK void ovcSaveState (VESSEL *vessel, FILEHANDLE scn)
{
	LOK* lok = (LOK*)vessel;
	// default vessel parameters
	lok->SaveState (scn);
	// custom parameters
//	oapiWriteScenario_int (scn, "CONFIGURATION_LOK", status);
}

DLLCLBK void ovcPostCreation (VESSEL *vessel)
{
//	vessel->SetNavRecv(0,(DWORD)588);
//	vessel->SetNavRecv(1,(DWORD)598);
}