// ==============================================================
// ORBITER modules - helper file.
//
// VESSEL-derived base class PRVESSEL for Laika lunar probe vessels, uses TProximator member class.
//
// TProximator class, handles:
//   - planet proximity detection;
//   - atmosphere detection;
//   - vessel proximity detection;
//   - contacts with planet surface and collisions with other vessels; 
//   - telemetry (output to log file).
//
// Developed by: Andrew Thielmann (igel)
//
// ==============================================================

#ifndef __PROXIMATOR_H
#define __PROXIMATOR_H

#include "..\..\include\orbitersdk.h"

// useful enumeration, better be visible from everywhere
enum fragment_id { FID_LANDER, FID_KTDU, FID_FRAME, FID_BO, FID_BG, FID_PO, FID_VNA, FID_RADAR };

	// useful random functions 
// random number between 0 and 1
inline double rand01() { return (double)rand() / (double)RAND_MAX; }
// random number between -1 and 1
inline double rand11() { return (double)rand() / (double)RAND_MAX * 2 - 0.5; }


class PRVESSEL: public VESSEL {
public:
	PRVESSEL (OBJHANDLE hVessel, int fmodel = 1);
	~PRVESSEL ();
	virtual void EnteringPlanet();				// call when 100 Mm to planet 
	virtual void ApproachingPlanet();           // call when 200 km to planet 
	virtual void Burn(OBJHANDLE HTarget);       // call when structural limit on maxQ is reached
	virtual void DoneLanding(OBJHANDLE HTarget, double V, double Vv, double Vh);
	virtual void Collide(OBJHANDLE HHit, double Vhit, OBJHANDLE HSwitch); 

	double dQTreshold;			// maximum allowed Q
};


class TProximator {
public:
		// initialize, link to a parent vessel
	TProximator(VESSEL *AVessel);
	~TProximator ();
	int ParsedScenarioLine (char *scnLine);
	void SaveState (FILEHANDLE scn);
	void Timestep (double simt);
    void UpdateFromVessel();

	// useful vars stored
	PRVESSEL *Vessel;
	char Name[256];
	double dSize;

	// important automation flags
	bool bAuto;
	bool bStruct;

		// planet and atmosphere proximity 
	void CheckPlanetProximity(double H);
	void CheckAtmosphere(double H);
	void CheckSurface(double H);
	void Burn();
	void DoneLanding(double H);
	
	// flags
	bool bEnteringPlanet;			// < 100 Mm
	bool bCloseToPlanet;			// < 200 km
	bool bInAtmosphere;				
	bool bOnSurface;

	// structural tresholds
	double dQTreshold;
	double dVCollTreshold;
	double dVLandTreshold;
	double dVCrashTreshold;

		// vessel proximity
	void CheckCollision();
	double CalcMinVesselDistance(OBJHANDLE *vHandle);
	void Collide(OBJHANDLE vHandle);

	void CalcVesselProximityDelay();
	double dVesselProximityDelay;		// 5 sec should be OK
	double dPrevStep;

		// telemetry, debugging, other
	void Tral (char *fmt, ...);
	void DebugSomething();
	OBJHANDLE FindFocusSwitchTarget();
    char TralFileName[256];

};
#endif // !__PROXIMATOR_H