// ==============================================================
// ORBITER modules - helper file.
//
// See Proximator.h for description.
//
// Developed by: Andrew Thielmann (igel)
//
// ==============================================================

#include "Proximator.h"
#include "stdio.h"


// ---------------- PRVESSEL (Vessel with a Proximator) superclass


PRVESSEL::PRVESSEL (OBJHANDLE hObj, int fmodel): VESSEL (hObj, fmodel)
{
}


PRVESSEL::~PRVESSEL ()
{
}


void PRVESSEL::EnteringPlanet()
{
}

void PRVESSEL::ApproachingPlanet() 
{
}


void PRVESSEL::Burn(OBJHANDLE HTarget) 
{
}


void PRVESSEL::DoneLanding(OBJHANDLE HTarget, double V, double Vv, double Vh)
{
}


void PRVESSEL::Collide(OBJHANDLE HHit, double Vhit, OBJHANDLE HSwitch)
{
}



// ----------------  Proximator class

TProximator::TProximator (VESSEL *AVessel)
{
	// store vessel info
	Vessel = (PRVESSEL*)AVessel;
	strcpy(Name, Vessel->GetName());

	// default initialization
	strcpy(TralFileName, "");

	bAuto = true;
	bStruct = true;

	bEnteringPlanet = false;
	bCloseToPlanet = false;
	bInAtmosphere = false;
	bOnSurface = false;

	dVesselProximityDelay = 5;		// start with 5 sec delay
	dPrevStep = oapiGetSimTime();

	UpdateFromVessel();
}


TProximator::~TProximator ()
{
}


void TProximator::SaveState (FILEHANDLE scn)
{
	oapiWriteScenario_int (scn, "PR_AUTO", bAuto);
	oapiWriteScenario_int (scn, "PR_STRUCT", bStruct);
	oapiWriteScenario_int (scn, "PR_PLANET", bCloseToPlanet);
	oapiWriteScenario_int (scn, "PR_INATM", bInAtmosphere);
	oapiWriteScenario_int (scn, "PR_SURFACE", bOnSurface);
	oapiWriteScenario_string (scn, "PR_TRALFILE", TralFileName);
}



int TProximator::ParsedScenarioLine (char *scnLine)
{
	if (!strnicmp (scnLine, "PR_TRALFILE", 11)) 
	{
		sscanf (scnLine + 11, "%s", TralFileName);
		if (strlen(TralFileName) > 0)
		{
			// reset Tral file
			FILE *TralFile;
			TralFile = fopen(TralFileName, "w");
			fclose(TralFile);
		}
		return TRUE;
	}
	else if (!strnicmp (scnLine, "PR_AUTO", 7)) 
	{
		sscanf (scnLine + 7, "%1d", &bAuto);
		return TRUE;
	}
	else if (!strnicmp (scnLine, "PR_STRUCT", 9)) 
	{
		sscanf (scnLine + 9, "%1d", &bStruct);
		return TRUE;
	}
	else if (!strnicmp (scnLine, "PR_PLANET", 9)) 
	{
		sscanf (scnLine + 9, "%1d", &bCloseToPlanet);
		return TRUE;
	}
	else if (!strnicmp (scnLine, "PR_INATM", 8)) 
	{
		sscanf (scnLine + 8, "%1d", &bInAtmosphere);
		return TRUE;
	}
	else if (!strnicmp (scnLine, "PR_SURFACE", 10)) 
	{
		sscanf (scnLine + 10, "%1d", &bOnSurface);
		return TRUE;
	}
	else 
		return FALSE;
}


void TProximator::Timestep (double simt)
{
	double H = Vessel->GetAltitude();

	CheckPlanetProximity(H);

	if (bCloseToPlanet)
	{
		CheckAtmosphere(H);
		CheckSurface(H);
	}

	if (bStruct)
		CheckCollision();
}


void TProximator::UpdateFromVessel()
{
	dSize = Vessel->GetSize();
	dQTreshold = Vessel->dQTreshold;      // ro * v**2 / 2
}


void TProximator::CheckPlanetProximity(double H)
{
	// close approach
	if (!(bCloseToPlanet) && (H < 200000))
	{
		bCloseToPlanet = true;
		Vessel->ApproachingPlanet();
	}
	else if (bCloseToPlanet && (H > 200000))
	{
		bCloseToPlanet = false;
	}

	// distant approach
	if (!bCloseToPlanet)
	{
		if (!(bEnteringPlanet) && (H < 100000000))
		{
			bEnteringPlanet = true;
			Vessel->EnteringPlanet();
		}
		else if (bEnteringPlanet && (H > 100000000))
		{
			bEnteringPlanet = false;
		}
	}
}


void TProximator::CheckAtmosphere(double H)
{
	if (dQTreshold < 0.1)
		return;

	double ADensity = Vessel->GetAtmDensity();

	if ( (ADensity == 0) && (bInAtmosphere) )
	{
		bInAtmosphere = false;
	}
	else if ( !(bInAtmosphere) && (ADensity > 0.0001) )
	{
		bInAtmosphere = true;
	}

	if (ADensity > 0.0001)
	{
		OBJHANDLE HVessel;
		double AVelocity;
		double AQ;			// dynamic pressure

		HVessel = Vessel->GetHandle();
		oapiGetAirspeed(HVessel, &AVelocity);
		AQ = ADensity * AVelocity * AVelocity / 2;
		if ( (AQ > dQTreshold) && bStruct )
			Burn();
	}
}


void TProximator::Burn()
{
	// pass back to vessel to let it decide what to do on burning.
	Vessel->Burn(FindFocusSwitchTarget());
}


void TProximator::CheckSurface(double H)
{
	if (bOnSurface)
	{
		if ( (H > 0) && !Vessel->GroundContact() )
		{
			Tral("Bounce up\n");
			bOnSurface = false;
		}
	}
	else
	{
		if ( (H < 0) || Vessel->GroundContact() )
			DoneLanding(H);
	}
}


void TProximator::DoneLanding(double H)
{
	double Vabs, Vhor;
	VECTOR3 V;
	VESSELSTATUS vs;

	H = Vessel->GetAltitude();
	Vessel->GetHorizonAirspeedVector(V);
	Vabs = sqrt(V.x*V.x + V.y*V.y + V.z*V.z);
	Vhor = sqrt(V.x*V.x + V.z*V.z);

	Tral("Surface contact! H: %.0f Vel: %.0f VelV: %.0f VelH: %.0f \n", H, Vabs, V.y, Vhor);
	bOnSurface = true;

	// force status to Landed
	Vessel->GetStatus(vs);
	vs.status = 2;
	Vessel->DefSetState(&vs);

	Vessel->DoneLanding(FindFocusSwitchTarget(), Vabs, fabs(V.y), Vhor);
}


double TProximator::CalcMinVesselDistance(OBJHANDLE *vHandle)
{
	OBJHANDLE targetHandle, myHandle;
	VECTOR3 rpos;
	double Distance, DistCurr;
	int VCount;

	Distance = -1;
	*vHandle = NULL;

	VCount = oapiGetVesselCount();
	for (int i=0; i<VCount; i++)
	{
		targetHandle = oapiGetVesselByIndex(i);
		myHandle = Vessel->GetHandle();
		if (myHandle != targetHandle)
		{
			try
			{
				oapiGetRelativePos(myHandle, targetHandle, &rpos);
				DistCurr = sqrt(rpos.x*rpos.x + rpos.y*rpos.y + rpos.z*rpos.z);
				if ((DistCurr < Distance) || (Distance < 0))
				{
					Distance = DistCurr;
					*vHandle = targetHandle;
				}
			}
			catch (...)
			{
			}
		}
	}

	return Distance;
}


void TProximator::CalcVesselProximityDelay()
{
	if ((oapiGetSimTime() - dPrevStep) > 1)
	{
		dPrevStep = oapiGetSimTime();
		dVesselProximityDelay--;
	}
}


void TProximator::CheckCollision()
{
	// delay collision checking when explicitly specified
	if (dVesselProximityDelay > 0)
	{
		CalcVesselProximityDelay();
		return;
	}

	double DMin, SizeMin;
	OBJHANDLE minHandle;

	DMin = CalcMinVesselDistance(&minHandle);
	if (minHandle == NULL)
		return;

	SizeMin = oapiGetSize(minHandle);
	if (DMin < ((SizeMin+dSize)/2))
		Collide(minHandle);
}


// try to find another vessel to switch focus to
OBJHANDLE TProximator::FindFocusSwitchTarget()
{
	OBJHANDLE targetHandle, myHandle, tempHandle, currFocHandle; 
	int VCount;

	myHandle = Vessel->GetHandle();

	try
	{
		currFocHandle = oapiGetFocusObject();

		if (currFocHandle != myHandle)
			targetHandle = currFocHandle;
		else
		{
			targetHandle = NULL;
			VCount = oapiGetVesselCount();
			for (int i=0; i<VCount; i++)
			{
				tempHandle = oapiGetVesselByIndex(i);

				if (myHandle != tempHandle)
				{
					targetHandle = tempHandle;
					break;
				}
			}
		}

		return targetHandle;
	}
	catch(...)
	{
		return myHandle;
	}
}



// collide with another vessel
void TProximator::Collide(OBJHANDLE vHandle)
{
	OBJHANDLE myHandle; 
	VECTOR3 rvel, rpos;
	VESSELSTATUS vs;
	double VMin, SizeMin, pos;

	myHandle = Vessel->GetHandle();
	oapiGetRelativeVel(myHandle, vHandle, &rvel);
	VMin = sqrt(rvel.x*rvel.x + rvel.y*rvel.y + rvel.z*rvel.z);

	if (!bOnSurface) // don't report when can't move away!
		Tral("Hit vessel at V=%f\n", VMin);

	// bounce back
	oapiGetRelativePos(myHandle, vHandle, &rpos);
	pos = sqrt(rpos.x*rpos.x + rpos.y*rpos.y + rpos.z*rpos.z);

	Vessel->GetStatus(vs);
	SizeMin = oapiGetSize(vHandle);

	vs.rvel.x -= rvel.x*1.02;
	vs.rvel.y -= rvel.y*1.02;
	vs.rvel.z -= rvel.z*1.02;
	
	vs.rpos.x += rpos.x * ((SizeMin + dSize)/2 - pos)/pos *1.01;
	vs.rpos.y += rpos.y * ((SizeMin + dSize)/2 - pos)/pos *1.01;
	vs.rpos.z += rpos.z * ((SizeMin + dSize)/2 - pos)/pos *1.01;

	vs.vrot.x = 0.3 - ((double)rand() / (double)RAND_MAX) * 0.6;
	vs.vrot.y = 0.3 - ((double)rand() / (double)RAND_MAX) * 0.6;
	vs.vrot.z = 0.1 - ((double)rand() / (double)RAND_MAX) * 0.2;

	Vessel->DefSetState(&vs);

	// pass to vessel to decide what to do next
	Vessel->Collide(vHandle, VMin, FindFocusSwitchTarget());
}


// telemetry function
void TProximator::Tral (char *fmt, ...)
{
	va_list ap;
	double simtime;
	int sec, min, hr, dy;
	FILE *TralFile;

	if (strcmp(TralFileName, "") == 0)
		return;

	simtime = oapiGetSimTime();
	dy = (int)(simtime / 86400);
	hr = (int)((simtime - dy * 86400) / 3600);
	min = (int)((simtime - dy * 86400 - hr * 3600) / 60);
	sec = (int)(simtime - dy * 86400 - hr * 3600 - min * 60);
	va_start (ap, fmt);

	TralFile = fopen(TralFileName, "a");
	fprintf(TralFile, "%d:%2.2d:%2.2d:%2.2d: %s:", dy, hr, min, sec, Name);
	vfprintf(TralFile, fmt, ap);
	fclose(TralFile);

	va_end (ap);
}
