// ==============================================================
// ORBITER vessel module: Laika lunar probe lander
//
// See LaikaLander.h for description.
//
// Developed by: Andrew Thielmann (igel)
//
// ==============================================================

#include "LaikaLander.h"
#include "LaikaFragment.h"
#include "stdio.h"
#include "math.h"
#include "time.h"


// ==============================================================
// Specialised vessel class LaikaLander
// ==============================================================

LaikaLander::LaikaLander (OBJHANDLE hObj, int fmodel): PRVESSEL (hObj, fmodel)
{
		// default initialization
	bJustDeployed = true;

	// lander-specific
	dAfterLandingOperationDecr = 0.5;

	// define animations
	bAnimate = false;
	CurrentAnimation = NONE;
	AnimSpeedClose = ANIM_SPEED_PALLET_NORMAL;
	for (int i = 0; i < 4; i++)
		animateBags[i] = false;
	for (i = 0; i < 3; i++)
		animatePallets[i] = false;
	NormalBagsStatus = STORED;
	NormalPalletsStatus = CLOSED;
	DefineAnimations();

	bWatingAfterLandingOperation = false;
	bInflateAirbags = false;

	// create a very own extention class member
	Proximator = new TProximator(this);
}


LaikaLander::~LaikaLander ()
{
	if (Proximator) 
		delete Proximator;
}


// General initialization of general params, called before config file
void LaikaLander::SetClassCaps (FILEHANDLE cfg)
{
	// size and geometry
	SetEmptyMass (300.0);

	// surface
	SetSurfaceFrictionCoeff (10.0, 10.0);
	SetCOG_elev (1);
	SetTouchdownPoints (_V(0, 4, 0), _V(-4, -4, 0), _V(4, -4, 0));

	// default configuration
	BagsStatus[0] = INITIAL;
	BagsStatus[1] = INITIAL;
	BagsStatus[2] = INITIAL;
	BagsStatus[3] = INITIAL;
	PalletsStatus[0] = P_CLOSED;
	PalletsStatus[1] = P_CLOSED;
	PalletsStatus[2] = P_CLOSED;

	UpdateLanderPhysics();

	// load mesh
	VECTOR3 MeshOffset = _V(0, 0, 0);
	ClearMeshes();
	AddMesh ("LaikaLander", &MeshOffset);
}


void LaikaLander::UpdateLanderPhysics()
{
	if (NormalPalletsStatus == OPENED)
	{
		// strength
		dQTreshold = 15000;
		dVCollTreshold = 5;
		dVLandTreshold = 5;
		dVCrashTreshold = 150;

		// size and geometry
		SetSize (2);
		SetPMI (_V(2, 2, 1));

		// aerodynamics
		SetCrossSections (_V(3.0, 1.0, 1.0));
		SetCW (5, 5, 1, 1);			// wind resistance
		SetRotDrag (_V(2, 2, 1));
		SetPitchMomentScale (0.001);
		SetBankMomentScale (0.001);
	}
	else if (NormalBagsStatus == DEPLOYED)
	{
		// strength
		dQTreshold = 50000;
		dVCollTreshold = 40;
		dVLandTreshold = 30;
		dVCrashTreshold = 150;

		// size and geometry
		SetSize (3);
		SetPMI (_V(1, 1, 1));

		// aerodynamics
		SetCrossSections (_V(3.0, 3.0, 3.0));
		SetCW (3, 3, 3, 3);			// wind resistance
		SetRotDrag (_V(3, 3, 3));
		SetPitchMomentScale (0.001);
		SetBankMomentScale (0.001);
	}
	else		// initial configuration
	{
		// strength
		dQTreshold = 30000;
		dVCollTreshold = 10;
		dVLandTreshold = 8;
		dVCrashTreshold = 150;

		// size and geometry
		SetSize (1);
		SetPMI (_V(3, 3, 3));

		// aerodynamics
		SetCrossSections (_V(1.5, 1.5, 1.5));
		SetCW (1, 1, 1, 1);			// wind resistance
		SetRotDrag (_V(1.5,1.5,1.5));
		SetPitchMomentScale (0.001);
		SetBankMomentScale (0.001);
	}

	Proximator->UpdateFromVessel();
}




// Read status from scenario file
void LaikaLander::LoadState (FILEHANDLE scn, void *vs)
{
    char *line;

	while (oapiReadScenario_nextline (scn, line)) 
	{
		// bags 
        if (!strnicmp (line, "BAG1", 4)) 
		{
            sscanf (line+4, "%d", &BagsStatus[0]);
		} 
        else if (!strnicmp (line, "BAG2", 4)) 
		{
            sscanf (line+4, "%d", &BagsStatus[1]);
		} 
        else if (!strnicmp (line, "BAG3", 4)) 
		{
            sscanf (line+4, "%d", &BagsStatus[2]);
		} 
        else if (!strnicmp (line, "BAG4", 4)) 
		{
            sscanf (line+4, "%d", &BagsStatus[3]);
		} 
        else if (!strnicmp (line, "BAGS", 4)) 
		{
            sscanf (line+4, "%d", &NormalBagsStatus);
		} 
		// pallets
        else if (!strnicmp (line, "PALLET2", 7)) 
		{
            sscanf (line+7, "%d", &PalletsStatus[0]);
		} 
        else if (!strnicmp (line, "PALLET3", 7)) 
		{
            sscanf (line+7, "%d", &PalletsStatus[1]);
		} 
        else if (!strnicmp (line, "PALLET4", 7)) 
		{
            sscanf (line+7, "%d", &PalletsStatus[2]);
		} 
        else if (!strnicmp (line, "PALLETS", 7)) 
		{
            sscanf (line+7, "%d", &NormalPalletsStatus);
		} 
		// try proximator's parse
		else if (!Proximator->ParsedScenarioLine(line))
			// unrecognised option - pass to Orbiter's generic parser
			ParseScenarioLineEx (line, vs);
    }

	bJustDeployed = false;
	SetExtraConfiguration ();
}


// Save status to scenario file
void LaikaLander::SaveState (FILEHANDLE scn)
{
	// default vessel parameters
	SaveDefaultState (scn);

	// save proximator parameters
	Proximator->SaveState (scn);

		// custom parameters
	oapiWriteScenario_int (scn, "BAG1", BagsStatus[0]);
	oapiWriteScenario_int (scn, "BAG2", BagsStatus[1]);
	oapiWriteScenario_int (scn, "BAG3", BagsStatus[2]);
	oapiWriteScenario_int (scn, "BAG4", BagsStatus[3]);
	oapiWriteScenario_int (scn, "BAGS", NormalBagsStatus);

	oapiWriteScenario_int (scn, "PALLET2", PalletsStatus[0]);
	oapiWriteScenario_int (scn, "PALLET3", PalletsStatus[1]);
	oapiWriteScenario_int (scn, "PALLET4", PalletsStatus[2]);
	oapiWriteScenario_int (scn, "PALLETS", NormalPalletsStatus);
}


void LaikaLander::JustDeployed()
{
	SetExtraConfiguration (); 

	bWatingAfterLandingOperation = true;
	bJustDeployed = false;
	bInflateAirbags = true;

	Proximator->Tral("AOS\n");
}


// called after loading scenario or ejection from carrier
void LaikaLander::SetExtraConfiguration()
{
	// restore any stored bags or pallets configuration
	if ( (BagsStatus[0] == INFLATED) || (BagsStatus[1] == INFLATED) || (BagsStatus[2] == INFLATED) || (BagsStatus[3] == INFLATED) )
		AirbagsInflateStart(ANIM_SPEED_FAST);
	if ( (PalletsStatus[0] == P_OPENED) || (PalletsStatus[1] == P_OPENED) || (PalletsStatus[2] == P_OPENED) )
		PalletsOpenStart(ANIM_SPEED_FAST);
}


void LaikaLander::Timestep (double simt)
{
	Proximator->Timestep (simt);
	
	if (bJustDeployed)
		JustDeployed();

	if (bWatingAfterLandingOperation)
		CheckAfterLandingOperationTime();

	if (bAnimate)
		Animate();
}


void LaikaLander::DefineAnimations()
{
	/* mesh indexes, are here for reference 
	Bag1 0
	Pallet1 1
	Pallet2 2
	Pallet3 3
	Pallet4 4
	Bag2 5
	Bag3 6
	Bag4 7
	PN1 8
	PN2 9
	*/

	// animate each element separately, to enable them to 'fail' by excluding them from animation

	// Bags

	MGROUP_SCALE *Bags[4];
	MGROUP_SCALE *Bag;
	void  *animBagHandles[4];

	static UINT grBag1[1] = {0}; 
	static UINT grBag2[1] = {5}; 
	static UINT grBag3[1] = {6}; 
	static UINT grBag4[1] = {7}; 

	Bags[0] = new MGROUP_SCALE(0, grBag1, 1, _V(0, 0, 0.165), _V(10, 10, 10));
	Bags[1] = new MGROUP_SCALE(0, grBag2, 1, _V(-0.149149, 0, -0.07057), _V(10, 10, 10));
	Bags[2] = new MGROUP_SCALE(0, grBag3, 1, _V(0.07457, 0.12917, -0.07057), _V(10, 10, 10));
	Bags[3] = new MGROUP_SCALE(0, grBag4, 1, _V(0.07457, -0.12917, -0.07057), _V(10, 10, 10));

	for (int i = 0; i < 4; i++)
	{
		BagAnimations[i] = CreateAnimation (0);
		Bag = Bags[i];
		animBagHandles[i] = AddAnimationComponent (BagAnimations[i], 0, 1, Bag);
	}

	// Pallets

	MGROUP_ROTATE *Pallets[3];
	MGROUP_ROTATE *Pallet;
	void  *animPalletHandles[3];

	static UINT grPallet2[2] = {2, 5}; 
	static UINT grPallet3[2] = {3, 6}; 
	static UINT grPallet4[2] = {4, 7}; 

	Pallets[0] = new MGROUP_ROTATE(0, grPallet2, 2, _V(-0.25, 0, 0.175), _V(0, 1, 0), (float)(100*PI/180));
	Pallets[1] = new MGROUP_ROTATE(0, grPallet3, 2, _V(0.125,  0.21651, 0.175), _V( 0.86603, -0.5, 0), (float)(100*PI/180));
	Pallets[2] = new MGROUP_ROTATE(0, grPallet4, 2, _V(0.125, -0.21651, 0.175), _V(-0.86603, -0.5, 0), (float)(100*PI/180));

	for (i = 0; i < 3; i++)
	{
		PalletAnimations[i] = CreateAnimation (0);
		Pallet = Pallets[i];
		animPalletHandles[i] = AddAnimationComponent (PalletAnimations[i], 0, 1, Pallet);
	}
}


void LaikaLander::DoneLanding(OBJHANDLE HTarget, double V, double Vv, double Vh)
{
	if (V > dVCrashTreshold)
		DestructiveCrash(HTarget);
	else if (V > dVLandTreshold)
		Burn(HTarget);
	else
	{
		bWatingAfterLandingOperation = true;
		dAfterLandingOperationDecr = DEFLATE_OPEN_DECR;  
	}
}


void LaikaLander::DestructiveCrash(OBJHANDLE HTarget)
{
	if (!Proximator->bStruct)
		return;

	oapiSetFocusObject(HTarget);
	oapiDeleteVessel (GetHandle(), HTarget);
}


// burn in the atmosphere
void LaikaLander::Burn(OBJHANDLE HTarget)
{
	if (!Proximator->bStruct)
		return;

	void *hFragment;
	VESSELSTATUS vs;
	VESSEL *NewFragment;

	GetStatus(vs);
	hFragment = oapiCreateVessel ("Lander Fragment", "LaikaFragment", vs);
	NewFragment = oapiGetVesselInterface(hFragment);
	((LaikaFragment*)NewFragment)->FragmentID = FID_LANDER;
	oapiSetFocusObject(hFragment);
	oapiDeleteVessel (GetHandle(), hFragment);
}


// collide with another vessel
void LaikaLander::Collide(OBJHANDLE HHit, double Vhit, OBJHANDLE HSwitch) 
{
	if (!Proximator->bStruct)
		return;

	if (Vhit > dVCrashTreshold)
		DestructiveCrash(HSwitch);
	else if (Vhit > dVCollTreshold)
		Burn(HSwitch);
	else
		;
}


void LaikaLander::Animate()
{
	double dPercent;

	if (CurrentAnimation == INFLATING) 
	{
		dPercent = oapiGetSimStep() * AnimSpeedInflate;
		if (BagsPercent < 1.0)
			BagsPercent = min (1.0, BagsPercent + dPercent);
		else
			AirbagsInflateEnd();

		for (int i = 0; i < 4; i++)
			if (animateBags[i])
				SetAnimation (BagAnimations[i], BagsPercent);
	} 
	
	if (CurrentAnimation ==  DEFLATING)
	{
		dPercent = oapiGetSimStep() * AnimSpeedDeflate;
		if (BagsPercent > 0.0)
			BagsPercent = max (0.0, BagsPercent - dPercent);
		else
			AirbagsDeflateEnd();

		for (int i = 0; i < 4; i++)
			if (animateBags[i])
				SetAnimation (BagAnimations[i], BagsPercent);
	}

	if (CurrentAnimation == OPENING) 
	{
		dPercent = oapiGetSimStep() * AnimSpeedOpen;
		if (PalletsPercent < 1.0)
			PalletsPercent = min (1.0, PalletsPercent + dPercent);
		else
			PalletsOpenEnd();

		for (int i = 0; i < 3; i++)
			if (animatePallets[i])
				SetAnimation (PalletAnimations[i], PalletsPercent);
	} 

	if (CurrentAnimation == CLOSING) 
	{
		dPercent = oapiGetSimStep() * AnimSpeedClose;
		if (PalletsPercent > 0.0)
			PalletsPercent = max (0.0, PalletsPercent - dPercent);
		else
			PalletsCloseEnd();

		for (int i = 0; i < 3; i++)
			if (animatePallets[i])
				SetAnimation (PalletAnimations[i], PalletsPercent);
	} 
}


void LaikaLander::AirbagsInflateStart(double Speed)
{
	// 'fast' indicates restoring configuration, not oprational inflation
	if (Speed == ANIM_SPEED_FAST)
	{
		for (int i = 0; i < 4; i++)
			if (BagsStatus[i] == INFLATED)
			{
				animateBags[i] = true;
			}
	}
	else
	// 'normal' indicates operational inflation
		for (int i = 0; i < 4; i++)
			if (BagsStatus[i] == INITIAL)
				animateBags[i] = true;

	// do we have any bags to inflate?
	if ( (!animateBags[0]) && (!animateBags[1]) && (!animateBags[2]) && 
							(!animateBags[3]) )
		return;

	// report only if it is operational inflation
	if (Speed != ANIM_SPEED_FAST)
		Proximator->Tral("Inflating airbags...\n");

	BagsPercent = 0;
	AnimSpeedInflate = Speed;
	bAnimate = true;
	CurrentAnimation = INFLATING;
}


void LaikaLander::AirbagsInflateEnd()
{
	CurrentAnimation = NONE;
	bAnimate = false;

	for (int i = 0; i < 4; i++)
	{
		if (animateBags[i])
			BagsStatus[i] = INFLATED;
		animateBags[i] = false;
	}

	// operational inflation, not config restoration
	if ( AnimSpeedInflate != ANIM_SPEED_FAST)
	{
		Proximator->Tral("Airbags inflated!\n"); 
		NormalBagsStatus = DEPLOYED;
		UpdateLanderPhysics();
	}
}



void LaikaLander::AirbagsDeflateStart(double Speed)
{
	// 'fast' indicates airbag explosion
/*
	if (Speed == ANIM_SPEED_FAST)
	{
		for (int i = 0; i < 4; i++)
			if (BagsStatus[i] == INFLATED)  // %%% + other criteria
			{
				animateBags[i] = true;
			}
	}
	else
*/
	// 'normal' indicates operational deflation
	{
		for (int i = 0; i < 4; i++)
		{
			if (BagsStatus[i] == INFLATED)
			{
				animateBags[i] = true;
			}
		}
	}

	// do we have any bags to deflate?
	if ( (!animateBags[0]) && (!animateBags[1]) && (!animateBags[2]) && 
							(!animateBags[3]) )
		return;

	if (Speed != ANIM_SPEED_FAST)
		Proximator->Tral("Deflating airbags...\n");

	BagsPercent = 1;
	AnimSpeedDeflate = Speed;
	bAnimate = true;
	CurrentAnimation = DEFLATING;
}


void LaikaLander::AirbagsDeflateEnd()
{
	CurrentAnimation = NONE;
	bAnimate = false;

	for (int i = 0; i < 4; i++)
	{
		if (animateBags[i])
			BagsStatus[i] = DEFLATED;
		animateBags[i] = false;
	}

	// operational inflation, not config restoration
	if ( AnimSpeedDeflate == ANIM_SPEED_FAST)
		Proximator->Tral("Airbag anomaly...\n"); 
	else
	{
		Proximator->Tral("Airbags deflated!\n"); 
		NormalBagsStatus = STORED;
		UpdateLanderPhysics();
	}
}


void LaikaLander::PalletsOpenStart(double Speed)
{
	if (bAnimate)
		return;

	if (Speed == ANIM_SPEED_FAST)
		for (int i = 0; i < 3; i++)
		{
			if (PalletsStatus[i] == P_OPENED)
				animatePallets[i] = true;
		}
	else
		for (int i = 0; i < 3; i++)
			if ( (PalletsStatus[i] == P_CLOSED) && (BagsStatus[i+1] != INFLATED) )
				animatePallets[i] = true;

	// do we have any pallets to open?
	if ( (!animatePallets[0]) && (!animatePallets[1]) && (!animatePallets[2]) ) 
		return;

	// 'fast' indicates restoring configuration, not oprational opening
	if (Speed != ANIM_SPEED_FAST)
		Proximator->Tral("Opening pallets...\n");

	PalletsPercent = 0;
	AnimSpeedOpen = Speed;
	bAnimate = true;
	CurrentAnimation = OPENING;
}


void LaikaLander::PalletsOpenEnd()
{
	CurrentAnimation = NONE;
	bAnimate = false;

	for (int i = 0; i < 3; i++)
	{
		if (animatePallets[i])
			PalletsStatus[i] = P_OPENED;
		animatePallets[i] = false;
	}

	// operational opening, not config restoration
	if ( AnimSpeedOpen != ANIM_SPEED_FAST)
	{
		Proximator->Tral("Pallets opened!\n");  
		NormalPalletsStatus = OPENED;
		UpdateLanderPhysics();
	}
}


void LaikaLander::PalletsCloseStart()
{
	if (bAnimate)
		return;

	for (int i = 0; i < 3; i++)
		if (PalletsStatus[i] == P_OPENED)
		{
			animatePallets[i] = true;
		}


	// do we have any pallets to open?
	if ( (!animatePallets[0]) && (!animatePallets[1]) && (!animatePallets[2]) ) 
		return;

	Proximator->Tral("Closing pallets...\n");

	PalletsPercent = 1;
	bAnimate = true;
	CurrentAnimation = CLOSING;
}


void LaikaLander::PalletsCloseEnd()
{
	CurrentAnimation = NONE;
	bAnimate = false;
	NormalPalletsStatus = CLOSED;

	for (int i = 0; i < 3; i++)
	{
		if (animatePallets[i])
			PalletsStatus[i] = P_CLOSED;
		animatePallets[i] = false;
	}

	Proximator->Tral("Pallets closed!\n");  
	UpdateLanderPhysics();
}


void LaikaLander::CheckAfterLandingOperationTime()
{
	double dStep = oapiGetSimStep();
	if ((dAfterLandingOperationDecr -= dStep) > 0)
		return;

	// inflate airbags
	if (bInflateAirbags)
	{
		AirbagsInflateStart(ANIM_SPEED_INFLATE_NORMAL);
		bWatingAfterLandingOperation = false;
		bInflateAirbags = false;
		return;
	}

	// in any case, delay next check to about 20-30 seconds
	dAfterLandingOperationDecr = DEFLATE_OPEN_DECR;  

	// automation off?
	if (!Proximator->bAuto)
		return;

	// not done with previous operation yet
	if (bAnimate)
		return;

	if (!Proximator->bOnSurface)
		return;

	// deflate airbags
	if (NormalBagsStatus == DEPLOYED)
	{
		AirbagsDeflateStart(ANIM_SPEED_DEFLATE_NORMAL);
		return;
	}

	// close pallets
	if (NormalPalletsStatus == CLOSED)
	{
		PalletsOpenStart(ANIM_SPEED_PALLET_NORMAL);
		bWatingAfterLandingOperation = false;
	}
}



void LaikaLander::DebugSomething()
{
}



// ==============================================================
// API interface
// ==============================================================

// Initialisation
DLLCLBK VESSEL *ovcInit (OBJHANDLE hvessel, int flightmodel)
{
	// seed random generator
	srand((unsigned)time( NULL ));

	return new LaikaLander (hvessel, flightmodel);
}


// Cleanup
DLLCLBK void ovcExit (VESSEL *vessel)
{
	if (vessel) 
		delete (LaikaLander*)vessel;
}


// Set the capabilities of the vessel class
DLLCLBK void ovcSetClassCaps (VESSEL *vessel, FILEHANDLE cfg)
{
	((LaikaLander*)vessel)->SetClassCaps (cfg);
}


// Read status from scenario file
DLLCLBK void ovcLoadStateEx (VESSEL *vessel, FILEHANDLE scn, void *vs)
{
	((LaikaLander*)vessel)->LoadState (scn, vs);
}


// Save status to scenario file
DLLCLBK void ovcSaveState (VESSEL *vessel, FILEHANDLE scn)
{
	((LaikaLander*)vessel)->SaveState (scn);
}


DLLCLBK void ovcTimestep (VESSEL *vessel, double simt)
{
	((LaikaLander*)vessel)->Timestep (simt);
}


// Keyboard interface handler
DLLCLBK int ovcConsumeBufferedKey (VESSEL *vessel, DWORD key, bool down, char *kstate)
{
	if (!down) return 0; // only process keydown events

	if (KEYMOD_SHIFT (kstate)) 
	{
		return 0; // shift combinations are reserved
	}

	LaikaLander *lander = (LaikaLander*)vessel;

	if ( !KEYMOD_CONTROL (kstate) && (key == OAPI_KEY_O))
	{
		if (lander->Proximator->bAuto)
		{
  			lander->Proximator->Tral("Autosequencer OFF\n");
			sprintf(oapiDebugString(), "Autosequencer OFF");
		}
		else
		{
  			lander->Proximator->Tral("Autosequencer ON\n");
			sprintf(oapiDebugString(), "Autosequencer ON");
		}

		lander->Proximator->bAuto = !lander->Proximator->bAuto;
		return 0;
	}

	if ( !KEYMOD_CONTROL (kstate) && (key == OAPI_KEY_S) )
	{
		if (lander->Proximator->bStruct)
		{
  			lander->Proximator->Tral("Structural limits OFF\n");
			sprintf(oapiDebugString(), "Structural limits OFF");
		}
		else
		{
  			lander->Proximator->Tral("Structural limits ON\n");
			sprintf(oapiDebugString(), "Structural limits ON");
		}

		lander->Proximator->bStruct = !lander->Proximator->bStruct;
		return 0;
	}

	if (KEYMOD_CONTROL (kstate) && (key == OAPI_KEY_P))
	{
		lander->Proximator->Tral("Unpaused\n");
		return 0;
	}

	if ( !KEYMOD_CONTROL (kstate) && (key == OAPI_KEY_D) ) 
	{
		lander->DebugSomething();
		return 1;
	}

	if (key == OAPI_KEY_T)
	{
		lander->Proximator->Tral("+Warp: %.0fx\n", oapiGetTimeAcceleration());
		return 0;
	}

	if (key == OAPI_KEY_R)
	{
		lander->Proximator->Tral("-Warp: %.0fx\n", oapiGetTimeAcceleration());
		return 0;
	}

	if ( !KEYMOD_CONTROL (kstate) && (key == OAPI_KEY_G) )
	{
		if (lander->NormalBagsStatus == STORED) 
			lander->AirbagsInflateStart(ANIM_SPEED_INFLATE_NORMAL);
		else
			lander->AirbagsDeflateStart(ANIM_SPEED_DEFLATE_NORMAL);

		return 0;
	}

	if ( !KEYMOD_CONTROL (kstate) && (key == OAPI_KEY_K) )
	{
		if (lander->NormalPalletsStatus == CLOSED) 
			lander->PalletsOpenStart(ANIM_SPEED_PALLET_NORMAL);
		else
			lander->PalletsCloseStart();

		return 0;
	}

	return 0;
}

