// ==============================================================
// ORBITER module - Laika lunar probe carrier
//
// Derived from base class PRVESSEL.
//
// Purpose: Deliver Laika lunar lander to the lunar surface
//
// Features:
// - is deployed at the begining of the out-of-Ecliptics translunar ellipse from Molniya's BlokL stage.
// - starts autopilot landing sequence 200 km above the lunar surface.
// - deploys Laika lunar lander when 20 meters above the lunar surface.
// - supports all features of TProximator class via Proximator member.
// - reacts to excessive structural loads from impacts, collisions and atmospheric dynamic pressure (Q).
//
// Developed by: Andrew Thielmann (igel)
//
// ==============================================================

#include "Proximator.h"


#define ISP 3234
#define ATT_THRUST 50

enum MCCAP_STATE { MCCAP_WAIT, MCCAP_ATT_BANK, MCCAP_ATT_YAW, MCCAP_ATT_PITCH, MCCAP_BURN, MCCAP_SAFE }; 
// autopilot states and velocity/altitude corridor:
// AP_WAIT   H > Hbr
// AP_HARD   Hbr > H > 5 km
// AP_HIGH   5 km > H > 1 km
// AP_LOW    1 km > H > 100 m
// AP_TUNE   100 m > H  > 10 m
// AP_SAFE   H < 10 m
enum AP_STATE { AP_WAIT, AP_HARD, AP_HIGH, AP_LOW, AP_TUNE, AP_SAFE }; 
double VMin[6] = {0, 500, 300, 50, 5, 0};
double VMax[6] = {0, 550, 320, 60, 10, 0};


class Laika: public PRVESSEL {
public:
	Laika (OBJHANDLE hObj, int fmodel);
	~Laika ();

	void LoadState (FILEHANDLE scn, void *vs);
	void SaveState (FILEHANDLE scn);
	void SetClassCaps (FILEHANDLE cfg);
	void Timestep (double simt);

	void SetState(const VESSELSTATUS *status);
	void ReloadMeshes();
	void JustDeployed();

	bool bJustDeployed;
	UINT ConfigAnimation;
	
	// main engine and fuel
	PROPELLANT_HANDLE fuel;
	THRUSTER_HANDLE KTDU[1];

	// attitude control
	THRUSTER_HANDLE th_rot[2];
	THGROUP_HANDLE thg_KTDU, tgh_pitchup, thg_pitchdown, thg_bankright, thg_bankleft, thg_yawright, thg_yawleft; 

	void DropProbe();
	void DropPO();
	bool bHasProbe;
	bool bHasPO1;
	bool bHasPO2;
	double dMLander;
	double dMPO1;

    char TralFileName[256];
	void DebugSomething();

	void EnteringPlanet();
	void ApproachingPlanet();          
	void Burn(OBJHANDLE HTarget);
	void DoneLanding(OBJHANDLE HTarget, double V, double Vv, double Vh);
	void Collide(OBJHANDLE HHit, double Vhit, OBJHANDLE HSwitch); 

	void Crash(OBJHANDLE HTarget);
	void DestructiveCrash(OBJHANDLE HTarget);
	double dVLandTreshold;
	double dVCrashTreshold;
	double dVCollTreshold;
    
	OBJHANDLE CreateFragment(char *MeshName, fragment_id FragmentID);
	void SetTimeAccelOff();

	// autopilot methods and vars
	bool bMccApOn;
	MCCAP_STATE MccApState;
	bool bMccApPitchStep2;
	double PastStepPericenter;
	void MccApActivate();
	void MccApDeactivate();
	void MccApUpdateState();

	void ApActivate();
	void ApUpdateState();
	void ApCheckParams();
	bool bApActive;
	AP_STATE apState;
	double dHApHard;
	double dHApHigh;
	double dHApLow;
	double dHApTune;
	double dHApSafe;

	char MeshName[256];
	TProximator *Proximator;

private:

};

