// ==============================================================
// ORBITER vessel module: R7 Molniya Module 
//   customized by Andrew Thielmann (igel)
//   to launch Laika lunar probe to trans-lunar ellipse.
//
// Portions taken from Rob Conley's code:
// GetCPitch()
//
// Thanks to:
// Rob Conley (misc functions), 
// Vinka (autopilot)
//
// Credits:
// Core module by Manuel Amorim (McDope)
// Guidance algorithm by Richard Croy (rjcroy)
// ==============================================================

#include "..\..\include\orbitersdk.h"
#include "LaikaMolniya.h"
#include "Laika.h"
#include "Proximator.h"
#include "stdio.h"
#include "math.h"
#include "time.h"

const VECTOR3 OFS_STAGE1 = { 0, 0, -0.75};
const VECTOR3 OFS_STRAPB = { 3.0, 0, -5};
const VECTOR3 OFS_STRAPV = { -3.0, 0, -5};
const VECTOR3 OFS_STRAPG = { 0, 3.0, -5};
const VECTOR3 OFS_STRAPO = { 0, -3.0, -5};
const VECTOR3 OFS_BLOCKI = { 0,0,17.15-18.25+0.25};
const VECTOR3 OFS_SHROUD = { 0.0,0,25.55-18.25};

VECTOR3 OFS_LAIKA = { 0,0,23.5};
char TralFileName[256] = "";
bool autopilot = false;
bool bStruct = true;
double LaikaMass = 1800;

bool bManualSeparate = false;
double stage_sep=0;
UINT stage = 0;
int status = 0;
bool ignition= false;
double ttime=0.0;

bool engineignited = false;
double last_simt=0.0;
double ignition_time=0.0;

bool first_guidance_pass = true;

// rjcroy Guidance variables

double INCLINATION = 51.63 ;// degrees

bool guidance = false;	// guidance switch
bool Alt_Flag = false;	// True if max launch altitude reached
bool MECO_Cond = false; // True if MECO conditions met
bool First_Guidance_Pass = false;
bool T_minus_10 = true;	// True at T minus 10 seconds.
double Vertical_Vel ; 
VECTOR3 Airspeed_Vector;
double Altitude;		// Current vessel altitude
double Pitch_SP;		// Pitch setpoint
double Pitch;			// Current vessel pitch
double Pitch_Error;
double MET;				// Mission Elasped Time
double TZero = -100;		// Time Zero - Launch time
double PC_D = 0.9;		// Pitch control coefficient D for d_theta /d_t
double PC_K = 0.6;		//    "    "         "       K  "  theta
double YC_D = 0.5;		// Yaw Control
double YC_K = 0.3;
VECTOR3 Angular_Velocity;  // vector x, y, z in rad/s
double Pitch_Dot;		// Rate of pitch vs time in degree/s
double Pitch_Error_Dot;  // degrees/s
double Pitch_Cntrl_Sig;  // Pitch control output signal
const double PC_D_FINAL = 0.9;  // Pitch control coefficients for final section.
const double PC_K_FINAL = 0.6; 
const double D_FINAL  = 0.8;	// Vertical velocity control coefficients
const double K_FINAL = 0.8;
const double ENG_THRST_2 = 294300 ;// Thrust rating of Stage 2 engine. (N)  
VECTOR3 Surface_Vel;	// surface relative velocity components m/s
double Vert_Vel_Last;	// vertical velocity in the last time step.
double SimT_Last;		// Time of last time step
double Vert_Acc;		// Vertical acceleration
double Earth_x_Vel;     // velocity of the earths surface at the current latitude (m/s) 
double i_last = 0;			// value of incliniation in previous iteration.
ELEMENTS elements;		// Current orbital element of the Soyuz
double MJD_Time ;		// current time in Modified Julian Date format.
double Alt_Apoapsis;	// Altitude of apoapsis (km)
double Alt_Periapsis;	// Altitude of periapsis (km)
double TC_Error_Dot;	// True course error rate
double TC_Error_Last ;	// Last true course error.
OBJHANDLE hVessel;
OBJHANDLE hEarth;
char Planet_Name[6] = "Earth";
VECTOR3 Rel_Vel; // vector of velocities relative to earth
VECTOR3 Rel_Pos ; // vector of our position relative to earth
VECTOR3 h_vector ; 
VECTOR3 temp;
double Velocity;
double a;		// semi-major axis
double e;		// eccentricity
double i;		// current inclination
double h;
double h_abs;   // magnitude of angular momentum (heading control)
double V_periapsis;  // Velocity at MECO/periapsis.
const double R_EARTH = 6371000; // radius of the earth (m)
const double U_EARTH = 3.986e5 * 1000 * 1000 * 1000; //( m^3/sec^2)
const double Ob_EARTH = 23.45 * RAD ; // Obliquity of the Earth (rad)
const double APO_ALT = 230 ;     // Altitude of apoapsis
double Clairaut_Constant ;
double Longitude ; // radians
double Latitude ;	// radians
double Radius ;
double True_Course_SP ;  // true course setpoint (degrees)
double True_Course ;	 // spacecraft true course.
double True_Course_Err;  // (deg)
double Error_Course ; // degrees
double Heading	;     // radians
double Heading_SP_Ajd ; //  heading setpoint adjustment
double Yaw_Error_Dot ;
double Yaw_Cntrl_Sig ;
double i_err;			// error in inclination (degrees)
double i_dot ;			// change rate of inclination (deg/s)
double ICF_K = 62.008 * INCLINATION*INCLINATION - 6848.7*INCLINATION + 189807;	//  inclination correction factor constant. (3580)
double ICF ;			// inclination correction factor
bool DisplayIncSP = false;  // toggle for displaying the inclination setpoint.
double PadHeading = 242.98; // Heading of the Soyuz on the pad (deg)
							// Default heading is for inclination 51.6 deg.
bool NewIncSP = false ;		// True if a user has entered a new inclination SP
bool PostMECO = false ;		// switch for cleaning up after MECO

bool TLICalcDone = false;
bool CoastingToTLI = false;
bool Prograded = false;
bool TLIBurning = false;
bool TLIBurnCompleted = false;
double TimeAtNode = 0;
double TimeAtTLIBurn = 0;
double TLIBurnTime = 0;
double TimeToTLIBurn = 0;
double CalcTimeToNode();
double CalcTLIBurnTime();
void TestCalc();

double TimeMECO;			// Time of MECO
bool BadHeading = false;	// true if soyuz is at an azimuth not suitable for the G&CS
double TimeMessage;

// End of rjcroy Guidance variables


OBJHANDLE hLaika;

const double ISP_FIRST_SL  = 257*9.81;
const double ISP_FIRST_VAC = 314*9.81;
const double ISP_MAIN_SL    = 248*9.81;
const double ISP_MAIN_VAC   = 315*9.81;
const double ISP_SECOND_SL    = 295*9.81;
const double ISP_SECOND_VAC   = 330*9.81;
const double ISP_BLOCKL_SL   = 272*9.81;
const double ISP_BLOCKL_VAC   = 360*9.81;
const double THRUST_FIRST_VAC = 995715;
const double THRUST_MAIN_VAC   = 235440;
const double THRUST_SECOND_VAC   = 294300;
const double THRUST_BLOCKL_VAC   = 85400;

const double MAX_ATT_FIRST = 45000;
const double MAX_ATT_MAIN = 12000;
const double MAX_ATT_SECOND = 2100;
const double MAX_ATT_BLOCKL = 800;
const double FIRST_PROP_MASS = 158400;
const double MAIN_PROP_MASS = 94000;
const double SECOND_PROP_MASS = 22700;
const double BLOCKL_PROP_MASS = 5350;


PROPELLANT_HANDLE ph_first, ph_main, ph_second, ph_blockl;	
THRUSTER_HANDLE th_first[8];
THRUSTER_HANDLE th_main[4];
THRUSTER_HANDLE th_second[1];            
THRUSTER_HANDLE th_blockl[1]; 
THGROUP_HANDLE thg_first; 
THGROUP_HANDLE thg_main; 
THGROUP_HANDLE thg_second;  
THGROUP_HANDLE thg_blockl;  
THGROUP_HANDLE thg_pitchup;
THGROUP_HANDLE thg_pitchdown;

static int refcount = 0;
static MESHHANDLE hR7stg1;
static MESHHANDLE hR7stg2;
static MESHHANDLE hR7shroud;
static MESHHANDLE hR7strapo;
static MESHHANDLE hR7strapb;
static MESHHANDLE hR7strapv;
static MESHHANDLE hR7strapg;
static MESHHANDLE hR7blockl;


#define N	14	// flight plan table size
const double met[N]    = { 0, 15, 30, 45, 60, 75, 90, 105, 120, 230, 280, 360, 420, 560};   // MET in sec
const double cpitch[N] = {90, 85, 80, 70, 60, 50, 40, 30, 20, 17.5, 15, 20, 10, 0};	// Commanded pitch in 

static double GetCPitch(double t)
{
	int i;
	if (t>met[N-1]) return cpitch[N-1];
	i=1;
	while (met[i]<t) i++;
	return cpitch[i-1]+(cpitch[i]-cpitch[i-1])/(met[i]-met[i-1])*(t-met[i-1]);
}

// rjcroy Lateral Guidance System ( called by Guidance_Routine() ).
static void Lateral_Guidance(VESSEL *vessel, double SimT, double SimT_Last)
{
	// Lateral Guidance System 
	
	// Only use heading control after T + 15 seconds.
	if (MET > 15 ) 
	{
		// Get state data
		oapiGetFocusAirspeedVector(&Airspeed_Vector);
		// Get current velocity
		oapiGetFocusRelativeVel (hEarth, &Rel_Vel);
		Velocity = sqrt((Rel_Vel.x*Rel_Vel.x) + (Rel_Vel.y*Rel_Vel.y) + (Rel_Vel.z*Rel_Vel.z));
			
		oapiGetFocusEquPos( &Longitude, &Latitude, &Radius );
		// Calculate earth rotation speed.
		Earth_x_Vel = (2*PI*(cos(fabs(Latitude)) * R_EARTH)) / 86164.09;
		Airspeed_Vector.x = Airspeed_Vector.x + Earth_x_Vel ;
		
		// Soyuz true course
		True_Course = atan(Airspeed_Vector.x / fabs(Airspeed_Vector.z)) * DEG; 
		
		//  Find the true course SP
		oapiGetFocusHeading( &Heading ) ;
		Clairaut_Constant = sin ( 90 * RAD - INCLINATION * RAD) ;
		
		if (True_Course < 90) 
		{
			True_Course_SP = (asin(Clairaut_Constant / cos(Latitude))) * DEG ;
		}
		else 
		{
			True_Course_SP = -((asin(Clairaut_Constant / cos(Latitude))) * DEG) + 180 ;
		}
		
		// Calculate the current orbital inclination
		oapiGetRelativeVel(hVessel, hEarth, &Rel_Vel);
		oapiGetRelativePos(hVessel, hEarth, &Rel_Pos);

		// Change ecliptic rel. vectors to equatorial vectors.
 		temp.x = Rel_Vel.x;
		temp.y = Rel_Vel.z;
		temp.z = Rel_Vel.y;
		veccpy( Rel_Vel, temp);
		temp.x = Rel_Pos.x;
		temp.y = Rel_Pos.z;
		temp.z = Rel_Pos.y;
		veccpy( Rel_Pos, temp);
		temp.x = Rel_Pos.x ;
		temp.y = cos(-Ob_EARTH)* Rel_Pos.y + sin(-Ob_EARTH)*Rel_Pos.z;
		temp.z =  -sin(-Ob_EARTH)* Rel_Pos.y + cos(-Ob_EARTH)*Rel_Pos.z;
		veccpy( Rel_Pos, temp);
		temp.x = Rel_Vel.x ;
		temp.y = cos(-Ob_EARTH)* Rel_Vel.y + sin(-Ob_EARTH)*Rel_Vel.z;
		temp.z =  -sin(-Ob_EARTH)* Rel_Vel.y + cos(-Ob_EARTH)*Rel_Vel.z;
		veccpy( Rel_Vel, temp);

		h_vector = crossp(Rel_Vel, Rel_Pos);
		h_abs = sqrt((h_vector.x*h_vector.x) + (h_vector.y*h_vector.y) + (h_vector.z*h_vector.z));
		i = 180 - acos(h_vector.z / h_abs) * DEG;
		i_dot = (i - i_last) / (MET - SimT_Last) ;
		i_last = i;
		ICF = Velocity / ICF_K;

		// Calculated error in inclination as output signal
		i_err = (i - INCLINATION) * (150 * ICF) + 250 * ICF * i_dot;
		if (i_err > ICF )
		{	
			i_err = ICF;
		}
		else if ( i_err < -ICF) 
		{
			i_err = -ICF;
		}
			
		// Feed the inclination error into the control loop after T + 25s 
		if (MET > 25) {
			True_Course_SP = True_Course_SP + i_err ;
		}		
			
		True_Course_Err = True_Course - True_Course_SP; 

		// True course error rate
		TC_Error_Dot = (True_Course_Err - TC_Error_Last) / (MET - SimT_Last) ;
		TC_Error_Last = True_Course_Err;
			
		// Adjusted heading setpoint
		Heading_SP_Ajd = True_Course_Err * 4 + TC_Error_Dot * 8 ;
			
		if (Heading_SP_Ajd > 4 )
		{	
			Heading_SP_Ajd = 4;
		}
		else if (Heading_SP_Ajd < -4) 
		{
			Heading_SP_Ajd = -4;
		}
		
		//	  Compare with the actual heading
		Error_Course = Heading * DEG - ( True_Course_SP - Heading_SP_Ajd); 
		Yaw_Error_Dot =  - Angular_Velocity.y * DEG ; //get the yaw rate of change
			
		//sprintf(oapiDebugString(), "MET: %f, TrueCourse: %f, Heading: %f, Pitch: %f, Pitch SP: %f", MET, True_Course, Heading * DEG, Pitch, Pitch_SP);
			
		//  Yaw control mechanism
		Yaw_Cntrl_Sig = - YC_D * Yaw_Error_Dot + YC_K * Error_Course;
	
		//sprintf(oapiDebugString(),"MET %f, Soyuz TC %f, TrueCourseSP, %f, TrueCrsError, %f, YawSig %f, Inc, %f, i_err*4, %f, idot, %f", MET, True_Course, True_Course_SP, True_Course_Err, Yaw_Cntrl_Sig, i, i_err, i_dot);

		if (Yaw_Cntrl_Sig > 1 )
		{
			vessel->SetAttitudeRotLevel( 1, -1); // 0 is y-axis
		}
		else if (Yaw_Cntrl_Sig < -1) 
		{
			vessel->SetAttitudeRotLevel( 1, 1);
		}
		else {
			vessel->SetAttitudeRotLevel( 1 , -Yaw_Cntrl_Sig);
		}
	}
		
	// sprintf(oapiDebugString(), "Time: %f, Pitch_SP: %f, Pitch: %f, PitchErrorDot: %f, Output: %f",MET, Pitch_SP, Pitch, Pitch_Error_Dot, Pitch_Cntrl_Sig);
		
	if ( (i - INCLINATION) < 0.006 && MET > 150)
	{
		sprintf(oapiDebugString(),"T + %.0f s  Guidance Converged",MET);
	}
	else 
	{
		sprintf(oapiDebugString(),"T + %.0f s",MET);
	}

}

// rjcroy Guidance & Control System
static void Guidance_Routine(VESSEL *vessel, double SimT, double SimT_Last)
{

	hVessel = oapiGetFocusObject ();
	hEarth = oapiGetGbodyByName ( Planet_Name );
	
	if (T_minus_10)
	{
		// Record launch time
		if (TZero == 0)
		{
			srand((int)hVessel);
			TZero = SimT + 11;
		}

		if (MET > -5 && MET < -3) {
			oapiSetEngineLevel(hVessel, ENGINE_MAIN, 0.3);
		}
		else if (MET > -3 && MET < 0) {
			oapiSetEngineLevel(hVessel, ENGINE_MAIN, (0.2*MET + 1));
		}
		else if (MET > 0)
		{
			T_minus_10 = false;
			First_Guidance_Pass = true;
		}
		sprintf(oapiDebugString(),"T - %.1f", fabs(MET));
	}		

	else if (First_Guidance_Pass) 
	{
		
		// Start the main engines
		oapiSetEngineLevel(hVessel, ENGINE_MAIN, 1);

		// Unset the first pass flag
		First_Guidance_Pass = false;
	}
	else if (!Alt_Flag) 
	{
		// Get correct pitch from table
		Pitch_SP=GetCPitch(MET);
			
		// Control to pitch setpoint
		oapiGetFocusPitch( &Pitch );
		Pitch = Pitch * DEG;
		Pitch_Error = Pitch - Pitch_SP;

		vessel->GetAngularVel( Angular_Velocity ); 
		Pitch_Error_Dot = Angular_Velocity.x * DEG; // get pitch change rate in degrees/s

		Pitch_Cntrl_Sig =  - PC_D * Pitch_Error_Dot + PC_K * Pitch_Error;
		
		if (Pitch_Cntrl_Sig > 1 )
		{
			vessel->SetAttitudeRotLevel( 0, 1); // 0 is y-axis
		}
		else if (Pitch_Cntrl_Sig < -1) 
		{
			vessel->SetAttitudeRotLevel( 0, -1);
		}
		else {
			vessel->SetAttitudeRotLevel( 0 , Pitch_Cntrl_Sig);
		}
		
		// Heading control
		Lateral_Guidance(vessel, SimT, SimT_Last);
		
		// Turn off warp for final section of orbital insertion.
		if (MET > 509 )  {
			oapiSetTimeAcceleration (1 );
		}
		
		//Timeout on Alt_Flag
		
		if (MET > 510) {
			Alt_Flag = true;
			// Record the current vertical speed and time for next section.
			vessel->GetHorizonAirspeedVector( Surface_Vel );
			Vert_Vel_Last = Surface_Vel.y ;
			SimT_Last = SimT;	
		}
		
	}
	else if (!MECO_Cond) // Final ~ 30 secs of launch
	{

		//Calculate if MECO velocity has been reached
		oapiGetFocusAltitude(&Altitude);
		a = ((APO_ALT * 1000 + R_EARTH) + (Altitude + R_EARTH)) / 2;
		e = ((APO_ALT * 1000 + R_EARTH) / a ) - 1;
		h = sqrt(U_EARTH * a * ( 1 - (e * e)));
		V_periapsis = h / (Altitude + R_EARTH);
		
		// Get current velocity
		oapiGetFocusRelativeVel (hEarth, &Rel_Vel);
		Velocity = sqrt((Rel_Vel.x*Rel_Vel.x) + (Rel_Vel.y*Rel_Vel.y) + (Rel_Vel.z*Rel_Vel.z));
		
		if (Velocity > V_periapsis) 
		{
			MECO_Cond = true;
			TimeMECO = SimT;
		}

		// Get vertical velocity and vertical acceleration
		vessel->GetHorizonAirspeedVector( Surface_Vel );
		Vert_Acc = (Surface_Vel.y - Vert_Vel_Last) / (MET - SimT_Last);
		Vert_Vel_Last = Surface_Vel.y;
		
		// Engine Throttling
		// Restricts acceleration in final stages to about 2.5G
		oapiSetEngineLevel(hVessel, ENGINE_MAIN, ( oapiGetMass(hVessel) * 25 / ENG_THRST_2 ));
	
		// Pitch Control
		Pitch_SP = - D_FINAL * Vert_Acc - K_FINAL * Surface_Vel.y;

		oapiGetFocusPitch( &Pitch );
		Pitch = Pitch * DEG;
		Pitch_Error = Pitch - Pitch_SP;

		vessel->GetAngularVel( Angular_Velocity ); 
		Pitch_Error_Dot = Angular_Velocity.x * DEG; // get pitch change rate in degrees/s

		Pitch_Cntrl_Sig = - PC_D_FINAL * Pitch_Error_Dot + PC_K_FINAL * Pitch_Error;
		
		if (Pitch_Cntrl_Sig > 1 )
		{
			vessel->SetAttitudeRotLevel( 0, 1); // 0 is y-axis
		}
		else if (Pitch_Cntrl_Sig < -1) 
		{
			vessel->SetAttitudeRotLevel( 0, -1);
		}
		else {
			vessel->SetAttitudeRotLevel( 0 , Pitch_Cntrl_Sig);
		}
		
		// Heading control
		Lateral_Guidance(vessel, SimT, SimT_Last);


		// sprintf(oapiDebugString(), "MET: %f, Surface_Vel.y, %f, Vert Acc: %f, PitchSP, %f, PC_DOut: %f, PC_KOut: %f", MET, Surface_Vel.y, Vert_Acc, Pitch_SP, Pitch_Error_Dot, Pitch_Error );
	}
	else if (MECO_Cond && !PostMECO)
	{
		// MECO.  Shutoff main engine, neutralize thrusters, turn off guidance
		oapiSetEngineLevel(hVessel, ENGINE_MAIN, 0);
		// Turn off pitch thrusters
		vessel->SetAttitudeRotLevel( 0 , 0);
		// Turn off the yaw thrusters
		vessel->SetAttitudeRotLevel( 1, 0);
//		vessel->ActivateNavmode(NAVMODE_KILLROT);

		// retrieve the final orbital elements
		MJD_Time = oapiGetSimMJD();
		vessel->GetElements( elements, MJD_Time );
		Alt_Apoapsis = (elements.a * ( 1 + elements.e )/1000) - 6371; 
		Alt_Periapsis = (elements.a * ( 1 - elements.e )/1000) - 6371; 
		
		// Inclination (equatorial)
		oapiGetRelativeVel(hVessel, hEarth, &Rel_Vel);
		oapiGetRelativePos(hVessel, hEarth, &Rel_Pos);
		
		// Change ecliptic rel. vectors to equatorial vectors.
 		temp.x = Rel_Vel.x;
		temp.y = Rel_Vel.z;
		temp.z = Rel_Vel.y;
		veccpy( Rel_Vel, temp);
		temp.x = Rel_Pos.x;
		temp.y = Rel_Pos.z;
		temp.z = Rel_Pos.y;
		veccpy( Rel_Pos, temp);
		temp.x = Rel_Pos.x ;
		temp.y = cos(-Ob_EARTH)* Rel_Pos.y + sin(-Ob_EARTH)*Rel_Pos.z;
		temp.z =  -sin(-Ob_EARTH)* Rel_Pos.y + cos(-Ob_EARTH)*Rel_Pos.z;
		veccpy( Rel_Pos, temp);
		temp.x = Rel_Vel.x ;
		temp.y = cos(-Ob_EARTH)* Rel_Vel.y + sin(-Ob_EARTH)*Rel_Vel.z;
		temp.z =  -sin(-Ob_EARTH)* Rel_Vel.y + cos(-Ob_EARTH)*Rel_Vel.z;
		veccpy( Rel_Vel, temp);

		h_vector = crossp(Rel_Vel, Rel_Pos);
		h_abs = sqrt((h_vector.x*h_vector.x) + (h_vector.y*h_vector.y) + (h_vector.z*h_vector.z));
		i = 180 - acos(h_vector.z / h_abs) * DEG;
		
		PostMECO = true;

	}
	else if (MECO_Cond && PostMECO) 
	{
		if ((SimT - TimeMECO) < 10) {
			sprintf(oapiDebugString(), "Inclination (Equatorial): %.3f", i); 
		}	
		else if (!TLICalcDone)
		{
			// Calculate TLI here!
			TimeAtNode = SimT + CalcTimeToNode();

//%%% to be implemented better, for now use pre-calculated value
//			TLIBurnTime = CalcTLIBurnTime();
			TLIBurnTime = 203.4;

			TimeAtTLIBurn = TimeAtNode - TLIBurnTime/2;
			TLICalcDone = true;
			CoastingToTLI = true;
			sprintf(oapiDebugString(), "Calculating TLI...");
		}
		else if (CoastingToTLI)
		{
			TimeToTLIBurn = TimeAtTLIBurn - SimT;

			// Turn off warp for TLI
			if ((TimeToTLIBurn < 60 )  && (!Prograded))
			{
				oapiSetTimeAcceleration (1 );
				vessel->ActivateNavmode(NAVMODE_PROGRADE);
				Prograded = true;
			}
			sprintf(oapiDebugString(), "Time To TLI burn: %.0f", TimeToTLIBurn);


			if (TimeToTLIBurn < 0)
			{
				//start TLI!
				oapiSetEngineLevel(hVessel, ENGINE_MAIN, 1);
				CoastingToTLI = false;
				TLIBurning = true;
			}
		}
		else if (TLIBurning)
		{
			TimeToTLIBurn = TimeAtTLIBurn - SimT;

			// Turn off warp for TLI
			if (TimeToTLIBurn < -(TLIBurnTime-10) ) 
			{
				oapiSetTimeAcceleration (1 );
			}
			sprintf(oapiDebugString(), "Time To MECO: %.2f", TLIBurnTime + TimeToTLIBurn);

			if (TimeToTLIBurn < -TLIBurnTime)
			{
				// stop TLI
				oapiSetEngineLevel(hVessel, ENGINE_MAIN, 0);
				vessel->ActivateNavmode(NAVMODE_PROGRADE);
				TLIBurning = false;
				TLIBurnCompleted = true;
			}

		}
		else
		{
			guidance = false;
			sprintf(oapiDebugString(), "");
		}
	}
}
// END Guidance_Routine()

// Guidance user interface
void GetInclinationSP()
{
	// Gets a new Inclination setting from the pilot
	bool IncInput (void *id, char *str, void *data);
	
	oapiOpenInputBox ("Inclination Setpoint, e.g. 5163 = 51.63 deg (47 to 56)", IncInput, 0, 25);
}

bool IncInput (void *id, char *str, void *data)
{
	sscanf(str, "%4lf", &INCLINATION);
	if (INCLINATION > 4700 && INCLINATION < 5600 ) {
		INCLINATION = INCLINATION / 100;
		NewIncSP = true;
	}

	return 1 ;
}

// END rjcroy Guidance System

void SetAttControlsFirst(VESSEL *vessel)
{
	THRUSTER_HANDLE th_rot[2];
	th_rot[0] = vessel->CreateThruster (_V(0,0, 5), _V(0, 1,0), MAX_ATT_MAIN, ph_main, ISP_MAIN_SL);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-5), _V(0,-1,0), MAX_ATT_MAIN, ph_main, ISP_MAIN_SL);
	vessel->AddExhaust (th_rot[0], 5.5, 0.15, _V( 0.5, 3.25,-13.95), _V(0,0.707,-0.707));
	vessel->AddExhaust (th_rot[0], 5.5, 0.15, _V( -0.5, 3.25,-13.95), _V(0,0.707,-0.707));
		thg_pitchup=vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_PITCHUP);
	th_rot[0] = vessel->CreateThruster (_V(0,0, 5), _V(0,-1,0), MAX_ATT_MAIN, ph_main, ISP_MAIN_SL);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-5), _V(0, 1,0), MAX_ATT_MAIN, ph_main, ISP_MAIN_SL);
	vessel->AddExhaust (th_rot[0], 5.5, 0.15, _V( 0.5,  -3.25,-13.95  ), _V(0,-0.707,-0.707));
	vessel->AddExhaust (th_rot[0], 5.5, 0.15, _V( -0.5, -3.25,-13.95  ), _V(0,-0.707,-0.707));
		thg_pitchdown=vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_PITCHDOWN);
	th_rot[0] = vessel->CreateThruster (_V(0,0, 4), _V(-1,0,0), MAX_ATT_MAIN, ph_main, ISP_MAIN_SL);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-4), _V( 1,0,0), MAX_ATT_MAIN, ph_main, ISP_MAIN_SL);
	vessel->AddExhaust (th_rot[0], 5.5, 0.15, _V(3.25, -0.5,-13.95 ), _V(0.707,0,-0.707));
	vessel->AddExhaust (th_rot[0], 5.5, 0.15, _V(3.25, 0.5,-13.95 ), _V(0.707,0,-0.707));
		vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_YAWLEFT);
	th_rot[0] = vessel->CreateThruster (_V(0,0, 4), _V( 1,0,0), MAX_ATT_MAIN, ph_main, ISP_MAIN_SL);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-4), _V(-1,0,0), MAX_ATT_MAIN, ph_main, ISP_MAIN_SL);
	vessel->AddExhaust (th_rot[0], 5.5, 0.15, _V( -3.25,-0.5,-13.95 ), _V(-0.707,0,-0.707));	
	vessel->AddExhaust (th_rot[0], 5.5, 0.15, _V( -3.25, 0.5,-13.95 ), _V(-0.707,0,-0.707));
		vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_YAWRIGHT);
	th_rot[0] = vessel->CreateThruster (_V(3,0,0), _V(0,-1,0), MAX_ATT_MAIN, ph_main, ISP_MAIN_SL);
	th_rot[1] = vessel->CreateThruster (_V(-3,0,0), _V(0, 1,0), MAX_ATT_MAIN, ph_main, ISP_MAIN_SL);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_BANKRIGHT);
	th_rot[0] = vessel->CreateThruster (_V(3,0, 0), _V(0, 1,0), MAX_ATT_MAIN, ph_main, ISP_MAIN_SL);
	th_rot[1] = vessel->CreateThruster (_V(-3,0,0), _V(0, -1,0), MAX_ATT_MAIN, ph_main, ISP_MAIN_SL);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_BANKLEFT);
}

void SetAttControlsMain(VESSEL *vessel)
{
	THRUSTER_HANDLE th_rot[2];
	th_rot[0] = vessel->CreateThruster (_V(0,0, 11), _V(0, 1,0), MAX_ATT_MAIN, ph_main, ISP_MAIN_SL);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-11), _V(0,-1,0), MAX_ATT_MAIN, ph_main, ISP_MAIN_SL);
	thg_pitchup=vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_PITCHUP);
	th_rot[0] = vessel->CreateThruster (_V(0,0, 11), _V(0,-1,0), MAX_ATT_MAIN, ph_main, ISP_MAIN_SL);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-11), _V(0, 1,0), MAX_ATT_MAIN, ph_main, ISP_MAIN_SL);
	thg_pitchdown=vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_PITCHDOWN);
	th_rot[0] = vessel->CreateThruster (_V( 3,0,0), _V(0, 1,0), MAX_ATT_MAIN, ph_main, ISP_MAIN_SL);
	th_rot[1] = vessel->CreateThruster (_V(-3,0,0), _V(0,-1,0), MAX_ATT_MAIN, ph_main, ISP_MAIN_SL);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_BANKLEFT);
	th_rot[0] = vessel->CreateThruster (_V( 3,0,0), _V(0,-1,0), MAX_ATT_MAIN, ph_main, ISP_MAIN_SL);
	th_rot[1] = vessel->CreateThruster (_V(-3,0,0), _V(0, 1,0), MAX_ATT_MAIN, ph_main, ISP_MAIN_SL);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_BANKRIGHT);
	th_rot[0] = vessel->CreateThruster (_V(0,0, 4), _V( 1,0,0), MAX_ATT_MAIN, ph_main, ISP_MAIN_SL);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-4), _V(-1,0,0), MAX_ATT_MAIN, ph_main, ISP_MAIN_SL);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_YAWRIGHT);
	th_rot[0] = vessel->CreateThruster (_V(0,0, 4), _V(-1,0,0), MAX_ATT_MAIN, ph_main, ISP_MAIN_SL);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-4), _V( 1,0,0), MAX_ATT_MAIN, ph_main, ISP_MAIN_SL);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_YAWLEFT);
}

void SetAttControlsSecond(VESSEL *vessel)
{
	THRUSTER_HANDLE th_rot[2];
	th_rot[0] = vessel->CreateThruster (_V(0,0, 8), _V(0, 1,0), MAX_ATT_SECOND , ph_second, ISP_SECOND_SL);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-8), _V(0,-1,0), MAX_ATT_SECOND , ph_second, ISP_SECOND_SL);
	thg_pitchup=vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_PITCHUP);
	th_rot[0] = vessel->CreateThruster (_V(0,0, 8), _V(0,-1,0), MAX_ATT_SECOND , ph_second, ISP_SECOND_SL);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-8), _V(0, 1,0), MAX_ATT_SECOND , ph_second, ISP_SECOND_SL);
	thg_pitchdown=vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_PITCHDOWN);
	th_rot[0] = vessel->CreateThruster (_V( 3,0,0), _V(0, 1,0), MAX_ATT_SECOND , ph_second, ISP_SECOND_SL);
	th_rot[1] = vessel->CreateThruster (_V(-3,0,0), _V(0,-1,0), MAX_ATT_SECOND , ph_second, ISP_SECOND_SL);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_BANKLEFT);
	th_rot[0] = vessel->CreateThruster (_V( 3,0,0), _V(0,-1,0), MAX_ATT_SECOND , ph_second, ISP_SECOND_SL);
	th_rot[1] = vessel->CreateThruster (_V(-3,0,0), _V(0, 1,0), MAX_ATT_SECOND , ph_second, ISP_SECOND_SL);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_BANKRIGHT);
	th_rot[0] = vessel->CreateThruster (_V(0,0, 4), _V( 1,0,0), MAX_ATT_SECOND , ph_second, ISP_SECOND_SL);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-4), _V(-1,0,0), MAX_ATT_SECOND , ph_second, ISP_SECOND_SL);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_YAWRIGHT);
	th_rot[0] = vessel->CreateThruster (_V(0,0, 4), _V(-1,0,0), MAX_ATT_SECOND , ph_second, ISP_SECOND_SL);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-4), _V( 1,0,0), MAX_ATT_SECOND , ph_second, ISP_SECOND_SL);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_YAWLEFT);
}

void SetAttControlsBlockL(VESSEL *vessel)
{
	THRUSTER_HANDLE th_rot[2];
	th_rot[0] = vessel->CreateThruster (_V(0,0, 5), _V(0, 1,0), MAX_ATT_BLOCKL , ph_blockl, ISP_BLOCKL_SL);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-5), _V(0,-1,0), MAX_ATT_BLOCKL , ph_blockl, ISP_BLOCKL_SL);
	thg_pitchup=vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_PITCHUP);
	th_rot[0] = vessel->CreateThruster (_V(0,0, 5), _V(0,-1,0), MAX_ATT_BLOCKL , ph_blockl, ISP_BLOCKL_SL);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-5), _V(0, 1,0), MAX_ATT_BLOCKL , ph_blockl, ISP_BLOCKL_SL);
	thg_pitchdown=vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_PITCHDOWN);
	th_rot[0] = vessel->CreateThruster (_V( 3,0,0), _V(0, 1,0), MAX_ATT_BLOCKL , ph_blockl, ISP_BLOCKL_SL);
	th_rot[1] = vessel->CreateThruster (_V(-3,0,0), _V(0,-1,0), MAX_ATT_BLOCKL , ph_blockl, ISP_BLOCKL_SL);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_BANKLEFT);
	th_rot[0] = vessel->CreateThruster (_V( 3,0,0), _V(0,-1,0), MAX_ATT_BLOCKL , ph_blockl, ISP_BLOCKL_SL);
	th_rot[1] = vessel->CreateThruster (_V(-3,0,0), _V(0, 1,0), MAX_ATT_BLOCKL , ph_blockl, ISP_BLOCKL_SL);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_BANKRIGHT);
	th_rot[0] = vessel->CreateThruster (_V(0,0, 4), _V( 1,0,0), MAX_ATT_BLOCKL , ph_blockl, ISP_BLOCKL_SL);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-4), _V(-1,0,0), MAX_ATT_BLOCKL , ph_blockl, ISP_BLOCKL_SL);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_YAWRIGHT);
	th_rot[0] = vessel->CreateThruster (_V(0,0, 4), _V(-1,0,0), MAX_ATT_BLOCKL , ph_blockl, ISP_BLOCKL_SL);
	th_rot[1] = vessel->CreateThruster (_V(0,0,-4), _V( 1,0,0), MAX_ATT_BLOCKL , ph_blockl, ISP_BLOCKL_SL);
	vessel->CreateThrusterGroup (th_rot, 2, THGROUP_ATT_YAWLEFT);
}

void SetFirstStage (VESSEL *vessel)
{
	vessel->SetSize (20);
	vessel->SetCOG_elev (19);
	vessel->SetEmptyMass (51400.0 + LaikaMass);
	vessel->SetTouchdownPoints (_V(0,-1.0,-15.25), _V(-.7,.7,-15.25), _V(.7,.7,-15.25));
	vessel->SetPMI (_V(75,75,5));
	vessel->SetCrossSections (_V(105,105,25));
	vessel->SetCW (0.1, 0.3, 1.4, 1.4);
	vessel->SetRotDrag (_V(0.7,0.7,1.2));
	vessel->SetPitchMomentScale (1e-6);
	vessel->SetBankMomentScale (1e-6);
	vessel->SetLiftCoeffFunc (0); 
	vessel->ClearMeshes();
	vessel->ShiftCentreOfMass (_V(0,0,5));
	VECTOR3 mesh_dir=_V(0,0,-0.75);
	vessel->AddMesh (hR7stg1, &mesh_dir);
    mesh_dir=_V(3.0,0,-5);
    vessel->AddMesh (hR7strapo, &mesh_dir);
	mesh_dir=_V(-3.0,0,-5);
    vessel->AddMesh (hR7strapv, &mesh_dir);
	mesh_dir=_V(0,3.0,-5);
    vessel->AddMesh (hR7strapb, &mesh_dir);
	mesh_dir=_V(0,-3.0,-5);
    vessel->AddMesh (hR7strapg, &mesh_dir);
	mesh_dir=_V(0,0,17.15);
	vessel->AddMesh (hR7stg2, &mesh_dir);
    mesh_dir=_V(0.0,0,25.55);
	vessel->AddMesh (hR7shroud, &mesh_dir);
	mesh_dir=_V(0,0,22.35);
	vessel->AddMesh (hR7blockl, &mesh_dir);
	mesh_dir=OFS_LAIKA;
	vessel->AddMesh ("Laika", &mesh_dir);
	if (!ph_first)  ph_first  = vessel->CreatePropellantResource (FIRST_PROP_MASS); 
	if (!ph_main)  ph_main  = vessel->CreatePropellantResource (MAIN_PROP_MASS); 
	vessel->SetDefaultPropellantResource (ph_first); 
	vessel->ClearThrusterDefinitions();
	th_first[0] = vessel->CreateThruster (_V(2.5,0,-14.25), _V(0,0,1), THRUST_FIRST_VAC, ph_first, ISP_FIRST_VAC, ISP_FIRST_SL);
	th_first[1] = vessel->CreateThruster (_V(-2.5,0,-14.25), _V(0,0,1), THRUST_FIRST_VAC, ph_first, ISP_FIRST_VAC, ISP_FIRST_SL);
	th_first[2] = vessel->CreateThruster (_V(0,2.5,-14.25), _V(0,0,1), THRUST_FIRST_VAC, ph_first, ISP_FIRST_VAC, ISP_FIRST_SL);
	th_first[3] = vessel->CreateThruster (_V(0,-2.5,-14.25), _V(0,0,1), THRUST_FIRST_VAC, ph_first, ISP_FIRST_VAC, ISP_FIRST_SL);
	th_first[4] = vessel->CreateThruster (_V(-0.25,0.25,-14.75), _V(0,0,1), THRUST_MAIN_VAC, ph_main, ISP_MAIN_VAC, ISP_MAIN_SL);
	th_first[5] = vessel->CreateThruster (_V(0.25,0.25,-14.75), _V(0,0,1), THRUST_MAIN_VAC, ph_main, ISP_MAIN_VAC, ISP_MAIN_SL);
	th_first[6] = vessel->CreateThruster (_V(0.25,-0.25,-14.75), _V(0,0,1), THRUST_MAIN_VAC, ph_main, ISP_MAIN_VAC, ISP_MAIN_SL);
	th_first[7] = vessel->CreateThruster (_V(-0.25,-0.25,-14.75), _V(0,0,1), THRUST_MAIN_VAC, ph_main, ISP_MAIN_VAC, ISP_MAIN_SL);
	thg_first = vessel->CreateThrusterGroup (th_first, 8, THGROUP_MAIN);
	for (int i = 0; i < 5; i++) vessel->AddExhaust (th_first[i], 10.0, 1.0);
	SetAttControlsFirst(vessel);
	status = 0;
}


void SetMainStage (VESSEL *vessel)
{
	vessel->SetSize (20);
	vessel->SetCOG_elev (19);
	vessel->SetEmptyMass (36200.0 + LaikaMass);
	vessel->SetPMI (_V(75,75,5));
	vessel->SetCrossSections (_V(105,105,25));
	vessel->SetCW (0.1, 0.3, 1.4, 1.4);
	vessel->SetRotDrag (_V(0.7,0.7,1.2));
	vessel->SetPitchMomentScale (1e-6);
	vessel->SetBankMomentScale (1e-6);
	vessel->SetLiftCoeffFunc (0); 
	vessel->ClearMeshes();
	VECTOR3 mesh_dir=_V(0,0,-0.75);
	vessel->AddMesh (hR7stg1, &mesh_dir);
	mesh_dir=_V(0,0,17.15);
	vessel->AddMesh (hR7stg2, &mesh_dir);
    mesh_dir=_V(0.0,0,25.55);
	vessel->AddMesh (hR7shroud, &mesh_dir);
	mesh_dir=_V(0,0,22.35);
	vessel->AddMesh (hR7blockl, &mesh_dir);
	mesh_dir=OFS_LAIKA;
	vessel->AddMesh ("Laika", &mesh_dir);
	vessel->ClearThrusterDefinitions();
	if (ph_first) vessel->DelPropellantResource (ph_first);
	if (!ph_main) ph_main  = vessel->CreatePropellantResource (MAIN_PROP_MASS); 
	vessel->SetDefaultPropellantResource (ph_main); 
	th_main[0] = vessel->CreateThruster (_V(-0.25,0.25,-14.25), _V(0,0,1), THRUST_MAIN_VAC, ph_main, ISP_MAIN_VAC, ISP_MAIN_SL);
	th_main[1] = vessel->CreateThruster (_V(0.25,0.25,-14.25), _V(0,0,1), THRUST_MAIN_VAC, ph_main, ISP_MAIN_VAC, ISP_MAIN_SL);
	th_main[2] = vessel->CreateThruster (_V(0.25,-0.25,-14.25), _V(0,0,1), THRUST_MAIN_VAC, ph_main, ISP_MAIN_VAC, ISP_MAIN_SL);
	th_main[3] = vessel->CreateThruster (_V(-0.25,-0.25,-14.25), _V(0,0,1), THRUST_MAIN_VAC, ph_main, ISP_MAIN_VAC, ISP_MAIN_SL);
	thg_main = vessel->CreateThrusterGroup (th_main, 4, THGROUP_MAIN);
	for (int i = 0; i < 4; i++) vessel->AddExhaust (th_main[i], 10.0, 1.0);
	vessel->SetThrusterGroupLevel (thg_main,1.0);
	SetAttControlsMain(vessel);
	status = 1;
}


void SetSecondStage (VESSEL *vessel)
{
	vessel->SetSize (10);
	vessel->SetCOG_elev (5);
	vessel->SetEmptyMass (8100.0 + LaikaMass);
	vessel->SetPMI (_V(22.75,22.75,2.25));
	vessel->SetCrossSections (_V(8.5,8.5,5));
	vessel->SetCW (0.1, 0.3, 1.4, 1.4);
	vessel->SetRotDrag (_V(0.7,0.7,1.2));
	vessel->SetPitchMomentScale (1e-6);
	vessel->SetBankMomentScale (1e-6);
	vessel->SetLiftCoeffFunc (0); 
	vessel->ClearMeshes();
	VECTOR3 mesh_dir=_V(0,0,17.15-18.25);
	vessel->AddMesh (hR7stg2, &mesh_dir);
    mesh_dir=_V(0.0,0,25.55-18.25);
	vessel->AddMesh (hR7shroud, &mesh_dir);
	mesh_dir=_V(0,0,22.35-18.25);
	vessel->AddMesh (hR7blockl, &mesh_dir);
	mesh_dir=OFS_LAIKA-_V(0,0,18.25);
	vessel->AddMesh ("Laika", &mesh_dir);
	if (ph_main) vessel->DelPropellantResource (ph_main);
	if (!ph_second)	ph_second  = vessel->CreatePropellantResource (SECOND_PROP_MASS);
	vessel->SetDefaultPropellantResource (ph_second); 
	vessel->ClearThrusterDefinitions();
	th_second[0] = vessel->CreateThruster (_V(0,0,-5.75), _V(0,0,1), THRUST_SECOND_VAC, ph_second, ISP_SECOND_VAC, ISP_SECOND_SL);
	thg_second = vessel->CreateThrusterGroup (th_second, 1, THGROUP_MAIN);
	vessel->AddExhaust (th_second[0], 3.0, 0.25);
	vessel->SetThrusterGroupLevel (thg_second,1.0);
	vessel->ShiftCentreOfMass (_V(0,0,18.25));
	SetAttControlsSecond(vessel);
	status = 2;    
}

void SetBlockIStage (VESSEL *vessel)
{
	vessel->SetSize (5);
	vessel->SetCOG_elev (2);
	vessel->SetEmptyMass (7200 + LaikaMass);
	vessel->SetPMI (_V(23.95,23.95,2.85));
	vessel->SetCrossSections (_V(7.5,7.5,6.15));
	vessel->SetCW (0.1, 0.3, 1.4, 1.4);
	vessel->SetRotDrag (_V(0.7,0.7,1.2));
	vessel->SetPitchMomentScale (1e-6);
	vessel->SetBankMomentScale (1e-6);
	vessel->SetLiftCoeffFunc (0); 
	vessel->ClearMeshes();
	vessel->ShiftCentreOfMass (_V(0,0,-0.25));
	VECTOR3 mesh_dir=_V(0,0,17.15-18.25+0.25);
	vessel->AddMesh (hR7stg2, &mesh_dir);
	mesh_dir=_V(0,0,22.35-18.25+0.25);
	vessel->AddMesh (hR7blockl, &mesh_dir);
	mesh_dir=OFS_LAIKA-_V(0,0,18.25-0.25);
	vessel->AddMesh ("Laika", &mesh_dir);
	if (!ph_second)	ph_second  = vessel->CreatePropellantResource (SECOND_PROP_MASS);
	vessel->SetDefaultPropellantResource (ph_second); 
	vessel->ClearThrusterDefinitions();
	th_second[0] = vessel->CreateThruster (_V(0,0,-5.5), _V(0,0,1), THRUST_SECOND_VAC, ph_second, ISP_SECOND_VAC, ISP_SECOND_SL);
	thg_second = vessel->CreateThrusterGroup (th_second, 1, THGROUP_MAIN);
	vessel->AddExhaust (th_second[0], 5.0, 0.25);
	vessel->SetThrusterGroupLevel (thg_second,1.0);
	SetAttControlsSecond(vessel);
	status = 3;
}

void SetBlockLStage (VESSEL *vessel)
{
	vessel->SetSize (5.0);
	vessel->SetEmptyMass (1200.0 + LaikaMass);
	vessel->ClearMeshes();
	vessel->SetCrossSections (_V(1.75,1.35,3.15));
	vessel->SetPMI (_V(3.15,3.15,1.75));
	vessel->SetCOG_elev (1.5);
	vessel->ShiftCentreOfMass (_V(0,0.0,4.0));
	vessel->SetPitchMomentScale (1e-6);
	vessel->SetBankMomentScale (1e-6);
	VECTOR3 mesh_dir=_V(0,0,22.35-18.25+0.25-4.0);
	vessel->AddMesh (hR7blockl, &mesh_dir);
	mesh_dir=OFS_LAIKA-_V(0,0,18.25-0.25+4.0);
	vessel->AddMesh ("Laika", &mesh_dir);
	if (ph_second) vessel->DelPropellantResource (ph_second);
	if (!ph_blockl)	ph_blockl  = vessel->CreatePropellantResource (BLOCKL_PROP_MASS);
	vessel->SetDefaultPropellantResource (ph_blockl); 
	vessel->ClearThrusterDefinitions();
	th_blockl[0] = vessel->CreateThruster (_V(0,0,-1.75), _V(0,0,1), THRUST_BLOCKL_VAC, ph_blockl, ISP_BLOCKL_VAC, ISP_BLOCKL_SL);
	thg_blockl = vessel->CreateThrusterGroup (th_blockl, 1, THGROUP_MAIN);
	vessel->AddExhaust (th_blockl[0], 5.0, 0.25);
	vessel->SetThrusterGroupLevel (thg_blockl,0.0);
	SetAttControlsBlockL(vessel);
	status = 4;
}

void SetDummyStage (VESSEL *vessel)
{
	vessel->SetSize (5.0);
	vessel->SetEmptyMass (1200.0);
	vessel->ClearMeshes();
	vessel->SetCrossSections (_V(1.75,1.35,3.15));
	vessel->SetPMI (_V(2.65,2.65,1.5));
	vessel->SetCOG_elev (1.5);
	vessel->SetPitchMomentScale (1e-6);
	vessel->SetBankMomentScale (1e-6);
	VECTOR3 mesh_dir=_V(0,0,22.35-18.25+0.25-4.0);
	vessel->AddMesh (hR7blockl, &mesh_dir);
	if (!ph_blockl)	ph_blockl  = vessel->CreatePropellantResource (BLOCKL_PROP_MASS);
	vessel->SetDefaultPropellantResource (ph_blockl); 
	vessel->ClearThrusterDefinitions();
	th_blockl[0] = vessel->CreateThruster (_V(0,0,-1.25), _V(0,0,1), THRUST_BLOCKL_VAC, ph_blockl, ISP_BLOCKL_VAC, ISP_BLOCKL_SL);
	thg_blockl = vessel->CreateThrusterGroup (th_blockl, 1, THGROUP_MAIN);
	vessel->AddExhaust (th_blockl[0], 5.0, 0.25);
	vessel->SetThrusterGroupLevel (thg_blockl,0.0);
	SetAttControlsBlockL(vessel);
	status = 5;
	vessel->AddForce(_V(0, 0, -30000), _V(rand11()*0.1, rand11()*0.1, 0));
}

void SeparateStage (VESSEL *vessel, UINT stage)
{
	VESSELSTATUS vs;
	VESSELSTATUS vs1;
	VESSELSTATUS vs2;
	VESSELSTATUS vs3;
	VESSELSTATUS vs4;
	VECTOR3 ofs = _V(0,0,0);
	VECTOR3 vel = _V(0,0,0);
	VECTOR3 ofs1 = _V(0,0,0);
	VECTOR3 vel1 = _V(0,0,0);
    VECTOR3 ofs2 = _V(0,0,0);
	VECTOR3 vel2 = _V(0,0,0);
    VECTOR3 ofs3 = _V(0,0,0);
	VECTOR3 vel3 = _V(0,0,0);
	VECTOR3 ofs4 = _V(0,0,0);
	VECTOR3 vel4 = _V(0,0,0);
	vessel->GetStatus (vs);
	vessel->GetStatus (vs1);
	vessel->GetStatus (vs2);
    vessel->GetStatus (vs3);
	vessel->GetStatus (vs4);


	if (stage == 0)
	{
		ofs1 = OFS_STRAPB;
		ofs2 = OFS_STRAPV;
		ofs3 = OFS_STRAPG;
		ofs4 = OFS_STRAPO;
		vel1 = _V(2.0,0, -2.0);
		vel2 = _V(-2.0, 0, -2.0);
		vel3 = _V(0,2.0, -2.0);
		vel4 = _V(0,-2.0, -2.0);
	}
	
	else if (stage == 1)
	{
		ofs = OFS_STAGE1;	
		vel = _V(0,0,-2.0);	
	}
	else if (stage == 2)
	{
		ofs = OFS_SHROUD;		
		vel = _V(0.0,1.0,2.0); 
	}
	else if (stage == 3)
	{
		ofs = OFS_BLOCKI;	
		vel = _V(0,0,-1.0);	
	}

	else if (stage == 4)
	{
		ofs = OFS_LAIKA - _V(0,0,18.25-0.25+4.0);
		vel = _V(0,0, 0.0);
	}


	VECTOR3 rofs, rvel = {vs.rvel.x, vs.rvel.y, vs.rvel.z};
	VECTOR3 rofs1, rvel1 = {vs1.rvel.x, vs1.rvel.y, vs1.rvel.z};
	VECTOR3 rofs2, rvel2 = {vs2.rvel.x, vs2.rvel.y, vs2.rvel.z};
	VECTOR3 rofs3, rvel3 = {vs3.rvel.x, vs3.rvel.y, vs3.rvel.z};
	VECTOR3 rofs4, rvel4 = {vs4.rvel.x, vs4.rvel.y, vs4.rvel.z};
	vessel->Local2Rel (ofs, vs.rpos);
	vessel->Local2Rel (ofs1, vs1.rpos);
	vessel->Local2Rel (ofs2, vs2.rpos);
	vessel->Local2Rel (ofs3, vs3.rpos);
	vessel->Local2Rel (ofs4, vs4.rpos);
	vessel->GlobalRot (vel, rofs);
	vessel->GlobalRot (vel1, rofs1);
	vessel->GlobalRot (vel2, rofs2);
	vessel->GlobalRot (vel3, rofs3);
	vessel->GlobalRot (vel4, rofs4);
	vs.rvel.x = rvel.x+rofs.x;
	vs.rvel.y = rvel.y+rofs.y;
	vs.rvel.z = rvel.z+rofs.z;
	vs1.rvel.x = rvel1.x+rofs1.x;
	vs1.rvel.y = rvel1.y+rofs1.y;
	vs1.rvel.z = rvel1.z+rofs1.z;
	vs2.rvel.x = rvel2.x+rofs2.x;
	vs2.rvel.y = rvel2.y+rofs2.y;
	vs2.rvel.z = rvel2.z+rofs2.z;
	vs3.rvel.x = rvel3.x+rofs3.x;
	vs3.rvel.y = rvel3.y+rofs3.y;
	vs3.rvel.z = rvel3.z+rofs3.z;
	vs4.rvel.x = rvel4.x+rofs4.x;
	vs4.rvel.y = rvel4.y+rofs4.y;
	vs4.rvel.z = rvel4.z+rofs4.z;
	
	if (stage == 0)
	{
		vs1.vrot.x = 0.0;
		vs1.vrot.y = -0.24;
		vs1.vrot.z = 0.0;		
		vs2.vrot.x = 0.0;
		vs2.vrot.y = 0.18;
		vs2.vrot.z = 0.0;		
		vs3.vrot.x = 0.12;
		vs3.vrot.y = 0.0;
		vs3.vrot.z = 0.0;		
		vs4.vrot.x = -0.14;
		vs4.vrot.y = 0.0;
		vs4.vrot.z = 0.0;		
        oapiCreateVessel("Strap-on Booster B", "r7strapbmol", vs3);
        oapiCreateVessel("Strap-on Booster V", "r7strapvmol", vs2);
		oapiCreateVessel("Strap-on Booster G", "r7strapgmol", vs4);
		oapiCreateVessel("Strap-on Booster O", "r7strapomol", vs1);

		vessel->DelPropellantResource (ph_first);

		vessel->DelThrusterGroup (thg_first, THGROUP_MAIN, true);
		vessel->DelThrusterGroup (thg_pitchup, THGROUP_ATT_PITCHUP, true);
		vessel->DelThrusterGroup (thg_pitchdown, THGROUP_ATT_PITCHDOWN, true);

		vessel->DelThrusterGroup (THGROUP_ATT_BANKLEFT, true);
		vessel->DelThrusterGroup (THGROUP_ATT_BANKRIGHT, true);
		vessel->DelThrusterGroup (THGROUP_ATT_YAWLEFT, true);
		vessel->DelThrusterGroup (THGROUP_ATT_YAWRIGHT, true);

		vessel->ClearThrusterDefinitions();

		SetMainStage (vessel);	
	}		

    else if (stage == 1)
	{
	    vs.vrot.x = 0.05;	
		vs.vrot.y = 0.05;
		vs.vrot.z = 0.0;

		oapiCreateVessel("R7 Molniya Block A", "r7stage1mol", vs);

		vessel->DelPropellantResource (ph_main);

		vessel->DelThrusterGroup (thg_main, THGROUP_MAIN, true);
		vessel->DelThrusterGroup (thg_pitchup, THGROUP_ATT_PITCHUP, true);
		vessel->DelThrusterGroup (thg_pitchdown, THGROUP_ATT_PITCHDOWN, true);

		vessel->DelThrusterGroup (THGROUP_ATT_BANKLEFT, true);
		vessel->DelThrusterGroup (THGROUP_ATT_BANKRIGHT, true);
		vessel->DelThrusterGroup (THGROUP_ATT_YAWLEFT, true);
		vessel->DelThrusterGroup (THGROUP_ATT_YAWRIGHT, true);

		vessel->ClearThrusterDefinitions();

		SetSecondStage (vessel);	
	}

	else if (stage == 2)
	{
		vs.vrot.x = 0.0;	
		vs.vrot.y = 0.05;	
		vs.vrot.z = 0.0;	

		oapiCreateVessel("R7 Molniya Shroud", "r7shroudmol", vs);

		vessel->DelThrusterGroup (thg_second, THGROUP_MAIN, true);
		vessel->DelThrusterGroup (thg_pitchup, THGROUP_ATT_PITCHUP, true);
		vessel->DelThrusterGroup (thg_pitchdown, THGROUP_ATT_PITCHDOWN, true);

		vessel->DelThrusterGroup (THGROUP_ATT_BANKLEFT, true);
		vessel->DelThrusterGroup (THGROUP_ATT_BANKRIGHT, true);
		vessel->DelThrusterGroup (THGROUP_ATT_YAWLEFT, true);
		vessel->DelThrusterGroup (THGROUP_ATT_YAWRIGHT, true);

		vessel->ClearThrusterDefinitions();

		SetBlockIStage (vessel);
	}
	else if (stage == 3)
	{
		vs.vrot.x = -0.05 + rand() / RAND_MAX * 0.1; //0.0;
		vs.vrot.y = -0.05 + rand() / RAND_MAX * 0.1; //0.05;
		vs.vrot.z = -0.01 + rand() / RAND_MAX * 0.02; // 0.0;
		vs1.vrot.x = -0.0005 + rand() / RAND_MAX * 0.001; //0.0;
		vs1.vrot.y = -0.0005 + rand() / RAND_MAX * 0.001; //0.0;
		vs1.vrot.z = -0.0001 + rand() / RAND_MAX * 0.0002; //0.0;
		oapiCreateVessel("R7 Molniya Block I", "r7blockimol", vs);

		vessel->DelPropellantResource (ph_second);

		vessel->DelThrusterGroup (thg_second, THGROUP_MAIN, true);
		vessel->DelThrusterGroup (thg_pitchup, THGROUP_ATT_PITCHUP, true);
		vessel->DelThrusterGroup (thg_pitchdown, THGROUP_ATT_PITCHDOWN, true);

		vessel->DelThrusterGroup (THGROUP_ATT_BANKLEFT, true);
		vessel->DelThrusterGroup (THGROUP_ATT_BANKRIGHT, true);
		vessel->DelThrusterGroup (THGROUP_ATT_YAWLEFT, true);
		vessel->DelThrusterGroup (THGROUP_ATT_YAWRIGHT, true);

		vessel->ClearThrusterDefinitions();

		SetBlockLStage (vessel);
	}

	else if (stage == 4)
	{
		vs.vrot.x = 0.0;
		vs.vrot.y = 0.0;
		vs.vrot.z = 0.0;
		vs1.vrot.x = -0.0005 + (double)rand() / (double)RAND_MAX * 0.001; 
		vs1.vrot.y = -0.0005 + (double)rand() / (double)RAND_MAX * 0.001;
		vs1.vrot.z = -0.0001 + (double)rand() / (double)RAND_MAX * 0.0002; 
		
		hLaika = oapiCreateVessel ("Laika", "Laika", vs);
		oapiSetFocusObject (hLaika);
		VESSEL *NewLaika;
		NewLaika = oapiGetVesselInterface(hLaika);
		if (strlen(TralFileName) > 0)
			strcpy(((Laika*)NewLaika)->Proximator->TralFileName, TralFileName); 

		((Laika*)NewLaika)->Proximator->bAuto = autopilot;
		((Laika*)NewLaika)->Proximator->bStruct = bStruct;
  		
		vessel->SetThrusterGroupLevel (thg_blockl,0.0);
				
		SetDummyStage (vessel);
	}


}

// ==============================================================
// API interface
// ==============================================================



// Initialisation
DLLCLBK VESSEL *ovcInit (OBJHANDLE hvessel, int flightmodel)
{
	// seed random generator
	srand((unsigned)time( NULL ));

	stage_sep = 0;
	bManualSeparate = false;
	
	if (!refcount++) {
		hR7stg1 = oapiLoadMeshGlobal ("r7stg1mol");
		hR7stg2 = oapiLoadMeshGlobal ("r7blockmol");
		hR7shroud = oapiLoadMeshGlobal ("r7molshroud");
		hR7strapo = oapiLoadMeshGlobal ("r7strapomol");
		hR7strapb = oapiLoadMeshGlobal ("r7strapbmol");
		hR7strapv = oapiLoadMeshGlobal ("r7strapvmol");
		hR7strapg = oapiLoadMeshGlobal ("r7strapgmol");
		hR7blockl = oapiLoadMeshGlobal ("r7blocklmol");
	}
	return new VESSEL (hvessel, flightmodel);
}

// Cleanup
DLLCLBK void ovcExit (VESSEL *vessel)
{
	--refcount;
	if (vessel) delete vessel;
}

// Keyboard interface handler
DLLCLBK int ovcConsumeKey (VESSEL *vessel, const char *keystate)
{

	if (KEYMOD_SHIFT (keystate)) 
	{
		return 0; // shift combinations are reserved
	}
	else if (KEYMOD_CONTROL (keystate)) 
	{
		// insert ctrl key combinations here
		if (KEYDOWN (keystate, OAPI_KEY_U))
		{ 
			if (oapiAcceptDelayedKey (OAPI_KEY_U, 1.0))
				// Report the inclination setpoint
				if(DisplayIncSP) {
					DisplayIncSP = false;
				}
				else {
					DisplayIncSP = true;
				}
																
			return 1;
		}
	}
	else 
	{ // unmodified keys
		if (KEYDOWN (keystate, OAPI_KEY_J)) { // "Jettison Stage"
			if (oapiAcceptDelayedKey (OAPI_KEY_J, 1.0))
				bManualSeparate = true;
			return 1;
		}

		if (KEYDOWN (keystate, OAPI_KEY_O))
		{ 
			if (oapiAcceptDelayedKey (OAPI_KEY_O, 1.0))
				if (!guidance) { 
					guidance=true;
					sprintf(oapiDebugString(),"Guidance ON");
				}
				else {
					guidance = false;
					sprintf(oapiDebugString(),"Guidance OFF");
				}
			return 1;
		}
		if (KEYDOWN (keystate, OAPI_KEY_U))
		{ 
			if (oapiAcceptDelayedKey (OAPI_KEY_U, 1.0))
				// Get a new Inclination setpoint from pilot.
				if(!guidance) {
					GetInclinationSP() ;
				}
				
			return 1;
		}

		if (KEYDOWN (keystate, OAPI_KEY_D))
		{ 
			if (oapiAcceptDelayedKey (OAPI_KEY_D, 1.0))
				// Test calc, debug!
				TestCalc();
				
			return 1;
		}
		
	}
	return 0;
}


DLLCLBK void ovcSetClassCaps (VESSEL *vessel, FILEHANDLE cfg)
{
	// propellant resources
	ph_first          = NULL;
	ph_second         = NULL;
	ph_main			 = NULL;
	ph_blockl		 = NULL;
	thg_first        = NULL;
	thg_second       = NULL;
	thg_main		 = NULL;
	thg_blockl		 = NULL;
	thg_pitchup		 = NULL;
	thg_pitchdown	 = NULL;	
}


DLLCLBK void ovcTimestep (VESSEL *vessel, double simt)
{
	if (!guidance && !MECO_Cond ) {
		if(!NewIncSP) {
			// Set the Inclination setpoint for the guidance computer
			vessel->GetEquPos( Longitude, Latitude, Radius );

			if (PadHeading < 258.40 && PadHeading > 233.44) {
				Clairaut_Constant = sin((PadHeading - 180)*RAD) * cos(Latitude);
				INCLINATION = acos(Clairaut_Constant)*DEG;
			} else if (stage < 4) {
				BadHeading = true ;
			}
			
		}
	}

		
	//Check if guidance is on and if so, call
	if (guidance)
	{
		if (BadHeading) {
			sprintf(oapiDebugString(),"Guidance system cannot launch from this heading.");
			guidance = false;
		} else {

			// Calculate MET
			if (TZero == -100) {
				MET = -10 ;
				TZero = 0;
			}
			else {
				MET = simt - TZero;
			}

			// Call the guidance routine
			Guidance_Routine(vessel, simt, SimT_Last);
			SimT_Last = MET;
		}
	}

	// Display the inclination setpoint if the switch is on.
	if (DisplayIncSP) {
		sprintf(oapiDebugString(),"Inclination Setpoint: %.2f deg", INCLINATION);
	}
	
	if (stage == 0)
	{
    	if (vessel->GetEngineLevel(ENGINE_MAIN) > .95 && !engineignited)
		{
			ignition_time = simt;
			engineignited = true;
		}

		last_simt = simt;            

		if (vessel->GetPropellantMass(ph_first) ==0 || bManualSeparate)
		{
			ttime=simt;
			SeparateStage (vessel, stage);
			bManualSeparate=false;
			stage = 1;
		} 

	}
	
	else if (stage == 1)
	{
		last_simt = simt;
		
		if (vessel->GetPropellantMass(ph_main) == 0 ||bManualSeparate)
		{
			ttime=simt;
			bManualSeparate=false;
			SeparateStage (vessel, stage);
			stage = 2;
		} 
		
	}
	else if (stage == 2)
	{
		last_simt = simt;
		
		if (simt - ttime > 38.0 || bManualSeparate)
		{
			stage_sep = simt;
			bManualSeparate=false;
			SeparateStage (vessel, stage);
			stage = 3;
		}
	}
    else if (stage == 3)
	{
		last_simt = simt;

		if (vessel->GetPropellantMass(ph_second) == 0 || bManualSeparate || MECO_Cond)
		{
			stage_sep = simt;
			bManualSeparate=false;
			SeparateStage (vessel, stage);
			stage = 4;
		}
	}

	else if (stage == 4)
	{		
		last_simt = simt;
		
		if (TLIBurnCompleted || bManualSeparate)
		{
			stage_sep = simt;
			bManualSeparate=false;
			SeparateStage (vessel, stage);
			stage = 5;
		}
	}
}

// Read status from scenario file

DLLCLBK void ovcLoadStateEx (VESSEL *vessel, FILEHANDLE scn, void *vs)

{
    char *line;
	double temp;

	while (oapiReadScenario_nextline (scn, line)) {
        if (!strnicmp (line, "CONFIGURATION", 13)) {
            sscanf (line+13, "%d", &status);
		} else if (!strnicmp (line, "TIME", 4)) {
            sscanf (line+4, "%lf", &temp);
			ignition_time=-temp;
		} else if (!strnicmp (line, "HEADING", 7)) {
			sscanf(line+7, "%lf", &PadHeading);
			vessel->ParseScenarioLineEx (line, vs);
		} 
		else if (!strnicmp (line, "TRALFILE", 8)) 
		{
			sscanf (line + 8, "%s", TralFileName);
			if (strlen(TralFileName) > 0)
			{
				// reset Tral file
				FILE *TralFile;
				TralFile = fopen(TralFileName, "w");
				fclose(TralFile);
			}
		}
		else if (!strnicmp (line, "AUTO", 4)) 
		{
			sscanf (line + 4, "%1d", &autopilot);
		}
		else if (!strnicmp (line, "STRUCT", 6)) 
		{
			sscanf (line + 6, "%1d", &bStruct);
		}
		else {
            vessel->ParseScenarioLineEx (line, vs);
			// unrecognised option - pass to Orbiter's generic parser
        }
    }
	switch (status) {
	case 0:
		stage=0;
		SetFirstStage(vessel);
		break;
	case 1:
		stage=1;
		SetMainStage(vessel);
		break;
	case 2:
		stage=2;
		SetSecondStage(vessel);
		break;
	case 3:
		stage=3;
		SetBlockIStage(vessel);
		break;
	case 4:
		stage=4;
		SetBlockLStage(vessel);
		break;
	case 5:
		stage=5;
		SetDummyStage(vessel);
		break;		
	}
}



// Save status to scenario file

DLLCLBK void ovcSaveState (VESSEL *vessel, FILEHANDLE scn)
{
	// default vessel parameters
	vessel->SaveDefaultState (scn);
	// custom parameters
	oapiWriteScenario_int (scn, "CONFIGURATION", status);
	oapiWriteScenario_float (scn, "TIME", last_simt-ignition_time);

	oapiWriteScenario_int (scn, "AUTO", autopilot);
	oapiWriteScenario_int (scn, "STRUCT", bStruct);
	oapiWriteScenario_string (scn, "TRALFILE", TralFileName);
}


// Returns the cross product of two vectors
inline VECTOR3 CrossProduct(const VECTOR3 &a, const VECTOR3 &b)
{
	VECTOR3 c;

	c.x = (a.y * b.z - a.z * b.y);
	c.y = (a.z * b.x - a.x * b.z);
	c.z = (a.x * b.y - a.y * b.x);

	return c;
}

// Returns the absolute value of a vector
double AbsOfVector(const VECTOR3 &Vec)
{
	double Result;
	Result = sqrt(Vec.x*Vec.x + Vec.y*Vec.y + Vec.z*Vec.z);
	return Result;
}


// Returns the Angular momentum vector (h)
inline VECTOR3 AngularMomentum(const VECTOR3 &R, const VECTOR3 &V)
{
	VECTOR3 h;

	h = CrossProduct(R, V);
	return h;
}

// Divide a vector by a scalar double
inline VECTOR3 DivideVector (const VECTOR3 &a, double b)
{
	VECTOR3 c;
	c.x = a.x / b;
	c.y = a.y / b;
	c.z = a.z / b;

	return c;
}


// change the order of the vectors. because they are weird!
VECTOR3 ReorderVector(const VECTOR3 &Vect)
{	
	VECTOR3 temp;
	
	// change the order of the vectors from left handed to normal.
	temp.x = Vect.x;
	temp.y = Vect.z;
	temp.z = Vect.y;
	return temp;
}


// Multiply a vector by a scalar double
inline VECTOR3 MultiplyVector (const VECTOR3 &a, double b)
{
	VECTOR3 c;
	c.x = a.x * b;
	c.y = a.y * b;
	c.z = a.z * b;

	return c;
}

// Dot product of two vectors
inline double DotProduct (const VECTOR3 &a, const VECTOR3 &b)
{
	double c;
	c = a.x * b.x + a.y * b.y + a.z * b.z;
	return c;
}

inline MATRIX3 MakeTranslationMatrix( const VECTOR3 &I, const VECTOR3 &J,const VECTOR3 &K )
{
	MATRIX3 TransM;

	TransM.m11 = I.x;
	TransM.m12 = I.y;
	TransM.m13 = I.z;
	
	TransM.m21 = J.x;
	TransM.m22 = J.y;
	TransM.m23 = J.z;

	TransM.m31 = K.x;
	TransM.m32 = K.y;
	TransM.m33 = K.z;

	return TransM;
}


void MultiplyByMatrix(const VECTOR3 &Initial, const MATRIX3 &RotMatrix, VECTOR3 &Result)
{

	Result.x =	(Initial.x * RotMatrix.m11) 
				+ (Initial.y * RotMatrix.m12) 
				+ (Initial.z * RotMatrix.m13);
	
	Result.y =	(Initial.x * RotMatrix.m21) 
				+ (Initial.y * RotMatrix.m22) 
				+ (Initial.z * RotMatrix.m23);
	
	Result.z =	(Initial.x * RotMatrix.m31) 
				+ (Initial.y * RotMatrix.m32) 
				+ (Initial.z * RotMatrix.m33);
}


// Returns the vector of eccentricity.
inline VECTOR3 EccentricityVector(double u, const VECTOR3 R2, const VECTOR3 V2)
{
	VECTOR3 ecc;
	double V_abs, R_abs, Part1a, Part2a;
	VECTOR3 Part1;
	VECTOR3 Part2;

	V_abs = AbsOfVector(V2);
	R_abs = AbsOfVector(R2);
	Part1a = V_abs*V_abs - (u / R_abs);
	Part1 = MultiplyVector(R2, Part1a);
	Part2a = DotProduct(R2, V2);
	Part2 = MultiplyVector(V2, Part2a); 

	Part1 = Part1 - Part2;
	ecc = Part1 * (1/u);
	return ecc;
}


// Returns Inclination
inline double Inclination(const VECTOR3 &h)
{
	double i;
	double h_abs;
	h_abs = AbsOfVector(h);
	i = acos(h.z / h_abs);
	return i;
}


// Returns Longitude of the Ascending Node
inline double LAN(const VECTOR3 &n, double i)
{
	double n_abs;
	double LAN;

	// calculate vector n and magnitude of n
	n_abs = AbsOfVector(n);
	
	// Longitude of the Ascending node.
	if (i == 0) {
		LAN = 0;
	} else if (n.y < 0) {
		LAN = 2*PI - acos(n.x / n_abs);
	} else {
		LAN = acos(n.x / n_abs);
	}
	return LAN;
}


// Returns the true anomaly
inline double TrueAnomaly(VECTOR3 &e, VECTOR3 &R, VECTOR3 &V)
{
	double f;
	double e_abs;
	double R_abs;

	e_abs = AbsOfVector(e);
	R_abs = AbsOfVector(R);

	f = acos((R.x * e.x + R.y * e.y + R.z * e.z) / (R_abs * e_abs));
	if ( DotProduct(R, V) < 0){
		f = 2*PI - f;
	}
	return f;
}

// Returns the argument of periapsis
inline double ArgOfPeriapsis(VECTOR3 &n, VECTOR3 &e)
{
	double omega;
	double n_abs;
	double e_abs;

	n_abs = AbsOfVector(n);
	e_abs = AbsOfVector(e);
	omega = acos((n.x * e.x + n.y * e.y + n.z * e.z) / (n_abs * e_abs));
	if (e.z < 0) {
		omega = 2*PI - omega;
	}
	return omega;

}


// Returns the Eccentric Anomaly
inline double EccentricAnomaly(const VECTOR3 &e, double f)
{
	double E;
	double e_abs;
	e_abs = AbsOfVector(e);
	E = 2 * atan(sqrt((1-e_abs)/(1+e_abs))*tan(f/2));
	return E;
}

// Returns the Mean Anomaly
inline double MeanAnomaly(double E, const VECTOR3 &e)
{
	double Me;
	double e_abs;
	e_abs = AbsOfVector(e);
	Me = E - e_abs*sin(E);
	return Me;
}


// Returns the Period of the orbit
inline double Period(double a, double u)
{
	double T;
	T = 2*PI*sqrt((a*a*a)/u);
	return T;

}

// Returns the time since periapsis for an anomaly
inline double TimeSincePeriapsis(double Me, double T)
{
	double TSP;
	TSP = Me / (2*PI)*T;
	return TSP;
}

// Returns the Semi-Major Axis of the orbit
inline double SemiMajorAxis(double E, double u)
{
	// E is specific mechanical energy
	double a;
	a = -u / (2 * E);
	return a;

}

// Returns Specific Mechanical Energy (E)
inline double SpecMechEnergy(const VECTOR3 &V, const VECTOR3 &R, double u)
{
	double SME;
	double V_abs, R_abs;
	V_abs = AbsOfVector(V);
	R_abs = AbsOfVector(R);
	SME = (V_abs*V_abs / 2) - (u / R_abs);
	return SME;

}


double CalcTimeToNode()
{

	VECTOR3 h;			// angular momentum of our ship.

	hTargetVessel = oapiGetGbodyByName("moon");
	
	// 1. Get the angular momentum vector on the target orbit
	hSetGbody = oapiGetGbodyByName("EARTH");
	oapiGetRelativeVel(hTargetVessel, hSetGbody, &V_targ);
	oapiGetRelativePos(hTargetVessel, hSetGbody, &R_targ);
	
	R_targ = ReorderVector(R_targ);
	V_targ = ReorderVector(V_targ);

	h_targ = AngularMomentum(R_targ, V_targ);

	// 2. Setup the target orbit plane relative reference
	GbodyMass = oapiGetMass(hSetGbody);
	Mu_Planet = GKSI * GbodyMass;

	h_targ_abs = AbsOfVector(h_targ);
	K_targ = DivideVector (h_targ, h_targ_abs);
		
	eVec = EccentricityVector(Mu_Planet, R_targ, V_targ);
	eVec_abs = AbsOfVector(eVec);
	I_targ = DivideVector(eVec, eVec_abs);

	RefYAxis = CrossProduct(K_targ, I_targ);
	RefYAxis_abs = AbsOfVector(RefYAxis);
	J_targ = DivideVector(RefYAxis, RefYAxis_abs);

	TMatrix = MakeTranslationMatrix( I_targ, J_targ, K_targ );
		
		// Get State vectors for our ship.
	Vessel=oapiGetFocusInterface();
	Vessel->GetRelativePos(hSetGbody, RPos);
	Vessel->GetRelativeVel(hSetGbody, VVel);
		
	RPos = ReorderVector(RPos);
	VVel = ReorderVector(VVel);
		
	MultiplyByMatrix(RPos, TMatrix, RPosF);
	MultiplyByMatrix(VVel, TMatrix, VVelF);
			
		// 4. Calculate n, i, then ascending node relative to the target orbit
	h = AngularMomentum(RPosF, VVelF);
	inc = Inclination(h);
	KAxis.x = 0;
	KAxis.y = 0;
	KAxis.z = 1;
	n = CrossProduct(KAxis, h);
	LongOfAN = LAN(n, inc);
		
	// Eccentricity (vector), True Anomaly and Argument of Periapsis,
	// and Argument of Ascension
		
	eVec = EccentricityVector(Mu_Planet, RPosF, VVelF);
	f = TrueAnomaly(eVec, RPosF, VVelF);
	ArgOfPeri = ArgOfPeriapsis(n,eVec);
	ArgOfAscension = f + ArgOfPeri; // angle since the ascending node.
	if (ArgOfAscension > 2*PI) 
	{
		ArgOfAscension = ArgOfAscension - 2*PI;
	}
		
	// 5.  Calculate the times of the ascending and descending nodes.

	// Time of Adjusted (inverse) Argument of periapsis
	//   - Eccentric anomaly
	EccAnomaly = EccentricAnomaly(eVec, (2*PI - ArgOfPeri));
	//   - Mean anomaly
	MnAnomaly = MeanAnomaly(EccAnomaly, eVec);
	//   - Period of orbit
	SpMechE = SpecMechEnergy(VVelF, RPosF, Mu_Planet);
	sma = SemiMajorAxis(SpMechE, Mu_Planet);
	PeriodT = Period(sma, Mu_Planet);
	// AdjAOP Time since periapsis is
	TimeOfAdjAOP = TimeSincePeriapsis(MnAnomaly, PeriodT);
	
	// Calculate the current time since periapsis
	EccAnomaly = EccentricAnomaly(eVec, f);
	MnAnomaly = MeanAnomaly(EccAnomaly, eVec);
	// Time since periapsis is
	TimeOfTrueAnomaly = TimeSincePeriapsis(MnAnomaly, PeriodT);

	// Time to ascending node.
	TimeToAscendingNode = TimeOfAdjAOP - TimeOfTrueAnomaly;
	if (TimeToAscendingNode < 0) {
		TimeToAscendingNode = PeriodT + TimeToAscendingNode;
	}

	// Time to the descending node
	ArgOfPeriPlusPI = ArgOfPeri + PI; 
	if (ArgOfPeriPlusPI > 2*PI) {
		ArgOfPeriPlusPI = ArgOfPeriPlusPI - 2*PI;
	}
	//   - Eccentric anomaly
	EccAnomaly = EccentricAnomaly(eVec, (2*PI - ArgOfPeriPlusPI));
	//   - Mean anomaly
	MnAnomaly = MeanAnomaly(EccAnomaly, eVec);
	// time between periapsis and the descending node
	TimeOfAdjAOPPlusPI = TimeSincePeriapsis(MnAnomaly, PeriodT);
	TimeToDescendingNode = TimeOfAdjAOPPlusPI - TimeOfTrueAnomaly;
	if (TimeToDescendingNode < 0) {
		TimeToDescendingNode = PeriodT + TimeToDescendingNode;
	}

	if (TimeToDescendingNode > TimeToAscendingNode)
	  return TimeToAscendingNode;
	else
	  return TimeToDescendingNode;
}



double CalcTLIBurnTime()
{
// %%% to be implemented
	return 0;
}


void TestCalc()
{
	// 1. Get the angular momentum vector on the target orbit
	OBJHANDLE hMoon = oapiGetGbodyByName("moon");
	OBJHANDLE hEarth = oapiGetGbodyByName("earth");
	VECTOR3 R_MoonE;
	VECTOR3 V_MoonE;

	oapiGetRelativeVel(hMoon, hEarth, &V_MoonE);
	oapiGetRelativePos(hMoon, hEarth, &R_MoonE);

	R_MoonE = ReorderVector(R_MoonE);
	V_MoonE = ReorderVector(V_MoonE);
	VECTOR3 AM_MoonE = AngularMomentum(R_MoonE, V_MoonE);

	// 2. Setup the target orbit plane relative reference
	double EarthMass = oapiGetMass(hEarth);
	double Mu_Earth = GKSI * EarthMass;

	double absAM_MoonE = AbsOfVector(AM_MoonE);
	VECTOR3 UnitZMoonVec = DivideVector (AM_MoonE, absAM_MoonE);

	VECTOR3 eMoonVecE = EccentricityVector(Mu_Earth, R_MoonE, V_MoonE);
	double eMoonE = AbsOfVector(eMoonVecE);
	VECTOR3 xMoonVec = DivideVector(eMoonVecE, eMoonE);

	VECTOR3 yRefVec = CrossProduct(UnitZMoonVec, xMoonVec);
	double yRef = AbsOfVector(yRefVec);
	VECTOR3 yMoonVec = DivideVector(yRefVec, yRef);

	MATRIX3 TranslMatrix = MakeTranslationMatrix( xMoonVec, yMoonVec, UnitZMoonVec);

		// 3. Get State vectors for our ship.
	VESSEL *Ams;
	VECTOR3 R_AmsE; 
    VECTOR3 V_AmsE;      
	Ams=oapiGetFocusInterface();
	Ams->GetRelativePos(hEarth, R_AmsE);
	Ams->GetRelativeVel(hEarth, V_AmsE);
		
	R_AmsE = ReorderVector(R_AmsE);
	V_AmsE = ReorderVector(V_AmsE);
		
	VECTOR3 R_AmsF;      
    VECTOR3 V_AmsF;      
	MultiplyByMatrix(R_AmsE, TranslMatrix, R_AmsF);
	MultiplyByMatrix(V_AmsE, TranslMatrix, V_AmsF);


		// 4. Calculate n, i, then ascending node relative to the target orbit
	VECTOR3 UnitZAmsVec;
	UnitZAmsVec.x = 0;
	UnitZAmsVec.y = 0;
	UnitZAmsVec.z = 1;

	VECTOR3 Am_AmsVec = AngularMomentum(R_AmsF, V_AmsF);
	double AmsIncl = Inclination(Am_AmsVec);
	VECTOR3 Node = CrossProduct(UnitZAmsVec, Am_AmsVec);
	LongOfAN = LAN(Node, AmsIncl);
		
	sprintf(oapiDebugString(), "i: %f", AmsIncl);
}