// ==============================================================
// ORBITER vessel module: Laika lunar probe fragment
//
// See LaikaFragment.h for description.
//
// Developed by: Andrew Thielmann (igel)
//
// ==============================================================

#include "LaikaFragment.h"
#include "time.h"
#include "stdio.h"


LaikaFragment::LaikaFragment (OBJHANDLE hObj, int fmodel): PRVESSEL (hObj, fmodel)
{
	bJustDeployed = true;
	Proximator = new TProximator(this);
}


LaikaFragment::~LaikaFragment ()
{
	if (Proximator) 
		delete Proximator;
}


// General initialization of general params, called before config file
void LaikaFragment::SetClassCaps (FILEHANDLE cfg)
{
}


// Read status from scenario file
void LaikaFragment::LoadState (FILEHANDLE scn, void *vs)
{
    char *line;

	while (oapiReadScenario_nextline (scn, line)) 
	{
        if (!strnicmp (line, "ID", 2)) 
            sscanf (line+2, "%d", &FragmentID);
		// try proximator's parse
		else if (!Proximator->ParsedScenarioLine(line))
			// unrecognised option - pass to Orbiter's generic parser
			ParseScenarioLineEx (line, vs);
    }

	bJustDeployed = false;
	SetExtraConfiguration ();
}


// Save status to scenario file
void LaikaFragment::SaveState (FILEHANDLE scn)
{
	// save default vessel parameters
	SaveDefaultState (scn);

	// save proximator parameters
	Proximator->SaveState (scn);

	// save custom parameters
	oapiWriteScenario_int (scn, "ID", FragmentID);
}


// called after loading scenario or after in-flight creation
void LaikaFragment::SetExtraConfiguration()
{
	if (FragmentID == FID_LANDER)
	{
		// mesh
		strcpy(MeshName, "LaikaLander");

		// consts
		dQTreshold = 150000;
		dVTotalDestructionTreshold = 150;

		// size and geometry
		SetEmptyMass (300.0);
		SetSize (1);
		SetPMI (_V(3, 3, 3));

		// aerodynamics
		SetCrossSections (_V(1.0, 1.0, 1.0));
		SetCW (1, 1, 1, 1);			// wind resistance
		SetRotDrag (_V(1.5,1.5,1.5));
		SetPitchMomentScale (0.001);
		SetBankMomentScale (0.001);

		// surface
		SetSurfaceFrictionCoeff (10.0, 10.0);
		SetCOG_elev (1);
		SetTouchdownPoints (_V(0, 4, -0.5), _V(-4, -4, -0.5), _V(4, -4, -0.5));
	}
	else if (FragmentID == FID_KTDU)
	{
		// mesh
		strcpy(MeshName, "LaikaKTDU");
		// consts
		dQTreshold = 150000;
		dVTotalDestructionTreshold = 150;

		// size and geometry
		SetEmptyMass (50.0);
		SetSize (1);
		SetPMI (_V(2, 2, 0.5));

		// aerodynamics
		SetCrossSections (_V(1.5, 1.5, 0.2));
		SetCW (0.5, 10, 5, 5);			// wind resistance
		SetRotDrag (_V(1.5,1.5,1.5));
		SetPitchMomentScale (0.001);
		SetBankMomentScale (0.001);

		// surface
		SetSurfaceFrictionCoeff (10.0, 10.0);
		SetCOG_elev (1);
		SetTouchdownPoints (_V(0, 0.5, 4), _V(-4, 0, -4), _V(4, -3, -4));
	}
	else if (FragmentID == FID_FRAME)
	{
		// mesh
		strcpy(MeshName, "LaikaFrame");
		// consts
		dQTreshold = 120000;
		dVTotalDestructionTreshold = 150;

		// size and geometry
		SetEmptyMass (100.0);
		SetSize (2);
		SetPMI (_V(1.5, 1.5, 3));

		// aerodynamics
		SetCrossSections (_V(0.5, 0.5, 2.0));
		SetCW (1.2, 1.2, 0.7, 0.7);			// wind resistance
		SetRotDrag (_V(1.5,1.5,0.5));
		SetPitchMomentScale (0.001);
		SetBankMomentScale (0.001);

		// surface
		SetSurfaceFrictionCoeff (10.0, 10.0);
		SetCOG_elev (1);
		SetTouchdownPoints (_V(0, 0.5, 4), _V(-4, 0, -4), _V(4, -3, -4));
	}
	else if (FragmentID == FID_BO)
	{
		// mesh
		strcpy(MeshName, "LaikaBO");
		// consts
		dQTreshold = 170000;
		dVTotalDestructionTreshold = 150;

		// size and geometry
		SetEmptyMass (50.0);
		SetSize (1);
		SetPMI (_V(2, 2, 2));

		// aerodynamics
		SetCrossSections (_V(1.0, 1.0, 1.0));
		SetCW (0.8, 1, 1.2, 1.2);			// wind resistance
		SetRotDrag (_V(1.5, 1.5, 1));
		SetPitchMomentScale (0.001);
		SetBankMomentScale (0.001);

		// surface
		SetSurfaceFrictionCoeff (10.0, 10.0);
		SetCOG_elev (1);
		SetTouchdownPoints (_V(0, 0.5, 4), _V(-4, 0, -4), _V(4, -3, -4));
	}
	else if (FragmentID == FID_BG)
	{
		// mesh
		strcpy(MeshName, "LaikaBG");
		// consts
		dQTreshold = 190000;
		dVTotalDestructionTreshold = 150;

		// size and geometry
		SetEmptyMass (25.0);
		SetSize (0.5);
		SetPMI (_V(2, 2, 2));

		// aerodynamics
		SetCrossSections (_V(0.25, 0.25, 0.25));
		SetCW (1, 1, 1, 1);			// wind resistance
		SetRotDrag (_V(1.5,1.5,1.5));
		SetPitchMomentScale (0.001);
		SetBankMomentScale (0.001);

		// surface
		SetSurfaceFrictionCoeff (10.0, 10.0);
		SetCOG_elev (1);
		SetTouchdownPoints (_V(0, 0.5, 4), _V(-4, 0, -4), _V(4, -3, -4));
	}
	else if (FragmentID == FID_PO)
	{
		// mesh
		strcpy(MeshName, "LaikaPO");
		// consts
		dQTreshold = 140000;
		dVTotalDestructionTreshold = 150;

		// size and geometry
		SetEmptyMass (250.0);
		SetSize (0.5);
		SetPMI (_V(2, 2, 2));

		// aerodynamics
		SetCrossSections (_V(0.2, 0.2, 0.2));
		SetCW (1.1, 1.1, 1.1, 1.1);			// wind resistance
		SetRotDrag (_V(1.8,1.8,1.8));
		SetPitchMomentScale (0.001);
		SetBankMomentScale (0.001);

		// surface
		SetSurfaceFrictionCoeff (10.0, 10.0);
		SetCOG_elev (1);
		SetTouchdownPoints (_V(0, 4, -0.5), _V(-4, -4, -0.5), _V(4, -4, -0.5));
	}
	else if (FragmentID == FID_VNA)
	{
		// mesh
		strcpy(MeshName, "LaikaVNA");
		// consts
		dQTreshold = 80000;
		dVTotalDestructionTreshold = 200;

		// size and geometry
		SetEmptyMass (1.0);
		SetSize (0.3);
		SetPMI (_V(1, 1, 1));

		// aerodynamics
		SetCrossSections (_V(0.02, 0.02, 0.02));
		SetCW (2.1, 2.1, 2.1, 2.1);			// wind resistance
		SetRotDrag (_V(1.3,1.3,1.3));
		SetPitchMomentScale (0.001);
		SetBankMomentScale (0.001);

		// surface
		SetSurfaceFrictionCoeff (10.0, 10.0);
		SetCOG_elev (0.1);
		SetTouchdownPoints (_V(0, 4, -0.5), _V(-4, -4, -0.5), _V(4, -4, -0.5));
	}
	else if (FragmentID == FID_RADAR)
	{
		// mesh
		strcpy(MeshName, "LaikaRadar");
		// consts
		dQTreshold = 70000;
		dVTotalDestructionTreshold = 200;

		// size and geometry
		SetEmptyMass (1.0);
		SetSize (0.2);
		SetPMI (_V(1, 1, 1));

		// aerodynamics
		SetCrossSections (_V(0.025, 0.025, 0.025));
		SetCW (2, 2, 2, 2);			// wind resistance
		SetRotDrag (_V(1, 1, 1));
		SetPitchMomentScale (0.001);
		SetBankMomentScale (0.001);

		// surface
		SetSurfaceFrictionCoeff (10.0, 10.0);
		SetCOG_elev (0.1);
		SetTouchdownPoints (_V(0, 4, -0.5), _V(-4, -4, -0.5), _V(4, -4, -0.5));
	}


	VECTOR3 MeshOffset = _V(0, 0, 0);
	ClearMeshes();
	AddMesh (MeshName, &MeshOffset);

	Proximator->UpdateFromVessel();

//%%% needs more rescearch, so implementation is still to be finalized
//	DefineAnimations();
}


void LaikaFragment::DefineAnimations()
{
	if (FragmentID == FID_LANDER)
	{
		/* mesh indexes, here for reference (GMax converter does not visualize them in the mesh file)
		Bag1 0
		Pallet1 1
		Pallet2 2
		Pallet3 3
		Pallet4 4
		Bag2 5
		Bag3 6
		Bag4 7
		PN1 8
		PN2 9
		*/

		UINT LanderAnimations[4];
		MGROUP_ROTATE *Pallets[3];
		MGROUP_ROTATE *PNs[2];

		static UINT grPallet2[2] = {2, 5}; 
		static UINT grPallet3[2] = {3, 6}; 
		static UINT grPallet4[2] = {4, 7}; 
		static UINT grPN1[1] = {8}; 
		static UINT grPN2[1] = {9}; 

		Pallets[0] = new MGROUP_ROTATE(0, grPallet2, 2, _V(rand11(), rand11(), rand11()), _V(rand11(), rand11(), rand11()), (float)rand01());
		Pallets[1] = new MGROUP_ROTATE(0, grPallet3, 2, _V(rand11(), rand11(), rand11()), _V( rand11(), rand11(), rand11()), (float)rand01());
		Pallets[2] = new MGROUP_ROTATE(0, grPallet4, 2, _V(rand11(), rand11(), rand11()), _V(rand11(), rand11(), rand11()), (float)rand01());
		PNs[0] = new MGROUP_ROTATE(0, grPN1, 1, _V(rand11(), rand11(), rand11()), _V(rand11(), rand11(), rand11()), (float)rand01());
		PNs[1] = new MGROUP_ROTATE(0, grPN2, 1, _V(rand11(), rand11(), rand11()), _V(rand11(), rand11(), rand11()), (float)rand01());

		for (int i = 0; i < 3; i++)
		{
			LanderAnimations[i] = CreateAnimation (0);
			AddAnimationComponent (LanderAnimations[i], 0, 1, Pallets[i]);
		}
		LanderAnimations[3] = CreateAnimation (0);
		AddAnimationComponent (LanderAnimations[3], 0, 1, PNs[0]);
		LanderAnimations[4] = CreateAnimation (0);
		AddAnimationComponent (LanderAnimations[4], 0, 1, PNs[1]);

		// animate!
		for (i = 0; i < 5; i++)
			SetAnimation (LanderAnimations[i], 1);
	}
}


void LaikaFragment::Timestep (double simt)
{
	Proximator->Timestep (simt);

	if (bJustDeployed)
		JustDeployed();
}


void LaikaFragment::JustDeployed()
{
	SetExtraConfiguration ();
	bJustDeployed = false;
}


void LaikaFragment::Burn(OBJHANDLE HTarget)
{
	if (GetHandle() != HTarget)
	{
		try
		{
			// switch focus wherever and destroy the current vessel
			oapiSetFocusObject(HTarget);
			oapiDeleteVessel (GetHandle(), HTarget);
		}
		catch(...)
		{
		}
	}
}


void LaikaFragment::Collide(OBJHANDLE HHit, double Vhit, OBJHANDLE HSwitch)
{
	if (Vhit > dVTotalDestructionTreshold)
		Burn(HSwitch);
	else
		;
}


void LaikaFragment::DoneLanding(OBJHANDLE HTarget, double V, double Vv, double Vh)
{
	if (V > dVTotalDestructionTreshold)
		Burn(HTarget);
	else
		;
}


void LaikaFragment::DebugSomething()
{
}



// ==============================================================
// API interface
// ==============================================================

// Initialisation
DLLCLBK VESSEL *ovcInit (OBJHANDLE hvessel, int flightmodel)
{
	// seed random generator
	srand((unsigned)time( NULL ));

	return new LaikaFragment (hvessel, flightmodel);
}

// Cleanup
DLLCLBK void ovcExit (VESSEL *vessel)
{
	if (vessel) 
		delete (LaikaFragment*)vessel;
}

// Set the capabilities of the vessel class
DLLCLBK void ovcSetClassCaps (VESSEL *vessel, FILEHANDLE cfg)
{
	((LaikaFragment*)vessel)->SetClassCaps (cfg);
}

// Read status from scenario file
DLLCLBK void ovcLoadStateEx (VESSEL *vessel, FILEHANDLE scn, void *vs)
{
	((LaikaFragment*)vessel)->LoadState (scn, vs);
}

// Save status to scenario file
DLLCLBK void ovcSaveState (VESSEL *vessel, FILEHANDLE scn)
{
	((LaikaFragment*)vessel)->SaveState (scn);
}

DLLCLBK void ovcTimestep (VESSEL *vessel, double simt)
{
	((LaikaFragment*)vessel)->Timestep (simt);
}

// Keyboard interface handler
DLLCLBK int ovcConsumeBufferedKey (VESSEL *vessel, DWORD key, bool down, char *kstate)
{
	if (!down) return 0; // only process keydown events

	if (KEYMOD_SHIFT (kstate)) 
	{
		return 0; // shift combinations are reserved
	}

	LaikaFragment *fragment = (LaikaFragment*)vessel;

	if ( !KEYMOD_CONTROL (kstate) && (key == OAPI_KEY_S) )
	{
		if (fragment->Proximator->bStruct)
		{
  			fragment->Proximator->Tral("Structural limits OFF\n");
			sprintf(oapiDebugString(), "Structural limits OFF");
		}
		else
		{
  			fragment->Proximator->Tral("Structural limits ON\n");
			sprintf(oapiDebugString(), "Structural limits ON");
		}

		fragment->Proximator->bStruct = !fragment->Proximator->bStruct;
		return 0;
	}

	if (KEYMOD_CONTROL (kstate) && (key == OAPI_KEY_P))
	{
		fragment->Proximator->Tral("Unpaused\n");
		return 0;
	}

	if (key == OAPI_KEY_T)
	{
		fragment->Proximator->Tral("%.0fx +Warp\n", oapiGetTimeAcceleration());
		return 0;
	}

	if (key == OAPI_KEY_R)
	{
		fragment->Proximator->Tral("%.0fx -Warp\n", oapiGetTimeAcceleration());
		return 0;
	}

	if ( !KEYMOD_CONTROL (kstate) && (key == OAPI_KEY_D) )
	{
		fragment->DebugSomething();
		return 0;
	}

	return 0;
}

