// ==============================================================
// ORBITER vessel module: Laika lunar probe carrier
//
// See Laika.h for description.
//
// Developed by: Andrew Thielmann (igel)
//
// ==============================================================

#include "Laika.h"
#include "LaikaFragment.h"
#include "LaikaLander.h"
#include "stdio.h"
#include "math.h"
#include "time.h"


// ********************** API 


// Initialisation
Laika::Laika (OBJHANDLE hObj, int fmodel): PRVESSEL (hObj, fmodel)
{
		// default initialization
	// common for the project
	dQTreshold = 40000;
	dVCollTreshold = 20;
	dVLandTreshold = 20;
	dVCrashTreshold = 150;
	bJustDeployed = true;

	// Laika-specific, may be overritten from the creator or scenario file
	bHasProbe = true;
	bHasPO1 = true;
	bHasPO2 = true;
	ConfigAnimation = -1;

	bMccApOn = false;
	MccApState = MCCAP_WAIT;
	bMccApPitchStep2 = false;
	bApActive = false;
	apState = AP_WAIT;

	// autopilot stage altitudes
	dHApHard = 0;    // will be calculated depending on remaining fuel and velocity
	dHApHigh = 5000;
	dHApLow = 1000;
	dHApTune = 100;
	dHApSafe = 20;

	// deployable items masses
	dMLander = 300;
	dMPO1 = 250;

	// create a very own extention class member
	Proximator = new TProximator(this);
}


Laika::~Laika ()
{
	if (Proximator) 
		delete Proximator;
}


// General initialization of general params, called before config file
void Laika::SetClassCaps (FILEHANDLE cfg)
{
	// size and geometry
	SetSize (3);
	SetPMI (_V(2, 2.25, 0.65));

	// aerodynamics
	SetCrossSections (_V(11,9,3));
	SetCW (0.3, 0.6, 1, 1);			// wind resistance
	SetRotDrag (_V(0.5, 0.5, 1));      // rotation resistance in atmosphere
	SetPitchMomentScale (0.001);
	SetBankMomentScale (0.0001);
	
	// surface
	SetCOG_elev (1);
	SetTouchdownPoints (_V(0, 0.5, 4), _V(-4, 0, -4), _V(4, -3, -4));
	SetSurfaceFrictionCoeff (10, 10);

	// main engine and fuel
	fuel = CreatePropellantResource (750); 
	SetDefaultPropellantResource (fuel); 
	ClearThrusterDefinitions();
	KTDU[0] = CreateThruster (_V(0, 0, -0.3), _V(0,0,1), 35000, fuel, ISP);
	thg_KTDU = CreateThrusterGroup (KTDU, 1, THGROUP_MAIN);
	AddExhaust (KTDU[0], 3.0, 0.6, _V( 0, 0, -1.5), _V(0, 0, -1));

	// attitude control
		// pitch
	th_rot[0] = CreateThruster (_V(0,  1.0, 0), _V(0, 0, -1), ATT_THRUST, fuel, ISP);
	th_rot[1] = CreateThruster (_V(0, -1.0, 0), _V(0, 0,  1), ATT_THRUST, fuel, ISP);
	AddExhaust (th_rot[0], 0.5, 0.1, _V( 0,  1.0,  0.35), _V(0, 0,  1));
	AddExhaust (th_rot[0], 0.5, 0.1, _V( 0, -1.0, -0.15), _V(0, 0, -1));
	tgh_pitchup = CreateThrusterGroup (th_rot, 2, THGROUP_ATT_PITCHUP);

	th_rot[0] = CreateThruster (_V(0,  1.0, 0), _V(0, 0,  1), ATT_THRUST, fuel, ISP);
	th_rot[1] = CreateThruster (_V(0, -1.0, 0), _V(0, 0, -1), ATT_THRUST, fuel, ISP);
	AddExhaust (th_rot[0], 0.5, 0.1, _V( 0,  1.0, -0.15), _V(0, 0, -1));
	AddExhaust (th_rot[0], 0.5, 0.1, _V( 0, -1.0,  0.35), _V(0, 0,  1));
	thg_pitchdown = CreateThrusterGroup (th_rot, 2, THGROUP_ATT_PITCHDOWN);

		// yaw
	th_rot[0] = CreateThruster (_V( 1.0, 0, 0), _V(0, 0,  1), ATT_THRUST, fuel, ISP);
	th_rot[1] = CreateThruster (_V(-1.0, 0, 0), _V(0, 0, -1), ATT_THRUST, fuel, ISP);
	AddExhaust (th_rot[0], 0.5, 0.1, _V( 1.0, 0, -0.15), _V(0, 0, -1));
	AddExhaust (th_rot[0], 0.5, 0.1, _V(-1.0, 0,  0.35), _V(0, 0,  1));
	thg_yawleft	= CreateThrusterGroup (th_rot, 2, THGROUP_ATT_YAWLEFT);

	th_rot[0] = CreateThruster (_V( 1.0, 0, 0), _V(0, 0, -1), ATT_THRUST, fuel, ISP);
	th_rot[1] = CreateThruster (_V(-1.0, 0, 0), _V(0, 0,  1), ATT_THRUST, fuel, ISP);
	AddExhaust (th_rot[0], 0.5, 0.1, _V( 1.0, 0,  0.35), _V(0, 0,  1));
	AddExhaust (th_rot[0], 0.5, 0.1, _V(-1.0, 0, -0.15), _V(0, 0, -1));
	thg_yawright = CreateThrusterGroup (th_rot, 2, THGROUP_ATT_YAWRIGHT);

		// bank
	th_rot[0] = CreateThruster (_V(0,  1.0, 0), _V( 1, 0, 0), ATT_THRUST, fuel, ISP);
	th_rot[1] = CreateThruster (_V(0, -1.0, 0), _V(-1, 0, 0), ATT_THRUST, fuel, ISP);
	AddExhaust (th_rot[0], 0.5, 0.1, _V( -0.25,  1.0, 0.1), _V(-1, 0, 0));
	AddExhaust (th_rot[0], 0.5, 0.1, _V(  0.25, -1.0, 0.1), _V( 1, 0, 0));
	thg_bankright = CreateThrusterGroup (th_rot, 2, THGROUP_ATT_BANKRIGHT);

	th_rot[0] = CreateThruster (_V(0,  1.0, 0), _V(-1, 0, 0), ATT_THRUST, fuel, ISP);
	th_rot[1] = CreateThruster (_V(0, -1.0, 0), _V( 1, 0, 0), ATT_THRUST, fuel, ISP);
	AddExhaust (th_rot[0], 0.5, 0.1, _V(  0.25,  1.0, 0.1), _V( 1, 0, 0));
	AddExhaust (th_rot[0], 0.5, 0.1, _V( -0.25, -1.0, 0.1), _V(-1, 0, 0));
	thg_bankleft = CreateThrusterGroup (th_rot, 2, THGROUP_ATT_BANKLEFT);
}


void Laika::ReloadMeshes()
{
	VECTOR3 FrameMesh_offs = _V(0, 0, 0);
	VECTOR3 KTDUMesh_offs = _V(0, 0, -0.2);
	VECTOR3 BG1Mesh_offs = _V(0,  0.4, -0.33);
	VECTOR3 BG2Mesh_offs = _V(0, -0.4, -0.33);
	VECTOR3 PO1Mesh_offs = _V( 0.45, 0, -0.22);
	VECTOR3 PO2Mesh_offs = _V(-0.45, 0, -0.22);
	VECTOR3 BOMesh_offs = _V(0, 0, 0.52);
	VECTOR3 YawMesh_offs = _V(-1, 0, 0.1);
	VECTOR3 PitchMesh_offs = _V( 0, -1, 0.1);

	VECTOR3 VNAMesh_offs = _V( -0.53, 0.1, 0.1);
	VECTOR3 Radar1Mesh_offs = _V( 0.45, -0.3, -0.04);
	VECTOR3 Radar2Mesh_offs = _V(-0.45, -0.3, -0.04);

	VECTOR3 ProbeMesh_offs = _V(0, 0, 1.5);

	ClearMeshes();

	SetEmptyMass (dMLander + 75 + 75 + 50 + 50 + dMPO1 + dMPO1);

	AddMesh ("LaikaFrame", &FrameMesh_offs);
	AddMesh ("LaikaKTDU", &KTDUMesh_offs);
	AddMesh ("LaikaBG", &BG1Mesh_offs);
	AddMesh ("LaikaBG2", &BG2Mesh_offs);
	AddMesh ("LaikaBO", &BOMesh_offs);
	AddMesh ("LaikaYaw", &YawMesh_offs);
//	int mindYaw2 = AddMesh ("LaikaYaw2", &YawMesh_offs);
	AddMesh ("LaikaPitch", &PitchMesh_offs);
//	int mindPitch2 = AddMesh ("LaikaPitch2", &PitchMesh_offs);
	AddMesh ("LaikaVNA", &VNAMesh_offs);
//	int mindVNA2 = AddMesh ("LaikaVNA2", &VNAMesh_offs);
	AddMesh ("LaikaRadar", &Radar1Mesh_offs);
	AddMesh ("LaikaRadar2", &Radar2Mesh_offs);

	// %%% this is a replacement of the broken animated meshes
	VECTOR3 Yaw2rMesh_offs = _V(1, 0, 0.1);
	VECTOR3 Pitch2rMesh_offs = _V( 0, 1, 0.1);
	VECTOR3 VNA2rMesh_offs = _V( 0.53, 0.1, 0.1);

	AddMesh ("LaikaYaw2r", &Yaw2rMesh_offs);
	AddMesh ("LaikaPitch2r", &Pitch2rMesh_offs);
	AddMesh ("LaikaVNA2r", &VNA2rMesh_offs);


	// not all parts may be present at all stages of the flight
	if (bHasProbe)
		AddMesh ("LaikaLander", &ProbeMesh_offs);
	else
		SetEmptyMass(GetEmptyMass() - dMLander);

	if (bHasPO1)
		AddMesh ("LaikaPO", &PO1Mesh_offs);
	else
		SetEmptyMass(GetEmptyMass() - dMPO1);

	if (bHasPO2)
		AddMesh ("LaikaPO2", &PO2Mesh_offs);
	else
		SetEmptyMass(GetEmptyMass() - dMPO1);

	// animate symmetrical meshes to their proper locations
/*	 //%%% this whole block is broken in the new Orbiter
	if (ConfigAnimation != -1) 
		return;

	MGROUP_ROTATE *mgrYaw2, *mgrPitch2, *mgrVNA2;
	static UINT grYaw2[4] = {0, 1, 2, 3}; 
	static UINT grPitch2[6] = {0, 1, 2, 3, 4, 5}; 
	static UINT grVNA2[1] = {0}; 

	mgrYaw2 = new MGROUP_ROTATE(mindYaw2, grYaw2, 4, _V(1, 0, 0), _V(0, 0, 1), (float)PI);
	mgrPitch2 = new MGROUP_ROTATE(mindPitch2, grPitch2, 6, _V(0, 1, 0), _V(0, 0, 1), (float)PI);
	mgrVNA2 = new MGROUP_ROTATE(mindVNA2, grVNA2, 1, _V(0.53, -0.55, 0), _V(0, 0, 1), -(float)PI*85/180);

	ConfigAnimation = CreateAnimation (0);
	AddAnimationComponent (ConfigAnimation, 0, 1, mgrYaw2);
	AddAnimationComponent (ConfigAnimation, 0, 1, mgrPitch2);
	AddAnimationComponent (ConfigAnimation, 0, 1, mgrVNA2);
	SetAnimation (ConfigAnimation, 1);
*/	
}


// Read status from scenario file
void Laika::LoadState (FILEHANDLE scn, void *vs)
{
    char *line;

	while (oapiReadScenario_nextline (scn, line)) 
	{
		if (!strnicmp (line, "LANDER", 6)) 
		{
            sscanf (line+6, "%ld", &bHasProbe);
		} 
		else if (!strnicmp (line, "PO1", 3)) 
		{
            sscanf (line+3, "%ld", &bHasPO1);
		} 
		else if (!strnicmp (line, "PO2", 3)) 
		{
            sscanf (line+3, "%ld", &bHasPO2);
		} 
		// try proximator's parse
		else if (!Proximator->ParsedScenarioLine(line))
			// unrecognised option - pass to Orbiter's generic parser
			ParseScenarioLineEx (line, vs);
    }

	bJustDeployed = false;
	ReloadMeshes();
}


// Save status to scenario file
void Laika::SaveState (FILEHANDLE scn)
{
	// default vessel parameters
	SaveDefaultState (scn);

	// save proximator parameters
	Proximator->SaveState (scn);

	// custom parameters
	oapiWriteScenario_int (scn, "LANDER", bHasProbe);
	oapiWriteScenario_int (scn, "PO1", bHasPO1);
	oapiWriteScenario_int (scn, "PO2", bHasPO2);
}


void Laika::Timestep (double simt)
{
	Proximator->Timestep (simt);

	if (bJustDeployed)
		JustDeployed();

	// midcourse correction autopilot
	if ( bMccApOn )
		MccApUpdateState();

	// autopilot
	if ( (bApActive) && (apState != AP_SAFE) && (Proximator->bAuto) )
	{
		ApUpdateState();
		ApCheckParams();
	}
}


void Laika::JustDeployed ()
{
	bJustDeployed = false;
	ReloadMeshes();
	Proximator->Tral("AOS\n");
}


void Laika::MccApActivate()
{
	SetTimeAccelOff();
	
	bMccApOn = true;
	MccApState = MCCAP_ATT_PITCH;

	Proximator->Tral("Midcourse correction autopilot sequence activated!\n");

	if (oapiGetFocusObject() == GetHandle())
		sprintf(oapiDebugString(),"Midcourse correction autopilot sequence activated!");
}



void Laika::MccApDeactivate()
{
	
	bMccApOn = false;
	MccApState = MCCAP_SAFE;

	Proximator->Tral("Midcourse correction autopilot sequence deactivated!\n");

	if (oapiGetFocusObject() == GetHandle())
		sprintf(oapiDebugString(),"Midcourse correction autopilot sequence deactivated!");
}


void Laika::MccApUpdateState()
{
	if ( (MccApState == MCCAP_WAIT) || (MccApState == MCCAP_SAFE) )
		return;


	VECTOR3 avel;
	GetAngularVel(avel);
	avel = avel * DEG;


	if (MccApState == MCCAP_ATT_PITCH)
	{
		// wait for stop
		if ( GetNavmodeState(NAVMODE_KILLROT) )
			return;

		// actual aoa, dg
		double aoaD = GetAOA() * DEG;

		// programmed turn: "correct" the sensor
		if (bMccApPitchStep2)
			aoaD += 90;

		double AbsAoaD;
		if (aoaD > 0)
			AbsAoaD = aoaD;
		else
			AbsAoaD = -aoaD;

		// max allowed angular vel on aoa
		double OmAoaMaxDs = 20; // deg/s
		if (AbsAoaD < 100)
			OmAoaMaxDs = AbsAoaD / 5;

		double pitchThrust = 0.5 + OmAoaMaxDs/20; 

		// have to pitch up?
		if ( (aoaD < 0) || ((aoaD > 0) && (avel.x < -OmAoaMaxDs)) )
		{
			SetThrusterGroupLevel(tgh_pitchup, pitchThrust);
			SetThrusterGroupLevel(thg_pitchdown, 0);
		}

		else if ( (aoaD > 0) || ((aoaD < 0) && (avel.x > OmAoaMaxDs)) )
		{
			SetThrusterGroupLevel(thg_pitchdown, pitchThrust);
			SetThrusterGroupLevel(tgh_pitchup, 0);
		}

		// done orientation?
		if ((AbsAoaD < 0.1) && (avel.x < 0.1) && (avel.x > -0.1))   
		{
			if (bMccApPitchStep2)
			{
				MccApState = MCCAP_BURN;
				Proximator->Tral("Midcourse correction pitch maneuver 2 completed!\n");
			}
			else
			{
				MccApState = MCCAP_ATT_YAW;
				Proximator->Tral("Midcourse correction pitch maneuver 1 completed!\n");
			}

			SetThrusterGroupLevel(thg_pitchdown, 0);
			SetThrusterGroupLevel(tgh_pitchup, 0);
			ActivateNavmode(NAVMODE_KILLROT);
		}

		if (oapiGetFocusObject() == GetHandle())
		{
			if (bMccApPitchStep2)
				sprintf(oapiDebugString(),"Midcourse correction - Step 4 of 5: Pitch %f", aoaD);
			else
				sprintf(oapiDebugString(),"Midcourse correction - Step 1 of 5: Pitch %f", aoaD);
		}
	}


	if (MccApState == MCCAP_ATT_YAW)
	{
		// wait for stop
		if ( GetNavmodeState(NAVMODE_KILLROT) )
			return;

		// actual slip, dg
		double slipD = GetSlipAngle() * DEG;
		double AbsSlipD;
		if (slipD > 0)
			AbsSlipD = slipD;
		else
			AbsSlipD = -slipD;

		// max allowed angular vel on slip
		double OmSlipMaxDs = 20; // deg/s
		if (AbsSlipD < 100)
			OmSlipMaxDs = AbsSlipD / 5;

		double yawThrust = 0.5 + OmSlipMaxDs/20; 

		// have to slip left?
		if ( (slipD > 0) || ((slipD < 0) && (avel.y < -OmSlipMaxDs)) )
		{
			SetThrusterGroupLevel(thg_yawleft, yawThrust);
			SetThrusterGroupLevel(thg_yawright, 0);
		}

		else if ( (slipD < 0) || ((slipD > 0) && (avel.y > OmSlipMaxDs)) )
		{
			SetThrusterGroupLevel(thg_yawright, yawThrust);
			SetThrusterGroupLevel(thg_yawleft, 0);
		}

		// done orientation?
		if ((AbsSlipD < 0.1) && (avel.y < 0.1) && (avel.y > -0.1))
		{
			MccApState = MCCAP_ATT_BANK;
			Proximator->Tral("Midcourse correction yaw maneuver completed!\n");
			SetThrusterGroupLevel(thg_yawright, 0);
			SetThrusterGroupLevel(thg_yawleft, 0);
			ActivateNavmode(NAVMODE_KILLROT);

		}

		if (oapiGetFocusObject() == GetHandle())
			sprintf(oapiDebugString(),"Midcourse correction - Step 2 of 5: Yaw %f", slipD);
	}

	if (MccApState == MCCAP_ATT_BANK)
	{
		// wait for stop
		if ( GetNavmodeState(NAVMODE_KILLROT) )
			return;

		// actual bank, dg
		double bankD = GetBank() * DEG;
		double AbsBankD;
		if (bankD > 0)
			AbsBankD = bankD;
		else
			AbsBankD = -bankD;

		// max allowed angular vel on bank
		double OmBankMaxDs = 20; // deg/s
		if (AbsBankD < 100)
			OmBankMaxDs = AbsBankD / 5;

		double bankThrust = OmBankMaxDs/20; 

		// have to bank left?
		if ( (bankD < 0) || ((bankD > 0) && (avel.z > OmBankMaxDs)) )
		{
			SetThrusterGroupLevel(thg_bankleft, bankThrust);
			SetThrusterGroupLevel(thg_bankright, 0);
		}

		else if ( (bankD > 0) || ((bankD < 0) && (avel.z < -OmBankMaxDs)) )
		{
			SetThrusterGroupLevel(thg_bankright, bankThrust);
			SetThrusterGroupLevel(thg_bankleft, 0);
		}

		// done orientation?
		if ((AbsBankD < 0.1) && (avel.z < 0.1) && (avel.z > -0.1))   
		{
			MccApState = MCCAP_ATT_PITCH;
			bMccApPitchStep2 = true;
			Proximator->Tral("Midcourse correction bank maneuver completed!\n");
			SetThrusterGroupLevel(thg_bankright, 0);
			SetThrusterGroupLevel(thg_bankleft, 0);
			ActivateNavmode(NAVMODE_KILLROT);
		}

		if (oapiGetFocusObject() == GetHandle())
			sprintf(oapiDebugString(),"Midcourse correction - Step 3 of 5: Bank %f", bankD);
	}

	if (MccApState == MCCAP_BURN)
	{
		// wait for stop
		if ( GetNavmodeState(NAVMODE_KILLROT) )
			return;

		// full throttle! 
		SetThrusterGroupLevel(thg_KTDU, 1);
		
		// check fuel
		bool bStopByFuel = GetPropellantMass(fuel)/GetPropellantMaxMass(fuel) < 0.95;

		// check pericenter
		OBJHANDLE HPlanet = GetSurfaceRef();
		ELEMENTS ElToMoon;
		ORBITPARAM OPtoMoon;
		GetElements(HPlanet, ElToMoon, &OPtoMoon, 0, FRAME_EQU);
		bool bStopByPericenter = OPtoMoon.PeD < 10000;

		// check pericenter var
		bool bStopByPericenterVar = PastStepPericenter < OPtoMoon.PeD;
		PastStepPericenter = OPtoMoon.PeD;

		if (oapiGetFocusObject() == GetHandle())
			sprintf(oapiDebugString(),"Midcourse correction - Step 5 of 5: Burn to pericenter %f", OPtoMoon.PeD);

		if (bStopByFuel || bStopByPericenter || bStopByPericenterVar)
		{
			// done!
			if (bStopByFuel)
				Proximator->Tral("Midcourse correction burn maneuver completed: fuel limit reached!\n");
			else if (bStopByPericenter)
				Proximator->Tral("Midcourse correction burn maneuver completed: required pericenter reached!\n");
			else if (bStopByPericenterVar)
				Proximator->Tral("Midcourse correction burn maneuver completed: pericenter limit reached!\n");

			MccApState = MCCAP_SAFE;
			SetThrusterGroupLevel(thg_KTDU, 0);
			ActivateNavmode(NAVMODE_KILLROT);
			MccApDeactivate();
		}
	}
}



void Laika::ApActivate() 
{
	if ( !(Proximator->bAuto && bHasPO1 && bHasPO2) )
		return;


	VECTOR3 VHor;

	SetTimeAccelOff();
	bApActive = true;
	ActivateNavmode(NAVMODE_RETROGRADE);

	double P = GetThrusterMax(KTDU[0]);
	double dM = (P / ISP);
	double Mfuel = (GetMass() - GetEmptyMass());
		
	double dMemp = 0;
	if (bHasPO1)
		dMemp += dMPO1;
	if (bHasPO2)
		dMemp += dMPO1;

	double VHar = ISP * log((GetMass() - dMemp) / (GetEmptyMass() - dMemp));
	GetHorizonAirspeedVector(VHor);
	double VLocAbs = sqrt(VHor.x * VHor.x + VHor.y * VHor.y + VHor.z * VHor.z);

	double H0 = GetAltitude();
	double Vv0 = VHor.y;
	double kV = fabs(Vv0 / VLocAbs);
	double Vbr = min(VHar, VLocAbs);

	double M = GetMass() - dMemp + dM;
	double a = 0;
	double V = 0;
	double Vv = 0;
	double H = 0;

	for (int t = 0; t < 100; t++)
	{
		M -= dM;
		a = P / M;
		V += a;
		Vv = V * kV;
		H += Vv;

		// %%% useful autopilot debugging
//		Proximator->Tral("t: %d M: %f a: %f V: %f Vv: %f H: %f\n", t, M, a, V, Vv, H);
		
		if ( (V > Vbr) || (H > H0) || (M < (GetEmptyMass() - dMemp)) )
			break;
	}

	dHApHard = H * 1.35;
	Proximator->Tral("Autopilot landing sequence activated! V: %.0f Vv: %.0f Hbr: %.0f\n", VLocAbs, VHor.y, dHApHard);

	if (oapiGetFocusObject() == GetHandle())
		sprintf(oapiDebugString(),"Autopilot landing sequence activated!");

}


void Laika::ApUpdateState()
{
	double H = GetAltitude();

	switch (apState)
	{
		case AP_WAIT:
			if (H < dHApHard)
			{
				apState = AP_HARD;

				if (bHasPO1)
					DropPO();

				Proximator->Tral("Hard brake initiated. H: %.0f\n", H);
				// %%% useful autopilot debugging
				//Proximator->Tral("M: %f Memp: %f Mfue: %f\n", GetMass(), GetEmptyMass(), GetFuelMass());

				sprintf(oapiDebugString(),"");
			}
			else if (H > 200000)
			{
				bApActive = false;
				Proximator->Tral("Autopilot deactivated. H: %.0f\n", H);
			}
			break;

		case AP_HARD:
			if (H < dHApHigh)
			{
				apState = AP_HIGH;
				Proximator->Tral("High brake initiated. H: %.0f\n", H);
			}
			else if (H > dHApHard)
			{
				apState = AP_WAIT;
				Proximator->Tral("Wait period re-initiated. H: %.0f\n", H);
			}
			break;

		case AP_HIGH:
			if (H < dHApLow)
			{
				apState = AP_LOW;
				Proximator->Tral("Low brake initiated. H: %.0f\n", H);
			}
			else if (H > dHApHigh)
			{
				apState = AP_HARD;
				Proximator->Tral("Hard period re-initiated. H: %.0f\n", H);
			}
			break;

		case AP_LOW:
			if (H < dHApTune)
			{
				apState = AP_TUNE;
				Proximator->Tral("Tune brake initiated. H: %.0f\n", H);
			}
			else if (H > dHApLow)
			{
				apState = AP_HIGH;
				Proximator->Tral("High period re-initiated. H: %.0f\n", H);
			}
			break;

		case AP_TUNE:
			if (H < dHApSafe)
			{
				VECTOR3 VHor;
				double VLocAbs;

				GetHorizonAirspeedVector(VHor);
				VLocAbs = sqrt(VHor.x * VHor.x + VHor.y * VHor.y + VHor.z * VHor.z);

				if (bHasProbe)
					DropProbe();

				// safe the autopilot forever
				apState = AP_SAFE;
				bApActive = false;
				SetThrusterGroupLevel(thg_KTDU, 0);
				ActivateNavmode(NAVMODE_KILLROT);
				DeactivateNavmode(NAVMODE_KILLROT);
				Proximator->Tral("Autopilot safed. H: %.0f V: %.0f Vv: %.0f\n", H, VLocAbs, VHor.y);
			}
			else if (H > dHApTune)
			{
				apState = AP_LOW;
				Proximator->Tral("Low period re-initiated. H: %.0f\n", H);
			}
			break;

		default:
			;
	}
}


void Laika::ApCheckParams()
{
	if ((apState == AP_SAFE) || (apState == AP_WAIT))
		return;

	VECTOR3 VHor;
	double VLocAbs;
	double H;

	H = GetAltitude();
	GetHorizonAirspeedVector(VHor);
	VLocAbs = sqrt(VHor.x * VHor.x + VHor.y * VHor.y + VHor.z * VHor.z);

	if (VHor.y > 0)
	{
		SetThrusterGroupLevel(thg_KTDU, 0);    
		return;
	}

	if (VLocAbs > VMax[apState])
	{
		if ( (GetThrusterGroupLevel(thg_KTDU) == 0) && (GetFuelMass() > 0) )
		{
			SetTimeAccelOff();
			SetThrusterGroupLevel(thg_KTDU, 100);    
			Proximator->Tral("KTDU full throttle. H: %.0f V: %.0f Vv: %.0f\n", H, VLocAbs, VHor.y);
		}
	}
	else if (VLocAbs < VMin[apState])
	{
		if (GetThrusterGroupLevel(thg_KTDU) > 0)
		{
			SetTimeAccelOff();
			SetThrusterGroupLevel(thg_KTDU, 0);    
			Proximator->Tral("KTDU off. H: %.0f V: %.0f Vv: %.0f\n", H, VLocAbs, VHor.y);
		}
	}
}


void Laika::DropPO()
{
	VESSELSTATUS vs;

	VECTOR3 PO1Mesh_offs = _V( 0.45, 0, -0.22);
	VECTOR3 PO2Mesh_offs = _V(-0.45, 0, -0.22);
	VECTOR3 LocVel1 = _V(1, 0, -1); 
	VECTOR3 LocVel2 = _V(-1, 0, -1);
	VECTOR3 GlobVel;
	VECTOR3 LocRot1 = _V(0, -3, 0);  
	VECTOR3 LocRot2 = _V(0, 3, 0);  
	OBJHANDLE hPO;
	VESSEL *vPO;

	SetTimeAccelOff();
	Proximator->Tral("Dropping avionics...\n");

	// special handling of priximity checks with a just-dropped POs
	Proximator->dVesselProximityDelay = 5;

	// reset meshes, create newly dropped vessel
	bHasPO1 = false;
	bHasPO2 = false;
	ReloadMeshes();

	// update future PO position and movement
	GetStatus(vs);
	Local2Rel(PO1Mesh_offs, vs.rpos);
	GlobalRot (LocVel1, GlobVel);
	vs.rvel += GlobVel;
	vs.vrot = LocRot1;

	hPO = oapiCreateVessel ("Laika PO1 Fragment", "LaikaFragment", vs);
	vPO = oapiGetVesselInterface(hPO);
	((LaikaFragment*)vPO)->FragmentID = FID_PO;
	((LaikaFragment*)vPO)->Proximator->bStruct = Proximator->bStruct;

	// update future PO position and movement
	GetStatus(vs);
	Local2Rel(PO2Mesh_offs, vs.rpos);
	GlobalRot (LocVel2, GlobVel);
	vs.rvel += GlobVel;
	vs.vrot = LocRot2;

	hPO = oapiCreateVessel ("Laika PO2 Fragment", "LaikaFragment", vs);
	vPO = oapiGetVesselInterface(hPO);
	((LaikaFragment*)vPO)->FragmentID = FID_PO;
	((LaikaFragment*)vPO)->Proximator->bStruct = Proximator->bStruct;

	Proximator->Tral("Avionics dropped\n");
}


void Laika::DropProbe()
{
	VESSELSTATUS vs;
	VECTOR3 LocPos = _V(0, 0, 1);
	VECTOR3 LocVel = _V(1, 0, 1), GlobVel;
	//%%% LocRot = _V(-3, 0, 0); will be a lot more realistic, 
	// but then Orbiter does not properly land the tumbling vessel, so:
	VECTOR3 LocRot = _V(0, 0, 0);  
	OBJHANDLE hProbe;


	SetTimeAccelOff();
	Proximator->Tral("Separating lander...\n");

	// special handling of priximity checks with a just-deployed lander
	Proximator->dVesselProximityDelay = 5;

	// update future probe position and movement
	GetStatus(vs);
	Local2Rel(LocPos, vs.rpos);
	GlobalRot (LocVel, GlobVel);
	vs.rvel += GlobVel;
	vs.vrot = LocRot;

	// reset meshes, create new vessel-probe
	bHasProbe = false;
	ReloadMeshes();
	hProbe = oapiCreateVessel ("Lander", "LaikaLander", vs);

	//  call new probe to update its configuration
	VESSEL *NewLander;
	NewLander = oapiGetVesselInterface(hProbe);
	strcpy(((LaikaLander*)NewLander)->Proximator->TralFileName, Proximator->TralFileName);
	((LaikaLander*)NewLander)->Proximator->bAuto = Proximator->bAuto;
	((LaikaLander*)NewLander)->Proximator->bStruct = Proximator->bStruct;

	AddForce(_V(0, 15, -15), _V(0, 1, -0.1));
	ShiftCentreOfMass (_V(0, 0, -1));
	Proximator->Tral("Lander separated\n");
}



OBJHANDLE Laika::CreateFragment(char *MeshName, fragment_id FragmentID)
{
	void *hFragment;
	VESSELSTATUS vs;
	VESSEL *NewFragment;
	VECTOR3 LocPos, LocVel, GlobVel, LocRot;

	// randomize new vesselstatus
	GetStatus(vs);
	LocPos = _V(rand11(), rand11(), rand11());
	LocVel = _V(rand11(), rand11(), rand11()),
	LocRot = _V(rand11(), rand11(), rand11());
	Local2Rel(LocPos, vs.rpos);
	GlobalRot (LocVel, GlobVel);
	vs.rvel += GlobVel;
	vs.vrot = LocRot;

	hFragment = oapiCreateVessel (MeshName, "LaikaFragment", vs);
	NewFragment = oapiGetVesselInterface(hFragment);
	((LaikaFragment*)NewFragment)->FragmentID = FragmentID;
	((LaikaFragment*)NewFragment)->Proximator->bStruct = Proximator->bStruct;

	// %%% For useful debugging output, pass a Tral file name to a new fragment. 
	// %%% Normally, fragments are inert.
	// strcpy(((LaikaFragment*)NewFragment)->Proximator->TralFileName, Proximator->TralFileName);
	
	return hFragment;
}


void Laika::EnteringPlanet() 
{
	char PlanetName[20];
	OBJHANDLE HPlanet;

	// do we need a mid-course correction? 
	HPlanet = GetSurfaceRef();
	oapiGetObjectName(HPlanet, PlanetName, 20);

	ELEMENTS ElToMoon;
	ORBITPARAM OPtoMoon;
	GetElements(HPlanet, ElToMoon, &OPtoMoon, 0, FRAME_EQU);
	PastStepPericenter = 6000000000;

	if (!strcmp(PlanetName, "Moon") && (OPtoMoon.PeD > 10000) && (OPtoMoon.PeD < 5000000000))
		MccApActivate();
}


void Laika::ApproachingPlanet() 
{
	char PlanetName[20];
	OBJHANDLE HPlanet;

	// activate autopilot if approaching Moon
	HPlanet = GetSurfaceRef();
	oapiGetObjectName(HPlanet, PlanetName, 20);
	if (!strcmp(PlanetName, "Moon"))
		ApActivate();
}


// burn in the atmosphere
void Laika::Burn(OBJHANDLE HTarget)
{
	 SetTimeAccelOff();
	Proximator->Tral("LOS\n");

	void *hFragment;

	if (bHasProbe)
	  CreateFragment("Laika Lander Fragment", FID_LANDER);
	if (bHasPO1)
		CreateFragment("Laika PO1 Fragment", FID_PO);
	if (bHasPO2)
		CreateFragment("Laika PO2 Fragment", FID_PO);

	CreateFragment("Laika KTDU Fragment", FID_KTDU);
	hFragment = CreateFragment("Laika Frame Fragment", FID_FRAME);
	CreateFragment("Laika BO Fragment", FID_BO);
	CreateFragment("Laika BG1 Fragment", FID_BG);
	CreateFragment("Laika BG2 Fragment", FID_BG);
	CreateFragment("Laika VNA1 Fragment", FID_VNA);
	CreateFragment("Laika VNA2 Fragment", FID_VNA);
	CreateFragment("Laika Radar1 Fragment", FID_RADAR);
	CreateFragment("Laika Radar2 Fragment", FID_RADAR);

	oapiSetFocusObject(hFragment);
	oapiDeleteVessel (GetHandle(), hFragment);
}


void Laika::DoneLanding(OBJHANDLE HTarget, double V, double Vv, double Vh)
{
	SetTimeAccelOff();
	if (!Proximator->bStruct)
		return;

	if (V > dVCrashTreshold)
		DestructiveCrash(HTarget);
	else if (V > dVLandTreshold)
		Crash(HTarget);
	else
		;
}


void Laika::Collide(OBJHANDLE HHit, double Vhit, OBJHANDLE HSwitch) 
{
	SetTimeAccelOff();
	if (!Proximator->bStruct)
		return;

	if (Vhit > dVCrashTreshold)
		DestructiveCrash(HSwitch);
	else if (Vhit > dVCollTreshold)
		Burn(HSwitch);
	else
		;
}


void Laika::Crash(OBJHANDLE HTarget)
{
	Burn(HTarget);
}


void Laika::DestructiveCrash(OBJHANDLE HTarget)
{
	Proximator->Tral("LOS\n");

	oapiSetFocusObject(HTarget);
	oapiDeleteVessel (GetHandle(), HTarget);
}


void Laika::SetTimeAccelOff()
{
	if (oapiGetTimeAcceleration() == 1)
		return;

	oapiSetTimeAcceleration(1);
	Proximator->Tral("Warp auto-off on Active Maneuver\n");
}


void Laika::DebugSomething()
{
	sprintf(oapiDebugString(),"Q: %f", GetDynPressure());
}



// ==============================================================
// API interface
// ==============================================================


// Initialisation
DLLCLBK VESSEL *ovcInit (OBJHANDLE hvessel, int flightmodel)
{
	// seed random generator
	srand((unsigned)time( NULL ));

	// create main vessel
	return new Laika (hvessel, flightmodel);
}


// Cleanup
DLLCLBK void ovcExit (VESSEL *vessel)
{
	if (vessel) 
		delete (Laika*)vessel;
}


// Set the capabilities of the vessel class
DLLCLBK void ovcSetClassCaps (VESSEL *vessel, FILEHANDLE cfg)
{
	((Laika*)vessel)->SetClassCaps (cfg);
}


// Read status from scenario file
DLLCLBK void ovcLoadStateEx (VESSEL *vessel, FILEHANDLE scn, void *vs)
{
	((Laika*)vessel)->LoadState (scn, vs);
}


// Save status to scenario file
DLLCLBK void ovcSaveState (VESSEL *vessel, FILEHANDLE scn)
{
	((Laika*)vessel)->SaveState (scn);
}


DLLCLBK void ovcTimestep (VESSEL *vessel, double simt)
{
	((Laika*)vessel)->Timestep (simt);
}


// Keyboard interface handler
DLLCLBK int ovcConsumeBufferedKey (VESSEL *vessel, DWORD key, bool down, char *kstate)
{
	if (!down) 
		return 0; // only process keydown events

	if (KEYMOD_SHIFT (kstate)) 
	{
		return 0; // shift combinations are reserved
	}

	Laika *laika = (Laika*)vessel;

	if ( !KEYMOD_CONTROL (kstate) && (key == OAPI_KEY_O) )
	{
		if (laika->Proximator->bAuto)
		{
  			laika->Proximator->Tral("Autopilot OFF\n");
			sprintf(oapiDebugString(), "Autopilot OFF");
		}
		else
		{
  			laika->Proximator->Tral("Autopilot ON\n");
			sprintf(oapiDebugString(), "Autopilot ON");
		}

		laika->Proximator->bAuto = !laika->Proximator->bAuto;
		return 0;
	}

	if ( !KEYMOD_CONTROL (kstate) && (key == OAPI_KEY_S) )
	{
		if (laika->Proximator->bStruct)
		{
  			laika->Proximator->Tral("Structural limits OFF\n");
			sprintf(oapiDebugString(), "Structural limits OFF");
		}
		else
		{
  			laika->Proximator->Tral("Structural limits ON\n");
			sprintf(oapiDebugString(), "Structural limits ON");
		}

		laika->Proximator->bStruct = !laika->Proximator->bStruct;
		return 0;
	}

	if ( !KEYMOD_CONTROL (kstate) && (key == OAPI_KEY_D) )
	{
		laika->DebugSomething();
		return 0;
	}

	if (KEYMOD_CONTROL (kstate) && (key == OAPI_KEY_P))
	{
		laika->Proximator->Tral("Unpaused\n");
		return 0;
	}

	if (key == OAPI_KEY_T)
	{
		laika->Proximator->Tral("%.0fx +Warp\n", oapiGetTimeAcceleration());
		return 0;
	}

	if (key == OAPI_KEY_R)
	{
		laika->Proximator->Tral("%.0fx -Warp\n", oapiGetTimeAcceleration());
		return 0;
	}

	if ( !KEYMOD_CONTROL (kstate) && (key == OAPI_KEY_J) )
	{
		if (laika->bHasPO1)
			laika->DropPO();
		else if (laika->bHasProbe)
			laika->DropProbe();

		return 0;
	}

	return 0;
}

