#define ORBITER_MODULE

#include "r7BlockL.h"
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <math.h>



// default constructor
r7BlockL::r7BlockL (OBJHANDLE hObj, int fmodel): VESSEL2 (hObj, fmodel)
{
	;
}


// Class initialization
void r7BlockL::clbkSetClassCaps(FILEHANDLE cfg)
{
	// Set geometry
	SetSize (2.0);

	// Set physics 
	SetEmptyMass (EMPTY_MASS + BOZ_MASS);
	SetCrossSections (_V(7, 7, 5));
	SetLiftCoeffFunc (0);
	SetSurfaceFrictionCoeff(10,10);

	// %%% to recalc
	SetPMI (_V(1.0, 1.0, .5));
	SetPitchMomentScale (1e-6);
	SetBankMomentScale (1e-6);
	SetRotDrag (_V(0.5, 0.5, 0.5));
	SetCW (0.2, 0.2, 0.2, 0.2);

	// set propellant, for everything, including BOZ
	MainTank = CreatePropellantResource(FUEL_MASS);
	BOZEnginesFuel  = CreatePropellantResource(BOZ_FUEL_MASS);
	SetPropellantMass(BOZEnginesFuel, 0);
	BOZRCSTanks = CreatePropellantResource(BOZ_GAS_MASS);
	SetPropellantMass(BOZRCSTanks, 0);

	// set main engine
	MainEngine = CreateThruster(_V(0, 0, GIMBAL_Z), _V(0,0,1), THRUST_MAIN, MainTank, ISP_MAIN, ISP_MAIN);
	MainTH[0] = MainEngine;
	AddExhaust(MainTH[0], 5.0, 0.4, 1.0);

	// set roll engines
	RollEngine1 = CreateThruster(_V( GIMBAL_ROLL_X, 0, GIMBAL_ROLL_Z), _V(0,0,1), THRUST_ROLL, MainTank, ISP_MAIN, ISP_ROLL);
	RollEngine2 = CreateThruster(_V(-GIMBAL_ROLL_X, 0, GIMBAL_ROLL_Z), _V(0,0,1), THRUST_ROLL, MainTank, ISP_MAIN, ISP_ROLL);
	MainTH[1] = RollEngine1;
	MainTH[2] = RollEngine2;
	AddExhaust(MainTH[1], ROLL_EXH_LEN, ROLL_EXH_WID, 0.1);
	AddExhaust(MainTH[2], ROLL_EXH_LEN, ROLL_EXH_WID, 0.1);

	// join main engine and roll engines in main group for manual control
	MainThGr = CreateThrusterGroup(MainTH,3,THGROUP_USER);

	// load meshes
	VECTOR3 Off = _V(0, 0, 0);
	MeshID = AddMesh (oapiLoadMeshGlobal("r7BlockL"), &Off);
	SetMeshVisibilityMode (MeshID, MESHVIS_ALWAYS);
	
	// initialize vars to defaults

	PayloadName[0] = 0;
	BOZName[0] = 0;
	PayREF = _V(0, 0, 0);
	PayDIR = _V(0, 0, -1);
	PayROT = _V(1, 0, 0);

	bNeedsCoastingInit = true;
	bCoasting = false;
	bPrograded = false;
	bIgniting = false;
	bBurning = false;
	bDone = false;
	bSwitchFocus = true;
	bBOZFired = false;
	bMEPrelim = false;
	bMEInterm = false;
	bBOZSeparated = false;
	bMEMain = false;
	bKickedBOZ = false;
	bVerboseCountdown = true;
	bManualControl = false;
	bEasyMode = false;

	TimeToIgnition = 0;
	TimeToBOZSeparation = 0;
	BurnLength = 0;
	TimeToMECO = 0;
	TargetApoR = 0;
	TargetApoRLess = 0;

	BOZMass = 0;
	hBOZ = 0;
	PayloadMass = 0;

	// start with down-looking camera
	CameraDir = CAMERA_DOWN;

	// %%% for now, default is moon - should it be Molniya?
	Target = TGT_MOON;

	// %%%
	double WW = 1280;
	double WH = 1024;

	MessagePort = oapiCreateAnnotation ( true, 0.5, _V(1, 1, 1) );
	oapiAnnotationSetPos (MessagePort, 0, 130/WH, 1, 1);
	msgStartTime = 0;
	bMessageDisplayed = false;
}


// SetCameraDefaultDirection has no effect when called from clbkSetClassCaps - bug in Orbiter? 
// Enforce camera dir switch here:
bool r7BlockL::clbkLoadGenericCockpit ()
{
	oapiSetDefRCSDisplay(0);
	ToggleOnboardCamera(CameraDir);
	return true;
}


// saving configuration to scn file
void r7BlockL::clbkSaveState (FILEHANDLE scn)
{
	// default vessel parameters
	SaveDefaultState (scn);

	// store visual separator
	oapiWriteScenario_string (scn, "------", "");

	// BOZ 
	if (BOZName[0] != 0)
	{
		oapiWriteScenario_string(scn, "BOZ", BOZName);
	}

	// Payload
	if (PayloadName[0] != 0)
	{
		oapiWriteScenario_string(scn, "PAYLOAD", PayloadName);
		oapiWriteScenario_vec   (scn, "APOINT", PayREF);
		oapiWriteScenario_vec   (scn, "PDIR", PayDIR);
		oapiWriteScenario_vec   (scn, "PROT", PayROT);
	}

	// state flags - write always
	long unsigned int StateFlags = 0;

	StateFlags += (bNeedsCoastingInit << 0);
	StateFlags += (bCoasting << 1);
	StateFlags += (bPrograded << 2);
	StateFlags += (bIgniting << 3);
	StateFlags += (bBurning << 4);
	StateFlags += (bDone << 5);
	StateFlags += (bSwitchFocus << 6);
	StateFlags += (bBOZFired << 7);
	StateFlags += (bMEPrelim << 8);
	StateFlags += (bMEInterm << 9);
	StateFlags += (bBOZSeparated << 10);
	StateFlags += (bMEMain << 11);
	StateFlags += (bKickedBOZ << 12);
	StateFlags += (bVerboseCountdown << 13);
	StateFlags += (bManualControl << 14);
	StateFlags += (bEasyMode << 15);

	oapiWriteScenario_int (scn, "STATEFLAGS", StateFlags);

	// long-term nav params
	if ( TimeToIgnition+TimeToBOZSeparation+TimeToMECO+
			BurnLength+TargetApoR+TargetApoR != 0)
	{
		oapiWriteScenario_vec   (scn, "NAVTIMES", _V(TimeToIgnition, TimeToBOZSeparation, TimeToMECO));
		oapiWriteScenario_vec   (scn, "NAVRADS", _V(BurnLength, TargetApoR, TargetApoRLess));
	}

	// %%% change consdition if default changes
	if (CameraDir != CAMERA_DOWN)
		oapiWriteScenario_int   (scn, "CAMERA", CameraDir);

	// %%% change consdition if default changes
	if ( (bNeedsCoastingInit) && (Target != TGT_MOON) )
	{
		if (Target == TGT_MOLNIYA)
		oapiWriteScenario_string(scn, "TARGET", "MOLNIYA");
	}
}


// updating the configuration from scn file
void r7BlockL::clbkLoadStateEx (FILEHANDLE scn, void *status)
{
	char *line;
	char tempstr[255];
	long unsigned int StateFlags = 0;

	// read the scenario file
	while (oapiReadScenario_nextline (scn, line)) 
	{
		// Payload description and attachment, 
		// cloned from the rest of the r7 package

		if (!strnicmp (line, "BOZ", 3))
		{
			sscanf (line+3, "%s", &BOZName);
		} 

		if (!strnicmp (line, "PAYLOAD", 7))
		{
			sscanf (line+7, "%s", &PayloadName);
		} 

		else if (!strnicmp (line, "APOINT", 6))
		{
			float x, y, z;
			sscanf (line+6, "%f %f %f", &x, &y, &z);
			PayREF.x = (double)x;
			PayREF.y = (double)y;
			PayREF.z = (double)z;
		} 

		else if (!strnicmp (line, "PDIR", 4))
		{
			float x, y, z;
			sscanf (line+4, "%f %f %f", &x, &y, &z);
			PayDIR.x = (double)x;
			PayDIR.y = (double)y;
			PayDIR.z = (double)z;
		} 

		else if (!strnicmp (line, "PROT", 4))
		{
			float x, y, z;
			sscanf (line+4, "%f %f %f", &x, &y, &z);
			PayROT.x = (double)x;
			PayROT.y = (double)y;
			PayROT.z = (double)z;
		} 

		else if (!strnicmp (line, "STATEFLAGS", 10)) 
		{
            sscanf (line+10, "%ld", &StateFlags);
		}

		else if (!strnicmp (line, "NAVTIMES", 8))
		{
			sscanf (line+8, "%lf %lf %lf", &TimeToIgnition, &TimeToBOZSeparation, &TimeToMECO);
		}

		else if (!strnicmp (line, "NAVRADS", 7))
		{
			sscanf (line+7, "%lf %lf %lf", &BurnLength, &TargetApoR, &TargetApoRLess);
		}

		else if (!strnicmp (line, "CAMERA", 6))
		{
			sscanf (line+6, "%d", &CameraDir);
		}

		else if (!strnicmp (line, "TARGET", 6))
		{
			sscanf (line+6, "%s", &tempstr);

			// %%% add moon if default changes from it
			if ( !strcmp(tempstr, "MOLNIYA") )
				Target = TGT_MOLNIYA;
		}

		// --- skip visual separator
		else if (!strnicmp (line, "------", 6)) 
		{
            ; 
		}

		// --- unrecognised option - pass to Orbiter's generic parser
		else 
			ParseScenarioLineEx (line, status);
	}

	if (StateFlags > 0)
	{
		bNeedsCoastingInit = 1 & (StateFlags >> 0);
		bCoasting = 1 & (StateFlags >> 1);
		bPrograded = 1 & (StateFlags >> 2);
		bIgniting = 1 & (StateFlags >> 3);
		bBurning = 1 & (StateFlags >> 4);
		bDone = 1 & (StateFlags >> 5);
		bSwitchFocus = 1 & (StateFlags >> 6);
		bBOZFired = 1 & (StateFlags >> 7);
		bMEPrelim = 1 & (StateFlags >> 8);
		bMEInterm = 1 & (StateFlags >> 9);
		bBOZSeparated = 1 & (StateFlags >> 10);
		bMEMain = 1 & (StateFlags >> 11);
		bKickedBOZ = 1 & (StateFlags >> 12);
		bVerboseCountdown = 1 & (StateFlags >> 13);
		bManualControl = 1 & (StateFlags >> 14);
		bEasyMode = 1 & (StateFlags >> 15);

		// any additional actions based on flags?
	}

}


// called after all vessel in scn are created, but before first timestep
void r7BlockL::clbkPostCreation ()
{
	// finish vessel "assembly"
	AttachBOZ();
	PayloadAttach();
	SetEmptyMass(EMPTY_MASS + BOZMass + PayloadMass);
	// just for ground test, keep stands at the bottom of the frame
	SetTouchdownPoints (_V( 2, 2, 0), _V( 2, -2, 0), _V(-2, 0, 0));

	// for mesh design convenience, 
	ShiftCG(_V(0, 0, CG_Z));


	Pitch = 0;
	Yaw = 0;
	Roll = 0;

	if (bPrograded)
		ActivateNavmode(NAVMODE_PROGRADE);
}


void r7BlockL::GetManualControls()
{
	// attitude controls grab, cloned from block I
	double lpp = GetManualControlLevel(THGROUP_ATT_PITCHUP, MANCTRL_ATTMODE, MANCTRL_ANYDEVICE);
	double lpm = GetManualControlLevel(THGROUP_ATT_PITCHDOWN, MANCTRL_ATTMODE, MANCTRL_ANYDEVICE);
	double lp = lpp - lpm;

	double lyp = GetManualControlLevel(THGROUP_ATT_YAWLEFT, MANCTRL_ATTMODE, MANCTRL_ANYDEVICE);
	double lym = GetManualControlLevel(THGROUP_ATT_YAWRIGHT, MANCTRL_ATTMODE, MANCTRL_ANYDEVICE);
	double ly = lyp - lym;
	
	double lbp = GetManualControlLevel(THGROUP_ATT_BANKLEFT, MANCTRL_ATTMODE, MANCTRL_ANYDEVICE);
	double lbm = GetManualControlLevel(THGROUP_ATT_BANKRIGHT, MANCTRL_ATTMODE, MANCTRL_ANYDEVICE);
	double lb = lbp - lbm;

	if (bManualControl)
	{

		// update attitude controls only if BOZ is NOT present!
		// pitch
		if (lp > 0) 
			GimbalDirPitch = ATT_POS;
		else if (lp < 0) 
			GimbalDirPitch = ATT_NEG;
		else
			GimbalDirPitch = ATT_NEUT;

		// yaw
		if (ly < 0) 
			GimbalDirYaw = ATT_POS;
		else if (ly > 0) 
			GimbalDirYaw = ATT_NEG;
		else
			GimbalDirYaw = ATT_NEUT;

		// roll
		if	(lb < 0)
			AttRollMode = ATT_R_NEG;
		else if	(lb > 0)
			AttRollMode = ATT_R_POS;
		else
			AttRollMode = ATT_R_NEUT;


		// %%% temp: finally, force gimbal if not autopiloting
		if (!bBurning) 
			Gimbal(oapiGetSimStep());
	}

	// autopilot: simply warn with a message if any controls are attmepted
	else if (lp + ly + lb != 0)
		ShowAnnotationMessage("Manual controls disabled. To enable, use M key");
}


int r7BlockL::clbkConsumeDirectKey (char *keystate)
{
	// main engine start
	if (KEYDOWN (keystate, OAPI_KEY_ADD)) 
	{
		if (bManualControl)
		{
			// already igniting or burning? just ignore
			if ( (bIgniting) || (bBurning) )
				;

			// do ignition sequence only is BOZ is present
			else if (GetAttachmentStatus(hForBOZ))
			{
				TimeToBOZSeparation = -42;
				bIgniting = true;
				bCoasting = false;

				ShowAnnotationMessage("Main engine ingition sequence started!");
			}
			else
				ShowAnnotationMessage("Can't ignite main engine without BOZ (ullage stage)!");
		}
		else
			ShowAnnotationMessage("Manual controls disabled. To enable, use M key");
	}
	
	// main engine stop
	if (KEYDOWN (keystate, OAPI_KEY_SUBTRACT)) 
	{
		if (bManualControl)
			SetThrusterGroupLevel(MainThGr, 0);
		else
			ShowAnnotationMessage("Manual controls disabled. To enable, use M key");
	}

	//
	return 0;
}


int r7BlockL::clbkConsumeBufferedKey(DWORD key, bool down, char *kstate)
{

	// only process keydown events
	if (!down) 
		return 0; 

	// shift combinations are reserved
	if (KEYMOD_SHIFT (kstate)) 
		return 0; 

	// Control combinations!
	if (KEYMOD_CONTROL (kstate))
	{
		if (key == OAPI_KEY_J) 
		{
			bSwitchFocus = !bSwitchFocus;

			if (bSwitchFocus)
				ShowAnnotationMessage("Switch focus to payload after separation? YES!");
			else
				ShowAnnotationMessage("Switch focus to payload after separation? NO!");
		}

		return 0; 
	}

	// toggle between autopilot and manual control
	if (key == OAPI_KEY_M) 
	{
		ToggleManualAndAutoplilot();
	}

	// %%% to go?
	if (key == OAPI_KEY_V) 
	{
		bVerboseCountdown = !bVerboseCountdown;
	}

	if (key == OAPI_KEY_J) 
	{
		if (!bManualControl)
			ShowAnnotationMessage("Manual controls disabled. To enable, use M key");

		else if (bIgniting)
			ShowAnnotationMessage("Jetison operations blocked during ignition sequence!");

		else if (GetAttachmentStatus(hForBOZ))
			SeparateBOZ();

		else if (GetAttachmentStatus(hForPayload))
			PayloadSeparate();

		else
			ShowAnnotationMessage("Nothing more to jetison!");
	}

	// toggle internal view
	if ( !KEYMOD_CONTROL (kstate) && (key == OAPI_KEY_C) )
	{

		// %%% attached? this will go away when/if Block I camera is implemented
		bool bAttached = false;

		DWORD ParentAttCount = AttachmentCount(true);
		if (ParentAttCount > 0)
		{
			OBJHANDLE hR7Obj = GetAttachmentStatus(GetAttachmentHandle(true, 0));
			bAttached = ( hR7Obj != NULL  );
		}

		if (bAttached)
		{
		
			if (CameraDir == CAMERA_DOWN)
				ToggleOnboardCamera(CAMERA_BLOCKI);

			else if (CameraDir == CAMERA_BLOCKI)
				ToggleOnboardCamera(CAMERA_UP);

			else if (CameraDir == CAMERA_UP)
				ToggleOnboardCamera(CAMERA_DOWN);
		}
		else
		{
			if (CameraDir == CAMERA_DOWN)
				ToggleOnboardCamera(CAMERA_UP);

			else if (CameraDir == CAMERA_BLOCKI)
				ToggleOnboardCamera(CAMERA_DOWN);

			else if (CameraDir == CAMERA_UP)
				ToggleOnboardCamera(CAMERA_DOWN);
		}
	}

	// manual debug
	if ( !KEYMOD_CONTROL (kstate) && (key == OAPI_KEY_D) )
		DebugSomething();

	return 0;
}


void r7BlockL::ToggleManualAndAutoplilot()
{
	// do not allow toggling during pre-burn!
	if (bIgniting)
	{
		ShowAnnotationMessage("Manual/Autopilot controls can't be toggled during the pre-burn sequence!");
		return;
	}

	// do not allow autopilot if we screwed up the whole thing while in manual control...
	if (bManualControl)
	{
		// BOZ jetisoned before main engine ignotion? Too bad.
	    if (!GetAttachmentStatus(hForBOZ) && (GetThrusterGroupLevel(MainThGr) == 0))
		{
			ShowAnnotationMessage("Autopilot can't be engaged without BOZ (ullage stage)!");
			return;
		}
	}

	// toggle flag
	bManualControl = !bManualControl;

	// implement switching to the appropriate mode
	if (bManualControl)
		SetManualControl();
	else
		SetAutopilotControl();
}



void r7BlockL::SetManualControl()
{
	ShowAnnotationMessage("Manual controls enabled. To return to autopilot, use M key");
	sprintf(oapiDebugString(), "");

	// no need to check if there is BOZ
	MakeBOZEngines();
	MakeBOZRCSSmall();
	MakeBOZRCSLarge();

	// deactivate prograde
	DeactivateNavmode(NAVMODE_PROGRADE);
}


void r7BlockL::SetAutopilotControl()
{
	ShowAnnotationMessage("Manual controls disabled. To enable, use M key");

	// no need to check if there is BOZ
	RemoveBOZEngines();
	RemoveBOZRCSSmall();
	RemoveBOZRCSLarge();

	// disable any possible residual navmodes
	DeactivateNavmode(NAVMODE_KILLROT);
	DeactivateNavmode(NAVMODE_HLEVEL);
	DeactivateNavmode(NAVMODE_RETROGRADE);
	DeactivateNavmode(NAVMODE_NORMAL);
	DeactivateNavmode(NAVMODE_ANTINORMAL);
	DeactivateNavmode(NAVMODE_HOLDALT);

	// activate prograde
	ActivateNavmode(NAVMODE_PROGRADE);
}


void r7BlockL::ToggleOnboardCamera(CameraDirs Dir)
{
	CameraDir = Dir;

	if (Dir == CAMERA_UP)
	{
		SetCameraOffset(_V(1.1, 0, 2.8-CG_Z));
		SetCameraDefaultDirection(_V(0,0,1));
	}

	else if (Dir == CAMERA_DOWN)
	{
		SetCameraOffset(_V(0, 1.1, 0.9-CG_Z));
		SetCameraDefaultDirection(_V(0,0,-1));
	}

	// %%% placeholder until (if) uimplemented on Block I
	else if (Dir == CAMERA_BLOCKI)
	{
		SetCameraOffset(_V(0, -1.5, -5));
		SetCameraDefaultDirection(_V(0,0,-1));
	}
}

void r7BlockL::MakeBOZRCSSmall()
{
	double TH_ROLL = TH_BOZ_S_ROLL;
	double TH = TH_BOZ_S;

	double Z = BOZ_S_Z-CG_Z;
	double R = BOZ_S_R;

	double RCSExhLen = BOZ_S_EXH_LEN;
	double RCSExhWid = BOZ_S_EXH_WID;

	THRUSTER_HANDLE thRot[2];

	// ROLL

	thRot[0] = CreateThruster (_V( R,  0, Z), _V(0, 1, 0), TH_ROLL, BOZRCSTanks, ISP_BOZ_RCS);
	thRot[1] = CreateThruster (_V( 0, -R, Z), _V(1, 0, 0), TH_ROLL, BOZRCSTanks, ISP_BOZ_RCS);
	LeftRollS = CreateThrusterGroup (thRot, 2, THGROUP_ATT_BANKLEFT);

	AddExhaust (thRot[0], RCSExhLen, RCSExhWid);
	AddExhaust (thRot[1], RCSExhLen, RCSExhWid);

	thRot[0] = CreateThruster (_V( R,  0, Z), _V( 0, -1, 0), TH_ROLL, BOZRCSTanks, ISP_BOZ_RCS);
	thRot[1] = CreateThruster (_V( 0, -R, Z), _V(-1,  0, 0), TH_ROLL, BOZRCSTanks, ISP_BOZ_RCS);
	RightRollS = CreateThrusterGroup (thRot, 2, THGROUP_ATT_BANKRIGHT);

	AddExhaust (thRot[0], RCSExhLen, RCSExhWid);
	AddExhaust (thRot[1], RCSExhLen, RCSExhWid);


	// YAW

	thRot[0] = CreateThruster (_V( R, 0, Z), _V( 1, 0, 0), TH, BOZRCSTanks, ISP_BOZ_RCS);
	LeftYawS = CreateThrusterGroup (thRot, 1, THGROUP_ATT_YAWLEFT);

	AddExhaust (thRot[0], RCSExhLen, RCSExhWid);

	thRot[0] = CreateThruster (_V( R, 0, Z), _V(-1, 0, 0), TH, BOZRCSTanks, ISP_BOZ_RCS);
	RightYawS = CreateThrusterGroup (thRot, 1, THGROUP_ATT_YAWRIGHT);

	AddExhaust (thRot[0], RCSExhLen, RCSExhWid);


	// PITCH

	thRot[0] = CreateThruster (_V(0, -R, Z), _V(0, -1, 0), TH, BOZRCSTanks, ISP_BOZ_RCS);
	PitchUpS = CreateThrusterGroup (thRot, 1, THGROUP_ATT_PITCHUP);

	AddExhaust (thRot[0], RCSExhLen, RCSExhWid);

	thRot[0] = CreateThruster (_V(0, -R, Z), _V(0,  1, 0), TH, BOZRCSTanks, ISP_BOZ_RCS);
	PitchDownS = CreateThrusterGroup (thRot, 1, THGROUP_ATT_PITCHDOWN);

	AddExhaust (thRot[0], RCSExhLen, RCSExhWid);

}


void r7BlockL::MakeBOZRCSLarge()
{
	double TH_ROLL = TH_BOZ_L_ROLL;
	double TH = TH_BOZ_L;

	double Z = BOZ_L_Z-CG_Z;
	double R = BOZ_L_R;

	double RCSExhLen = BOZ_L_EXH_LEN;
	double RCSExhWid = BOZ_L_EXH_WID;
	double RCSExhSep = BOZ_L_EXH_SEP;

	THRUSTER_HANDLE thRot[2];

	// ROLL

	thRot[0] = CreateThruster (_V( R,  0, Z), _V(0, 1, 0), TH_ROLL, BOZRCSTanks, ISP_BOZ_RCS);
	thRot[1] = CreateThruster (_V( 0, -R, Z), _V(1, 0, 0), TH_ROLL, BOZRCSTanks, ISP_BOZ_RCS);
	LeftRollL = CreateThrusterGroup (thRot, 2, THGROUP_ATT_BANKLEFT);

	AddExhaust (thRot[0], RCSExhLen, RCSExhWid, RCSExhSep);
	AddExhaust (thRot[1], RCSExhLen, RCSExhWid, RCSExhSep);

	thRot[0] = CreateThruster (_V( R,  0, Z), _V( 0, -1, 0), TH_ROLL, BOZRCSTanks, ISP_BOZ_RCS);
	thRot[1] = CreateThruster (_V( 0, -R, Z), _V(-1,  0, 0), TH_ROLL, BOZRCSTanks, ISP_BOZ_RCS);
	RightRollL = CreateThrusterGroup (thRot, 2, THGROUP_ATT_BANKRIGHT);

	AddExhaust (thRot[0], RCSExhLen, RCSExhWid, RCSExhSep);
	AddExhaust (thRot[1], RCSExhLen, RCSExhWid, RCSExhSep);


	// YAW

	thRot[0] = CreateThruster (_V( R, 0, Z), _V( 1, 0, 0), TH, BOZRCSTanks, ISP_BOZ_RCS);
	LeftYawL = CreateThrusterGroup (thRot, 1, THGROUP_ATT_YAWLEFT);

	AddExhaust (thRot[0], RCSExhLen, RCSExhWid, RCSExhSep);

	thRot[0] = CreateThruster (_V( R, 0, Z), _V(-1, 0, 0), TH, BOZRCSTanks, ISP_BOZ_RCS);
	RightYawL = CreateThrusterGroup (thRot, 1, THGROUP_ATT_YAWRIGHT);

	AddExhaust (thRot[0], RCSExhLen, RCSExhWid, RCSExhSep);


	// PITCH

	thRot[0] = CreateThruster (_V(0, -R, Z), _V(0, -1, 0), TH, BOZRCSTanks, ISP_BOZ_RCS);
	PitchUpL = CreateThrusterGroup (thRot, 1, THGROUP_ATT_PITCHUP);

	AddExhaust (thRot[0], RCSExhLen, RCSExhWid, RCSExhSep);

	thRot[0] = CreateThruster (_V(0, -R, Z), _V(0,  1, 0), TH, BOZRCSTanks, ISP_BOZ_RCS);
	PitchDownL = CreateThrusterGroup (thRot, 1, THGROUP_ATT_PITCHDOWN);

	AddExhaust (thRot[0], RCSExhLen, RCSExhWid, RCSExhSep);

}


// make ullage engines, solid fuel, 2 
void r7BlockL::MakeBOZEngines()
{
	THRUSTER_HANDLE thBOZEngines[2];

	thBOZEngines[0] = CreateThruster (_V( BOZ_R,  BOZ_R, BOZ_Z-CG_Z), _V(-BOZ_DIR_XY, -BOZ_DIR_XY, BOZ_DIR_Z), THRUST_BOZ_ENGINE, BOZEnginesFuel, ISP_BOZ_ENGINE);
	thBOZEngines[1] = CreateThruster (_V(-BOZ_R, -BOZ_R, BOZ_Z-CG_Z), _V(BOZ_DIR_XY, BOZ_DIR_XY, BOZ_DIR_Z), THRUST_BOZ_ENGINE, BOZEnginesFuel, ISP_BOZ_ENGINE);
	BOZEngines = CreateThrusterGroup (thBOZEngines, 2, THGROUP_USER);

	AddExhaust (thBOZEngines[0], BOZ_EXH_LEN, BOZ_EXH_WID, BOZ_EXH_SEP);
	AddExhaust (thBOZEngines[1], BOZ_EXH_LEN, BOZ_EXH_WID, BOZ_EXH_SEP);
}


void r7BlockL::RemoveBOZRCSSmall()
{
	DelThrusterGroup(LeftRollS, THGROUP_ATT_BANKLEFT, true);
	DelThrusterGroup(RightRollS, THGROUP_ATT_BANKRIGHT, true); 
	DelThrusterGroup(LeftYawS, THGROUP_ATT_YAWLEFT, true);
	DelThrusterGroup(RightYawS, THGROUP_ATT_YAWRIGHT, true); 
	DelThrusterGroup(PitchUpS, THGROUP_ATT_PITCHUP, true);
	DelThrusterGroup(PitchDownS, THGROUP_ATT_PITCHDOWN, true); 
}


void r7BlockL::RemoveBOZRCSLarge()
{
	DelThrusterGroup(LeftRollL, THGROUP_ATT_BANKLEFT, true);
	DelThrusterGroup(RightRollL, THGROUP_ATT_BANKRIGHT, true); 
	DelThrusterGroup(LeftYawL, THGROUP_ATT_YAWLEFT, true);
	DelThrusterGroup(RightYawL, THGROUP_ATT_YAWRIGHT, true); 
	DelThrusterGroup(PitchUpL, THGROUP_ATT_PITCHUP, true);
	DelThrusterGroup(PitchDownL, THGROUP_ATT_PITCHDOWN, true); 
}


void r7BlockL::RemoveBOZEngines()
{
	DelThrusterGroup(BOZEngines, THGROUP_USER);
}



void r7BlockL::clbkPreStep(double simt, double simdt, double mjd)
{

}


void r7BlockL::clbkPostStep(double simt, double simdt, double mjd)
{

		// first, run checks and events that can happen both in automated and manual mode



	// message display timer - always give it a chance to finish
	if (bMessageDisplayed)
	{
		if ( (oapiGetSysTime() - msgStartTime) > MSG_DISPLAY_TIME )
		{
			oapiAnnotationSetText (MessagePort, "");
			bMessageDisplayed = false;
		}
	}



	// First thing first: are we fully done? Mission accomplished? 
	// If yes, ignore all of the following, regardless of automated or manual mode.
	// Free up CPU for more important things!
	if (bDone)
		return;



	// Are we still attached to parent rocket? If yes, no further actions or even checks 
	// until we separate... unless we later implement some "failure" modes that allow 
	// block L to do some "actions" when still attached.
	// Do it regardless of the control mode
	DWORD ParentAttCount = AttachmentCount(true);  // true means "to parent"
	if (ParentAttCount > 0)
	{		
		// still attached?
		OBJHANDLE hR7Obj = GetAttachmentStatus(GetAttachmentHandle(true, 0));
		if (hR7Obj != NULL)
			return;

		// No longer attached... but are we initialized for coasting?
		// For now (before easy mode) we need it only once; no need to re-initialize
		if (bNeedsCoastingInit)
			InitializeCoastingPhase();
	}



	// ignition sequence? Also can be in both modes.
	if (bIgniting)
		UpdateIgnitingPhase(simdt);

	// get manual controls, aslo in both modes - just behave differently
	GetManualControls();

	// turn nozzle, also in any mode or state, except when safed
	Gimbal(simdt);



	// coastng? 
	if (bCoasting)
	{
		// this gets updated regardless of the control mode...
		TimeToIgnition -= simdt;

		// ... while the rest only makes sense for autopilot
		if (!bManualControl)
		{
			// show countdown message
			if ( (bVerboseCountdown) && (oapiGetFocusObject() == GetHandle()) )
				sprintf(oapiDebugString(), "Burn countdown: T-%.0f s", TimeToIgnition);

			if (TimeToIgnition < 60 )  
			{
				// Turn off warp if burn is too close
				oapiSetTimeAcceleration (1);

				// toggle to prograde
				if (!bPrograded)
				{
					MakeBOZEngines();
					MakeBOZRCSSmall();
					MakeBOZRCSLarge();

					bPrograded = true;
					ActivateNavmode(NAVMODE_PROGRADE);
				}

				// set separation sequence timer
				TimeToBOZSeparation = -90;

				// start ignition sequence 
				bIgniting = true;
				bCoasting = false;
			}
		}
	}



	// burning?
	if (bBurning)
	{
		// watch time and actual orbit
		TimeToMECO -= simdt;
		double CurrApoR;
		GetApDist(CurrApoR);


		// autopilot
		if (!bManualControl)
		{
			// Turn off warp if MECO is too close
			if (TimeToMECO < 10) 
				oapiSetTimeAcceleration (1 );

			if ( (bVerboseCountdown) && (oapiGetFocusObject() == GetHandle()) )
				sprintf(oapiDebugString(), "Cutoff countdown: T-%.0f s", TimeToMECO);

			// close to MECO? throttle down?
			if ( (CurrApoR > TargetApoRLess) && (GetThrusterGroupLevel(MainThGr) > 0.1) )
				SetThrusterGroupLevel(MainThGr, 0.1);

			// done? MECO time!
			if (CurrApoR > TargetApoR)
				SetThrusterGroupLevel(MainThGr, 0);
		}


		// Turn off warp and throttle down if very low on fuel
		if (GetPropellantMass(MainTank)/FUEL_MASS < 0.01) 
			oapiSetTimeAcceleration (1);

		if (GetPropellantMass(MainTank)/FUEL_MASS < 0.002) 
			SetThrusterGroupLevel(MainThGr, 0.1);

		// engin cutoff? for whatever reason - run the end of mission sequence!
		if (GetThrusterGroupLevel(MainThGr) == 0) 
		{
			// stop all timestep processing from this moment!
			bDone = true;
			bBurning = false; // this one just in case, bDone should be enough...

			// engine shutdown
			SetThrusterGroupLevel(MainThGr, 0);

			// if we have payload, separate it
			PayloadSeparate();

			// clear oputput string
			sprintf(oapiDebugString(), "");

			// toggle ProGrade mode off
			bPrograded = false;
			DeactivateNavmode(NAVMODE_PROGRADE);

			// say bye...
			ShowAnnotationMessage("Mission compelte! Block L is safed.");
		}
	}
}



// can be called not only from autopilot!
void r7BlockL::UpdateIgnitingPhase(double simdt)
{
	// update current time
	TimeToBOZSeparation += simdt;

	// show countdown string
	if ( (bVerboseCountdown) && (oapiGetFocusObject() == GetHandle()) )
		sprintf(oapiDebugString(), "Burn countdown: T%.0f s", TimeToBOZSeparation);
			

	// T-41: ignite solid rocket engines, engage large RCS valves
	if ((!bBOZFired) && (TimeToBOZSeparation > -41))
	{
		SetThrusterGroupLevel(BOZEngines, 1);
		bBOZFired = true;
	}

	// T-6: ignite main engine and roll engines to 10%, initalize platform
	if ((!bMEPrelim) && (TimeToBOZSeparation > -6))
	{
		SetThrusterGroupLevel(MainThGr, 0.1);
		bMEPrelim = true;
	}

	// T-3: throttle up to 75%
	if ((!bMEInterm) && (TimeToBOZSeparation > -3))
	{
		SetThrusterGroupLevel(MainThGr, 0.75);
		bMEInterm = true;
	}

	// T0: separate BOZ
	if ((!bBOZSeparated) && (TimeToBOZSeparation > 0))
	{
		SeparateBOZ();
		bBOZSeparated = true;
	}

	// T+1: kick BOZ
	if ( (!bKickedBOZ) && (TimeToBOZSeparation > 1) )
	{

		if (hBOZ)
		{
			VESSEL	*BOZIntf;
			BOZIntf = oapiGetVesselInterface(hBOZ);
			BOZIntf->AddForce(_V(0, 0, -100000), _V(DRand(-50, 100), DRand(-50, 100), 0) );
		}

		bKickedBOZ = true;
	}

	// T+3: full throttle, end of ignition sequence
	if ((!bMEMain) && (TimeToBOZSeparation > 3))
	{
		SetThrusterGroupLevel(MainThGr, 1);
		bMEMain = true;

		bIgniting = false;
		bBurning = true;
		TimeToMECO = BurnLength-3;

		if (bManualControl)
			sprintf(oapiDebugString(), "");
	}
}


void r7BlockL::Gimbal(double simdt)
{
	// store current controls positions
	double Pitch0 = Pitch;
	double Yaw0 = Yaw;
	double Roll0 = Roll;

	// get angular velocities
	VECTOR3 omegas;
	GetAngularVel(omegas);
	omegas *= DEG;

	// start from newtral positions. 
	// If there will ne no inputs, corresponding cameras will be newtralized.
	Pitch = 0;
	Yaw = 0;
	Roll = 0;

	// autopilot ALWAYS uses prograde, so we check it first
	if (GetNavmodeState(NAVMODE_PROGRADE))
	{
		GetWantedNavmodeAttiude(NAVMODE_PROGRADE, &LocalAtt);
		Pitch = GetWantedActuatorPosition(EU_PITCH, LocalAtt.x, omegas.x);
		Yaw = GetWantedActuatorPosition(EU_YAW, LocalAtt.y, omegas.y);
		Roll = GetWantedActuatorPosition(EU_ROLL, LocalAtt.z, omegas.z);
	}

	// see if we use something different in manual control
	if (bManualControl)
	{
		// check for other supported navmodes

		if (GetNavmodeState(NAVMODE_KILLROT))
		{
			Pitch = GetWantedActuatorPosition1KillRot(EU_PITCH, omegas.x);
			Yaw = GetWantedActuatorPosition1KillRot(EU_YAW, omegas.y);
			Roll = GetWantedActuatorPosition1KillRot(EU_ROLL, omegas.z);
		}

				
		else if (GetNavmodeState(NAVMODE_RETROGRADE))
		{
			GetWantedNavmodeAttiude(NAVMODE_RETROGRADE, &LocalAtt);

			Pitch = GetWantedActuatorPosition(EU_PITCH, LocalAtt.x, omegas.x);
			Yaw = GetWantedActuatorPosition(EU_YAW, LocalAtt.y, omegas.y);
			Roll = GetWantedActuatorPosition(EU_ROLL, LocalAtt.z, omegas.z);
		}


		// check for pilot inputs - they override navmodes

		// Pitch
		if (GimbalDirPitch == ATT_POS)
			Pitch = -MAX_GIMBAL;
		else if (GimbalDirPitch == ATT_NEG)
			Pitch = MAX_GIMBAL;

		// Yaw
		if (GimbalDirYaw == ATT_POS)
			Yaw = -MAX_GIMBAL;
		else if (GimbalDirYaw == ATT_NEG)
			Yaw = MAX_GIMBAL;
		
		// Roll
		if (AttRollMode == ATT_R_POS)
			Roll = MAX_GIMBAL_ROLL;
		else if (AttRollMode == ATT_R_NEG)
			Roll = -MAX_GIMBAL_ROLL;
	}


	// Convert desired control positions to physically possible positions
	Pitch = GetPhysicalActuatorPosition(EU_PITCH, simdt, Pitch, Pitch0);
	Yaw = GetPhysicalActuatorPosition(EU_YAW, simdt, Yaw, Yaw0);
	Roll = GetPhysicalActuatorPosition(EU_ROLL, simdt, Roll, Roll0);

	
	// for main engine gimbal, calc thrust vector
	double sP = sin(Pitch * RAD);
    double sY = sin(Yaw * RAD);
    double cP = cos(Pitch * RAD);
    double cY = cos(Yaw * RAD);
      
    VECTOR3 VEngine = _V(sY * cP, sP * cY, cP * cY);

	// point main engine thrust
    SetThrusterDir(MainTH[0], VEngine);
	

	double RollRad = Roll * RAD;

    VECTOR3 VRollEngine1 = _V(0,  sin(RollRad), cos(RollRad));
    VECTOR3 VRollEngine2 = _V(0, -sin(RollRad), cos(RollRad));

	// point roll engine thrusts
    SetThrusterDir(MainTH[1], VRollEngine1);
    SetThrusterDir(MainTH[2], VRollEngine2);

	//animate mesh
	if (vis) 
	{
		// main engine - pitch
		const VECTOR3 CE_AXIS01 = {-1, 0, 0};
		MESHGROUP_TRANSFORM	mt01;
		mt01.transform = MESHGROUP_TRANSFORM::ROTATE;
		mt01.nmesh = MeshID;
		mt01.ngrp = 30;
		mt01.P.rotparam.angle = float((Pitch-Pitch0)*RAD);
		mt01.P.rotparam.ref = _V(0, 0, 1.349);
		mt01.P.rotparam.axis = CE_AXIS01;
		MeshgroupTransform(vis, mt01);

		// yaw
		const VECTOR3 CE_AXIS02 = {0, -1, 0};
		MESHGROUP_TRANSFORM	mt02;
		mt02.transform = MESHGROUP_TRANSFORM::ROTATE;
		mt02.nmesh = MeshID;
		mt02.P.rotparam.angle = (float)((Yaw-Yaw0)*RAD);
		mt02.P.rotparam.ref = _V(0, 0, 1.349);
		mt02.P.rotparam.axis = CE_AXIS02;

		// main engine
		mt02.ngrp = 30;
		MeshgroupTransform(vis, mt02);

		// gimbal frame 
		mt02.ngrp = 27;
		MeshgroupTransform(vis, mt02);

		// gimbal axix - yaw
		mt02.ngrp = 25;
		MeshgroupTransform(vis, mt02);

		// roll engine 1
		const VECTOR3 CE_AXIS03 = {-1, 0, 0};
		MESHGROUP_TRANSFORM	mt03;
		mt03.transform = MESHGROUP_TRANSFORM::ROTATE;
		mt03.nmesh = MeshID;
		mt03.ngrp = 23;
		mt03.P.rotparam.angle = (float)((Roll-Roll0)*RAD);
		mt03.P.rotparam.ref = _V(0.551, 0, 0.811);
		mt03.P.rotparam.axis = CE_AXIS03;
		MeshgroupTransform(vis, mt03);

		// roll engine 2
		const VECTOR3 CE_AXIS04 = {1, 0, 0};
		MESHGROUP_TRANSFORM	mt04;
		mt04.transform = MESHGROUP_TRANSFORM::ROTATE;
		mt04.nmesh = MeshID;
		mt04.ngrp = 21;
		mt04.P.rotparam.angle = (float)((Roll-Roll0)*RAD);
		mt04.P.rotparam.ref = _V(-0.551, 0, 0.811);
		mt04.P.rotparam.axis = CE_AXIS04;
		MeshgroupTransform(vis, mt04);
	}
}


void r7BlockL::GetWantedNavmodeAttiude(int NavMode, VECTOR3 *Angles)
{
	VECTOR3 rVel, lVel;
	OBJHANDLE hEarth = oapiGetGbodyByName("earth");
	GetRelativeVel(hEarth, rVel);
	LocalRot(rVel, lVel);
	VECTOR3 vAngs = lVel / length(lVel);

	double sign;
	double bank;

	if (NavMode == NAVMODE_PROGRADE)
	{
		sign = -1;

		bank = GetBank() * DEG - 90;
		if (bank < -180)
			bank += 360;
	}
	else if (NavMode == NAVMODE_RETROGRADE)
	{
		sign = 1;

		bank = GetBank() * DEG + 90;
		if (bank > 180)
			bank -= 360;
	}

	Angles->x = asin(vAngs.y) * sign * DEG;
	Angles->y = asin(vAngs.x) * sign * DEG;
	Angles->z = bank;
}


double r7BlockL::GetWantedActuatorPosition1KillRot(EuAngle Channel, double omega)
{
	// sensitivity zones for pitc/yaw and roll, in degrees/s
	const double EPS_OM = 5;
	const double EPS_OM_ROLL = 0.5;

	//
	double angle = 0;
	double fabsOmega = fabs(omega);
	double eps;
	double maxAngle;
	double sign;

	// depending on channel, set up conditional variables

	if (Channel == EU_PITCH)
	{
		eps = EPS_OM;
		maxAngle = MAX_GIMBAL;
		sign = omega / fabsOmega;
	}

	else if (Channel == EU_YAW)
	{
		eps = EPS_OM;
		maxAngle = MAX_GIMBAL;
		sign = - omega / fabsOmega;
	}

	else if (Channel == EU_ROLL)
	{
		eps = EPS_OM_ROLL;
		maxAngle = MAX_GIMBAL_ROLL;
		sign = omega / fabsOmega;
	}

	// if omega is already slow, reduce the inputs as well!
	if (fabsOmega > eps) 
		angle = maxAngle * sign;
	else
		angle = maxAngle * sign / 10;

	// return resulting angle (to be yet processed for physical limits)
	return angle;
}


// for desired attitude angle, pitch or yaw
double r7BlockL::GetWantedActuatorPosition(EuAngle Channel, double angle, double omega)
{
	// sensitivity zones for pitc/yaw and roll, in degrees
	const double EPS_ANG = 5;
	const double EPS_ANG_ROLL = 15;

	// some values need to be calculated early...
	double fabsAngle = fabs(angle);
	double fabsOmega = fabs(omega);
	double sign = angle / fabsAngle;

	// others will be defined later
	double angleNew;
	bool bReachingZero;
	bool bSameDirection;
	double omegaMax;
	double maxGimbal;

	// are we arelady moving in wanted direction?
	// This is the only difference between pitch and yaw (and roll goes with pitch)
	if (Channel == EU_PITCH)
  		bSameDirection = angle * omega < 0;
	else 
  		bSameDirection = angle * omega > 0;

	// account for differences between roll and pitch-yaw
	if (Channel == EU_ROLL)
	{
		bReachingZero = fabs(angle) < EPS_ANG_ROLL;
		maxGimbal = MAX_GIMBAL_ROLL;
		omegaMax = OMEGA_ROLL_MAX;

		sign = - sign;
	}
	else
	{
		bReachingZero = fabs(angle) < EPS_ANG;
		maxGimbal = MAX_GIMBAL;

		omegaMax = OMEGA_MAX / 2;
		if (bReachingZero)
			omegaMax /= 4;
	}

	// conditions
	bool bOmegaNominal = fabsOmega < omegaMax;
	bool bFlipBell = (bSameDirection) && (bReachingZero);

	// if working close to zero, shift smaller...
	double DeviationMultiplier = fabs(angle) / MAX_ATT_SENSOR + 0.1;
	if (DeviationMultiplier > 1)
		DeviationMultiplier = 1;

	// when just flipped bell, don't rush back!
	double SignMultiplier;
	if (!bFlipBell)
		SignMultiplier = 1;
	else
		SignMultiplier = 0.1;

	// if we will be wishing to move the bell, it is will be by this amount
	angleNew =  maxGimbal * DeviationMultiplier * SignMultiplier;


	// now that we have all pre-requisites, decisions look simple

	// speed up?
	if ((!bSameDirection) || (bOmegaNominal))
		angleNew *= sign;

	// slow down?
	else if (bReachingZero)
		angleNew *= -sign;
		
	// just coast?
	else
		angleNew = 0;

	return angleNew;
}



double r7BlockL::GetPhysicalActuatorPosition(EuAngle Channel, double simdt, double angle, double angle0)
{
	double NewAngle;
	double GimbalVel;
	double maxGimbal;

	//
	if (Channel == EU_ROLL)
	{
		GimbalVel = GIMBAL_ROLL_VEL;
		maxGimbal = MAX_GIMBAL_ROLL;
	}
	else
	{
		GimbalVel = GIMBAL_VEL;
		maxGimbal = MAX_GIMBAL;
	}
		

	// maximum possible actuator shift during time step
	double dActuatorMove = GimbalVel * simdt;

	// can we jump directly to where we want to be, or are we limited by timestep?
	double dAngle = angle - angle0;
	if (fabs(dAngle) > dActuatorMove)
	{
		if (dAngle > 0)
			dAngle = dActuatorMove;
		else
			dAngle = -dActuatorMove;
	}

	// finally, the required _and_ possible new pitch actuator position...
	NewAngle = angle0 + dAngle;

	// ...well, almost... check if we hit the stopper
	if (NewAngle > maxGimbal)
		NewAngle = maxGimbal;
	else if (NewAngle < -maxGimbal)
		NewAngle = -maxGimbal;

	return NewAngle;
}


/*
double r7BlockL::GetWantedActuatorPosition2(double simdt, double angle, double angle0)
{
	double NewAngle;

	// maximum possible actuator shift during time step
	double dActuatorMove = GIMBAL_VEL * simdt;

	// can we jump directly to where we want to be, or are we limited by timestep?
	double dAngle = angle - angle0;
	if (fabs(dAngle) > dActuatorMove)
	{
		if (dAngle > 0)
			dAngle = dActuatorMove;
		else
			dAngle = -dActuatorMove;
	}

	// finally, the required _and_ possible new pitch actuator position...
	NewAngle = angle0 + dAngle;

	// ...well, almost... check if we hit the stopper
	if (NewAngle > MAX_GIMBAL)
		NewAngle = MAX_GIMBAL;
	else if (NewAngle < -MAX_GIMBAL)
		NewAngle = -MAX_GIMBAL;

	return NewAngle;
}


double r7BlockL::GetWantedActuatorPositionRoll2(double simdt, double angle, double angle0)
{
	double NewAngle;

	// maximum possible actuator shift during time step
	double dActuatorMove = GIMBAL_ROLL_VEL * simdt;

	// can we jump directly to where we want to be, or are we limited by timestep?
	double dAngle = angle - angle0;
	if (fabs(dAngle) > dActuatorMove)
	{
		if (dAngle > 0)
			dAngle = dActuatorMove;
		else
			dAngle = -dActuatorMove;
	}

	// finally, the required _and_ possible new pitch actuator position...
	NewAngle = angle0 + dAngle;

	// ...well, almost... check if we hit the stopper
	if (NewAngle > MAX_GIMBAL_ROLL)
		NewAngle = MAX_GIMBAL_ROLL;
	else if (NewAngle < -MAX_GIMBAL_ROLL)
		NewAngle = -MAX_GIMBAL_ROLL;

	return NewAngle;
}
*/

void r7BlockL::InitializeCoastingPhase()
{
	// %%% won't need when camera is on Block I
	if (CameraDir = CAMERA_BLOCKI)
		ToggleOnboardCamera(CAMERA_DOWN);

	// Update state flags
	bNeedsCoastingInit = false;
	bCoasting = true;

	if (Target == TGT_MOON)
	{
		// calculate time to TLI 
		TimeToIgnition = CalcTimeToNode();

		// calc TLI burn time
		BurnLength = CalcTLIBurnTime(); 
	}

	else if (Target == TGT_MOLNIYA)
	{
		// for Molniya, we can hard-code this
		TargetApoR = 45371000;	
		TargetApoRLess = TargetApoR * 0.98; 

		// calculate time to south point - burning there makes axis normal to nodes line
		TimeToIgnition = CalcTimeToSouth();

		// %%% for now, use TLI function; will need to be fixed when base orbit becomes elliptic too
		BurnLength = CalcTLIBurnTime(); 
	}

	// start ignition earlier to account for burn length
	TimeToIgnition -= (BurnLength / 1.8);
}




void r7BlockL::AttachBOZ()
{
	// create attachment point on us - always, or it screws many things
	hForBOZ = CreateAttachment(false,
				                           _V(0,0,0),
										   _V(0,0,-1),
										   _V(1,0,0),
										   "ForBOZ",
										   false);

	// should we attach BOZ?
	if (BOZName[0] == 0)
		return;

	OBJHANDLE hBOZ = oapiGetObjectByName(BOZName);
	if (!hBOZ)
		return;

	// get BOZ vessel interface
	VESSEL	*BOZIntf;
	BOZIntf = oapiGetVesselInterface(hBOZ);
	BOZMass = BOZ_MASS;

	// create attachment point on BOZ
	OBJHANDLE hToMe = BOZIntf->CreateAttachment(true,
				                           _V(0,0,0),
										   _V(0,0,1),
	 									   _V(1,0,0),
								           "ToBlockL",
								           false);
	// Attach!
	AttachChild(hBOZ, hForBOZ, hToMe);

	// make fuel and gas resources for BOZ engines ans RCS
	SetPropellantMass(BOZEnginesFuel, BOZ_FUEL_MASS);
	SetPropellantMass(BOZRCSTanks, BOZ_GAS_MASS);
}


// this, of course, happens during scenario reading, not in-flight :-)
void r7BlockL::PayloadAttach()
{
	// create attachment point on us - always, or it screws many things
	hForPayload = CreateAttachment(false,
				                           PAYPOINT,
										   PAYDIR,
										   PAYROT,
										   "ForPayload",
										   false);

	// do we have a payload at all?
	if (PayloadName[0] == 0)
		return;

	OBJHANDLE hPayload = oapiGetObjectByName(PayloadName);
	if (!hPayload)
		return;

	// get payload vessel interface, get its mass
	VESSEL	*PayloadIntf;
	PayloadIntf = oapiGetVesselInterface(hPayload);
	PayloadMass = PayloadIntf->GetMass();

	// create attachment point on payload
	OBJHANDLE hToMe = PayloadIntf->CreateAttachment(true,
		                                    PayREF,
								            PayDIR,
								            PayROT,
								            "ToBlockL",
								            false);

	// Attach!
	AttachChild(hPayload, hForPayload, hToMe);
}


// by J key, by autopilot or by accident :-)
void r7BlockL::SeparateBOZ()
{
	// do we have someting to separate?
    if (!GetAttachmentStatus(hForBOZ))
		return;

	// No need to randomize: visual effect is not noticeable.
	// %%% Maybe just to add a touch of rotation...
	// Better to randomize the kick effect
	double vDetach = 0.1; 

	// separate whatever is attached to the payload attacment, indiscriminatory
	DetachChild(hForBOZ, vDetach);

	BOZMass = 0;
	SetEmptyMass(EMPTY_MASS + BOZMass + PayloadMass);

	// before killing BOZ name, store BOZ handle for the last kick 
	hBOZ = oapiGetObjectByName(BOZName);
	BOZName[0] = 0;

	RemoveBOZRCSSmall();
	RemoveBOZRCSLarge();
	SetPropellantMass(BOZRCSTanks, 0);
	RemoveBOZEngines();
}


// by J key, by autopilot or by accident :-)
void r7BlockL::PayloadSeparate()
{
	// do we have someting to separate?
    if (!GetAttachmentStatus(hForPayload))
		return;

	// just in case - shutdown main engine if running
	SetThrusterGroupLevel(MainThGr, 0);

	//%%% calc or randomize separation params
	double vDetach = 0.1; 

	// separate whatever is attached to the payload attacment, indiscriminatory
	DetachChild(hForPayload, vDetach);

	PayloadMass = 0;
	SetEmptyMass(EMPTY_MASS + BOZMass + PayloadMass);

	// get payload handle by name before killing the name 
	OBJHANDLE hPayload = oapiGetObjectByName(PayloadName);
	PayloadName[0] = 0;

	// change focus to payload - only if we have focus in the first place!
	if ( (bSwitchFocus) && (oapiGetFocusObject() == GetHandle()) )
	{
		// Make sure that the originally attached object still exists.
		// This won't work if the object was destroyed, and only some piece remains...
		// but it won't CTD either, just keeps the focus at L.
		if (hPayload) 
		{
			VESSEL	*PlIntf;
			PlIntf = oapiGetVesselInterface(hPayload);

			// one last check: is payload allowed to receive focus?
			if (PlIntf->GetEnableFocus()) 
				oapiSetFocusObject(hPayload);
		}
	}
}


// not only this calculates the time to node,
// but, as a side effect, sets the targeting orbit apogee that touches the Moon orbit
double r7BlockL::CalcTimeToNode()
{

	// get planets data
	OBJHANDLE hMoon = oapiGetGbodyByName("moon");
	OBJHANDLE hEarth = oapiGetGbodyByName("earth");

	double MEarth = oapiGetMass(hEarth);
	double MMoon = oapiGetMass(hMoon);

	double MuEarth = GRAVITYCONTANT * MEarth;
	double MuBoth = GRAVITYCONTANT * (MEarth + MMoon); // Moon is a big body, not a point!

	// Get angular momentum vector R*V of the target (Moon) orbit rel to Earth
	VECTOR3 VMoonPos, VMoonVel;	  
	oapiGetRelativeVel(hMoon, hEarth, &VMoonVel);
	oapiGetRelativePos(hMoon, hEarth, &VMoonPos);
	double MoonPos = length(VMoonPos);
	double MoonVel = length(VMoonVel);

	VECTOR3 VMoonAngMomentum = crossp(VMoonPos, VMoonVel); 
	double MoonAngMomentum = length(VMoonAngMomentum);

	// Angular momentum is the "pole" of the orbit, so be it our Z axis.
	// Make z unit vector from it.
	VECTOR3 VMoon1z = VMoonAngMomentum/MoonAngMomentum;    

	// Get eccentricity vector, AKA normalized Laplace vector
	VECTOR3 VE = VEccentricity(MuBoth, VMoonPos, VMoonVel); 
	double e = length(VE);

	// Make x unit vector from it
	VECTOR3 VMoon1x = VE / e;

	// Supplement the orts with y unit vector
	VECTOR3 VMoon1y = crossp(VMoon1z, VMoon1x); 
	double Moon1y = length(VMoon1y);
	VMoon1y /= Moon1y;

	// Make tranlslation matrix from ecliptic to moon orbit reference frame
	MATRIX3 TmEcliToMoon = MakeTranslationMatrix(VMoon1x, VMoon1y, VMoon1z);
		
	// Get our pos and vel in the new frame
	VECTOR3 VPos, VVel;	  
	GetRelativePos(hEarth, VPos);
	GetRelativeVel(hEarth, VVel);

	VPos = mul(TmEcliToMoon, VPos);
	VVel = mul(TmEcliToMoon, VVel);
	double Pos = length(VPos);
	double Vel = length(VVel);
			
	// Calculate Normal vector and i, then ascending node in the new frame

	// Get our angular mmentum (orbit pole) in the new frame
	VECTOR3 VAngMomentum = crossp(VPos, VVel);   
	double AngMomentum = length(VAngMomentum);

	// Get inclination (z projection angle of ang momentum)
	double i = acos(VAngMomentum.z / AngMomentum);  

	// Get normal vector (parallel to the Moon orbit, and lying in the plane of our orbit)
	VECTOR3 V1z = _V(0, 0, 1);
	VECTOR3 VNormal = crossp(V1z, VAngMomentum); 

		// Calculate moon orbit R at the poit of arrival

	// make unit vector from normal vector 
	VECTOR3 VNormal1 = VNormal / length(VNormal);

	// get true anomaly of the crossing point
	double MoonCrossTrueAno = PI-acos(VNormal1.x);

	// get extra orbit params (Orbiter does not have GetElements for planets
	double MoonSpecificMechanicalEnergy = (MoonVel * MoonVel / 2) - (MuBoth / MoonPos);
	double a = -MuBoth / (2 * MoonSpecificMechanicalEnergy);
	double p = a * (1 - e * e);

	// Done - this will be our targeting apogee!
	double MoonCrossR = p / (1 + e * cos(MoonCrossTrueAno));
	TargetApoR = MoonCrossR + 1736000;	// add Moon radius to get a wider envelope at the Moon
	TargetApoRLess = TargetApoR * 0.95;   // this is a criteria to switch off time accel during TLI

		// Calculate the times of the ascending and descending nodes.

	// Eccentricity, True Anomaly, Argument of Perigee of our orbit
	VE = VEccentricity(MuEarth, VPos, VVel);  
	double E = length(VE);
	double f = TrueAnomaly(VE, VPos, VVel); 
	double ArgPeri = ArgOfPeri(VNormal, VE);
		
	// period
	double SpecificMechanicalEnergy = (Vel * Vel / 2) - (MuEarth / Pos);
	double SemiMajorAxis = -MuEarth / (2 * SpecificMechanicalEnergy);
	double SemiMajorAxis3 = SemiMajorAxis * SemiMajorAxis * SemiMajorAxis;
	double Period = 2 * PI * sqrt(SemiMajorAxis3 / MuEarth);

	// Time of Adjusted (inverse) Argument of periapsis
	double EccAnomaly = EccentricAnomaly(E, 2*PI - ArgPeri);
	double MnAnomaly = MeanAnomaly(EccAnomaly, E);
	double TimeOfAdjAOP = TimeSincePeriapsis(MnAnomaly, Period);
	
	// Current time since periapsis
	EccAnomaly = EccentricAnomaly(E, f);
	MnAnomaly = MeanAnomaly(EccAnomaly, E);
	double TimeOfTrueAnomaly = TimeSincePeriapsis(MnAnomaly, Period);

	// Time to ascending node
	double TimeToAscendingNode = TimeOfAdjAOP - TimeOfTrueAnomaly;

	if (TimeToAscendingNode < 0) 
		TimeToAscendingNode = Period + TimeToAscendingNode;

	// time between periapsis and the descending node
	double ArgOfPeriPlusPI = ArgPeri + PI; 
	if (ArgOfPeriPlusPI > 2*PI)
		ArgOfPeriPlusPI = ArgOfPeriPlusPI - 2*PI;

	EccAnomaly = EccentricAnomaly(E, 2*PI - ArgOfPeriPlusPI);
	MnAnomaly = MeanAnomaly(EccAnomaly, E);
	double TimeOfAdjAOPPlusPI = TimeSincePeriapsis(MnAnomaly, Period);

	// Time to the descending node
	double TimeToDescendingNode = TimeOfAdjAOPPlusPI - TimeOfTrueAnomaly;

	if (TimeToDescendingNode < 0)
		TimeToDescendingNode = Period + TimeToDescendingNode;

	// done! select the shortest time and return!
	if (TimeToDescendingNode > TimeToAscendingNode)
	  return TimeToAscendingNode;
	else
	  return TimeToDescendingNode;
}


// to the southmost point of orbit, assuming it is closer to apogee than to perigee
double r7BlockL::CalcTimeToSouth()
{
	// get current orbit
	OBJHANDLE hEarth = oapiGetGbodyByName("earth");
	ELEMENTS OrbEls;
	ORBITPARAM OrbParam;
	GetElements(hEarth, OrbEls, &OrbParam, 0, FRAME_EQU);

	// us to south - in degrees
	double UsToSouthD = 270 - (OrbParam.TrL - OrbEls.theta)* DEG;

	// true anomaly of south point
	double TrASouthD = OrbParam.TrA * DEG + UsToSouthD;
	double TrASouthR = TrASouthD*RAD;

	// my time to perigee, can benegative if already passed
	double MyTime = OrbParam.PeT;
	if (OrbParam.PeT > OrbParam.ApT)
		MyTime = -MyTime;

	// time from perigee to south point
	double EccAnomaly = EccentricAnomaly(OrbEls.e, TrASouthR);
	double MnAnomaly = MeanAnomaly(EccAnomaly, OrbEls.e);
	double SouthTime = OrbParam.T + TimeSincePeriapsis(MnAnomaly, OrbParam.T);
	if (OrbParam.PeT > OrbParam.ApT)
		SouthTime = -TimeSincePeriapsis(MnAnomaly, OrbParam.T);;

	// add  ot subtract times
	double TravelTime = MyTime + SouthTime;

	// done!
	return TravelTime;

}


// Approximate. Actual MECO is commanded by sensor
double r7BlockL::CalcTLIBurnTime()
{
	// get apogee radius
	double RA = TargetApoR;

	// get current orbit
	OBJHANDLE hEarth = oapiGetGbodyByName("earth");
	ELEMENTS OrbEls;
	ORBITPARAM OrbParam;
	GetElements(hEarth, OrbEls, &OrbParam);

	// assume orbit circular, %%% later can calculate actual anomaly/radius
	double RP = (OrbParam.PeD + OrbParam.ApD) / 2;

	double a = (RA + RP) / 2;
	double e = (a - RP) / a;
	double p = RP * (1 + e);

	double mu = 3.986E+14;

	// calc delta v 
	double V0 = sqrt(mu / RP);
	double Vk = sqrt(mu / p) * (1+e);
	double dV = Vk - V0;

	// calc delta m
	double M0 = GetMass() - BOZ_MASS - GetPropellantMass(BOZRCSTanks) - GetPropellantMass(BOZEnginesFuel);
	double mdot = ISP_MAIN / THRUST_MAIN;  // %%% can add roll engines if needed)
	double mratio = exp(dV / ISP_MAIN);
	double Mk = M0 / mratio;
	double Mt = M0 - Mk; 

	// calc burn length
	double t = Mt * mdot;

	return t;
}


void r7BlockL::LocalRot(const VECTOR3 vecGlob, VECTOR3 &vecLoc)
{
	VESSELSTATUS2 vs;
	double dCosZ, dSinZ, dCosY, dSinY, dCosX, dSinX;
	vs.version = 2;
	vs.flag = 0;
	GetStatusEx((void*) &vs);

	dCosZ = cos(vs.arot.z);
	dSinZ = sin(vs.arot.z);
	dCosY = cos(vs.arot.y);
	dSinY = sin(vs.arot.y);
	dCosX = cos(vs.arot.x);
	dSinX = sin(vs.arot.x);

	vecLoc.x = vecGlob.x*dCosZ*dCosY + vecGlob.y*(-dSinZ*dCosX+dCosZ*dSinY*dSinX) + vecGlob.z*(dSinZ*dSinX+dCosZ*dSinY*dCosX);
	vecLoc.y = vecGlob.x*dSinZ*dCosY + vecGlob.y*(dCosZ*dCosX+dSinZ*dSinY*dSinX) + vecGlob.z*(-dCosZ*dSinX+dSinZ*dSinY*dCosX);
	vecLoc.z = vecGlob.x*-dSinY + vecGlob.y*dCosY*dSinX + vecGlob.z*dCosY*dCosX;
}


void r7BlockL::clbkVisualCreated(VISHANDLE visual, int refcount)
{
	if (refcount > 1) 
		return;

	vis = visual;
}


void r7BlockL::clbkVisualDestroyed(VISHANDLE visual, int refcount)
{
	if (visual == vis) 
		vis = NULL;
}


void r7BlockL::DebugSomething()
{
//	sprintf(oapiDebugString(), "%f", temp);
}


void r7BlockL::ShowAnnotationMessage(char *Message)
{
	oapiAnnotationSetText (MessagePort, Message);
	msgStartTime = oapiGetSysTime();
	bMessageDisplayed = true;
}


void r7BlockL::clbkNavMode (int mode, bool active)
{
	// in manual mode, allow everything
	if (bManualControl) 
		return;


	// in autoplilot, allow only prograde, and only when really needed...
	// and protect it from overriding!
	if ((bPrograded) && (mode = NAVMODE_PROGRADE) )
	{
		if (!active) 
		{
			ActivateNavmode(mode);
			ShowAnnotationMessage("Manual controls disabled. To enable, use M key");
		}

		// else do nothing, just continue prograding
		
		// ...and just return
		return;
	}

	// for all other deactivation events, simply ignore them, no harm
	if (!active) 
		return;

	// finally, for all other activation events - override by immediately deactivating it!
	DeactivateNavmode(mode);
	ShowAnnotationMessage("Manual controls disabled. To enable, use M key");
}



DLLCLBK VESSEL *ovcInit (OBJHANDLE hVessel, int flightmodel)
{
	srand(time(NULL));
	return new r7BlockL (hVessel,flightmodel);
}


DLLCLBK void ovcExit (VESSEL *vessel)
{
	if (vessel) delete (r7BlockL*)vessel;
}



