#define ORBITER_MODULE

#include "Quad.h"
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <math.h>



// default constructor
Quad::Quad (OBJHANDLE hObj, int fmodel): VESSEL2 (hObj, fmodel)
{
	;
}


// This class initialization is called for both runtime and scenario created objects.
// Realistically, I DO NOT expect object to be created at runtime.
// For scenario object,is is called BEFORE the .scn file is read!
void Quad::clbkSetClassCaps(FILEHANDLE cfg)
{

	// Set geometry
	SetSize (2.0);
	SetTouchdownPoints (_V( 1, 1,-0.9), _V( 1, -1,-0.9), _V(-1, 0,-0.9));
	SetCameraOffset(_V(0, 0, 0));
	SetCOG_elev(0.9);

	// Set physics 
	SetEmptyMass (300);
	SetPMI (_V(.35, .35, .35));
	SetCrossSections (_V(1, 1, 2));
	SetPitchMomentScale (1e-4);
	SetBankMomentScale (1e-4);
	SetRotDrag (_V(0.1, 0.1, 0.1));
	SetLiftCoeffFunc (0);
	SetCW (0.55, 0.55, 0.55, 0.55);
	SetSurfaceFrictionCoeff(10,10);

	// set main engine
	MainFuel = 0;
	MainTank = CreatePropellantResource(FUEL_CAPACITY);
	MainEngine = CreateThruster(_V(0, 0, GIMBAL_Z), _V(0,0,1), 13336, MainTank, 2000, 2000);
	MainTH[0] = MainEngine;
	CreateThrusterGroup(MainTH,1,THGROUP_MAIN);
	AddExhaust(MainTH[0], 10.0, 0.3, 0.8);


	// load mesh, shifting Z of CG
	VECTOR3 Off = _V(0, 0, MESH_Z);
	MeshID = AddMesh (oapiLoadMeshGlobal("Quad"), &Off);
	SetMeshVisibilityMode (MeshID, MESHVIS_ALWAYS);
	SetCameraDefaultDirection(_V(0,0,-1));
	
	// create additional roll control thrusters
	MakeRollRCS();

	// create dust cloud when low
	PARTICLESTREAMSPEC Dust = 
	{
		0,			// flags, reserved
		0.1,			// srcsize, particle size at creation, m
		5,			// srcrate, average particle generation rate, Hz  
		5.0,		// V0, average particle emission velocity, m/s
		0,			// srcspread, emission velocity distribution randomisation
		3,			// lifetime, average particle lifetime [s]
		10,			// growthrate, particle growth rate [m/s]
		5,			// atmslowdown, deceleration rate b in atmosphere, defined as v = v0 e- bdt 
		PARTICLESTREAMSPEC::DIFFUSE,
		PARTICLESTREAMSPEC::LVL_PSQRT, 
		0, 
		0.5,
		PARTICLESTREAMSPEC::ATM_PLOG, 
		1.22, 
		1.25
	};

	AddExhaustStream (MainTH[0], _V(0,0,-40.0), &Dust); 
}


// updating the configuration from scn file
void Quad::clbkLoadStateEx (FILEHANDLE scn, void *status)
{
	char *line;

	// read the scenario file
	while (oapiReadScenario_nextline (scn, line)) 
	{
/*		if (!strnicmp (line, "VAR1", 4)) 
		{
            sscanf (line+4, "%lf", &var1);
		}
		else if (!strnicmp (line, "VAR2", 4)) 
		{
            sscanf (line+4, "%lf", &var2);
		}

		// --- unrecognised option - pass to Orbiter's generic parser
		else 
*/
		{

			if (!strnicmp (line, "PRPLEVEL 0:", 11)) 
			{
			    sscanf (line+11, "%lf", &MainFuel);
			}
            
			ParseScenarioLineEx (line, status);
        }
	}

	// Initialize some variables

	// main engine controls
	MainFuel *= FUEL_CAPACITY;
	MainThrottleManualMode = THM_OFF;
	LastThrottleManualCommand = THM_OFF;
	ThrottleManualKeyTime = 0;

	// attitude controls
	Pitch = 0;
	Yaw = 0;
	oapiGetHeading(GetHandle(), &Heading0);
	Heading0 *= DEG;

	// running vars
	met = 0;

	// flag internal camera for auto-redirection when visual is created
	CameraDir = CAMERA_UP;

}


// saving configuration to scn file
void Quad::clbkSaveState (FILEHANDLE scn)
{
	// default vessel parameters
	SaveDefaultState (scn);

/*	// store visual separator
	oapiWriteScenario_string (scn, "------", "");

	// store auto-generated entries, except for bRealistic (will be part of the StateFlags)
	oapiWriteScenario_float (scn, "VAR1", var1);
*/
}


// Keyboard interface handlers

int Quad::clbkConsumeDirectKey (char *keystate)
{
	// throttle

	// up!
	if (KEYDOWN (keystate, OAPI_KEY_ADD)) 
	{
		// update for how long pilot is holding the key
		if (LastThrottleManualCommand == THM_UP_FAST)
			ThrottleManualKeyTime += oapiGetSimStep();
		else
			ThrottleManualKeyTime =0;

		// set the up mode depending on elapsed key time
		if (ThrottleManualKeyTime < THM_TIME_TRESHOLD_UP)
			MainThrottleManualMode = THM_UP_SLOW;
		else
			MainThrottleManualMode = THM_UP_FAST;

		// store key flag for next step reference
		LastThrottleManualCommand = THM_UP_FAST;

		// disable default key processing
		RESETKEY (keystate, OAPI_KEY_ADD);
	}

	// down!
	else if (KEYDOWN (keystate, OAPI_KEY_SUBTRACT)) 
	{
		// update for how long pilot is holding the key
		if (LastThrottleManualCommand == THM_DOWN_FAST)
			ThrottleManualKeyTime += oapiGetSimStep();
		else
			ThrottleManualKeyTime =0;

		// set the up mode depending on elapsed key time
		if (ThrottleManualKeyTime < THM_TIME_TRESHOLD_DOWN)
			MainThrottleManualMode = THM_DOWN_SLOW;
		else
			MainThrottleManualMode = THM_DOWN_FAST;

		// store key flag for next step reference
		LastThrottleManualCommand = THM_DOWN_FAST;

		// disable default key processing
		RESETKEY (keystate, OAPI_KEY_SUBTRACT);
	}

	// none - hover!
	else
	{
		MainThrottleManualMode = THM_HOVER;
		LastThrottleManualCommand = THM_HOVER;
	}


	// direction: yaw

	if	(	
//			KEYDOWN (keystate, OAPI_KEY_NUMPAD3) || 
			KEYDOWN (keystate, OAPI_KEY_NUMPAD6) //|| 
//			KEYDOWN (keystate, OAPI_KEY_NUMPAD9) 
		) 
	{
		GimbalDirYaw = ATT_POS;
	}

	else if	(	
//			KEYDOWN (keystate, OAPI_KEY_NUMPAD1) || 
			KEYDOWN (keystate, OAPI_KEY_NUMPAD4) //|| 
//			KEYDOWN (keystate, OAPI_KEY_NUMPAD7) 
		) 
	{
		GimbalDirYaw = ATT_NEG;
	}

	else
		GimbalDirYaw = ATT_NEUT;


	// direction: pitch

	if	(	
//			KEYDOWN (keystate, OAPI_KEY_NUMPAD7) || 
			KEYDOWN (keystate, OAPI_KEY_NUMPAD8) //|| 
//			KEYDOWN (keystate, OAPI_KEY_NUMPAD9) 
		) 
	{
		GimbalDirPitch = ATT_NEG;
	}

	else if	(	
//			KEYDOWN (keystate, OAPI_KEY_NUMPAD1) || 
			KEYDOWN (keystate, OAPI_KEY_NUMPAD2) //|| 
//			KEYDOWN (keystate, OAPI_KEY_NUMPAD3) 
		) 
	{
		GimbalDirPitch = ATT_POS;
	}

	else
		GimbalDirPitch = ATT_NEUT;


	// direction: roll
	if	(KEYDOWN (keystate, OAPI_KEY_NUMPAD0))
		AttRollMode = ATT_R_NEG;
	else if	(KEYDOWN (keystate, OAPI_KEY_DECIMAL))
		AttRollMode = ATT_R_POS;
	else
		AttRollMode = ATT_R_NEUT;


	//
	return 0;
}


int Quad::clbkConsumeBufferedKey(DWORD key, bool down, char *kstate)
{

	// only process keydown events
	if (!down) return 0; 

	// shift combinations are reserved
	if (KEYMOD_SHIFT (kstate)) return 0; 

	// down ground view
	if ( !KEYMOD_CONTROL (kstate) && (key == OAPI_KEY_G) )
	{
		SetCameraOffset(_V(1, 1, 0.4));
		SetCameraDefaultDirection(_V(0,0,-1));
		CameraDir = CAMERA_DOWN;
		return 0;
	}

	// forward  view
	if ( !KEYMOD_CONTROL (kstate) && (key == OAPI_KEY_K) )
	{
		SetCameraOffset(_V(-1, 1, 0.5));
		SetCameraDefaultDirection(_V(0,1,0), PI);
		CameraDir = CAMERA_FORWARD;
		return 0;
	}

	// refuel, only on the ground, by 10%
	if ( !KEYMOD_CONTROL (kstate) && (key == OAPI_KEY_PERIOD) )
	{
		if (!bInFlight)
		{
			MainFuel += (FUEL_CAPACITY*0.10);

			if (MainFuel > FUEL_CAPACITY)
				MainFuel = FUEL_CAPACITY;

			SetPropellantMass(MainTank, MainFuel);
			OutputString();
		}

		return 0;
	}

	// manual debug
	if ( !KEYMOD_CONTROL (kstate) && (key == OAPI_KEY_D) )
	{
		DebugSomething();
		return 0;
	}

	return 0;
}



void Quad::MakeRollRCS()
{
	// tune-up consts
	const RCSThrust = 5;
	const RCSISP = 49000000;
	const double RCSExhLen = 0.5;
	const double RCSExhWid = 0.02;

	// set virtual RCS (to simulate vanes/rudders)
	PROPELLANT_HANDLE RCSTank  = CreatePropellantResource (1);
	THRUSTER_HANDLE thRot[4];


	thRot[0] = CreateThruster (_V(-0.9,  0.9, -0.25), _V( 0.707,  0.707, 0), RCSThrust, RCSTank, RCSISP);
	thRot[1] = CreateThruster (_V( 0.9, -0.9, -0.25), _V(-0.707, -0.707, 0), RCSThrust, RCSTank, RCSISP);
	thRot[2] = CreateThruster (_V(-0.9, -0.9, -0.25), _V(-0.707,  0.707, 0), RCSThrust, RCSTank, RCSISP);
	thRot[3] = CreateThruster (_V( 0.9,  0.9, -0.25), _V( 0.707, -0.707, 0), RCSThrust, RCSTank, RCSISP);
	RightRoll = CreateThrusterGroup (thRot, 4, THGROUP_USER);

	AddExhaust (thRot[0], RCSExhLen, RCSExhWid);
	AddExhaust (thRot[1], RCSExhLen, RCSExhWid);
	AddExhaust (thRot[2], RCSExhLen, RCSExhWid);
	AddExhaust (thRot[3], RCSExhLen, RCSExhWid);

	thRot[0] = CreateThruster (_V(-0.9,  0.9, -0.25), _V(-0.707, -0.707, 0), RCSThrust, RCSTank, RCSISP);
	thRot[1] = CreateThruster (_V( 0.9, -0.9, -0.25), _V( 0.707,  0.707, 0), RCSThrust, RCSTank, RCSISP);
	thRot[2] = CreateThruster (_V(-0.9, -0.9, -0.25), _V( 0.707, -0.707, 0), RCSThrust, RCSTank, RCSISP);
	thRot[3] = CreateThruster (_V( 0.9,  0.9, -0.25), _V(-0.707,  0.707, 0), RCSThrust, RCSTank, RCSISP);
	LeftRoll = CreateThrusterGroup (thRot, 4, THGROUP_USER);

	AddExhaust (thRot[0], RCSExhLen, RCSExhWid);
	AddExhaust (thRot[1], RCSExhLen, RCSExhWid);
	AddExhaust (thRot[2], RCSExhLen, RCSExhWid);
	AddExhaust (thRot[3], RCSExhLen, RCSExhWid);
}





void Quad::clbkPostStep(double simt, double simdt, double mjd)
{
}


void Quad::clbkPreStep(double simt, double simdt, double mjd)
{

	// fuel autofill prevention
	if (GetPropellantMass(MainTank) > MainFuel + 1)
		SetPropellantMass(MainTank, MainFuel);

		// read sensors

	// altitute
	Altitude = GetAltitude();
	if (Altitude > MaxAltitude)
		MaxAltitude = Altitude;

	// velocity, horizon frame
	VECTOR3 HAV, LAV; 
	GetHorizonAirspeedVector(HAV);
	Vv = HAV.y;
	Vh = sqrt(HAV.x*HAV.x + HAV.z*HAV.z);

	// velocity, local frame
	GetShipAirspeedVector(LAV);
	Vforw = LAV.y;
	Vsidew = - LAV.x;

	// g, acceleration
	double CurrentThrust = GetThrusterLevel(MainEngine);
	VECTOR3 W, T; 
	GetWeightVector(W);
	GetThrustVector(T);
	double Wlen = sqrt(W.x*W.x + W.y*W.y + W.z*W.z);
	double Tlen = sqrt(T.x*T.x + T.y*T.y + T.z*T.z);
	n = Tlen / Wlen;

	if (!bInFlight && (CurrentThrust > 0))
		SetThrusterLevel(MainEngine, 0);

	// launch
	if (!bInFlight && (MainThrottleManualMode == THM_UP_SLOW))
		Launch();


	if (bInFlight)
	{
		// update met, do it only here!
		met += simdt;
		Simdt = simdt;

		// calc throttle

		// start from current
		NewMainThrust = CurrentThrust;

		// manual nputs - where pilot wants it to go?
		if (MainThrottleManualMode == THM_UP_FAST)
			NewMainThrust += TH_PERCENT_UP_FAST * simdt;

		else if (MainThrottleManualMode == THM_UP_SLOW)
			NewMainThrust += TH_PERCENT_UP_SLOW * simdt;

		else if (MainThrottleManualMode == THM_DOWN_FAST)
		{
			if (NewMainThrust > TH_PERCENT_MIN)
				NewMainThrust -= TH_PERCENT_DOWN_FAST * simdt;
		}

		else if (MainThrottleManualMode == THM_DOWN_SLOW)
		{
			if (NewMainThrust > TH_PERCENT_MIN)
				NewMainThrust -= TH_PERCENT_DOWN_SLOW * simdt;
		}

		// hovering: auto adjust!
		else if (MainThrottleManualMode == THM_HOVER)
		{
			if (Vv > 0)
			{
				if (NewMainThrust > TH_PERCENT_MIN)
					NewMainThrust -= TH_PERCENT_UP_FAST * simdt;
			}
			else
			{
				NewMainThrust += TH_PERCENT_UP_FAST * simdt;
			}
		}

		// check pre-programmed limits - only when going up
		if (Vv > 0)
		{
			if (Vv > MAX_VERT_VEL)
				NewMainThrust -= TH_PERCENT_UP_FAST * simdt;
			if (n > MAX_N)
				NewMainThrust -= TH_PERCENT_UP_FAST * simdt;
		}

		// bound to physical limits
		if (NewMainThrust > 1) 
			NewMainThrust = 1;
		else if (NewMainThrust < 0) 
			NewMainThrust = 0;

		// adjust the control
		SetThrusterLevel(MainEngine, NewMainThrust);


		//
		Gimbal();

	// landed?
	if ((met > 5) && GroundContact())
	{
		bInFlight = false;
		SetThrusterLevel(MainEngine, 0);
		SetThrusterGroupLevel(RightRoll, 0);
		SetThrusterGroupLevel(LeftRoll, 0);

		double Longitude, Latitude, Radius;
		VESSELSTATUS vs;

		GetEquPos(Longitude, Latitude, Radius);
		GetStatus(vs);

		vs.status = 1;
		vs.vdata[0].x = Longitude;
		vs.vdata[0].y = Latitude;
		vs.vdata[0].z = LocalAtt.z * RAD;

		DefSetState(&vs);

		MainFuel = GetPropellantMass(MainTank);

	}
		OutputString();
	}
}


void Quad::Gimbal()
{
	double Pitch0 = Pitch;
	double Yaw0 = Yaw;
	double dActuatorMove = GIMBAL_VEL * Simdt;

	// pitch censor
	VECTOR3 Y1 = _V(0, 1, 0);
	VECTOR3 Y1H;
	HorizonRot(Y1, Y1H);
	LocalAtt.x = Y1H.y;

	// yaw censor
	VECTOR3 X1 = _V(1, 0, 0);
	VECTOR3 X1H;
	HorizonRot(X1, X1H);
	LocalAtt.y = X1H.y;

	// roll censor
	LocalAtt.z = - asin(X1H.z);
	if (Y1H.z > 0)
	{
		if (LocalAtt.z < 0)
			LocalAtt.z = -PI - LocalAtt.z;
		else
			LocalAtt.z = PI - LocalAtt.z;
	}

	// normalize LocalAtt to degrees
	LocalAtt *= DEG;

	// pitch actuator from control input
	if (GimbalDirPitch == ATT_POS)
		Pitch = -MAX_GIMBAL;
	else if (GimbalDirPitch == ATT_NEG)
		Pitch = MAX_GIMBAL;
	else
	{
		// velocity priority
		if (Vforw < 0)
			Pitch =  MAX_GIMBAL;
		else 
			Pitch = -MAX_GIMBAL;

		// attitude secondary priority
		if (fabs(Vforw) < 0.1)
		{
			if (LocalAtt.x > 0)
				Pitch =  MAX_GIMBAL;
			else 
				Pitch = -MAX_GIMBAL;
		}
	}

	// pitch actuator limit from sensor
	if (LocalAtt.x < -5) 
		Pitch = -MAX_GIMBAL;
	else if (LocalAtt.x > 5) 
		Pitch = MAX_GIMBAL;

	// pitch actuator real move
	if (Pitch > 0)
	  Pitch = Pitch0 + dActuatorMove;
	else
	  Pitch = Pitch0 - dActuatorMove;

	// pitch actuator physical limit
	if (Pitch > MAX_GIMBAL)
		Pitch = MAX_GIMBAL;
	else if (Pitch < -MAX_GIMBAL)
		Pitch = -MAX_GIMBAL;


	// yaw actuator from control input
	if (GimbalDirYaw == ATT_POS)
		Yaw = -MAX_GIMBAL;
	else if (GimbalDirYaw == ATT_NEG)
		Yaw = MAX_GIMBAL;
	else
	{
		// velocity priority
		if (Vsidew > 0)
			Yaw = MAX_GIMBAL;
		else 
			Yaw = -MAX_GIMBAL;

		// attitude secondary priority
		if (fabs(Vsidew) < 0.1)
		{
			if (LocalAtt.y > 0)
				Yaw =  MAX_GIMBAL;
			else 
				Yaw = -MAX_GIMBAL;
		}
	}

	// yaw actuator limit from sensor
	if (LocalAtt.y < -5) 
		Yaw = -MAX_GIMBAL;
	else if (LocalAtt.y > 5) 
		Yaw = MAX_GIMBAL;

	// yaw actuator real move
	if (Yaw > 0)
	  Yaw = Yaw0 + dActuatorMove;
	else
	  Yaw = Yaw0 - dActuatorMove;

	// pitch actuator physical limit
	if (Yaw > MAX_GIMBAL)
		Yaw = MAX_GIMBAL;
	else if (Yaw < -MAX_GIMBAL)
		Yaw = -MAX_GIMBAL;

	
	if (AttRollMode == ATT_R_POS)
		Roll = 1;
	else if (AttRollMode == ATT_R_NEG)
		Roll = -1;
	else
	{
		Roll = 0;
	}

	double sP = sin(Pitch*RAD);
    double sY = sin(Yaw*RAD);
    double cP = cos(Pitch*RAD);
    double cY = cos(Yaw*RAD);
      
    VECTOR3 VEngine = _V(sY * cP, sP * cY, cP * cY);

    SetThrusterDir(MainTH[0], VEngine);

	// roll
	SetThrusterGroupLevel(RightRoll, -Roll);
	SetThrusterGroupLevel(LeftRoll, Roll);
        
	if (!vis) return;
	const VECTOR3 CE_AXIS01 = {-1, 0, 0};
	MESHGROUP_TRANSFORM	mt01;
	mt01.transform = MESHGROUP_TRANSFORM::ROTATE;
	mt01.nmesh = MeshID;
	mt01.ngrp = 2;
	mt01.P.rotparam.angle = float((Pitch-Pitch0)*RAD);
	mt01.P.rotparam.ref = _V(0, 0, GIMBAL_Z);
	mt01.P.rotparam.axis = CE_AXIS01;
	MeshgroupTransform(vis, mt01);

	if (!vis) return;
	const VECTOR3 CE_AXIS02 = {0, -1, 0};
	MESHGROUP_TRANSFORM	mt02;
	mt02.transform = MESHGROUP_TRANSFORM::ROTATE;
	mt02.nmesh = MeshID;
	mt02.ngrp = 2;
	mt02.P.rotparam.angle = (float)((Yaw-Yaw0)*RAD);
	mt02.P.rotparam.ref = _V(0, 0, GIMBAL_Z);
	mt02.P.rotparam.axis = CE_AXIS02;
	MeshgroupTransform(vis, mt02);
}



void Quad::Launch()
{
	bInFlight = true;
	met = 0;
	oapiGetHeading(GetHandle(), &Heading0);
	Heading0 *= DEG;

	// initialize intertial
	InitStartLocalRot();
	GetGlobalOrientation(GlobalAtt0);
	LocalAtt0 = GetAttitudeDeltas();
	GetRotationMatrix(R0);

	VESSELSTATUS VS0;
	GetStatus(VS0);
	rvel0 = VS0.rvel;
	rpos0 = VS0.rpos;

	GetEquPos(Lon0, Lat0, Rad0);
}


VECTOR3 Quad::GetAttitudeDeltas()
{

	MATRIX3 R;
	GetRotationMatrix(R);

	double	dB, dP, dY;
	dP = atan2(R.m23, R.m33);   //alfa, x
	dY = -asin(R.m13);			//beta, y
	dB = atan2(R.m12, R.m13);	//gamma, z

	return _V(dB, dP, dY);
} 



void Quad::LocalRot(const VECTOR3 vecGlob, VECTOR3 &vecLoc)
{
	VESSELSTATUS2 vs;
	double dCosZ, dSinZ, dCosY, dSinY, dCosX, dSinX;
	vs.version = 2;
	vs.flag = 0;
	GetStatusEx((void*) &vs);

	dCosZ = cos(vs.arot.z);
	dSinZ = sin(vs.arot.z);
	dCosY = cos(vs.arot.y);
	dSinY = sin(vs.arot.y);
	dCosX = cos(vs.arot.x);
	dSinX = sin(vs.arot.x);

	vecLoc.x = vecGlob.x*dCosZ*dCosY + vecGlob.y*(-dSinZ*dCosX+dCosZ*dSinY*dSinX) + vecGlob.z*(dSinZ*dSinX+dCosZ*dSinY*dCosX);
	vecLoc.y = vecGlob.x*dSinZ*dCosY + vecGlob.y*(dCosZ*dCosX+dSinZ*dSinY*dSinX) + vecGlob.z*(-dCosZ*dSinX+dSinZ*dSinY*dCosX);
	vecLoc.z = vecGlob.x*-dSinY + vecGlob.y*dCosY*dSinX + vecGlob.z*dCosY*dCosX;
}


void Quad::InitStartLocalRot()
{
	VESSELSTATUS2 vs;
	vs.version = 2;
	vs.flag = 0;
	GetStatusEx((void*) &vs);

	dCosZs = cos(vs.arot.z);
	dSinZs = sin(vs.arot.z);
	dCosYs = cos(vs.arot.y);
	dSinYs = sin(vs.arot.y);
	dCosXs = cos(vs.arot.x);
	dSinXs = sin(vs.arot.x);
}


void Quad::StartLocalRot(const VECTOR3 vecGlob, VECTOR3 &vecLoc)
{
	vecLoc.x = vecGlob.x*dCosZs*dCosYs + vecGlob.y*(-dSinZs*dCosXs+dCosZs*dSinYs*dSinXs) + vecGlob.z*(dSinZs*dSinXs+dCosZs*dSinYs*dCosXs);
	vecLoc.y = vecGlob.x*dSinZs*dCosYs + vecGlob.y*(dCosZs*dCosXs+dSinZs*dSinYs*dSinXs) + vecGlob.z*(-dCosZs*dSinXs+dSinZs*dSinYs*dCosXs);
	vecLoc.z = vecGlob.x*-dSinYs + vecGlob.y*dCosYs*dSinXs + vecGlob.z*dCosYs*dCosXs;
}


void Quad::clbkVisualCreated(VISHANDLE visual, int refcount)
{
	if (refcount > 1) 
		return;

	vis = visual;

	if (CameraDir == CAMERA_UP)
	{
		SetCameraOffset(_V(1, 1, 0.4));
		SetCameraDefaultDirection(_V(0,0,-1));
		CameraDir = CAMERA_DOWN;
	}
}


void Quad::clbkVisualDestroyed(VISHANDLE visual, int refcount)
{
	if (visual == vis) 
		vis = NULL;
}


void Quad::DebugSomething()
{
//	sprintf(oapiDebugString(), "MET: %f   Vforw: %f   Vsidew: %f", met, Vforw, Vsidew);
}



void Quad::OutputString()
{
	sprintf(oapiDebugString(), "MET: %.0f   Fuel: %.1f   Alt: %.1fm   Vhor: %.1fms   Vvert: %.1fms", 
		met, 
		GetPropellantMass(MainTank)/FUEL_CAPACITY*100,
		GetAltitude(),
		Vh,
		Vv
		);
}


DLLCLBK VESSEL *ovcInit (OBJHANDLE hVessel, int flightmodel)
{
	srand(time(NULL));
	return new Quad (hVessel,flightmodel);
}


DLLCLBK void ovcExit (VESSEL *vessel)
{
	if (vessel) delete (Quad*)vessel;
}

